/********************************************************************/
/*          FILE: Ilp.C                                             */
/********************************************************************/

#include "Ilp.h"

/********************************************************************/
//Convert this Ilp to the standard form. Works only if there are no
//integer variables. An LP in the standard form can be written as:
//
//  MINIMIZE C.X subject to
//  A_i.X >= b_i, for each i in M
//  A_i.X == b_i, for each i in M'
//  -INF <= x_j <= INF, for each x_j in N
//  0 <= x_j <= INF, for each x_j in N'
//
//  where A_i is the ith row of the constraint matrix,
//  b_i is the RHS of the ith constraint,
//  M is the set of greater-than-or-equal-to constraints,
//  M' is the set of all constraints not in M,
//  N is the set of unconstrained variables,
//  N' is the set of variables not in N
void Ilp::convert_to_standard_form()
{
  register int i, j;

  FRITS_SET_MESSAGE("convert_to_standard_form");

  //if the objective function is MAXIMIZE, make it MINIMIZE and
  //multiply all objective_coeffs by -1
  if(probtype == MAXIMIZE) {
    for(i = 0; i < get_num_variables(); i++) {
      set_objective_coeff( i, (-1*get_objective_coeff(i)) );
    }
    probtype = MINIMIZE;
  }

  //process the constraints
  for(i = 0; i < get_num_constraints(); i++) {
    switch(get_constraint_type(i)) {
#ifdef PHYSICAL	    
    case EQU:
#else
    case EQ:
#endif	    
      break;
    case GEQ:
      break;
    case LEQ:
      //convert this constraint to GEQ and multiply all constraint_coeffs and
      //the rhs for this constraint by -1
      for(j = 0; j < get_num_variables(); j++) {
	set_constraint_coeff( i, j, (-1*get_constraint_coeff(i,j)) );
      }
      set_constraint_rhs( i, (-1*get_constraint_rhs(i)) );
      set_constraint_type(i, GEQ);
      break;
    default:
      assert_force(0);
      break;
    }
  }

  //process the variable bounds
  for(i = 0; i < get_num_variables(); i++) {
    if( (get_lower_bound(i) == -ILP_INFBOUND && get_upper_bound(i) == ILP_INFBOUND) ||
	(get_lower_bound(i) == 0 && get_upper_bound(i) == ILP_INFBOUND) ) {
      //No need to do anything
    } else if(get_lower_bound(i) == -ILP_INFBOUND && get_upper_bound(i) == 0) {
      for(j = 0; j < get_num_constraints(); j++) {
	set_constraint_coeff( j, i, (-1*get_constraint_coeff(j,i)) );
      }
      set_objective_coeff( i, (-1*get_objective_coeff(i)) );
      set_lower_bound(i, 0);
      set_upper_bound(i, ILP_INFBOUND);
    } else {
      assert_force(0);
      //for now, we cannot handle the case of variables with finite bounds
    }
  }

  return;
}
/********************************************************************/
//This routine creates a new Ilp object representing the dual of this
//Ilp and returns it. An Ilp MUST be in the standard form before its dual
//can be computed.
Ilp *Ilp::get_dual()
{
  register int i, j;
  Ilp *newilp;

  FRITS_SET_MESSAGE("get_dual");

  //assert that this Ilp is an LP and is in the standard form
  assert(probtype == MINIMIZE);
  for(i = 0; i < num_variables; i++) {
    assert(variable_types[i] == CONT);
    assert( (lower_bounds[i] == 0 || lower_bounds[i] == -ILP_INFBOUND) &&
	   upper_bounds[i] == ILP_INFBOUND );
  }
  for(i = 0; i < num_constraints; i++) {
#ifdef PHYSICAL	  
    assert(constraint_types[i] == GEQ || constraint_types[i] == EQU);
#else
    assert(constraint_types[i] == GEQ || constraint_types[i] == EQ);
#endif
  }

  //num of variables in dual = num of constraints in primal
  //num of constraints in dual = num of variables in primal
  newilp = new Ilp(get_num_constraints(), get_num_variables());

  //create the objective function for the dual
  newilp->probtype = MAXIMIZE;
  for(i = 0; i < newilp->get_num_variables(); i++) {
    newilp->set_objective_coeff(i, get_constraint_rhs(i));
  }

  //Create the constraint coeff matrix for the dual
  //This is just the transpose of the constraint matrix for the primal
  for(i = 0; i < newilp->get_num_constraints(); i++) {
    for(j = 0; j < newilp->get_num_variables(); j++) {
      newilp->set_constraint_coeff(i, j, get_constraint_coeff(j, i));
    }
  }

  //Set the constraint types for the dual
  //Unconstrained primal variables <==> EQ in dual
  //Nonnegative primal variables <==> LEQ in dual
  for(i = 0; i < newilp->get_num_constraints(); i++) {
    if(get_upper_bound(i) == ILP_INFBOUND && get_lower_bound(i) == -ILP_INFBOUND) {
#ifdef PHYSICAL	    
      newilp->set_constraint_type(i, EQU);
#else
      newilp->set_constraint_type(i, EQ);
#endif      
 	      
    } else if(get_upper_bound(i) == ILP_INFBOUND && get_lower_bound(i) == 0) {
      newilp->set_constraint_type(i, LEQ);
    } else {
      assert_force(0);
    }
  }

  //RHS values of the dual are equal to the objective coeffs in the primal
  for(i = 0; i < newilp->get_num_constraints(); i++) {
    newilp->set_constraint_rhs(i, get_objective_coeff(i));
  }

  //Set the variable bounds for the dual
  //EQ in primal <==> unconstrained variables in dual
  //GEQ in primal <==> nonnegative variables in dual
  for(i = 0; i < newilp->get_num_variables(); i++) {
    switch(get_constraint_type(i)) {
#ifdef PHYSICAL	    
    case EQU:
#else
    case EQ:
#endif	    
      newilp->set_upper_bound(i, ILP_INFBOUND);
      newilp->set_lower_bound(i, -ILP_INFBOUND);
      break;
    case GEQ:
      newilp->set_upper_bound(i, ILP_INFBOUND);
      newilp->set_lower_bound(i, 0);      
      break;
    default:
      assert_force(0);
      break;
    }
  }

  return(newilp);
}
/********************************************************************/
// Solve the ILP by calling the appropriate ILP solver
// For now, only CPLEX is supported. Returns T if the
// problem is feasible and an optimal solution is found,
// F otherwise
Boolean Ilp::solve(Array<double> &solutions, double &objective_value)
{
  Boolean retval = F;

#ifdef _USE_CPLEX_
  retval = solve_using_cplex(solutions, objective_value);
#endif

  return(retval);
}
/********************************************************************/
#ifdef _USE_CPLEX_
//This routine solves the ILP using the CPLEX(TM) Linear Optimizer
//callable library routines (Version 3.0)
//If an optimal solution is found, the optimal objective function value
//and values for the variables are written in to the provided arguments
//and the message returns T. Else, the routine returns F and the arguments
//are untouched
Boolean Ilp::solve_using_cplex(Array<double> &solutions, double &objective_value)
{
  register int i, j, k;
  Boolean int_vars_present, optimal_soln_found = F;
  double tmpval;
  int tmpcount;

  //Variables used by the cplex package
  CPXLPptr lp;
  char *probname = "My_ilp_problem", *senx, *ctype;
  int objective, matsize, *matbeg, *matcnt, *matind, status, lpstat;
  double *objx, *rhsx, *matval, *bdl, *bdu, *lpx, lpobj;
  int pnodes, parcs, pitcnt;

  objx = new double[get_num_variables()];
  for(i = 0; i < get_num_variables(); i++) {
    objx[i] = get_objective_coeff(i);
  }

  rhsx = new double[get_num_constraints()];
  senx = new char[get_num_constraints()];
  for(i = 0; i < get_num_constraints(); i++) {
    rhsx[i] = get_constraint_rhs(i);
    switch(get_constraint_type(i)) {
#ifdef PHYSICAL	    
    case EQU:
#else
    case EQ:
#endif	    
      senx[i] = 'E';
      break;
    case GEQ:
      senx[i] = 'G';
      break;
    case LEQ:
      senx[i] = 'L';
      break;
    default:
      assert_force(0);
      break;
    }
  }

  bdl = new double[get_num_variables()];
  bdu = new double[get_num_variables()];
  for(i = 0; i < get_num_variables(); i++) {
    bdu[i] = get_upper_bound(i);
    bdl[i] = get_lower_bound(i);
  }

  matbeg = new int[get_num_variables()];
  matcnt = new int[get_num_variables()];

  matsize = 0;
  for(i = 0; i < get_num_variables(); i++) {
    for(j = 0; j < get_num_constraints(); j++) {
      tmpval = get_constraint_coeff(j, i);
      if(tmpval != 0.0) {
	matsize++;
      }
    }
  }

  matval = new double[matsize];
  matind = new int[matsize];

  k = 0;
  for(i = 0; i < get_num_variables(); i++) {
    tmpcount = 0;
    for(j = 0; j < get_num_constraints(); j++) {
      tmpval = get_constraint_coeff(j, i);
      if(tmpval != 0.0) {
	matval[k+tmpcount] = tmpval;
	matind[k+tmpcount] = j;
	tmpcount++;
      }
    }
    matbeg[i] = k;
    matcnt[i] = tmpcount;
    k += tmpcount;
  }

  assert(k == matsize);

  assert(probtype == MAXIMIZE || probtype == MINIMIZE);
  objective = ((probtype == MAXIMIZE)? -1 : 1);

  lpx = new double[get_num_variables()];

  int_vars_present = F;
  for(i = 0; i < get_num_variables(); i++) {
    if(get_variable_type(i) != CONT) {
      int_vars_present = T;
      break;
    }
  }
  if(int_vars_present) {
    ctype = new char[get_num_variables()];
    for(i = 0; i < get_num_variables(); i++) {
      switch(get_variable_type(i)) {
      case CONT:
	ctype[i] = 'C';
	break;
      case BIN:
	ctype[i] = 'B';
	break;
      case INT:
	ctype[i] = 'I';
	break;
      default:
	assert_force(0);
	break;
      }
    }

    lp = loadmprob(probname, get_num_variables(), get_num_constraints(), 0,
		  objective, objx, rhsx, senx, matbeg, matcnt,
		  matind, matval, bdl, bdu, (double *)NULL, (int *)NULL,
		  (int *)NULL, (int *)NULL, (int *)NULL, (int *)NULL,
		  (double *)NULL, (char *)NULL, (char *)NULL, (char *)NULL,
		  (char *)NULL, (char *)NULL, (char **)NULL, (char *)NULL,
		  (char **)NULL, (char *)NULL, (char **)NULL, (char *)NULL,
		  get_num_variables(), get_num_constraints(), matsize,
		  0, 0, (unsigned)0, (unsigned)0, (unsigned)0, ctype);
    if (!lp) {
      fprintf(stderr, "ERROR: CPLEX: loadmprob() returned NULL\n");
      exit(-1);
    }

    lpstat = mipoptimize(lp);
    if(lpstat) {
      cerr << "ERROR: CPLEX: mipoptimize() returned " << lpstat << endl;
      exit(-1);
    }

    lpstat = getstat(lp);
    if(lpstat == CPX_OPTIMAL) {
      lpstat = getmobjval(lp, &lpobj);
      if(lpstat) {
	cerr << "ERROR: CPLEX: getmobjval() returned " << lpstat << endl;
	exit(-1);
      }

      lpstat = getmx(lp, lpx, 0, get_num_variables()-1);
      if(lpstat) {
	cerr << "ERROR: CPLEX: getmx() returned " << lpstat << endl;
	exit(-1);
      }
      optimal_soln_found = T;
    }
  } else { //No integer variables, LP problem
    lp = loadprob(probname, get_num_variables(), get_num_constraints(), 0, 
		  objective, objx, rhsx, senx, matbeg, matcnt,
		  matind, matval, bdl, bdu, (double *)NULL, (int *)NULL,
		  (int *)NULL, (int *)NULL, (int *)NULL, (int *)NULL,
		  (double *)NULL, (char *)NULL, (char *)NULL, (char *)NULL,
		  (char *)NULL, (char *)NULL, (char **)NULL, (char *)NULL,
		  (char **)NULL, (char *)NULL, (char **)NULL, (char *)NULL,
		  get_num_variables(), get_num_constraints(), matsize,
		  0, 0, (unsigned)0, (unsigned)0, (unsigned)0);
    if (!lp) {
      fprintf(stderr, "ERROR: CPLEX: loadprob() returned NULL\n");
      assert(0);
    }

    //Try our luck with netopt. This will save time if the problem
    //of a significant sub-portion of it is a network-flow problem
    lpstat = netopt(lp, &lpstat, &pnodes, &parcs, &pitcnt);

    if (lpstat != -1) {
      //netopt() was not able to find an optimal solution, so we have to
      //call optimize()
      lpstat = optimize(lp);
      if(lpstat) {
	cerr << "ERROR: CPLEX: optimize() returns " << lpstat << endl;
	exit(-1);
      }
    }

    solution(lp, &lpstat, &lpobj, lpx, (double *)NULL, (double *)NULL, (double *)NULL);
    if(lpstat == CPX_OPTIMAL) {
      optimal_soln_found = T;
    }
  }

#ifdef ILP_DEBUG
  if (lpwrite(lp, "CPLEX.LP")) {
    fprintf(stderr, "ERROR: Could not write the lp to file %s\n", "CPLEX.LP");
    exit(-1);
  }
#endif
  
  if(optimal_soln_found) {
    objective_value = lpobj;
    solutions.resize(get_num_variables());
    for(i = 0; i < get_num_variables(); i++) {
      solutions[i] = lpx[i];
    }
  } 

  //Clean up all memory used by cplex
  freeprob(&lp);
  delete [] objx; objx = NULL;
  delete [] rhsx; rhsx = NULL;
  delete [] senx; senx = NULL;
  delete [] bdl; bdl = NULL;
  delete [] bdu; bdu = NULL;
  delete [] matbeg; matbeg = NULL;
  delete [] matcnt; matcnt = NULL;
  delete [] matval; matval = NULL;
  delete [] matind; matind = NULL;
  delete [] lpx; lpx = NULL;
  if(int_vars_present) {
    delete [] ctype; ctype = NULL;
  }

  return(optimal_soln_found);
}
#endif
/********************************************************************/
