/********************************************************************/
/*          FILE: Ilp.h                                             */
/********************************************************************/
#ifndef ANAND_ILP_H
#define ANAND_ILP_H

#include "matrix.h"
#ifdef _USE_CPLEX_
#include "/u/cplex/cpxdefs.inc"
#endif
/********************************************************************/
typedef enum ilp_variable_type {
  NOVARTYPE=0,
  CONT=1,
  BIN=2,
  INT=3
} Variable_type;
/********************************************************************/
typedef enum ilp_constraint_type {
  NOCONSTRTYPE=0,
  GEQ=1,
  LEQ=2,
#ifdef PHYSICAL  
  EQU=3
#else
  EQ=3
#endif  
} Constraint_type;
/********************************************************************/
typedef enum ilp_problem_type {
  NOPROBTYPE=0,
  MINIMIZE=1,
  MAXIMIZE=2
} Ilp_type;
/********************************************************************/
//Any value greater than or equal to this number is treated as infinity
//CPLEX uses the same value, but it may differ for other solvers
#define ILP_INFBOUND 1.0E+20
/********************************************************************/
#ifdef _SCALP_
class Ilp : public Assurance {
#else
class Ilp {
#endif
private:
  int num_variables;
  int num_constraints;
  Ilp_type probtype;                       //minimization or maximation
  Matrix<double> constraint_coeffs;        //variable coeffs in constraints
  Array<Constraint_type> constraint_types; //LHS-RHS relationship in constraints
  Array<double> constraint_rhs;            //RHS constants in the constraints
  Array<double> objective_coeffs;          //variable coeffs in objective fn.
  Array<Variable_type> variable_types;     //is a variable cont., binary, or integer?
  Array<double> upper_bounds;              //upper bounds on variable values
  Array<double> lower_bounds;              //lower bounds on variable values

  //Functions used by other member functions
#ifdef _USE_CPLEX_
  //Call CPLEX to solve the ILP
  Boolean solve_using_cplex(Array<double> &, double &);
#endif

public:

  Ilp(int nv = 0, int nc = 0) : constraint_coeffs(nc, nv), constraint_types(nc), constraint_rhs(nc), objective_coeffs(nv), variable_types(nv), upper_bounds(nv), lower_bounds(nv) {
    FRITS_SET_CLASS("Ilp");
    num_variables = nv;
    num_constraints = nc;
    probtype = NOPROBTYPE;
    constraint_coeffs.reset(0);
    constraint_types.reset(NOCONSTRTYPE);
    constraint_rhs.reset(0);
    objective_coeffs.reset(0);
    variable_types.reset(NOVARTYPE);
    upper_bounds.reset(ILP_INFBOUND);
    lower_bounds.reset(-ILP_INFBOUND);
  }

  ~Ilp() {}

  inline const int get_num_variables() {return num_variables;}
  inline const int get_num_constraints() {return num_constraints;}
  inline const double get_constraint_coeff(const int constrindex, const int varindex) {
    return constraint_coeffs[constrindex][varindex];
  }
  inline const Constraint_type get_constraint_type(const int constrindex) {
    return constraint_types[constrindex];
  }
  inline const double get_constraint_rhs(const int constrindex) {
    return constraint_rhs[constrindex];
  }
  inline const double get_objective_coeff(const int varindex) {
    return objective_coeffs[varindex];
  }
  inline const Variable_type get_variable_type(const int varindex) {
    return(variable_types[varindex]);
  }
  inline const double get_upper_bound(const int varindex) {
    return upper_bounds[varindex];
  }
  inline const double get_lower_bound(const int varindex) {
    return lower_bounds[varindex];
  }
  inline void set_constraint_coeff(const int constrindex, const int varindex, const double val) {
    assert(ABS(val) < ILP_INFBOUND);
    constraint_coeffs[constrindex][varindex] = val;
  }
  inline void set_objective_coeff(const int varindex, const float val) {
    assert(ABS(val) < ILP_INFBOUND);
    objective_coeffs[varindex] = val;
  }
  inline void set_constraint_rhs(const int constrindex, const double val) {
    constraint_rhs[constrindex] = val;
  }
  inline void set_variable_type(const int varindex, const Variable_type vtype) {
    variable_types[varindex] = vtype;
  }
  inline void set_constraint_type(const int constrindex, const Constraint_type ctype) {
#ifdef PHYSICAL		    
    assert(ctype == GEQ || ctype == LEQ ||ctype == EQU);
#else
    assert(ctype == GEQ || ctype == LEQ ||ctype == EQ);
#endif    
		    
    constraint_types[constrindex] = ctype;
  }
  inline void set_problem_type(const Ilp_type ptype) {
    assert(ptype == MINIMIZE || ptype == MAXIMIZE);
    probtype = ptype;
  }
  inline void set_upper_bound(const int varindex, const double newbound) {
    upper_bounds[varindex] = newbound;
  }
  inline void set_lower_bound(const int varindex, const double newbound) {
    lower_bounds[varindex] = newbound;
  }

  void resize(const int nv, const int nc) {
    num_variables = nv;
    num_constraints = nc;
    constraint_coeffs.resize(nc, nv);
    constraint_types.resize(nc);
    objective_coeffs.resize(nv);
    variable_types.resize(nv);
    upper_bounds.resize(nv);
    lower_bounds.resize(nv);
  };

  //Convert this problem to standard form
  void convert_to_standard_form();

  //Create the dual of this Ilp problem
  //Only works when there are no integer variables and the problem
  //is in standard form
  Ilp *get_dual();

  //Solve the Ilp. Returns T if problem feasible and
  //optimal solution, F otherwise
  Boolean solve(Array<double> &, double &);
};
/********************************************************************/
#endif
