/********************************************************************/
/*              FILE: Main.c                                        */
/********************************************************************/

#include "Main.h"
#include <unistd.h>
#include <fstream.h>
#ifdef PHYSICAL
#include <time.h>
#include "i_macro.h"
#endif
const char *Main::allowed_arguments = "Ad:e:f:Hl:p:Rs:u:v:t:";
/********************************************************************/
/*This routine prints out the command line usage details.
*/
void Main::print_usage(ostream &output) {
  output << endl;
  output << "Usage: scalp [-" << allowed_arguments << "]" << endl << endl;
  output << "Options:" << endl;
  output << "  -A              use AREA as the cost function" << endl;
  output << "  -d <filename>   read the DFG from the file named <filename>" << endl;
  output << "  -e <n>          perform at most <n> synthesis iterations" << endl;
  output << "  -f <int>        use a laxity factor of <int> over the fastest possible"<< endl;
  output << "                  non-pipelined implementation" << endl;
  output << "  -H              print out this help message" << endl;
  output << "  -l <filename>   read the library from the file named <filename>" << endl;
  output << "  -p <int>        produce a pipelined data path with <int> stages" << endl;
  output << "  -R              use Leiserson-Saxe retiming to satisfy" << endl;
  output << "                  sample period constraints" << endl;
  output << "  -u <int>        duplicate the CDFG <int> times" << endl;
  output << "  -s <int>        sample period constraint in nanoseconds" << endl;
  output << "  -v <filename>   use simulation vectors from the file named <filename>" << endl;
  output << "                  for extracting switched capacitance matrices" << endl;
  output << "  -t <0-5>          specify technology:"<<endl;
  output << "		       0-0.25um 1-0.18um, 2-0.15um, 3-0.13um, 4-0.10um, 5-0.07um" << endl;
  return;
}
/********************************************************************/
void Main::print_header(int argc, char **argv)
{
  char pathname[MAXPATHLEN], hostname[MAXHOSTNAMELEN];
  int i;
  time_t clock[1];

//  extern time_t time(time_t *);
  //extern char *getwd(char pathname[]);
#if 0 
  commented out by wwang on 9/24 
  if (gethostname(hostname, MAXHOSTNAMELEN)) {
    cerr << "ERROR: Cannot determine current machine name";
  }
#endif  
  cout << "*" << endl;
  cout << "* Command Line:      " << argv[0];
  for (i = 1; i < argc; i++) {
    cout << " " << argv[i];
  }
  cout << endl;

  time(clock);
#if 0
  //commented by wwang on 9/24
  //ctime returns a string that has a newline character in it, so,
  //no explicit endl is required.
  cout << "* Beginning Time:    " << ctime(clock);
  if(!getwd(pathname)) {
    cerr << "ERROR: Cannot determine current working directory" << endl;
  }
  cout << "* Working Directory: " << pathname << endl;
  cout << "* Machine Name:      " << hostname << endl;
//  cout << "* Make Mode:         " << MAKEMODE << endl;
  cout << "*" << endl;
#endif
  return;
}
/********************************************************************/
/*This routine processes the command line options
 */
void Main::process_arguments(int argc, char **argv) {
  int c;
  Boolean print_help;
  extern char *optarg;
  cout<<"sample period ="<<sample_constraint<<endl;

  while((c = getopt(argc, argv, allowed_arguments)) != EOF) {
    switch(c) {
    case 'A':
      obj = AREA;
      break;
    case 'd':
      dfg_filename = strdup(optarg);
      break;
    case 'e':
      max_synthesis_iterations = atoi(optarg);
      if(max_synthesis_iterations < 1) {
	cerr << "ERROR: bad value (" << max_synthesis_iterations << ") specified for maximum number of synthesis iterations" << endl;
	exit(-1);
      }
      break;
    case 'f':
      if(sample_constraint > 0.0) {
	cerr << "ERROR: cannot specify both SAMPLE CONSTRAINT and LAXITY FACTOR" << endl;
	print_usage(cerr);
	exit(-1);
      }

      laxity_factor = atof(optarg);
      if(laxity_factor < 1.0) {
	cerr << "ERROR: invalid value " << optarg << " specified for LAXITY FACTOR" << endl;
	print_usage(cerr);
	exit(-1);
      }
      break;
    case 'H':
      print_usage();
      exit(0);
      break;
    case 'l':
      lib_filename = strdup(optarg);
      break;
    case 'p':
      pipeline_stages = atoi(optarg);
      if(pipeline_stages < 1) {
 	cerr << "ERROR: Invalid value (" << pipeline_stages << ") specified for # of pipeline stages" << endl;
	print_usage(cerr);
	exit(-1);
      }
      break;
    case 'R':
      do_retiming = T;
      break;
    case 's':
      if(laxity_factor >= 1.0) {
 	cerr << "ERROR: Cannot specify both laxity factor and sample constraint" << endl;
	print_usage(cerr);
	exit(-1);
      }
      sample_constraint = atof(optarg);
      if(sample_constraint <= 0.0) {
	cerr << "ERROR: Invalid value (" << sample_constraint << ") specified for SAMPLE CONSTRAINT" << endl;
	print_usage(cerr);
	exit(-1);
      }
      break;
    case 'u':
      num_copies = atoi(optarg);
      if(num_copies < 1) {
	cerr << "ERROR: Invalid value (" << num_copies << ") specified for NUMBER of COPIES" << endl;
	print_usage(cerr);
	exit(-1);
      }
      break;
    case 'v':
      vec_filename = strdup(optarg);
      break;
#ifdef PHYSICAL
    case 't':
      tech = atoi(optarg);
      assert(tech<6);
      break;
#endif      
    case '?':
      cerr << "ERROR: Unknown command line option" << (char)c << endl;
      print_usage(cerr);
      exit(-1);
      break;
    default:
      assert_force(0);
      break;
    }
  }
  return;
}
/********************************************************************/
#ifdef _SCALP_
// Extract the library from a VHDL package
void Main::extract_library_from_package (Package *pack)
{
  Library_package *libpack;

  assert(pack);

  libpack = new Library_package(pack);

  lib = new library;  
  lib->set_library_package(libpack);

  return;
}
#endif
/********************************************************************/
#ifndef _SCALP_
/*This routine parses the input files to read the DFG and LIBRARY
 */
void Main::parse_files(void)
{
  assert(!dataflowgraph && !lib && !dp);

  //allocate the dfg and library
  dataflowgraph = new Dfg;
  lib = new library;

  if(!dfg_filename) {
    cerr << "ERROR: No filename specified for reading the DFG" << endl;
    print_usage(cerr);
    exit(-1);
  }
  dataflowgraph->read(dfg_filename);
  if(!lib_filename) {
    cerr << "ERROR: No filename specified for reading the LIBRARY" << endl;
    print_usage(cerr);
    exit(-1);
  }
  lib->read(lib_filename);

#ifdef PHYSICAL
  //Note the base is 0.18um tech
  wire_cap Wire; gate_cap Gate;
  double scale = Wire.get_unit_cap(tech)/Wire.get_unit_cap(1);
  cout<<"Wire scale "<<scale;
  lib->set_wire_scale_down(scale);
  scale = Gate.get_cap(tech)/Gate.get_cap(1);
  cout<<" Gate scale "<<scale;
  lib->set_gate_scale_down(scale);
  scale = Wire.get_lamda(tech)/Wire.get_lamda(1);
  lib->set_lamda_scale_down(scale);
  cout<<" Lamda scale "<<scale<<endl;
#endif 

  cout << "*  BEGIN LEVELIZING DFG..." << endl;
  dataflowgraph->levelize();
  cout << "*  END LEVELIZING DFG..." << endl;
  cout << "*" << endl;;

  return;
}
#endif
/********************************************************************/
/*This routine performs the DFG simulation to extract the switched
 *capacitance matrices.
 */
void Main::sc_simulation() {
  assert_force(dataflowgraph && lib);
  assert(!scms);

  scms = new Scm;

  cout << "*  BEGIN SC_SIMULATION..." << endl;

  scms->extract_scmatrices(*dataflowgraph, lib, vec_filename);
  cout << "*  END SC_SIMULATION..." << endl;
  cout << "*" << endl;

  return;
}
/********************************************************************/
/*This routine performs the actual low power data path synthesis
 */
void Main::scalp(void) {
  assert_force(dataflowgraph && lib && !dp);
  assert_force(get_obj() == POWER);
#ifdef PHYSICAL  
  sch.set_scm(scms);
#endif  
  
  cout<<"Sample constraint"<<sample_constraint<<endl;
  cout<<"laxity factor is "<<laxity_factor<<endl;
  cout << "* BEGIN LOW POWER SCHEDULING..." << endl;
  if(sample_constraint < 0.0001 && laxity_factor< 0.00001) {
    cerr << "WARNING: No SAMPLE PERIOD or LAXITY FACTOR constraint specified, assuming a SAMPLE PERIOD of "
    //cerr << "WARNING: No SAMPLE PERIOD or LAXITY FACTOR constraint specified, assuming a LAXITY_FACTOR of "
      << DEFAULT_SAMPLING_PERIOD <<"ns"<< endl;
    sample_constraint = DEFAULT_SAMPLING_PERIOD;
  }
  cout<<"Sample constraint "<<sample_constraint<<endl;
#ifdef PHYSICAL
  clock_t start_time = clock();
  time_t start, end;
  time(&start);
#endif  
  dp = sch.lpschedule(*dataflowgraph, lib, sample_constraint, laxity_factor, scms, do_retiming, vec_filename, max_synthesis_iterations);
#ifdef PHYSICAL
 time(&end);
  clock_t end_time = clock();
  double time_used = difftime(end, start);
#endif
 
  cout << "* END LOW POWER SCHEDULING..." << endl;

  cout << "Best solution at Vdd: " << dp->get_vdd() << " sample period: "
    << dp->get_sample_period()  << " control steps: " << dp->get_csteps();
  cout<<endl;
  char tmpfilename[MAXSTRLEN];

#ifdef PHYSICAL
  cout<<"Final after floorplanning: "<<endl;
  cout<<"Note: the VDD of the lib energy data is not known. Assumed 1 V"<<endl;
  dp->do_floorplan(lib,scms);
  float total = dp->compute_sccost(lib, scms);
  //total = total*dp->get_vdd()*dp->get_vdd();
  cout<<"Total Power = "<<total/dp->get_sample_period()<<endl;
  cout<<"Total Energy = "<<total<<endl;
  
  char tech_name[3];
  sprintf(tech_name,"_%d", tech);

  fstream fp_file;
  strcpy(tmpfilename,dataflowgraph->get_name());
  strcat(tmpfilename, tech_name);
  strcat(tmpfilename,"_fp");
  fp_file.open(tmpfilename, ios::out|ios::trunc);
  dp->display_floorplan(fp_file);
  dp->display_module_map(fp_file);
  dp->display_link_map(fp_file);
  fp_file.close();
//////////////////////////////////////////////////////////////////////////////
  fstream sched_file, alloc_file;
  strcpy(tmpfilename,dataflowgraph->get_name());
  strcat(tmpfilename, tech_name);
  strcat(tmpfilename,"_sched");
  sched_file.open(tmpfilename, ios::out|ios::trunc);
  dataflowgraph->display_schedule(sched_file);
  sched_file.close();

  strcpy(tmpfilename,dataflowgraph->get_name());
  strcat(tmpfilename, tech_name);
  strcat(tmpfilename,"_alloc");
  alloc_file.open(tmpfilename, ios::out|ios::trunc);
  dataflowgraph->display_allocation(alloc_file);
  alloc_file.close();
//////////////////////////////////////////////////////////////////////////////
  float real_module_sc = dp->compute_module_sc(lib,scms);
  float reduced_module_sc = dp->compute_reduced_module_sc(lib, scms);
  fstream result_data;
  strcpy(tmpfilename, dataflowgraph->get_name());
  strcat(tmpfilename, tech_name);
  strcat(tmpfilename,"_rslt");
  result_data.open(tmpfilename, ios::out|ios::trunc);
  result_data<<dataflowgraph->get_name()<<"  "<<real_module_sc<<
          "  "<<dp->get_wire_sc()<<"  "<<dp->get_mux_sc()<<"  ";
  cout<<"Wire energy is "<<dp->get_wire_sc()<<endl;
  cout<<"Mux energy is "<<dp->get_mux_sc()<<endl;
  cout<<"Logic energy is "<<real_module_sc<<endl;
  float reduced_sc = dp->compute_reduced_wire_sc(lib, scms);
  pair<float, float> tmp =dp->compute_shared_wire_sc(lib, scms);
  float shared_wire_sc =  tmp.first;
  float reduced_shared_wire_sc = tmp.second;
  float new_sc = dp->compute_wire_sc_new(lib, scms);
  cout<<"Reduced module energy is "<<reduced_module_sc<<endl;
  cout<<"Reduced wire energy is "<<reduced_sc<<endl;
  cout<<"Shared wire energy would be "<<shared_wire_sc<<" reduced is "<<reduced_shared_wire_sc<<endl;
  cout<<"MST wire energy would be "<<new_sc<<endl;
  dp->compute_reduced_mux_sc(lib, scms);
  cout<<"Reduced mux energy is "<<dp->get_mux_sc()<<endl;
  result_data <<reduced_module_sc<<" "<<reduced_sc<<" "
	  <<dp->get_mux_sc()<<" "<<dp->get_clock_tree_sc()<<" "
         <<dp->get_sample_period()<<" "<<dp->get_csteps()<<
	 " "<<dp->get_area_after_floorplan()<<" "<<dp->get_floorplan().area_efficiency()<<" "
	 <<shared_wire_sc<<" "<<reduced_shared_wire_sc<<" "<<new_sc<<" "
	 <<dp->get_control_area_overhead()<<" "<<dp->get_control_energy_overhead()<<endl;

  result_data.close();

  strcpy(tmpfilename, dataflowgraph->get_name());
  strcat(tmpfilename, tech_name);
  strcat(tmpfilename,"_len");
  result_data.open(tmpfilename, ios::out|ios::trunc);
  unsigned long module_num = dp->get_functional_units().get_size() 
	  + dp->get_storage_units().get_size();
  AssocVec<double> links( dp->get_link());
  dp->generate_module_link(lib, scms);
  for(unsigned long i = 0; i<module_num; i++) {
	for(unsigned long j = i + 1; j <module_num; j++) {
		if(dp->get_floorplan().tie(i,j)>0.0)
			result_data<<dp->get_floorplan().link_length(i, j)
				<<"   "<<(dp->get_link())(i,j) 	
				<<"   "<<links(i,j)<<endl;
	}
  }
  result_data.close();


  
#endif


  return;
}
/********************************************************************/
/*This routine performs the actual low power data path synthesis
 */
void Main::area_optimize(void) {
  assert_force(dataflowgraph && lib && !dp);
  assert_force(obj == AREA);
#ifdef PHYSICAL  
  sc_simulation();
  assert_force(scms);
  sch.set_scm(scms);
#endif  

  cout << "*  ATTEMPTING AREA OPTIMIZATION..." << endl;
  if(sample_constraint==0.0 && laxity_factor==0.0) {
    cerr << "WARNING: No SAMPLE PERIOD or LAXITY FACTOR constraint specified, assuming a SAMPLE PERIOD of "
      << DEFAULT_SAMPLING_PERIOD << "ns" << endl;
    sample_constraint = DEFAULT_SAMPLING_PERIOD;
  }
  dp = sch.areaschedule(*dataflowgraph, lib, sample_constraint, laxity_factor, max_synthesis_iterations);
  
  cout << "*  END AREA OPTIMIZATION" << endl;

  cout << "Best solution at Vdd: " << dp->get_vdd() << " sample period: "
    << dp->get_sample_period()  << " control steps: " << dp->get_csteps();
  cout<<endl;
  char tmpfilename[MAXSTRLEN];
#ifdef PHYSICAL
  cout<<"Final after floorplanning: "<<endl;
  cout<<"Note: the VDD of the lib energy data is not known. Assumed 1 V"<<endl;
  dp->do_floorplan(lib,scms);
  float total = dp->compute_sccost(lib, scms);
  //total = total*dp->get_vdd()*dp->get_vdd();
  cout<<"Total Power = "<<total/dp->get_sample_period()<<endl;
  cout<<"Total Energy = "<<total<<endl;

  char tech_name[3];
  sprintf(tech_name,"_%d", tech);

  fstream fp_file;
  strcpy(tmpfilename,dataflowgraph->get_name());
  strcat(tmpfilename, tech_name);
  strcat(tmpfilename,"_fp");
  fp_file.open(tmpfilename, ios::out|ios::trunc);
  dp->display_floorplan(fp_file);
  dp->display_module_map(fp_file);
  dp->display_link_map(fp_file);
  fp_file.close();
//////////////////////////////////////////////////////////////////////////////
  fstream sched_file, alloc_file;
  strcpy(tmpfilename,dataflowgraph->get_name());
  strcat(tmpfilename, tech_name);
  strcat(tmpfilename,"_sched");
  sched_file.open(tmpfilename, ios::out|ios::trunc);
  dataflowgraph->display_schedule(sched_file);
  sched_file.close();

  strcpy(tmpfilename,dataflowgraph->get_name());
  strcat(tmpfilename, tech_name);
  strcat(tmpfilename,"_alloc");
  alloc_file.open(tmpfilename, ios::out|ios::trunc);
  dataflowgraph->display_allocation(alloc_file);
  alloc_file.close();
///////////////	///////////////////////////////////////////////////////////////
  float reduced_module_sc = dp->compute_reduced_module_sc(lib, scms);
  float real_module_sc = dp->compute_module_sc(lib,scms);
  fstream result_data;
  strcpy(tmpfilename, dataflowgraph->get_name());
  strcat(tmpfilename, tech_name);
  strcat(tmpfilename,"_rslt");
  result_data.open(tmpfilename, ios::out|ios::trunc);
  result_data<<dataflowgraph->get_name()<<"  "<<real_module_sc<<
          "  "<<dp->get_wire_sc()<<"  "<<dp->get_mux_sc()<<"  ";
  cout<<"Wire energy is "<<dp->get_wire_sc()<<endl;
  cout<<"Mux energy is "<<dp->get_mux_sc()<<endl;
  cout<<"Logic energy is "<<real_module_sc<<endl;
  float reduced_sc = dp->compute_reduced_wire_sc(lib, scms);
  pair<float, float> tmp =dp->compute_shared_wire_sc(lib, scms);
  float shared_wire_sc =  tmp.first;
  float reduced_shared_wire_sc = tmp.second;
  float new_sc = dp->compute_wire_sc_new(lib, scms);
  cout<<"Reduced module sc is "<<reduced_module_sc<<endl;
  cout<<"Reduced wire energy is "<<reduced_sc<<endl;
  cout<<"Shared wire energy would be "<<shared_wire_sc<<"reduced is "<<reduced_shared_wire_sc<<endl;
  cout<<"MST wire energy would be "<<new_sc<<endl;
  dp->compute_reduced_mux_sc(lib, scms);
  cout<<"Reduced mux energy is "<<dp->get_mux_sc()<<endl;

  result_data <<reduced_module_sc<<" "<<reduced_sc<<" "
	  <<dp->get_mux_sc()<<" "<<dp->get_clock_tree_sc()<<
         " "<<dp->get_sample_period()<<" "<<dp->get_csteps()<<
         " "<<dp->get_area_after_floorplan()<<" "<<dp->get_floorplan().area_efficiency()
	 <<" "<<shared_wire_sc
	 <<" "<<reduced_shared_wire_sc<<" "<<new_sc<<" "
	 <<dp->get_control_area_overhead()<<" "<<dp->get_control_energy_overhead()<<endl;

  result_data.close();

  strcpy(tmpfilename, dataflowgraph->get_name());
  strcat(tmpfilename, tech_name);
  strcat(tmpfilename,"_len");
  result_data.open(tmpfilename, ios::out|ios::trunc);
  unsigned long module_num = dp->get_functional_units().get_size()
	  +dp->get_storage_units().get_size();
  AssocVec<double> links(dp->get_link());	
  dp->generate_module_link(lib, scms);
  for(unsigned long i = 0; i<module_num; i++) {
        for(unsigned long j = i + 1; j <module_num; j++) {
                if(dp->get_floorplan().tie(i,j)>0.0)
                        result_data<<dp->get_floorplan().link_length(i, j)
				<<"  "<<(dp->get_link())(i,j)
				<<"   "<<links(i,j)<<endl;
        }
  }
  result_data.close();
#endif
  return;
}
/********************************************************************/
#ifdef PHYSICAL
void Main::power_measurement(void) {
	dp->do_floorplan(lib, scms);
	delete scms;
	scms = new Scm;
	cout<<"* Begin measuring the synthesized circuit..."<<endl;
	scms->extract_scmatrices(*dataflowgraph, lib, vec_filename);
	dp->compute_module_sc(lib, scms);
	dp->compute_mux_sc(lib, scms);
	dp->compute_wire_sc(lib,scms);
	fstream result_data;
	char tmpfilename[MAXSTRLEN];
	char tech_name[3];
	sprintf(tech_name,"_%d", tech);
  	strcpy(tmpfilename, dataflowgraph->get_name());
  	strcat(tmpfilename, tech_name);
  	strcat(tmpfilename,"_real");
  	result_data.open(tmpfilename, ios::out|ios::trunc);
  	result_data<<dataflowgraph->get_name()<<"  "<<dp->get_module_sc()<<
          "  "<<dp->get_wire_sc()<<"  "<<dp->get_mux_sc()<<"  ";
  	cout<<" Real power measurement with random input"<<endl;
  	cout<<"Wire energy is "<<dp->get_wire_sc()<<endl;
  	cout<<"Mux energy is "<<dp->get_mux_sc()<<endl;
  	cout<<"Logic energy is "<<dp->get_module_sc()<<endl;
  	result_data <<dp->get_clock_tree_sc()<<"  " ;
  	dp->compute_reduced_wire_sc(lib, scms);
  	cout<<"Reduced wire energy is "<<dp->get_wire_sc()<<endl;
  	dp->compute_reduced_mux_sc(lib, scms);
  	cout<<"Reduced mux energy is "<<dp->get_mux_sc()<<endl;
  	result_data <<dp->get_wire_sc()<<"  "<<dp->get_mux_sc()<<"  "<<dp->get_clock_tree_sc()<<"  "
         <<dp->get_sc_after_floorplan()<<
         "  "<<dp->get_sample_period()<<"  "<<dp->get_csteps()<<
               "  "<<dp->get_area_after_floorplan()<<"  "<<0<<endl;

	
  	result_data.close();
}
#endif
