/********************************************************************/
/*             FILE: array.h                                        */
/********************************************************************/
#ifndef ANAND_ARRAY_H
#define ANAND_ARRAY_H

#include "basic.h"

/********************************************************************/
template<class type>
#ifdef _SCALP_
class Array : public Assurance {
#else
class Array {
#endif
private:
  int size;
  type *elements;

public:
  Array(int count = 0) {
    FRITS_SET_CLASS("Array<type>");
    assert(count >= 0);
    size = count;
    if (count) {
      elements = new type[size];
      assert(elements);
    } else {
      elements = NULL;
    }
  }

  ~Array () {
    if (elements) delete [] elements;
  }

  //reset all elements of an Array
  void reset(const type &initval) {
    register int i;
    for(i = 0; i < size; i++) {
      elements[i] = initval;
    }
  }

  inline const int get_size() {
    FRITS_SET_MESSAGE("get_size");
    return size;
  }

  //resize the Array to hold newsize elements
  void resize(int newsize) {
    register int i;
    register type *newelements;
    FRITS_SET_MESSAGE("resize");
    assert(size >= 0 && newsize >= 0);
    
    if (newsize) {
      newelements = new type[newsize];
      assert(newelements);
    } else {
      newelements = NULL;
    }
    // CAREFUL: If newsize < size, overflow is discarded
    for(i = 0; i < MIN(size, newsize); i++) {
      newelements[i] = elements[i];
    }
    if (elements) delete [] elements;
    elements = newelements;
    size = newsize;
    return;
  }

  //Resize the array, and RESET any extra elements with the given value
  void resize(int newsize, const type &initval) {
    register int i;
    register type *newelements;
    FRITS_SET_MESSAGE("resize");
    assert(size >= 0 && newsize >= 0);
    
    if (newsize) {
      newelements = new type[newsize];
      assert(newelements);
    } else {
      newelements = NULL;
    }
    // CAREFUL: If newsize < size, overflow is discarded
    for(i = 0; i < MIN(size, newsize); i++) {
      newelements[i] = elements[i];
    }
    //If any elements are left in the new array, reset them to initval
    for( ; i < newsize; i++) {
      newelements[i] = initval;
    }

    if (elements) delete [] elements;
    elements = newelements;
    size = newsize;
    return;
  }

  //accessing Array elements
  inline type &operator[](int index) { 
    assert(index >= 0 && index < size);
    return elements[index];
  }

  //copy one Array into another
  void copy(Array<type> &source) {
    register int i;

    FRITS_SET_MESSAGE("copy");
    assert(source.get_size() == size);
    for(i = 0; i < size; i++) {
      elements[i] = source.elements[i];
    }
  }

  //Copy one Array into another
  //This can only be done with Arrays of the same size
  //SLOW - USE WITH CAUTION
  inline Array<type> &operator = (Array<type> &array1) {
    register int i;
    if(this == &array1) {
      return *this;
    }
    assert(size == array1.size);
    for(i = 0; i < size; i++) {
      elements[i] = array1.elements[i];
    }
    return *this;
  }

  void display(ostream &output = cout) {
    int i;

    FRITS_SET_MESSAGE("display");
    assert(size >= 0);
    output << "Array: <";
    for(i = 0; i < size; i++) {
      output << " " << elements[i];
    }
    output << " >" << endl;
  }

};

//Syntactic sugar for displaying an Array
template<class Type>
inline ostream &operator <<(ostream &output, Array<Type> &ar)
{ar.display(output);return(output);}

/********************************************************************/
#endif
