/********************************************************************/
/*              FILE: datapath.C                                    */
/********************************************************************/

#include "datapath.h"

#define bitwidth 8

/********************************************************************/
void Datapath::display(ostream &output)
{
  output << "Datapath: " << this << endl;
  output << "  Functional units: " << functional_units << endl;
  output << "  Storage units: " << storage_units << endl;
  output << "  Interconnect units: " << interconnect_units << endl;
  output << "  PI nets: " << pi_nets << endl;
  output << "  PO nets: " << po_nets << endl;
  return;
}
/********************************************************************/
Datapath *Datapath::add_functional_unit(Functional_unit *new_fu)
{
  FRITS_SET_MESSAGE("add_functional_unit");
  assert(new_fu);
#ifndef NDEBUG
  if(functional_units.find(new_fu)) {
    cerr << "ERROR: Attempting to add Functional_unit " << new_fu <<
      " twice to Datapath " << this << endl;
    exit(-1);
  }
#endif
  functional_units.append(new_fu);
  return(this);
}
/********************************************************************/
Datapath *Datapath::delete_functional_unit(Functional_unit *old_fu)
{
  List_node<Functional_unit *> *lnode;
  int i;
  Net *net;

  FRITS_SET_MESSAGE("delete_functional_unit");
  assert(old_fu);
  assert(!strcmp(old_fu->get_id(), "Functional_unit"));

  if(lnode = functional_units.find(old_fu)) {
    //Disconnect each port of old_fu from the net that it is connected to
    int numnets = old_fu->get_number_of_nets();
    for(i = 1; i <= numnets; i++) {
      net = old_fu->get_net(i);
      if(net) {
	old_fu->disconnect_from_net(net, i);
      }
    }
    functional_units.remove_node(lnode);
  } else {
    cerr << "ERROR: Attempting to delete Functional_unit" << old_fu <<
      " that is not a part of Datapath " << this << endl;
    exit(-1);
  }
  return(this);
}
/********************************************************************/
Datapath *Datapath::add_storage_unit(Storage_unit *new_su)
{
  FRITS_SET_MESSAGE("add_storage_unit");
  assert(new_su);
#ifndef NDEBUG
  if(storage_units.find(new_su)) {
    cerr << "ERROR: Attempting to add Storage_unit " << new_su <<
      "twice to Datapath " << this << endl;
    exit(-1);
  }
#endif
  storage_units.append(new_su);
  return(this);
}
/********************************************************************/
Datapath *Datapath::delete_storage_unit(Storage_unit *old_su)
{
  List_node<Storage_unit *> *lnode;
  Net *net;

  FRITS_SET_MESSAGE("delete_storage_unit");
  assert(old_su);
  if(lnode = storage_units.find(old_su)) {
    net = old_su->get_net(REGIN);
    if(net) {
      old_su->disconnect_from_net(net, REGIN);
    }
    net = old_su->get_net(REGOUT);
    if(net) {
      old_su->disconnect_from_net(net, REGOUT);
    }
    storage_units.remove_node(lnode);
  } else {
    cerr << "ERROR: Attempting to delete storage_unit" << old_su <<
      " that is not a part of Datapath " << this << endl;
    exit(-1);
  }
  return(this);
}
/********************************************************************/
Datapath *Datapath::add_interconnect_unit(Interconnect_unit *new_iu)
{
  FRITS_SET_MESSAGE("add_interconnect_unit");
  assert(new_iu);
#ifndef NDEBUG
  if(interconnect_units.find(new_iu)) {
    cerr << "ERROR: Attempting to add interconnect_unit " << new_iu <<
      "twice to Datapath " << this << endl;
    exit(-1);
  }
#endif 
 interconnect_units.append(new_iu);
  return(this);
}
/********************************************************************/
Datapath *Datapath::delete_interconnect_unit(Interconnect_unit *old_iu)
{
  List_node<Interconnect_unit *> *lnode;
  int i;
  char portname[10];
  Net *net;

  FRITS_SET_MESSAGE("delete_interconnect_unit");
  assert(old_iu);
  if(lnode = interconnect_units.find(old_iu)) {

    //Disconnect each port of old_iu from the net that it is connected to
    //NOTE: Assumes that an interconnect_unit has only 1 output port
    int numnets = old_iu->get_number_of_nets();
    for(i = 1; i < numnets; i++) {
      sprintf(portname, "in%d", i);
      net = old_iu->get_net(portname);
      if(net) {
	old_iu->disconnect_from_net(net, portname);
      }
    }
    net = old_iu->get_net(MUXOUT);
    if(net) {
      old_iu->disconnect_from_net(net, MUXOUT);
    }

    interconnect_units.remove_node(lnode);
  } else {
    cerr << "ERROR: Attempting to delete Interconnect_unit" << old_iu <<
      " that is not a part of Datapath " << this << endl;
    exit(-1);
  }
  return(this);
}
/********************************************************************/
Datapath *Datapath::add_net(Net *new_net)
{
  FRITS_SET_MESSAGE("add_net");
  assert(new_net);
#ifndef NDEBUG
  if(nets.find(new_net)) {
    cerr << "ERROR: Attempting to add Net " << new_net <<
      " twice to Datapath " << this << endl;
    exit(-1);
  }
#endif

  nets.append(new_net);
  return(this);
}
/********************************************************************/
Datapath *Datapath::delete_net(Net *old_net)
{
  int i;
  //old_net->get_port_connections().display();
  //cout << old_net->get_port_connections().get_size() <<"\n";
  List_node<Net *> *lnode;
  PORTCONNPTR connection;
  List_iterator<PORTCONNPTR> portconscan(old_net->get_port_connections());
  Datapath_element *dpelem = NULL;
// added by wwang for debugging
  List_ar<Datapath_element *> conn_to_delete;
  List_ar<Boolean> name_or_num;
  List_ar<int> conn_num;
  List_ar<const char *> conn_name;
  FRITS_SET_MESSAGE("delete_net");
  assert(old_net);
  if(lnode = nets.find(old_net)) {

    //Disconnect old_net from all datapath elements that it is connected to
    //NOTE: The port_connections list in the Net, is NOT freed here - this
    //is done in the desctructor of net
    FOR_EACH_LISTNODE(old_net->get_port_connections(), portconscan) {
      connection = portconscan.get_item();
      assert(connection);
      //old_net->get_port_connections().display();
      //cout << old_net->get_port_connections().get_size() <<"\n";
      //cout << connection->get_port_name() <<"," << connection->get_datapath_element() <<"," << "\n";
      assert(connection->get_datapath_element() &&
        (connection->get_port_name() || (connection->get_port_number()>=0)));
      if ( connection->get_port_name() ) {
      //  connection->get_datapath_element()->disconnect_from_net(old_net, connection->get_port_name());
        conn_to_delete.append(connection->get_datapath_element());
        name_or_num.append(T);
        conn_name.append(connection->get_port_name());
        conn_num.append(-1);
      } else if ( connection->get_port_number()>=0 ) {
      //  connection->get_datapath_element()->disconnect_from_net(old_net, connection->get_port_number());
       conn_to_delete.append(connection->get_datapath_element());
       name_or_num.append(F);
       conn_name.append(NULL);
       conn_num.append(connection->get_port_number());
      }
    }


   assert(conn_to_delete.get_size() == name_or_num.get_size());
   for(i = 0; i < conn_to_delete.get_size(); i++) {
     if(name_or_num[i])
       conn_to_delete[i]->disconnect_from_net(old_net, conn_name[i]);
     else
       conn_to_delete[i]->disconnect_from_net(old_net, conn_num[i]);
  }
    old_net->disconnect_from_all_ports();

    nets.remove_node(lnode);
  } else {
    cerr << "ERROR: Attempting to delete Net " << old_net <<
      " that is not a part of Datapath " << this << endl;
    exit(-1);
  }
  return(this);
}
/********************************************************************/
Datapath *Datapath::mark_pi(Net *new_pi_net)
{
  FRITS_SET_MESSAGE("mark_pi");

  assert(new_pi_net);
#ifndef NDEBUG
  if(!nets.find(new_pi_net)) {
    cerr << "ERROR: Must add Net " << new_pi_net << " to Datapath " << this
      << " before attempting to make it a PI" << endl;
  }
  if(pi_nets.find(new_pi_net)) {
    cerr << "ERROR: Attempting to add PI Net " << new_pi_net <<
      " twice to Datapath " << this << endl;
    exit(-1);
  }
#endif

  pi_nets.append(new_pi_net);

  return(this);
}
/********************************************************************/
Datapath *Datapath::unmark_pi(Net *old_pi_net)
{
  List_node<Net *> *lnode;

  FRITS_SET_MESSAGE("unmark_pi");

#ifndef NDEBUG
  if(!nets.find(old_pi_net)) {
    cerr << "ERROR: Attempting to unmark PI Net " << old_pi_net <<
      " that was not added to Datapath " << this << endl;
  }
#endif

  if(lnode = pi_nets.find(old_pi_net)) {
    pi_nets.remove_node(lnode);
  } else {
    cerr << "ERROR: Attempting to unmark PI Net " << old_pi_net <<
      " that is not marked as a PI net in Datapath " << this << endl;
    exit(-1);
  }

  return(this);
}
/********************************************************************/
Datapath *Datapath::mark_po(Net *new_po_net)
{
  FRITS_SET_MESSAGE("mark_po");

  assert(new_po_net);
#ifndef NDEBUG
  if(!nets.find(new_po_net)) {
    cerr << "ERROR: Must add Net " << new_po_net << " to Datapath " << this
      << " before attempting to make it a PO" << endl;
  }
  if(po_nets.find(new_po_net)) {
    cerr << "ERROR: Attempting to add PO Net " << new_po_net <<
      " twice to Datapath " << this << endl;
    exit(-1);
  }
#endif

  po_nets.append(new_po_net);

  return(this);
}
/********************************************************************/
Datapath *Datapath::unmark_po(Net *old_po_net)
{
  List_node<Net *> *lnode;

  FRITS_SET_MESSAGE("unmark_po");

#ifndef NDEBUG
  if(!nets.find(old_po_net)) {
    cerr << "ERROR: Attempting to unmark PO Net " << old_po_net <<
      " that was not added to Datapath " << this << endl;
  }
#endif

  if(lnode = po_nets.find(old_po_net)) {
    po_nets.remove_node(lnode);
  } else {
    cerr << "ERROR: Attempting to unmark PO Net " << old_po_net <<
      " that is not marked as a PO net in Datapath " << this << endl;
    exit(-1);
  }

  return(this);
}
/********************************************************************/
void Datapath::copy(Datapath &source_dp)
{
  vdd = source_dp.vdd;
  sample_period = source_dp.sample_period;
  csteps = source_dp.csteps;
  functional_units.copy(source_dp.functional_units);
  storage_units.copy(source_dp.storage_units);
  interconnect_units.copy(source_dp.interconnect_units);
  nets.copy(source_dp.nets);
  pi_nets.copy(source_dp.pi_nets);
  po_nets.copy(source_dp.po_nets);
  return;
}
/********************************************************************/
void Datapath::clear()
{
  Functional_unit *fu;
  List_iterator<Functional_unit *> fuscan;
  Storage_unit *su;
  List_iterator<Storage_unit *> suscan;
  Mux *intcon;
  List_iterator<Interconnect_unit *> intconscan;
  Net *net;
  List_iterator<Net *>netscan;

  FOR_EACH_LISTNODE(functional_units, fuscan) {
    fu = fuscan.get_item();
    delete fu;
  }
  functional_units.clear();

  FOR_EACH_LISTNODE(storage_units, suscan) {
    su = suscan.get_item();
    delete su;
  }
  storage_units.clear();

  FOR_EACH_LISTNODE(interconnect_units, intconscan) {
    intcon = (Mux *) intconscan.get_item();
    delete intcon;
  }
  interconnect_units.clear();

  FOR_EACH_LISTNODE(nets, netscan) {
    net = netscan.get_item();
    delete net;
  }
  nets.clear();
  pi_nets.clear();
  po_nets.clear();
  vdd = sample_period = csteps = -1;
  return;
}
/********************************************************************/
/*Update the interconnect units that feed a functional unit
 *NOTE: Does not handle the case of chaining yet.
 */
void Datapath::generate_interconnect_network(Functional_unit *fu, INTCON_LIBELPTR mux_libelement)
{
  register int i, j;
  int num_input_ports;
  List_iterator<Net *> netscan;
  List_ar<Net *> inputlist;
  Net *net, *tmpnet;
  STORPTR su;
  Mux *intcon_unit;
  List_iterator<NODEPTR> nodescan;
  NODEPTR node;
  EDGEPTR edge;
  char portname[10];

  assert(fu);
  assert(functional_units.find(fu));

  num_input_ports = fu->get_libelement()->number_of_input_ports();
  for(i = 1; i <= num_input_ports; i++) {

    //If this input port is already connected to an Intconnect_unit,
    //just blow away all connections to input ports of the Interconnect_unit.
    //Else, create a new Interconnect_unit and connect it to the input port.
    net = fu->get_net(i);
    if(net) {
      assert(net->number_of_connections() == 2);
      intcon_unit = (Mux *)net->get_driver();
      assert(intcon_unit && !strcmp(intcon_unit->get_id(), "Mux"));
      //disconnect all input connections of intcon_unit
      //NOTE: Assumes that an intcon_unit has only 1 output port,
      //and rest are input ports
      int numnets = intcon_unit->get_number_of_nets();
      for(j = 1; j < numnets; j++) {
	sprintf(portname, "in%d", j);
	tmpnet = intcon_unit->get_net(portname);
	assert(tmpnet);
	su = (STORPTR) tmpnet->get_driver();
	assert(su && !strcmp(su->get_id(), "Register"));
	intcon_unit->disconnect_from_net(tmpnet, portname);
      }
    } else {
      mem_ok( net = new Net );
      fu->connect_to_net(net, i);
      add_net(net);
      mem_ok( intcon_unit = new Mux(mux_libelement) );
      intcon_unit->connect_to_net(net, MUXOUT);
      add_interconnect_unit(intcon_unit);
    }

    //Build a list of all storage units that must feed the functional unit fu
    inputlist.clear();
    FOR_EACH_LISTNODE(fu->get_operations(), nodescan) {
      node = nodescan.get_item();
      assert(node);
      edge = (node->get_input_edges())[i-1];
      assert(edge);
      su = edge->get_storage_unit();
      tmpnet = su->get_net(REGOUT);
      assert(tmpnet);
      if(!inputlist.find(tmpnet)) {
	inputlist.append(tmpnet);
      }
    }

    //Connect each net in inputlist to an input port of intcon_unit
    assert(intcon_unit->get_number_of_nets() == 1);
    assert(inputlist.get_size() > 0);
    FOR_EACH_LISTNODE(inputlist, netscan) {
      tmpnet = netscan.get_item();
      sprintf(portname, "in%d", intcon_unit->get_number_of_nets());
      intcon_unit->connect_to_net(tmpnet, portname);
    }
  }/*END FOR EACH INPUT PORT OF fu*/
  return;
}
/********************************************************************/
/*Update the interconnect unit in front of a
 *storage unit
 */
void Datapath::generate_interconnect_network(Storage_unit *su, INTCON_LIBELPTR mux_libelement)
{
  register int j;
  List_iterator<Net *> netscan;
  List_ar<Net *> inputlist;
  Net *net, *tmpnet;
  Mux *intcon_unit;
  List_iterator<EDGEPTR> edgescan;
  Datapath_element *dpelem;
  STORPTR source_reg;
  FUPTR source_fu;
  NODEPTR node;
  EDGEPTR edge, loopout_edge;
  char portname[10];

  assert(su);
  assert(storage_units.find(su)!=NULL);
  //NOTE: assumes that a Storage_unit has ONLY 1 input port numbered REGIN
  //and 1 output port numbered REGOUT
  
  //If this input port is already connected to an Intconnect_unit,
  //just blow away all connections to input ports of the Interconnect_unit.
  //Else, create a new Interconnect_unit and connect it to the input port.
  net = su->get_net(REGIN);
  if(net) {
    assert(net->number_of_connections() == 2);
    intcon_unit = (Mux *)net->get_driver();
    assert(intcon_unit && !strcmp(intcon_unit->get_id(), "Mux"));
    //disconnect all input connections of intcon_unit
    //NOTE: Assumes that an intcon_unit has only 1 output port,
    //and rest are input ports
    int numnets = intcon_unit->get_number_of_nets();
    for(j = 1; j < numnets; j++) {
      sprintf(portname, "in%d", j);
      tmpnet = intcon_unit->get_net(portname);
      assert(tmpnet);
      dpelem = tmpnet->get_driver();
      if(dpelem) {
	if(!strcmp(dpelem->get_id(), "Register")) {
	  source_reg = (STORPTR) dpelem;
	  intcon_unit->disconnect_from_net(tmpnet, portname);
	} else {
	  source_fu = (FUPTR) dpelem;
	  assert(!strcmp(source_fu->get_id(), "Functional_unit"));
	  intcon_unit->disconnect_from_net(tmpnet, portname);
	}
      } else {
	//this is a "hanging" net created for an input port - blow it away
	assert(pi_nets.find(tmpnet));

	intcon_unit->disconnect_from_net(tmpnet, portname);
	unmark_pi(tmpnet);
	delete_net(tmpnet);
	assert(tmpnet->number_of_connections() == 0);
	delete tmpnet;
      }
    }
  } else {
    mem_ok( net = new Net );
    su->connect_to_net(net, REGIN);
    add_net(net);

    mem_ok( intcon_unit = new Mux(mux_libelement) );
    intcon_unit->connect_to_net(net, MUXOUT);
    add_interconnect_unit(intcon_unit);
  }

  //Build a list of all nets that must feed the storage unit
  inputlist.clear();
  FOR_EACH_LISTNODE(((STORPTR)su)->get_variables(), edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    node = edge->input_node();
    if(node) {
      source_fu = node->get_functional_unit();
      assert(source_fu);
      assert(functional_units.find(source_fu)!=NULL);
      tmpnet = source_fu->get_net(OUT1);
      assert(tmpnet);
      if(!inputlist.find(tmpnet)) {
	inputlist.append(tmpnet);
      }
    } else {
      assert(edge->is_pi() || edge->is_loopin() || edge->is_constant());

      //connect a hanging net to an input port of the mux - this net
      //should be later connected to a primary input pin, or hardwired
      //to Vdd/GND in the case of a constant/loopin initial value
      mem_ok(tmpnet = new Net);
      add_net(tmpnet);
      mark_pi(tmpnet);
      assert(!inputlist.find(tmpnet));
      inputlist.append(tmpnet);

      //connect a register that stores a loopin to the register that stores the
      //corresponding loopout
      if(edge->is_loopin()) {
	loopout_edge = edge->get_loopout_link();
	assert(loopout_edge && loopout_edge->is_loopout());
	source_reg = loopout_edge->get_storage_unit();
	assert(source_reg);
	if(source_reg != su) {
	  tmpnet = source_reg->get_net(REGOUT);
	  assert(tmpnet);
	  if(!inputlist.find(tmpnet)) {
	    inputlist.append(tmpnet);
	  }
	}
      }//end of if edge->is_loopin()
    }//end of else
  }//end of FOR_EACH 

  //Connect each net in inputlist to an input port of intcon_unit
  FOR_EACH_LISTNODE(inputlist, netscan) {
    tmpnet = netscan.get_item();
    assert(tmpnet);
    sprintf(portname, "in%d", intcon_unit->get_number_of_nets());
    intcon_unit->connect_to_net(tmpnet, portname);
  }

  return;
}
/********************************************************************/
/*Given the two functional units, old_fu1 and old_fu2 that were merged
 *by a class B move, update the datapath to reflect the move.
 *The output of new_fu feeds the units that were previously fed by
 *fu1 and fu2.
 */
/*void Datapath::merge_functional_units(Functional_unit *old_fu1,
			  Functional_unit *old_fu2, Functional_unit *new_fu)
{
  register int i;
  List_iterator<Net *> netscan;
  List_iterator<Storage_unit *>regscan;
  List_ar<Storage_unit *> inputlist;
  Net *net, *outnet;
  STORPTR su;
  Mux *intcon_unit;
  List_iterator<NODEPTR> nodescan;
  NODEPTR node;
  EDGEPTR edge;
  char portname[10];

  assert(old_fu1 && old_fu2 && new_fu);

  delete_functional_unit(old_fu1);
  delete_functional_unit(old_fu2);
  add_functional_unit(new_fu);

  assert(old_fu1->get_number_of_nets() == 3 &&
	 old_fu2->get_number_of_nets() == 3);

  for(i = 1; i < old_fu1->get_libelement()->number_of_ports(); i++) {
    //Collect all storage units connected to input port i
    //of old_fu1 and old_fu2 into inputlist
    inputlist.clear();
    sprintf(portname, "in%d", i);

    net1 = old_fu1->get_net(portname);
    assert(net1);
    assert(net1->number_of_connections() == 2);
    assert(net1->is_connected_by_name(fu, portname));
    intcon_unit1 = (Mux *)net1->get_driver();
    assert(intcon_unit1 && !strcmp(intcon_unit1->get_id(), "Mux"));

    FOR_EACH_LISTNODE(fu->get_operations(), nodescan) {
      node = nodescan.get_item();
      assert(node);
      edge = (node->get_input_edges())[i];
      assert(edge);
      su = edge->get_storage_unit();
      if(!inputlist.find(su)) {
	inputlist.append(su);
      }
    }
    //If a storage unit in inputlist is not already connected to
    //an input port of intcon_unit, do so now
    FOR_EACH_LISTNODE(inputlist, regscan) {
      su = (STORPTR) regscan.get_item();
      assert(su);
      if(!intcon_unit->search_input_net(su)) {
	sprintf(portname, "in%d", intcon_unit->get_number_of_nets());
	outnet = su->get_net(REGOUT);
	assert(outnet);
	intcon_unit->connect_to_net(outnet, portname);
      }
    }
  }
  return;
}*/
/********************************************************************/
// Extract RTL VHDL code for the Datapath
// Plainly loops through lists.
#ifdef _SCALP_
void Datapath::extract_vhdl(Architecture* arch, int base_signal_type)
{
  FRITS_SET_MESSAGE("extract_vhdl");
  List_iterator<Functional_unit *> functional_unit_iterator;
  List_iterator<Storage_unit *> storage_unit_iterator;
  List_iterator<Interconnect_unit *> interconnect_unit_iterator;
  List_iterator<Net *> net_iterator;
  Concurrent_body* cb;
  Block_declaration* bd;
  extern void* manager;
  Manager* management = (Manager*)manager;
  int net_count = 0;
  Signal* s;
  Subtype_indication* si;
  Range* r;
  Type_mark *bit, *bit_vector;
  STring net_name;
  STring instance_name;
  Net* net;
  char index[7];

  switch (base_signal_type) {
    case _BIT_ :
	bit = management->search_package("standard")->search_type("bit");
  	bit_vector = management->search_package("standard")->
		search_type("bit_vector");
	break;
    case _STD_LOGIC_ :
	bit = management->search_package("std_logic_1164")->
		search_type("std_logic");
  	bit_vector = management->search_package("std_logic_1164")->
		search_type("std_logic_vector");
	break;
    default:
	error("Incorrect VHDL base type specified for VHDL extraction");
    }

  if ( !(cb = arch->get_concurrent_body()) ) {
    mem_ok(cb = new Concurrent_body);
    arch->set_concurrent_body(cb); 
  } 

  if ( !(bd = arch->get_block_declaration()) ) {
    mem_ok(bd = new Block_declaration);
    arch->set_block_declaration(bd); 
  } 

  // Add the "clk" port to the entity
  Port* p; p = arch->get_entity()->get_port();
  Interface_element* clk;
  if ( !p->search_interface_element("clk") ) {
    mem_ok(s=new Signal);
    s->set_identifier("clk");
    mem_ok(si = new Subtype_indication);
    si->set_type_mark(bit);
    s->set_subtype_indication(si);
    mem_ok(clk = new Interface_element);
    clk->set_signal(s);
    p->add_interface_element(clk);
  }
  

  FOR_EACH_LISTNODE(nets, net_iterator) {
    mem_ok(s = new Signal);
    if (!((net=net_iterator.get_item())->get_name())) {
      net_name.set("net");
      net_name.cat(itoa(net_count++, index, 10));
      net->set_name(net_name.get());
      s->set_identifier(net_name.get());
    }
    mem_ok(si=new Subtype_indication);
    if (net->get_bitwidth() > 1) {
      mem_ok(r = new Range(net->get_bitwidth()-1, 0, _DOWNTO_));
      si->set_range(r);
      si->set_type_mark(bit_vector);
    } else {
      si->set_type_mark(bit);
    }
    s->set_subtype_indication(si);
    bd->add_signal(s);
  }  /* FOR_EACH_LISTNODE */

  // Make sure all addresses of datapath elements are set.
  assert_force(addresses_set);

  FOR_EACH_LISTNODE(functional_units, functional_unit_iterator) {
    instance_name.set("FU_");
    instance_name.cat(itoa(functional_unit_iterator.get_item()->get_address(), index, 10));
    cb->add_component_instantiation(functional_unit_iterator.get_item()->
	create_component_instantiation(arch, instance_name.get()));
  }

  FOR_EACH_LISTNODE(storage_units, storage_unit_iterator) {
    instance_name.set("SU_");
    instance_name.cat(itoa(storage_unit_iterator.get_item()->get_address(), index, 10));
    cb->add_component_instantiation(storage_unit_iterator.get_item()->
	create_component_instantiation(arch, instance_name.get()));
  }

  FOR_EACH_LISTNODE(interconnect_units, interconnect_unit_iterator) {
    instance_name.set("IU_");
    instance_name.cat(itoa(interconnect_unit_iterator.get_item()->get_address(), index, 10));
    cb->add_component_instantiation(interconnect_unit_iterator.get_item()->
	create_component_instantiation(arch, instance_name.get()));
  }
  return;
}
#endif
/********************************************************************/
// This message loops through all lists of datapath_elements and
// sets their addresses in sequence. 
// This message can be called only ONCE, after the datapath is fixed.
Boolean Datapath::set_addresses_of_datapath_elements(void) 
{
  if (addresses_set) {
    return F;
  } else {
    addresses_set = T;
    List_iterator<Functional_unit *> functional_unit_iterator;
    List_iterator<Storage_unit *> storage_unit_iterator;
    List_iterator<Interconnect_unit *> interconnect_unit_iterator;
    int current_address = 1;
  
    FOR_EACH_LISTNODE(functional_units, functional_unit_iterator) {
      functional_unit_iterator.get_item()->set_address(current_address++);
    }

    FOR_EACH_LISTNODE(storage_units, storage_unit_iterator) {
      storage_unit_iterator.get_item()->set_address(current_address++);
    }

    FOR_EACH_LISTNODE(interconnect_units, interconnect_unit_iterator) {
      interconnect_unit_iterator.get_item()->set_address(current_address++);
    }
  }

  return(T);
}
/********************************************************************/
void Datapath::write_bdnet(char *dfgname, float laxity)
{
  //a list of pointers to nets seen
  List_iterator<Storage_unit *> regscan;
  List_iterator<Interconnect_unit *> muxscan;
  List_iterator<Net *> netscan;
  List_iterator<Functional_unit *> fuscan;
  Mux *intcon_unit;
  Register *reg;
  Net *tmpnet;
  FUPTR fu;
  FSM *controller;
  char controller_name[MAXSTRLEN];
  char netname[10];
  int i;

  //writing the inputs & outputs into a file
  char out_name[MAXSTRLEN];
  strcpy(out_name, dfgname);
#ifdef _DMR_
  sprintf(out_name, "%s%f%s", dfgname, laxity, "dmr");
#endif
#ifdef _GUARANTEE_
  sprintf(out_name, "%s%f%s", dfgname, laxity, "guar");
#endif
#ifdef _ALPS_
  sprintf(out_name, "%s%f%s", dfgname, laxity, "alps");
#endif

  char outfile_name[MAXSTRLEN];
  sprintf(outfile_name, "%s%s", out_name, ".bdnet");
  ofstream fout(outfile_name);
  fout << "Model \"" << out_name << "\";\n";
  fout << "\tTECHNOLOGY    scmos;\n";
  fout << "\tVIEWTYPE SYMBOLIC;\n";
  fout << "\tEDITSTYLE SYMBOLIC;\n";
  fout << endl;

  //declare the primary input nets
  fout << "INPUT" << endl;
  i = 0;
  int j = 0;
  FOR_EACH_LISTNODE(pi_nets, netscan)
    {
      tmpnet = netscan.get_item();
      assert(!tmpnet->get_driver());
      for (i = 0; i < bitwidth; i++) {
	//Need to use a net name - currently using pointer
	fout << "\t\"Net" << (int) tmpnet << "[" << i << "]\"\t:\t";
	fout << "\"Net" << (int) tmpnet << "[" << i << "]\"";
	if ((j == pi_nets.get_size()-1)&&(i == bitwidth -1))
	  fout << ";" << endl;
	else
	  fout << endl;
      }
      j++;
    }

  fout << endl;

  //declare the primary output nets
  i = 0;
  fout << "OUTPUT" << endl; 
  j  =0;
  FOR_EACH_LISTNODE (po_nets, netscan) { 
    tmpnet = netscan.get_item();
    for (i = 0; i < bitwidth; i++) {
      fout << "\t\"Net" << (int) tmpnet << "[" << i << "]\"\t:\t";
      fout << "\"Net" << (int) tmpnet << "[" << i << "]\"";
      if ((j == po_nets.get_size()-1)&&(i == bitwidth -1)) {
	fout << ";" << endl;
      } else {
	fout << endl;
      }
    }
    j++;
  }
  fout << endl;

  fout << "SUPPLY \"Vdd!\";" << endl;
  fout << "GROUND \"GND!\";" << endl;
  fout << "CLOCK clk;" << endl;

  //I am doing the path definitions here for now,
  //will have to moved to the library at a later date
  const char *ADD_SUB = "~ganesh/layouts/add_8";
  const char *MULT = "~ganesh/layouts/mult_8";
  const char *MUX = "~ganesh/layouts/mux_8";
  const char *REG =  "~ganesh/layouts/reg_8";
  const char *NOTEQ = "~ganesh/layouts/noteq_8";

  //Now, the connectivity info is printed out
  FOR_EACH_LISTNODE(functional_units, fuscan) {
    //Finding the instance path
    //We need to find the fu-name. Temporarily, I am using the operation 
    //to find the name as there is only one FU of each type. This will
    //have to be got from the library.
    int oper_type;
    List_iterator<NODEPTR> nodescan;
    fu = fuscan.get_item();
    assert(fu);

    oper_type = fu->get_operations()[0]->get_func();
    if (oper_type == MULT_) {
      fout << "INSTANCE \"" << MULT << "\":\"wolfe\"" << endl;
      //connect the component supply and ground to the global supply and ground
      fout << "\t\"Vdd!\"" << "\t:\t" << "\"Vdd!\""<< ";"  << endl;
      fout << "\t\"GND!\"" << "\t:\t" << "\"GND!\""<< ";"  << endl;

      //Connect the left input of the FU
      Net *temp_net = fu->get_net(IN1);
      //If the net is driven by a 1-input MUX, connect the net directly
      //to the ouput of the dpelement driving the MUX input
      Datapath_element *IN = temp_net->get_driver();
      //assert that IN is a Mux
      if (IN->get_number_of_nets() == 2)
	temp_net = ((Mux*)IN)->get_net("in1");
      for (i = 0; i < bitwidth; i++) {
	fout << "\t\"MUL1in1node_" << i << "\"\t:\t";
	fout << "\"Net" << (int)temp_net << "[" << i << "]\"" 
	     << ";"  << endl;
      }

      //connect the right input of the FU
      temp_net = fu->get_net(IN2);
      IN = temp_net->get_driver();
      if (IN->get_number_of_nets() == 2)
	temp_net = ((Mux*)IN)->get_net("in1");
      for (i = 0; i < bitwidth; i++) {
	fout << "\t\"MUL1in2node_" << i << "\"\t:\t";
	fout << "\"Net" << (int)temp_net << "[" << i << "]\"" 
	     << ";"  << endl;
      }

      //connect the output of the FU
      temp_net = fu->get_net(OUT1);
      for (i = 0; i < bitwidth; i++) {
	fout << "\t\"MUL1outnode_" << i << "\"\t:\t";
	fout << "\"Net" << (int)temp_net << "[" << i << "]\"" 
	     << ";"  << endl;
      }
    }

      if ((oper_type == PLUS_)||(oper_type == MINUS_))
	{
	  fout << "INSTANCE \"" << ADD_SUB << "\":\"wolfe\"" << endl;
	  fout << "\t\"Vdd!\"" << "\t:\t" << "\"Vdd!\""<< ";"  << endl;
	  fout << "\t\"GND!\"" << "\t:\t" << "\"GND!\""<< ";"  << endl;

	  Net *temp_net = fu->get_net(IN1);
	  Datapath_element *IN = temp_net->get_driver();
	  if (IN->get_number_of_nets() == 2)
	    temp_net = ((Mux*)IN)->get_net("in1");
	  for (i = 0; i < bitwidth; i++)
	    {
	      fout << "\t\"ADD1in1node_" << i << "\"\t:\t";
	      fout << "\"Net" << (int)temp_net << "[" << i << "]\"" 
		<< ";"  << endl;
	    }

	  temp_net = fu->get_net(IN2);
	  IN = temp_net->get_driver();
	  if (IN->get_number_of_nets() == 2)
	    temp_net = ((Mux*)IN)->get_net("in1");
	  for (i = 0; i < bitwidth; i++)
	    {
	      fout << "\t\"ADD1in2node_" << i << "\"\t:\t";
	      fout << "\"Net" << (int)temp_net << "[" << i << "]\"" 
		<< ";"  << endl;
	    }

	  temp_net = fu->get_net(OUT1);
	  for (i = 0; i < bitwidth; i++)
	    {
	      fout << "\t\"ADD1outnode_" << i << "\"\t:\t";
	      fout << "\"Net" << (int)temp_net << "[" << i << "]\"" 
		<< ";"  << endl;
	    }
	}

      if (oper_type == NOTEQ_)
	{
	  fout << "INSTANCE \"" << NOTEQ << "\":\"wolfe\"" << endl;
	  fout << "\t\"Vdd!\"" << "\t:\t" << "\"Vdd!\""<< ";"  << endl;
	  fout << "\t\"GND!\"" << "\t:\t" << "\"GND!\""<< ";"  << endl;

	  Net *temp_net = fuscan.get_item()->get_net(IN1);
	  Datapath_element *IN = temp_net->get_driver();
	  if (IN->get_number_of_nets() == 2)
	    temp_net = ((Mux*)IN)->get_net("in1");
	  for (i = 0; i < bitwidth; i++)
	    {
	      fout << "\t\"in1node_" << i << "\"\t:\t";
	      fout << "\"Net" << (int)temp_net << "[" << i << "]\"" 
		<< ";"  << endl;
	    }

	  temp_net = fuscan.get_item()->get_net(IN2);
	  IN = temp_net->get_driver();
	  if (IN->get_number_of_nets() == 2)
	    temp_net = ((Mux*)IN)->get_net("in1");
	   for (i = 0; i < bitwidth; i++)
	    {
	      fout << "\t\"in2node_" << i << "\"\t:\t";
	      fout << "\"Net" << (int)temp_net << "[" << i << "]\"" 
		<< ";"  << endl;
	    }

	  temp_net = fuscan.get_item()->get_net(OUT1);
	  fout << "\t\"out\"\t:\t";
	  fout << "\"Net" << (int)temp_net << "[" << i << "]\"" 
	    << ";"  << endl;
	}

    } //END FOR EACH FUNCTIONAL UNIT

    FOR_EACH_LISTNODE(storage_units, regscan) {
      reg = (Register *) regscan.get_item();
      assert(reg);
      //IGNORE CONSTANT REGISTERS, FOR NOW - NEED TO IMPROVE
      if (is_constant_register(reg)) {
	continue;
      }

      fout << "INSTANCE \"" << REG << "\":\"wolfe\"" << endl;
      fout << "\t\"Vdd!\"" << "\t:\t" << "\"Vdd!\"" << ";"  << endl;
      fout << "\t\"GND!\"" << "\t:\t" << "\"GND!\"" << ";"  << endl;
      fout << "\t\"CLOCK\"" << "\t:\t" << "\"clk\"" << ";" << endl;

      //Connect the input port
      Net *temp_net = reg->get_net(REGIN);
      Datapath_element *IN = temp_net->get_driver();
      //If the net is driven by a 1-input MUX, connect the net directly
      //to the ouput of the dpelement driving the MUX input
      if (IN->get_number_of_nets() == 2) {
	temp_net = ((Mux*)IN)->get_net("in1");
      }
      for (i = 0; i < bitwidth; i++) {
	fout << "\t\"dnode_" << i << "\"\t:\t";
	fout << "\"Net" << (int)temp_net << "[" << 
	  i << "]\""<< ";"  << endl;
      }

      //Connect the output port
      temp_net = reg->get_net(REGOUT);
      for (i = 0; i < bitwidth; i++) {
	fout << "\t\"qnode_" << i << "\"\t:\t";
	fout << "\"Net" << (int)temp_net << "[" << i << "]\""<< ";"  << endl;
      }

      //Connect the load enable port
      temp_net = reg->get_net(LE);
      fout << "\t\"le" <<  "\"\t:\t";
      fout << "\"Net" << (int)temp_net << "\""<< ";"  << endl;

    } //END FOR EACH STORAGE UNIT

    FOR_EACH_LISTNODE(interconnect_units, muxscan) {
      Mux *intcon_unit;
      intcon_unit = (Mux *) muxscan.get_item();
      assert(intcon_unit);
      int num_inputs = intcon_unit->get_number_of_nets()-1;
      if (num_inputs > 1)
	write_nx1_mux(intcon_unit, fout);
    }

  //Write out the controller instance
  sprintf(controller_name, "%s_control", dfgname);

  fout << "INSTANCE \"" << controller_name  << "\":\"wolfe\"" << endl;  
  fout << "\t\"Vdd!\"" << "\t:\t" << "\"Vdd!\""<< ";"  << endl;  
  fout << "\t\"GND!\"" << "\t:\t" << "\"GND!\""<< ";"  << endl;
  fout << "\t\"CLOCK\"" << "\t:\t" << "\"clk\""<< ";"  << endl;

  //ANAND NEED TO FIX
  //connect the controller input ports
  //for each controller input() {
  //get the net connected to the current input;
  //print out the connection;
  //}

  //connect the controller output ports  
  for(i = 0; i < controller->get_number_of_outputs(); i++) {
    char *tmpname = controller->get_output_name(i);
    tmpnet = controller->get_net(tmpname);

    fout << "\t" << tmpname << "\"\t:\t"; 
    fout << "\"Net" << (int)tmpnet << "\""<< ";" << endl; 
  }

//  int countreg = 1;  
//  int countmux = 1;  
//Write out the load enables
//   FOR_EACH_LISTNODE(storage_units, regscan)
//     {
//       reg = (Register *) regscan.get_item();  
//       if (is_constant_register(reg) == 0)
//         {
//           Net *temp_net = reg->get_net(2);  
//           fout << "\t\"le" << countreg << "node" << "\"\t:\t"; 
//           fout << "\"Net" << (int)temp_net << "\""<< ";" << endl; 
//           countreg += 1; 
//         }
//     }      

//   //Write out the selects
//   FOR_EACH_LISTNODE(interconnect_units, muxscan)
//     {
//       Mux *intcon_unit;  
//       intcon_unit = (Mux *) muxscan.get_item(); 
//       assert(intcon_unit); 
//       int num_inputs = intcon_unit->get_number_of_nets()-1; 
//       if (num_inputs > 1)
//         {
//           Net *temp_net;  
//           int num_select = 0; 
//           num_select = ceil( log2( num_inputs ) );

//           for (i=0; i < num_select; i += 1); 
//             {
//               sprintf(netname, "select%d", i+1);  
//               temp_net = multiplexer->get_net(netname); 
//               fout << "\t" << "mux" << countmux << netname << "\t:\t\"Net" << (int) temp_net << "\";\n"; 
//             }
//           countmux += 1;  
//         }
//     }  


  fout << "ENDMODEL;\n";
  fout.close();
}
/********************************************************************/
Boolean Datapath::is_input(Net *test_net)
{
  List_ar<PORTCONNPTR> temp_ports;
  List_iterator<PORTCONNPTR> portconnscan;
  PORTCONNPTR connection;

  FOR_EACH_LISTNODE(test_net->get_port_connections(), portconnscan)
    {
      connection = portconnscan.get_item();
      assert(connection);

      if (!strcmp(connection->get_datapath_element()->get_id(), "Register"))
	if (connection->get_port_number() == REGOUT)
	  {
	    return F;
	  }
      if (!strcmp(connection->get_datapath_element()->get_id(), "Mux"))
	if (!strcmp(connection->get_port_name(),"out1"))
	  {
	    return F;
	  }
      if (!strcmp(connection->get_datapath_element()->get_id(), "Functional_unit"))
	if (connection->get_port_number() == OUT1)
	  {
	    return F;
	  }
    }
  return T;
}
/********************************************************************/
Boolean Datapath::is_output(Register *test_reg)
{
  List_ar<EDGEPTR> variables;
  variables.copy(test_reg->get_variables());
  List_iterator<EDGEPTR> varscan;
  FOR_EACH_LISTNODE(variables, varscan)
    {
      if (varscan.get_item()->is_po())
	return T;
    }
  return F;
}
/********************************************************************/
//This routine writes a nx1 mux into the BDNET file,
//assuming that nx1 muxes are available as
//
void Datapath::write_nx1_mux(Mux *multiplexer, ofstream& fout)
{
  int i = 0;
  char netname[20];
  int num_in = multiplexer->get_number_of_inputs();

  fout << "INSTANCE \"~ganesh/layouts/mux" << num_in << "x1_8\"" << ":\"wolfe\"" << endl;
  fout << "\t\"Vdd!\"" << "\t:\t" << "\"Vdd!\";" << endl;
  fout << "\t\"GND!\"" << "\t:\t" << "\"GND!\";" << endl;

  //Connect the input nets
  Net **input_nets = new Net*[multiplexer->get_number_of_nets()-1];
  Net *temp_net;
  for(i = 0; i < multiplexer->get_number_of_nets()-1; i++) {
    sprintf(netname, "in%d", i+1); 
    temp_net = multiplexer->get_net(netname);
    for (int j = 0; j < bitwidth; j++)
      fout << "\t" << netname << "node_" << j << "\t:\t\"Net" << (int) temp_net 
	   << "[" << j << "]\";\n";
  }

  //Connect the output port
  temp_net = multiplexer->get_net(MUXOUT); 
  for (i = 0; i < bitwidth; i++) {
    fout << "\t" << "resultnode_" << i << "\t:\t" << "\"Net" 
      << (int)temp_net << "[" << i << "]\";\n";
  }

  //Determine number of selects
  int num_select = (int) ceil(LOG2((double)num_in));

  //Connect the select nets
  for(i = 0; i < num_select; i += 1) {
    sprintf(netname, "select%d", i+1); 
    temp_net = multiplexer->get_select_net(netname);
    assert(temp_net);
    fout << "\t" << netname <<  "\t:\t\"Net" << (int) temp_net << "\";\n"; 
  }

  return;
}
/********************************************************************/
// void Datapath::recursive_mux_create(int num_nets, Net **input_nets, 
// 				    int *output_net, ofstream& fout, int level)
// {
//   //Check for termination
//   if (num_nets <= 2)
//     {
//       //connect up the inputs & outputs, return.
//       write_2x1_mux(input_nets, output_net, fout, level);
//       return;
//     }
//   else
//     {
//       int num_muxes = (int) ceil((float)num_nets/(float)2); 
//       int *new_inputs = new int[num_muxes];
//       for (int i = 0; i < floor((float)num_nets/(float)2); i++)
// 	{
// 	  //Get the name of the output net created by the MUX (pointer to int)
// 	  new_inputs[i] =  write_2x1_mux(input_nets+2*i, NULL, fout, level);
// 	}
//       if (num_nets%2 == 1)
// 	{
// 	  new_inputs[num_muxes-1] = (int) input_nets[num_nets-1];
// 	}
//       level++;
//       recursive_mux_create(num_muxes, (Net **)new_inputs, output_net, fout, level);
//       //Note that the absence of the delete instruction is intentional
//       //guarantees uniqueness of net IDs
//     }
// }
// /********************************************************************/
// int Datapath:: write_2x1_mux(Net **input_nets, int *output_net, 
// 			     ofstream& fout, int level)
// {
//   int bitwidth = 16;
//   int out_label;
//   int i = 0;
//   int constant_in1, constant_in2;
//   constant_in1 = constant_in2 = 0;
//   int *unique_id;
//   if (output_net)
//     {
//       out_label = (int)output_net;
//     }
//   else
//     {
//       unique_id = new int[1];
//       out_label = (int)unique_id;
//     }
  
//   if (!level)
//     {
//       if (!is_input(input_nets[0]))
// 	if (!strcmp(input_nets[0]->get_driver()->get_id(), "Register"))
// 	  {
// 	    if (is_constant_register((Register*)((input_nets[0])->get_driver())) == 1)
// 	      constant_in1 = 1;
// 	  }
//       if (!is_input(input_nets[1]))
// 	if (!strcmp(input_nets[1]->get_driver()->get_id(), "Register"))
// 	  {
// 	    if (is_constant_register((Register*)(input_nets[1])->get_driver()) == 1)
// 	      constant_in2 = 1;
// 	  }
//     }
//   //Actually do the write into the file
//   const char *MUX = "~ganesh/layouts/mux32";
//   fout << "INSTANCE \"" << MUX << "\":\"wolfe\"" << endl;
//   fout << "\t\"Vdd!\"" << "\t:\t" << "\"Vdd!\"" << ";" << endl;
//   fout << "\t\"GND!\"" << "\t:\t" << "\"GND!\"" << ";" << endl;
  
//   for (i = 0; i < bitwidth; i++)
//     {
//       fout << "\t\"aanode_" << i << "\"\t:\t";
//       if (!constant_in1)
// 	fout << "\"Net" << (int)input_nets[0] << "[" << i << "]\"" 
// 	  << ";"  << endl;
//       else
// 	fout << "\"Vdd!\";" << endl;
//     }
//   for (i = 0; i < bitwidth; i++)
//     {
//       fout << "\t\"bbnode_" << i << "\"\t:\t";
//       if (!constant_in2)
// 	fout << "\"Net" << (int)input_nets[1] << "[" << i << "]\"" 
// 	  << ";"  << endl;
//       else
// 	fout << "\"Vdd!\";" << endl;
//     }
//   for (i = 0; i < bitwidth; i++)
//     {
//       fout << "\t\"resultnode_" << i << "\"\t:\t";
//       fout << "\"Net" << out_label << "[" << i << "]\"" 
// 	<< ";"  << endl; 
//     }
//   fout << "\tselect\t:\tUNCONNECTED;" << endl;
//   return out_label;
// }*/
/********************************************************************/
int Datapath::is_constant_register(Register *reg)
{
  List_iterator<EDGEPTR> varscan;
  if (reg->get_variables().get_size() == 1)
    {
      FOR_EACH_LISTNODE(reg->get_variables(), varscan)
	{
	  if ((varscan.get_item())->is_constant())
	    return 1;
	}
    }
  return 0;
}
/********************************************************************/
void Datapath::write_composite_bdnet(char *dfgname, float laxity)
{
  //wites out a file where the each functional unit and register is 
  //merged with the mux that feeds it.


  //a list of pointers to nets seen
  List_iterator<Storage_unit *> regscan;
  List_iterator<Interconnect_unit *> muxscan;
  List_iterator<Net *> netscan;
  List_iterator<Functional_unit *> fuscan;
  Mux *intcon_unit;
  Register *reg;
  Net *tmpnet, *tempnet, *tempnet0, *tempnet1;
  char netname[10];
  int i;

  //Now, we find all the output nets in the datapath
/*  FOR_EACH_LISTNODE(storage_units, regscan)
    {
      //If the net that connects to the input port of the 
      //register connects to only other input ports, and
      //to no ouput port, then the input is a PI.
      //I am arbitrarily assuming that port 0 is the input
      //port of the register and port 1 is the output port.
      Net *temp_net = regscan.get_item()->get_net(REGIN);
      //get all ports connected to this net
      if (is_output((Register *)regscan.get_item()))
   	{
	  //Mark the terminal as an input
	  if (Output_Nets.find(temp_net) == NULL)
	    Output_Nets.append(temp_net);
	}
    }
*/

  //writing the inputs & outputs into a file
  char out_name[30];
  strcpy(out_name, dfgname);
  #ifdef _DMR_
  sprintf(out_name, "%s%f%s", dfgname, laxity, "dmr");
  #endif
  #ifdef _GUARANTEE_
  sprintf(out_name, "%s%f%s", dfgname, laxity, "guar");
  #endif
  #ifdef _ALPS_
  sprintf(out_name, "%s%f%s", dfgname, laxity, "alps");
  #endif
  char outfile_name[50];
  sprintf(outfile_name, "%s%s", out_name, ".bdnet");
  ofstream fout(outfile_name);
  fout << "Model \"" << out_name << "\";\n";
  fout << "\tTECHNOLOGY    scmos;\n";
  fout << "\tVIEWTYPE SYMBOLIC;\n";
  fout << "\tEDITSTYLE SYMBOLIC;\n";
  fout << endl;
  fout << "INPUT" << endl;
  i = 0;
  int j = 0;
  FOR_EACH_LISTNODE(pi_nets, netscan)
    {
      tmpnet = netscan.get_item();
      assert(!tmpnet->get_driver());
      for (i = 0; i < bitwidth; i++)
	{
	  fout << "\t\"Net" << (int) tmpnet << "[" << i << "]\"\t:\t";
	  fout << "\"Net" << (int) tmpnet << "[" << i << "]\"";
	  if ((j == pi_nets.get_size()-1)&&(i == bitwidth -1))
	    fout << ";" << endl;
	  else
	    fout << endl;
	}
      j++;
    }
  fout << endl;
  i = 0;
  fout << "OUTPUT" << endl; 
  j  =0;
  FOR_EACH_LISTNODE (po_nets, netscan)
    { 
      tmpnet = netscan.get_item();
      for (i = 0; i < bitwidth; i++)
	{
	  fout << "\t\"Net" << (int) tmpnet << "[" << i << "]\"\t:\t";
	  fout << "\"Net" << (int) tmpnet << "[" << i << "]\"";
	  if ((j == po_nets.get_size()-1)&&(i == bitwidth -1))
	    fout << ";" << endl;
	  else
	    fout << endl;
	}
      j++;
    }
  fout << endl;
  fout << "SUPPLY \"Vdd!\";" << endl;
  fout << "GROUND \"GND!\";" << endl;
  fout << "CLOCK clk;" << endl;
  //I am doing the path definitions here for now,
  //will have to moved to the library at a later date

  const char *ADD_SUB = "~ganesh/layouts/add";
  const char *MULT = "~ganesh/layouts/mult";
  const char *MUX = "~ganesh/layouts/mux_8";
  const char *REG =  "~ganesh/layouts/reg";
  const char *NOTEQ = "~ganesh/layouts/noteq";

  //Now, the connectivity info is printed out
  FOR_EACH_LISTNODE(functional_units, fuscan)
    {
      //Finding the instance path
      //We need to find the fu-name. Temporarily, I am using the operation 
      //to find the name as there is only one FU of each type. This will
      //have to be got from the library.
      int oper_type;
      List_iterator<NODEPTR> nodescan;
      FOR_EACH_LISTNODE(fuscan.get_item()->get_operations(), nodescan)
	{
	  oper_type = nodescan.get_item()->get_func();
	  break;
	}
      if (oper_type == MULT_)
	{ 
	  Net *tempnet0 = fuscan.get_item()->get_net(IN1);
	  Datapath_element *IN_0 = tempnet0->get_driver();
	  Net *tempnet1 = fuscan.get_item()->get_net(IN2); 
	  Datapath_element *IN_1 = tempnet1->get_driver();
	  
	  int in1 = IN_0->get_number_of_nets() -1;
	  int in2 = IN_1->get_number_of_nets() -1;

	  //write out the BDNET description of this new composite 
	  //unit
	  int  in3, in4;
	  get_element(in1, in2, in3, in4);
	  //in3 and in4 identify the functional unit that has 
	  //in3 and in4 inputs & 1 output
	  fout << "INSTANCE \"" << MULT << "l" << in3 << "r" << in4 << "\":\"wolfe\"" << endl;
	  fout << "\t\"Vdd!\"" << "\t:\t" << "\"Vdd!\""<< ";"  << endl;
	  fout << "\t\"GND!\"" << "\t:\t" << "\"GND!\""<< ";"  << endl;
	 
	  //In gives the name of the input mux
	  //we now use in3 and in4 to decide what to put down
	  char input[50];
	  for ( i = 1; i <= in3; i++)
	    {
	      sprintf(input, "%s%d", "in", i);
	      tempnet = ((Mux *)IN_0)->get_net(input);
	      for (int j = 0; j < bitwidth; j++)
		{
		  fout << "\t\"MUL1l" << i << "node_" << j << "\"\t:\t";
		  if (i <= in1)
		    fout << "\"Net" << (int)tempnet << "[" << j << "]\"" 
			 << ";"  << endl;
		  else
		    fout << "UNCONNECTED;" << endl;
		}
	    } 
	  for (i = 1; i <= in4; i++)
	    {
	      sprintf(input, "%s%d", "in", i);
	      tempnet = ((Mux *)IN_1)->get_net(input);
	      for (int j = 0; j < bitwidth; j++)
		{
		  fout << "\t\"MUL1r" << i << "node_" << j << "\"\t:\t";
		  if (i <= in2)
		    fout << "\"Net" << (int)tempnet << "[" << j << "]\"" 
			 << ";"  << endl;
		  else
		    fout << "UNCONNECTED;" << endl;
		}
	    } 
	  
	  tempnet = fuscan.get_item()->get_net(OUT1);
	  for (i = 0; i < bitwidth; i++)
	    {
	      fout << "\t\"MUL1outnode_" << i << "\"\t:\t";
	      fout << "\"Net" << (int)tempnet << "[" << i << "]\"" 
		   << ";"  << endl;
	    }
	}
      if ((oper_type == PLUS_)||(oper_type == MINUS_))
	{	  
	  Net *tempnet0 = fuscan.get_item()->get_net(IN1); 
	  Datapath_element *IN_0 = tempnet0->get_driver();
	  Net *tempnet1 = fuscan.get_item()->get_net(IN2); 
	  Datapath_element *IN_1 = tempnet1->get_driver();
	  
	  int in1 = IN_0->get_number_of_nets() -1;
	  int in2 = IN_1->get_number_of_nets() -1;

	  //write out the BDNET description of this new composite 
	  //unit
	
	  int  in3, in4;
	  get_element(in1, in2, in3, in4);
	  //in3 and in4 identify the functional unit that has 
	  //in3 and in4 inputs & 1 output
	  fout << "INSTANCE \"" << ADD_SUB << "l" << in3 << "r" << in4 << "\":\"wolfe\"" << endl;
	  fout << "\t\"Vdd!\"" << "\t:\t" << "\"Vdd!\""<< ";"  << endl;
	  fout << "\t\"GND!\"" << "\t:\t" << "\"GND!\""<< ";"  << endl;
	 	  
	  //In gives the name of the input mux
	  //we now use in3 and in4 to decide what to put down
	  char input[50];
	  for ( i = 1; i <= in3; i++)
	    {
	      sprintf(input, "%s%d", "in", i);
	      tempnet = ((Mux *)IN_0)->get_net(input);
	      for (int j = 0; j < bitwidth; j++)
		{
		  fout << "\t\"ADD1l" << i << "node_" << j << "\"\t:\t";
		  if (i <= in1)
		    fout << "\"Net" << (int)tempnet << "[" << j << "]\"" 
			 << ";"  << endl;
		  else
		    fout << "UNCONNECTED;" << endl;
		}
	    } 
	  for (i = 1; i <= in4; i++)
	    {
	      sprintf(input, "%s%d", "in", i);
	      tempnet = ((Mux *)IN_1)->get_net(input);
	      for (int j = 0; j < bitwidth; j++)
		{
		  fout << "\t\"ADD1r" << i << "node_" << j << "\"\t:\t";
		  if (i <= in2)
		    fout << "\"Net" << (int)tempnet << "[" << j << "]\"" 
			 << ";"  << endl;
		  else
		    fout << "UNCONNECTED;" << endl;
		}
	    } 
	  
	  tempnet = fuscan.get_item()->get_net(OUT1);
	  for (i = 0; i < bitwidth; i++)
	    {
	      fout << "\t\"ADD1outnode_" << i << "\"\t:\t";
	      fout << "\"Net" << (int)tempnet << "[" << i << "]\"" 
		   << ";"  << endl;
	    }
	}
      if (oper_type == NOTEQ_)
	{ 
		  
	  Net *tempnet0 = fuscan.get_item()->get_net(IN1); 
	  Datapath_element *IN_0 = tempnet0->get_driver();
	  Net *tempnet1 = fuscan.get_item()->get_net(IN2); 
	  Datapath_element *IN_1 = tempnet1->get_driver();
	  
	  int in1 = IN_0->get_number_of_nets() -1;
	  int in2 = IN_1->get_number_of_nets() -1;
	  //write out the BDNET description of this new composite 
	  //unit
	  int  in3, in4;
	  get_element(in1, in2, in3, in4);
	  //in3 and in4 identify the functional unit that has 
	  //in3 and in4 inputs & 1 output
	  fout << "INSTANCE \"" << NOTEQ << "l" << in3 << "r" << in4 << "\":\"wolfe\"" << endl;
	  fout << "\t\"Vdd!\"" << "\t:\t" << "\"Vdd!\""<< ";"  << endl;
	  fout << "\t\"GND!\"" << "\t:\t" << "\"GND!\""<< ";"  << endl;
	 
	  
	  //In gives the name of the input mux
	  //we now use in3 and in4 to decide what to put down
	  char input[50];
	  for (i = 1; i <= in3; i++)
	    {
	      sprintf(input, "%s%d", "in", i);
	      tempnet = ((Mux *)IN_0)->get_net(input);
	      for (int j = 0; j < bitwidth; j++)
		{
		  fout << "\t\"COMPl" << i << "node_" << j << "\"\t:\t";
		  if (i <= in1)
		    fout << "\"Net" << (int)tempnet << "[" << j << "]\"" 
			 << ";"  << endl;
		  else
		    fout << "UNCONNECTED;" << endl;
		}
	    } 
	  for (i = 1; i <= in4; i++)
	    {
	      sprintf(input, "%s%d", "in", i);
	      tempnet = ((Mux *)IN_1)->get_net(input);
	      for (int j = 0; j < bitwidth; j++)
		{
		  fout << "\t\"COMPr" << i << "node_" << j << "\"\t:\t";
		  if (i <= in2)
		    fout << "\"Net" << (int)tempnet << "[" << j << "]\"" 
			 << ";"  << endl;
		  else
		    fout << "UNCONNECTED;" << endl;
		}
	    } 
	}
    }
  FOR_EACH_LISTNODE(storage_units, regscan)
    {
      reg = (Register *) regscan.get_item();
      assert(reg);
      if (is_constant_register(reg) == 0)
	{ 
	  Net *tempnet = reg->get_net(REGIN);
	  Datapath_element *IN = tempnet->get_driver();
	  int num_inputs = IN->get_number_of_nets()-1; 
	  int match = get_element(num_inputs, num_inputs, num_inputs, num_inputs);
	  fout << "INSTANCE \"" << REG << match << "\":\"wolfe\"" << endl;
	  fout << "\t\"Vdd!\"" << "\t:\t" << "\"Vdd!\""<< ";"  << endl;
	  fout << "\t\"GND!\"" << "\t:\t" << "\"GND!\""<< ";"  << endl;
	  //clk and other control signals not yet initialized
	
	  //If the driver is a single input mux, then, connect to input of the mux
	
	 ;
	  char input[50];
	  for (i = 1; i <= match; i++)
	    {
	      sprintf(input, "%s%d", "in", i);
	      tempnet = ((Mux *)IN)->get_net(input);
	      for (int j = 0; j < bitwidth; j++)
		{
		  fout << "\t\"d" << i << "node_" << j << "\"\t:\t";
		  if (i <= num_inputs)
		    fout << "\"Net" << (int)tempnet << "[" << j << "]\"" 
			 << ";"  << endl;
		  else
		    fout << "UNCONNECTED;" << endl;
		}
	    } 
	  tempnet = reg->get_net(REGOUT);
	  for (i = 0; i < bitwidth; i++)
	    {
	      fout << "\t\"qnode_" << i << "\"\t:\t";
	      fout << "\"Net" << (int)tempnet << "[" << i << "]\""<< ";"  << endl;
	    }
	  fout << "\tle\t:\tUNCONNECTED;" << endl;
	}
    }
    fout << "ENDMODEL;\n";
    fout.close();
}
/************************************************************************/
int Datapath::get_element(int i1, int i2, int& i3, int& i4)
{
  int max = i2;
  if (i1 > i2)
    max = i1;
  float p = log(max)/log(2);
  i3 = i4 = (int)pow(2, ceil(p));
  return i3;
}
/************************************************************************/
