/********************************************************************/
/*              FILE: datapath.h                                    */
/* Contains a description of a Datapath_element class		    */
/********************************************************************/

#ifndef ANAND_DATAPATH_H
#define ANAND_DATAPATH_H

#include<assert.h>
#include "dpunit.h"
#include "fsm.h"
#include "net.h"
#ifdef _SCALP_
#include "architec.h"
#include "manager.h"
#endif
#ifdef PHYSICAL
#include "library_patch.h"
#include "dfg.h"
#include "FloorPlan2.h"
#include "i_macro.h"

#include <iostream>
#include <utility>
#include <vector>
#endif
/********************************************************************/
#ifdef _SCALP_
class Datapath : public Assurance {
#else
class Datapath {
#endif
private:
  float vdd;
  float sample_period;
  //no. of control steps
  int csteps;

  List_ar<Functional_unit *> functional_units;
  List_ar<Storage_unit *> storage_units;
  List_ar<Interconnect_unit *> interconnect_units;
  List_ar<Net *> nets;
  //List of PI ports - just a list of nets, for now
  List_ar<Net *> pi_nets;
  //List of PO ports - just a list of nets, for now
  List_ar<Net *> po_nets;
  Boolean addresses_set;

  //related to writing out the interconnect network
  int write_2x1_mux(Net **input_nets, int *output_net, 
		    ofstream& fout, int level);
  void recursive_mux_create(int num_nets, Net **input_nets, 
			    int *output_net, ofstream& fout, int level);
  void write_nx1_mux(Mux *multiplexer, ofstream& fout);
  int is_constant_register(Register *reg);
#ifdef PHYSICAL
protected:

  	//This is only a function object pointer.
 	 //will be used only in compute_*cost
	FloorPlan2 floorplan;  
	RVector< Rect<2, double> > module_size;
	AssocVec<double> module_link;
	RVector<double> output_sw;
	int bitwidth;
	float area_after_floorplan;
	float module_sc;
	float mux_sc;
	float wire_sc;
	float clock_tree_sc;
	float control_energy_overhead;
	float control_area_overhead;
	float neighborhood_crowd;


public:	
	//Neighborhood sensitive binding
	float compute_nc(library *lib);
	bool check_nc(float local_threshold, float global_threshold);
	float get_nc(void) const {return neighborhood_crowd;}

	int get_bitwidth(void) { return bitwidth;}
	void set_bitwidth(int bd) { bitwidth = bd;}
	void set_link(AssocVec<double> link){ module_link.clear();module_link = link;return;}
	AssocVec<double> & get_link() {return module_link;}
	RVector< Rect<2, double> > & get_size(){return module_size;}
	void set_size(RVector< Rect<2, double> > size){module_size.clear() ; module_size = size;}

	void  do_floorplan(library *lib, Scm *scm);
	
	float compute_sccost(library *lib, Scm *scm);
	float compute_module_sc(library *lib, Scm *scm);
	float compute_wire_sc(library *lib);
	float compute_wire_sc(library *lib,Scm *scm);
	//07212002
	float compute_wire_sc_new(library *lib,Scm *scm);
	pair<float, float> compute_shared_wire_sc(library *lib,Scm *scm);
	float generate_fu_output_network_energy(library *lib, Scm *scm, FUPTR fu,
			unsigned long index);
        float generate_reg_output_network_energy(library *lib, Scm *scm, Storage_unit *reg,
		                        unsigned long index);
        pair<float, float> generate_fu_shared_output_network_energy(library *lib, Scm *scm, FUPTR fu,
			                        unsigned long index);
	pair<float, float> generate_reg_shared_output_network_energy(library *lib, Scm *scm, Storage_unit *reg,
				                                        unsigned long index);

	float compute_mux_sc(library *lib, Scm *scm);
	float compute_mux_sc(Interconnect_unit &mux, library *lib, Scm *scm) const;
	float compute_reduced_module_sc(library *lib, Scm *scm);
	float compute_reduced_wire_sc(library *lib, Scm *scm);
	float compute_reduced_mux_sc(library *lib, Scm *scm);
	float compute_reduced_mux_sc(Interconnect_unit & mux, library *lib, Scm *scm)const;
	//return <energy, area> 07252002
	pair<float, float> compute_gating_overhead(short tech, library *lib, Scm *scm);
	float get_control_area_overhead(void) { return control_area_overhead; }
	float get_control_energy_overhead(void) {return control_energy_overhead;}	

	void create_graph(void);
	void generate_module_size(library *lib);
	void generate_module_link(library *lib, Scm *scm);
	void get_fu_output(Register &reg_temp, vector<int> & reg_list, FUPTR fu) const;
        void get_fu_reduced_output(Register &reg_temp, vector<int> &reg_list, 
                        vector<float> &reduced_link, FUPTR fu, Scm *scm) const;
        void get_fu_input(Register &reg_temp,FUPTR fu) const;
        void get_fu_input(vector<int> &reg_list,FUPTR fu) const;

	void get_reg_output(Register &reg_temp, vector<int> &fu_list, Storage_unit * su) const;
	void get_reg_reduced_output(Register &reg_temp, vector<int> &fu_list, 
			vector<float> &reduced_link, Storage_unit * reg, Scm *scm) const;
	void get_reg_input(vector<int> &fu_list, Storage_unit * su) const;

	void generate_reduced_module_link(library *lib, Scm *scm);
	
	float get_area_after_floorplan() const {return area_after_floorplan;}
	void set_area_after_floorplan(float area){area_after_floorplan = area;return;}
	float get_sc_after_floorplan(void) const 
		{ return module_sc + wire_sc + mux_sc;} 
	
	float get_module_sc(void) const { return module_sc;}
	float get_wire_sc(void) const {return wire_sc;}
	float get_mux_sc(void) const { return mux_sc; }
	float get_clock_tree_sc(void) const { return clock_tree_sc; }
	
	void set_module_sc(float sc) { module_sc = sc; return;}
	void set_wire_sc(float sc) {wire_sc = sc; return;}
	void set_mux_sc(float sc) { mux_sc = sc; return; }
	void set_clock_tree_sc(float sc) { clock_tree_sc = sc; }

	void display_floorplan(ostream & os) const;
	void display_module_map(ostream &os) const;
	void display_link_map(ostream &os) const;
	const FloorPlan2  & get_floorplan(void) const { return floorplan; }
#endif	
public:
  Datapath() : functional_units(), storage_units(), interconnect_units(), nets(),
  pi_nets(), po_nets()  
#ifdef PHYSICAL
	, floorplan(), module_size(), module_link(), output_sw(),
	control_area_overhead(0.0), control_energy_overhead(0.0)
#endif      	
  {
    FRITS_SET_CLASS("Datapath");
    vdd = 0.0;
    sample_period = 0.0;
    csteps = 0;
    addresses_set = F;
  }
  ~Datapath() {
    clear();
  }

  void display(ostream & = cout);

  inline const float get_vdd() {return vdd;}
  inline const float get_sample_period() {return sample_period;}
  inline const int get_csteps() {return csteps;}
  inline const List_ar<FUPTR> &get_functional_units() {return functional_units;}
  inline const List_ar<Storage_unit *> &get_storage_units() {return storage_units;}
  inline const List_ar<Interconnect_unit *> &get_interconnect_units() {return interconnect_units;}
  inline const List_ar<Net *> &get_nets() {return nets;}
  inline const List_ar<Net *> &get_pi_nets() {return pi_nets;}
  inline const List_ar<Net *> &get_po_nets() {return po_nets;}

  //set the vdd
  inline Datapath *set_vdd(const float new_vdd) {vdd = new_vdd; return this;}

  inline Datapath *set_sample_period(const float new_sample_period) {sample_period = new_sample_period; return this;}

  inline Datapath *set_csteps(const int new_csteps) {csteps = new_csteps; return this;}

  //add a functional unit
  Datapath *add_functional_unit(Functional_unit *);

  //delete a functional unit
  Datapath *delete_functional_unit(Functional_unit *);

  //add a storage unit
  Datapath *add_storage_unit(Storage_unit *);

  //delete a storage unit
  Datapath *delete_storage_unit(Storage_unit *);

  //add an interconnect unit
  Datapath *add_interconnect_unit(Interconnect_unit *);

  //delete an interconnect unit
  Datapath *delete_interconnect_unit(Interconnect_unit *);

  //add a net
  Datapath *add_net(Net *);

  //delete a net
  Datapath *delete_net(Net *);

  //add a PI net to the Datapath
  //NOTE: The net should ALREADY have been added to the datapath using add_net()
  Datapath *mark_pi(Net *);

  //remove a PI net from the Datapath
  //NOTE: The net is NOT removed from the nets list here - must do that separately
  //by calling delete_net()
  Datapath *unmark_pi(Net *);

  //add a PO net to the Datapath
  //NOTE: The net should ALREADY have been added to the datapath using add_net()
  Datapath *mark_po(Net *);

  //remove a PO net from the Datapath
  //NOTE: The net is NOT removed from the nets list here - must do that separately
  //by calling delete_net()
  Datapath *unmark_po(Net *);

  //clear this datapath
  void clear();

  //copy from another datapath
  void copy(Datapath &);

  //generate or re-generate the interconnect unit in front of a functional
  //unit or register
  void generate_interconnect_network(Functional_unit *fu, INTCON_LIBELPTR mux_libelement);

  //generate or re-generate the interconnect unit in front of a functional
  //unit or register
  void generate_interconnect_network(Storage_unit *su, INTCON_LIBELPTR mux_libelement);

  //change the library element associated with a functional unit -
  //used for SCALP class A move
  //void implement_move(Class_a_move &move);

  Boolean set_addresses_of_datapath_elements(void);

#ifdef _SCALP_
  // Writes the datapath into the specified architecture
  // 'base_signal_type' is used to get scalar and vector types for signals
  void extract_vhdl(Architecture* arch, int base_signal_type);
#endif

  //writes out a BDNET description of the datapath into the file dfgname##laxilty
  void write_bdnet(char *dfgname, float laxity);
  void write_composite_bdnet(char *, float);
  int get_element(int, int, int&, int&);

  //used to figure out the inputs and outputs of the 
  //RTL
  Boolean is_input(Net *test_net);
  Boolean is_output(Register *test_reg);

};
/********************************************************************/
#endif

