/********************************************************************/
/*              FILE: dpunit.C                                      */
/* Contains implementations of Functional_unit and Register classes */
/********************************************************************/

#include "dpunit.h"
#include "net.h"
#ifdef _SCALP_
#include "entity.h"
#include "architec.h"
#endif

/********************************************************************/
//add an operation to this Functional_unit
//the list of operations is maintained in INCREASING order of birth times.
Functional_unit *Functional_unit::add_operation(const NODEPTR new_op) 
{
  register NODEPTR node;
  List_iterator<NODEPTR> nodescan;
  Boolean added;

  FRITS_SET_MESSAGE("add_operation");
  assert(new_op);
#ifndef NDEBUG
  if(operations.find(new_op)) {
    cerr << "ERROR: Attempting to add NODE " << new_op << " twice to Functional_unit "
      << address << endl;
    exit(-1);
  }
#endif
  added = F;
  FOR_EACH_LISTNODE(operations, nodescan) {
    node = nodescan.get_item();
    assert(node);
    if(node->get_birth() > new_op->get_birth()) {
      operations.add_before(nodescan, new_op);
      added = T;
      break;
    }
  }
  if(!added) operations.append(new_op);
  return this;
}
/********************************************************************/
//added by wwang on 9/24/01 to determine whether or not a functional unit is special

Boolean Functional_unit::is_special_unit()
{
/*
  NODEPTR node;
  List_iterator<NODEPTR> nodescan;
  Operator func;

  FOR_EACH_LISTNODE(operations, nodescan) {
    node = nodescan.get_item();
    func = node->get_func();
    if(func == ELP_ || func == BLP_ || func == SLP_
      || func == SPS_ || func == EPS_ || func = EIF_)
    {
      return T;
    }
  }
*/
  return F;
} 
/********************************************************************/
//delete an operation from this Functional_unit
Functional_unit *Functional_unit::delete_operation(const NODEPTR op)
{
  List_node<NODEPTR> *lnode;

  FRITS_SET_MESSAGE("delete_operation");
  assert(op);
  if(lnode = operations.find(op)) {
    operations.remove_node(lnode);
    // assert(!operations.find(op)); -- waste of time
  } else {
    cerr << "ERROR: Attempt to delete operation " << op << " that is not mapped to Functional_unit " << address << endl;
    exit(-1);
  }
  return this;
}

/***************************************************************************/
void Functional_unit::display(ostream &output)
{
  FRITS_SET_MESSAGE("display");

  //print out the list of operations mapped to this functional unit
  output << "Functional unit: " << this << ", Operations: " <<
    operations << endl;
  return;
}

/***************************************************************************/
Register *Register::add_variable(const EDGEPTR new_var)
{
  register EDGEPTR edge;
  List_iterator<EDGEPTR> edgescan;
  Boolean added = T;

  FRITS_SET_MESSAGE("add_variable");
  assert(new_var);
#ifndef NDEBUG
  if(variables.find(new_var)) {
    cerr << "ERROR: Attempting to add EDGE " << new_var << " twice to Register "
      << address << endl;
    exit(-1);
  }
#endif

  added = F;
  FOR_EACH_LISTNODE(variables, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    if(edge->get_birth() > new_var->get_birth()) {
      variables.add_before(edgescan, new_var);
      added = T;
      break;
    }
  }
  if(!added) {
    variables.append(new_var);
  }
  return this;
}

/********************************************************************/
Register *Register::delete_variable(EDGEPTR var)
{
  List_node<EDGEPTR> *lnode;
  FRITS_SET_MESSAGE("delete_variable");
  assert(var);
  if(lnode = variables.find(var)) {
    variables.remove(var);
    // assert(!variables.find(var));
  } else {
    cerr << "ERROR: Attempt to delete variable " << var << " that is not mapped to Register "
      << address << endl;
    exit(-1);
  }
  return this;
}
/********************************************************************/
void Register::display(ostream &output)
{
  //print out the list of variables
  output << "Register " << (void *)this << ": Variables: ";
  display_edgelist(variables, output);
  return;
}

/********************************************************************/
#ifdef _SCALP_
// The primary difference between this message and the same message of
// the Datapath_element base class is a generic map of BitWidth.
// Possibly, we could move this to the Datapath_element:: message.
Component_instantiation *Register::create_component_instantiation(Architecture* arch, const char* label)
{
  FRITS_SET_MESSAGE("create_component_instantiation");
  Component_instantiation* compinst;
  mem_ok(compinst = new Component_instantiation);
  Port_map* portmap;
  Association_element* assoelem;
  Formal_designator* fd;
  Actual_designator* ad;
  FRITS_object* clk;
  Name* compname;
  int i = 0;
  SigVarCon* net;

  List_iterator<Library_element_port *> portlist_iterator;
  mem_ok(compinst = new Component_instantiation);
  compinst->set_identifier(label);
  mem_ok(compname = new Name);
  Net* n;
#ifdef _OLD_LIBRARY_
  compname->set_identifier(library_element->get_frits_libelement()->get_name());
#else
  compname->set_identifier(library_element->get_name());
#endif
  compinst->set_component_name(compname);

  // We will create a Port_map here, Generic_map will have to be postponed
  // until we figure out what generics we have to pass to components.
  assert(library_element->number_of_ports() == nets.get_size());
  mem_ok(portmap = new Port_map);
  compinst->set_port_map(portmap);

// First, map input and output
#ifdef _OLD_LIBRARY_
  FOR_EACH_LISTNODE(library_element->get_frits_libelement()->data_ports, portlist_iterator) {
#else
  FOR_EACH_LISTNODE(library_element->data_ports, portlist_iterator) {
#endif
    if ( i < 2 ) assert(nets[i]);
    if (nets[i]) {
      mem_ok(assoelem = new Association_element);
      mem_ok(fd = new Formal_designator);
      mem_ok(ad = new Actual_designator);
      fd->set_name(portlist_iterator.get_item()->get_name());
      assoelem->set_formal_designator(fd);
      arch->st_search(nets[i]->get_name(), (FRITS_object*&)net);
      assert(net);  // net exists in the architecture
      ad->set_actual_object(net); 
      assoelem->set_actual_designator(ad);
      portmap->add_association_element(assoelem);
    }
    i++;
  }

  // connnect the 'clk' port, assuming there is a global 'clk' signal
  // other Register-specific global signals should also be connected here
  // If we include Vdd and Vss rails, they should be attached in 
  // Datapath_element::create_component_instantiation
  mem_ok(assoelem = new Association_element);
  mem_ok(fd = new Formal_designator);
  mem_ok(ad = new Actual_designator);
  fd->set_name("clk");
  assoelem->set_formal_designator(fd);
  arch->st_search("clk", clk);
  assert(clk);  // net exists in the architecture
  ad->set_actual_object((SigVarCon*)clk); 
  assoelem->set_actual_designator(ad);
  portmap->add_association_element(assoelem);

  Storage_library_element* frits_le;
#ifdef _OLD_LIBRARY_
  frits_le = (Storage_library_element*)library_element->get_frits_libelement();
#else
  frits_le = (Storage_library_element*)library_element;
#endif

  if ( frits_le->has_load_enable() ) {		// loadable register
    char addr[5];
    mem_ok(assoelem = new Association_element);
    mem_ok(fd = new Formal_designator);
    mem_ok(ad = new Actual_designator);
    fd->set_name("le");
    assoelem->set_formal_designator(fd);
    STring output_name("X");
    FRITS_object* le;
    output_name.cat(itoa(address, addr, 10));
    output_name.cat("_le");
    arch->st_search(output_name.get(), le);
//    assert(le);  // net exists in the architecture
    if (le) ad->set_actual_object((SigVarCon*)le);
    else ad->set_open(TRUE); 
    assoelem->set_actual_designator(ad);
    portmap->add_association_element(assoelem);
  }

  if (bitwidth) {  // if we need to scale the register
#ifdef _OLD_LIBRARY_
    assert(library_element->get_frits_libelement()->get_frits_component()->
      get_entity()->get_generic()->search_interface_element("BitWidth"));
#else
    assert(library_element->get_frits_component()->
      get_entity()->get_generic()->search_interface_element("BitWidth"));
#endif
    Generic_map* genmap;
    Expression* ex;
    Primary* p;
    if ( !(genmap = compinst->get_generic_map()) ) {
      mem_ok(genmap = new Generic_map);
      compinst->set_generic_map(genmap);
    }
    mem_ok(assoelem = new Association_element);
    mem_ok(fd = new Formal_designator);
    mem_ok(ad = new Actual_designator);
    fd->set_name("BitWidth");
    assoelem->set_formal_designator(fd);
    mem_ok(p = new Primary);
    p->set_int(bitwidth);
    mem_ok(ex = new Expression(p));
    ad->set_expression(ex); 
    assoelem->set_actual_designator(ad);
    genmap->add_association_element(assoelem);
    compinst->set_generic_map(genmap);
  } /* if (bitwidth) */
  return compinst;
}
#endif /* _SCALP_ */
  
/********************************************************************/
// Searches for the first input net connected to a dpelement.
// Since input mux lines are normally connected to outputs, and
// a dpelement is assumed to have a single output, the net returned
// is most likely the only one connected to the dpelement.
// Any risk?
// ANAND_TO_SERGEI: Sounds fine with me, as long as we assert the
// invariant. My construction procedures ensure that a Mux will NOT
// have a dpelement connected to more than 1 of its input ports.
// Search is done by going through the nets_list of the mux and looking
// for dpelement in port_connections of each net.
// Note that nets array is not used for Muxes
Net* Mux::search_input_net(Datapath_element* dpelement)
{
  List_iterator<Port_connection*> pc_iterator;
  List_iterator<Port_map_by_name*> pmbn_iterator;

  FRITS_SET_MESSAGE("search_input_net");
  assert(dpelement);

  //assert that the same dpelemnt is not connected to multiple input ports
  //of this Mux
#ifndef NDEBUG
  Boolean check = F;
  FOR_EACH_LISTNODE(nets_list, pmbn_iterator) {
    FOR_EACH_LISTNODE(pmbn_iterator.get_item()->net->get_port_connections(),
		                                                pc_iterator) {
      if (pc_iterator.get_item()->get_datapath_element() == dpelement) {
	assert(!check);
	check = T;
      }
    }
  }
#endif

  FOR_EACH_LISTNODE(nets_list, pmbn_iterator) {
    FOR_EACH_LISTNODE(pmbn_iterator.get_item()->net->get_port_connections(),
		                                                pc_iterator) {
      if (pc_iterator.get_item()->get_datapath_element() == dpelement)
	return pmbn_iterator.get_item()->net;
    } /*FOR_EACH_LISTNODE*/
  } /*FOR_EACH_LISTNODE*/
  return NULL;  // search unsuccessful
}

/********************************************************************/
//connect one of the Datapath_element's ports to a net - BY NAME
void Interconnect_unit::connect_select_to_net(Net *new_net, const char* select_name)
{
  FRITS_SET_MESSAGE("connect_select_to_net");
  assert(new_net != NULL && select_name != NULL);
  List_iterator<Port_map_by_name*> select_list_iterator;
  FOR_EACH_LISTNODE(select_list, select_list_iterator) {
    if( !strcmp(select_list_iterator.get_item()->port_name, select_name) ) {
// HACK: see connect_to_net above      !!assert(F);!!
      select_list_iterator.get_item()->net->disconnect_from_port(this, select_name);
      disconnect_from_net(select_list_iterator.get_item()->net, select_name);
    }
  }  // make sure the port with this name is not mapped
  Port_map_by_name* pmbn;
  mem_ok(pmbn = new Port_map_by_name(new_net, select_name));
  nets_list.append(pmbn);
  new_net->connect_to_port(this, select_name);
  return;
}

/********************************************************************/
//disconnect one of the Datapath_element's ports from a net - BY NAME
void Interconnect_unit::disconnect_select_from_net(Net *old_net, const char* select_name)
{
  FRITS_SET_MESSAGE("disconnect_select_from_net");
  assert(old_net != NULL);
  assert(select_name != NULL);
  List_iterator<Port_map_by_name*> select_list_iterator;
  Boolean found = F;
  Port_map_by_name* pmbn;
  FOR_EACH_LISTNODE(select_list, select_list_iterator) {
    pmbn = select_list_iterator.get_item();
    if ( pmbn->net == old_net && !strcmp(pmbn->port_name, select_name) ) {
      found = T;
      break;
    }
  }  // make sure the port with this name is not mapped
  assert(found);
  old_net->disconnect_from_port(this, select_name);
  select_list.remove(pmbn);
  return;
}
/********************************************************************/
// Search for a select net of an interconnect unit, by port name
Net *Interconnect_unit::get_select_net(char *selectname)
{
  Net *retptr = NULL;
  List_iterator<Port_map_by_name *> pmbn_iterator;

  FOR_EACH_LISTNODE(select_list, pmbn_iterator) {
    if(!strcmp(pmbn_iterator.get_item()->port_name, selectname)) {
      retptr = pmbn_iterator.get_item()->net;
      break;
    }
  } /* FOR */

  return(retptr);
}
/********************************************************************/
#ifdef _SCALP_
// The primary difference between this message and the same message of
// the Datapath_element base class is a generic map of BitWidth, and
// sizing of the mux according to the size of 'nets'
// Also, port mapping is done by name for Mux. (uses nets_list)
// Since we create a bunch of real 2-1 muxes to realize a "symbolic" mux,
// Component_instantiation* is NULL, because it's meaningless to return one
// out of many created. If we encapsulate a Mux as an entity later, we can
// return Component_instantiation*. 'label' for an instance name gets
// appended an integer for 2-1 mux index.

Component_instantiation *Mux::create_component_instantiation(Architecture* arch, const char* label)
{
  FRITS_SET_MESSAGE("create_component_instantiation");
  Component_instantiation* compinst;
  Name* compname;
  Generic_map* genmap;
  Port_map* portmap;
  Association_element* assoelem;
  Formal_designator* fd;
  Actual_designator* ad;
  FRITS_object* net;
  List_iterator<Port_map_by_name *> portmap_iterator;
  Port_map_by_name *pmbn, *pmbn_out1 = NULL;
  STring instance_name;
  STring select_name;
  char mux_index[3];
  int mux_count = -1;
  Net* out1;
  char* pn = NULL;
  Entity* mux_entity;
  Architecture* mux_architecture;

  mem_ok(compinst = new Component_instantiation);
  compinst->set_identifier(label);
  mem_ok(compname = new Name);
  STring entity_name("IU_");
  char c_address[5];
  entity_name.cat(itoa(address, c_address, 10));
  compname->set_identifier(entity_name.get());
  compinst->set_entity_name(compname);
  mem_ok(portmap = new Port_map);
  compinst->set_port_map(portmap);

  Library_element* frits_le;
#ifdef _OLD_LIBRARY_
  frits_le = library_element->get_frits_libelement();
#else
  frits_le = library_element;
#endif
  assert(((Interconnect_library_element*)frits_le)->is_tristate());	// ONLY tri-states for now

    if (bitwidth) {  // if we need to scale the mux bitwidth-wise
      Expression* ex;
      Primary* p;
      mem_ok(genmap = new Generic_map);
      compinst->set_generic_map(genmap);
      mem_ok(assoelem = new Association_element);
      mem_ok(fd = new Formal_designator);
      mem_ok(ad = new Actual_designator);
      fd->set_name("BitWidth");
      assoelem->set_formal_designator(fd);
      mem_ok(p = new Primary);
      p->set_int(bitwidth);
      mem_ok(ex = new Expression(p));
      ad->set_expression(ex); 
      assoelem->set_actual_designator(ad);
      genmap->add_association_element(assoelem);
      compinst->set_generic_map(genmap);
    } /* if (bitwidth) */

  // map the output net first
  out1 = get_net(MUXOUT);
  assert(out1);

    // map the output line "out1" to a common net
    mem_ok(assoelem = new Association_element);
    mem_ok(fd = new Formal_designator);
    mem_ok(ad = new Actual_designator);
    fd->set_name("out1");
    assoelem->set_formal_designator(fd);
    arch->st_search(out1->get_name(), net);
    assert(net);  // net exists in the architecture
    ad->set_actual_object((SigVarCon*)net); 
    assoelem->set_actual_designator(ad);
    portmap->add_association_element(assoelem);

    // map inputs by scrolling through the nets_list
  FOR_EACH_LISTNODE(nets_list, portmap_iterator) {
    pmbn = portmap_iterator.get_item();
    mem_ok(assoelem = new Association_element);
    mem_ok(fd = new Formal_designator);
    mem_ok(ad = new Actual_designator);
    fd->set_name(pmbn->port_name);
    assoelem->set_formal_designator(fd);
    arch->st_search(pmbn->net->get_name(), net);
    assert(net);  // net exists in the architecture
    ad->set_actual_object((SigVarCon*)net); 
    assoelem->set_actual_designator(ad);
    portmap->add_association_element(assoelem);
  } /* FOR */

    // now, map the selects, if any are specified
  if ( select_list.get_size() ) {
      FOR_EACH_LISTNODE(select_list, portmap_iterator) {
	pmbn = portmap_iterator.get_item();
	mem_ok(assoelem = new Association_element);
	mem_ok(fd = new Formal_designator);
	mem_ok(ad = new Actual_designator);
	fd->set_name(pmbn->port_name);
	assoelem->set_formal_designator(fd);
	arch->st_search(pmbn->net->get_name(), net);
	assert(net);  // net exists in the architecture
	ad->set_actual_object((SigVarCon*)net); 
	assoelem->set_actual_designator(ad);
	portmap->add_association_element(assoelem);
      } /* FOR */
  } /* if ( select_list.get_size() ) */

return compinst;
}

// The following code creates an architecture for the mux
/*
Architecture*
Mux::create_architecture(const char* architecture_name)
{
FRITS_SET_MESSAGE("create_architecture");
  FOR_EACH_LISTNODE(nets_list, portmap_iterator) {
    pmbn = portmap_iterator.get_item();
    if ( !pmbn_out1 && !(strcmpl(pmbn->port_name, "out1")) ) {
      pmbn_out1 = pmbn;  // flag, just to prevent subsequent calls to strcmpl  
      portmap_iterator.increment();  // filter out the output port map
      if ( portmap_iterator.not_done() )
        pmbn = portmap_iterator.get_item();
      else
	break;
    }

    // Creates component instantiation and attaches a port map
    mem_ok(compinst = new Component_instantiation);
    instance_name.set(label);
    instance_name.cat(itoa(mux_count++, mux_index, 10));
    compinst->set_identifier(instance_name.get());
    mem_ok(compname = new Name);
    compname->set_identifier(frits_le->get_name());
    compinst->set_component_name(compname);
    mem_ok(portmap = new Port_map);
    compinst->set_port_map(portmap);

    if (bitwidth) {  // if we need to scale the mux bitwidth-wise
      assert(frits_le->get_frits_component()->
        get_entity()->get_generic()->search_interface_element("BitWidth"));
      Generic_map* genmap;
      Expression* ex;
      Primary* p;
      mem_ok(genmap = new Generic_map);
      compinst->set_generic_map(genmap);
      mem_ok(assoelem = new Association_element);
      mem_ok(fd = new Formal_designator);
      mem_ok(ad = new Actual_designator);
      fd->set_name("BitWidth");
      assoelem->set_formal_designator(fd);
      mem_ok(p = new Primary);
      p->set_int(bitwidth);
      mem_ok(ex = new Expression(p));
      ad->set_expression(ex); 
      assoelem->set_actual_designator(ad);
      genmap->add_association_element(assoelem);
    } // if (bitwidth) 

    // map the output line "out1" to a common net
    mem_ok(assoelem = new Association_element);
    mem_ok(fd = new Formal_designator);
    mem_ok(ad = new Actual_designator);
    fd->set_name("out1");
    assoelem->set_formal_designator(fd);
    arch->st_search(out1->get_name(), net);
    assert(net);  // net exists in the architecture
    ad->set_actual_object((SigVarCon*)net); 
    assoelem->set_actual_designator(ad);
    portmap->add_association_element(assoelem);

    // map the first line of the mux
    mem_ok(assoelem = new Association_element);
    mem_ok(fd = new Formal_designator);
    mem_ok(ad = new Actual_designator);
    fd->set_name("in1");
    assoelem->set_formal_designator(fd);
    arch->st_search(pmbn->net->get_name(), net);
    assert(net);  // net exists in the architecture
    ad->set_actual_object((SigVarCon*)net); 
    assoelem->set_actual_designator(ad);
    portmap->add_association_element(assoelem);
    
    // find out the select net and map the select port
    select_name.set("select");
    // Get the index by skipping "in"    
    if (pn) free(pn);
    select_name.cat(++(++(pn=strdup(pmbn->port_name))));
    mem_ok(assoelem = new Association_element);
    mem_ok(fd = new Formal_designator);
    mem_ok(ad = new Actual_designator);
    fd->set_name("select1");
    assoelem->set_formal_designator(fd);
    // Trick is to get the net mapped to select port of this input 
    // and search for a signal with the net's name
    arch->st_search(get_net(select_name.get())->get_name(), net);
    assert(net);  // net exists in the architecture
    ad->set_actual_object((SigVarCon*)net); 
    assoelem->set_actual_designator(ad);
    portmap->add_association_element(assoelem);

    // prepare stuff for the second in/select pair
    portmap_iterator.increment();  // get the next port map
    if ( portmap_iterator.not_done() )
      pmbn = portmap_iterator.get_item();
    else
      pmbn = NULL;   

    mem_ok(assoelem = new Association_element);
    mem_ok(fd = new Formal_designator);
    mem_ok(ad = new Actual_designator);
    fd->set_name("in2");
    assoelem->set_formal_designator(fd);
    if (pmbn) {
      arch->st_search(pmbn->net->get_name(), net);
      assert(net);  // net exists in the architecture
      ad->set_actual_object((SigVarCon*)net);
    } else {
      ad->set_open(TRUE);
    } // if (pmbn) 
    assoelem->set_actual_designator(ad);
    portmap->add_association_element(assoelem);

    mem_ok(assoelem = new Association_element);
    mem_ok(fd = new Formal_designator);
    mem_ok(ad = new Actual_designator);
    fd->set_name("select2");
    assoelem->set_formal_designator(fd);
    if (pmbn) {
      // find out the select net and map the select port
      select_name.set("select");
      // Get the index by skipping "in"    
      if (pn) free(pn);  // garbage collect
      select_name.cat(++(++(pn = strdup(pmbn->port_name))));
      // Trick is to get the net mapped to select port of this input 
      // and search for a signal with the net's name
      arch->st_search(get_net(select_name.get())->get_name(), net);
      assert(net);  // net exists in the architecture
      ad->set_actual_object((SigVarCon*)net); 
    } else {
      ad->set_open(TRUE);
    } // if (pmbn) 
    assoelem->set_actual_designator(ad);
    portmap->add_association_element(assoelem);

    arch->get_concurrent_body()->add_component_instantiation(compinst);

    if (!pmbn) break;
  } // FOR 
}
*/
#endif /* _SCALP_ */
/********************************************************************/
int Mux::find_input_position(Net* net)
{
  if (!net) {
    return (-1);
  } else {
    List_iterator<Port_map_by_name *> inputs_iterator;
    int i = 0;
    FOR_EACH_LISTNODE (nets_list, inputs_iterator) {
      if ( inputs_iterator.get_item()->net == net ) {
        return i; 
      }
      i++;
    } /* FOR */
  } /* if */
  return (-1);
}
/********************************************************************/
