/********************************************************************/
/*              FILE: dpunit.h                                      */
/* Contains a description of a Datapath_element class		    */
/********************************************************************/

#ifndef ANAND_DATAPATH_UNIT_H
#define ANAND_DATAPATH_UNIT_H
#include <assert.h>
#include "dpelemnt.h"
#include "dfgnode.h"
#ifdef PHYSICAL
#include <list>
#include <utility>
#include"scm.h"
#include"library_patch.h"
#endif
/********************************************************************/
class Functional_unit : public Datapath_element {
#ifdef PHYSICAL
public:
	void compute_neighbors(void);
	float get_nc(void) const { return _NC;}
	float compute_nc(void);	
	float compute_reduced_sc(Scm *);
	float compute_sc(Scm *);
#endif
private:
  List_ar<NODEPTR> operations;         //list of CDFG operations performed
public:
  Functional_unit(FUNC_LIBELPTR libelem, int bw = 0) : Datapath_element(libelem), operations() {
    FRITS_SET_CLASS("Functional_unit");
    assert(bw >= 0);
    bitwidth = bw;
  }

  ~Functional_unit() {}

  inline const List_ar<NODEPTR> &get_operations() {
    FRITS_SET_MESSAGE("get_operations");
    return operations;
  }

  Boolean is_special_unit();

  //add an operation to this Functional_unit
  //the list of operations is maintained in INCREASING order of birth times.
  Functional_unit* add_operation(NODEPTR new_op); 

  //delete an operation from this Functional_unit
  Functional_unit* delete_operation(NODEPTR op);

  virtual const char* get_id(void) {return "Functional_unit";};

  virtual int get_number_of_nets(void)	{ return nets.get_size();}

  void display(ostream &output = cout);
};
//Syntactic sugar for displaying a Functional_unit
inline ostream &operator <<(ostream &output, Functional_unit &fu)
{fu.display(output);return(output);}
/********************************************************************/
/*Currently, class Storage_unit is just a placeholder so that components like
 *memory and register files can be handled in the future, apart from registers.
 */
class Storage_unit : public Datapath_element {
public:
  Storage_unit(STOR_LIBELPTR libelem) : Datapath_element(libelem) {
    FRITS_SET_CLASS("Storage_unit");
#ifndef _SCALP_
#ifdef _OLD_LIBRARY_
    //TEMPORARY HACK PUT IN BY ANAND
    //In the STAND ALONE MODE, libelem will be NULL, and hence,
    //the nets array is NOT initialized in Datapath_element().
    //However, the nets array needs to be allocated appropriately
    //since registers are mapped by order
    nets.resize(2);
    nets.reset((Net *)NULL);
#endif
#endif
  }
  ~Storage_unit() {
  }
  virtual const char* get_id(void) {return "Storage_unit";};
};

/********************************************************************/
class Register : public Storage_unit {
private:
  //list of variables stored
  List_ar<EDGEPTR> variables;
public:
#ifdef PHYSICAL
  	void compute_neighbors(void);
	float get_nc(void) const { return _NC;}
        float compute_nc(float reg_area);
#endif	
  Register(STOR_LIBELPTR libelem, int bw = 0) : Storage_unit(libelem), variables() {
    FRITS_SET_CLASS("Register");
    assert(bw >= 0);
    bitwidth = bw;
  }
#ifdef PHYSICAL  
  Register():Storage_unit(NULL), variables() {
    FRITS_SET_CLASS("Register");
    bitwidth = 0;
  }
#endif
  
  ~Register() { variables.clear();}

  inline const List_ar<EDGEPTR> &get_variables() {
    FRITS_SET_MESSAGE("get_variables");
    return(variables);
  }

  Register* add_variable(const EDGEPTR new_var);

  Register* delete_variable(EDGEPTR var);
#ifdef PHYSICAL
  void clear_variables() { variables.clear();}  
#endif  
  virtual const char* get_id(void) {return "Register";};

  void display(ostream &output);

  virtual int get_number_of_nets(void)	{return nets.get_size();}

#ifdef _SCALP_
  // Creates a component instantiation for mapping to VHDL.
  // The instantiation gets 'label'.
  virtual Component_instantiation* create_component_instantiation(Architecture* arch, const char* label);
#endif
};
//Syntactic sugar for displaying a Register
inline ostream &operator <<(ostream &output, Register &reg)
{reg.display(output);return(output);}
/********************************************************************/
/*Again, Interconnect_unit is currently just a place holder - we will
 *add fields if necessary as we go along.
 */
class Interconnect_unit : public Datapath_element {
 protected:
  List_ar<Port_map_by_name *> select_list;  // port_mapping by name for selects

  Interconnect_unit(INTCON_LIBELPTR libelem) : select_list(), Datapath_element(libelem) {
    FRITS_SET_CLASS("Interconnect_unit");
  }
  ~Interconnect_unit() {}
  virtual const char* get_id(void) {return "Interconnect_unit";};
 public:
  void connect_select_to_net(Net *new_net, const char* select_name);
  void disconnect_select_from_net(Net *old_net, const char* select_name);
  Net *get_select_net(char *selectname);
};

/********************************************************************/

/********************************************************************
  A Mux appears to be somewhat different than other Datapath_elements.
  The root of the difference may even belong to the Interconnect_unit,
  but we won't know until there is another subclass of Interconnect_unit.
  The difference stems from the fact that a Mux has a varying number of
  input ports and select signals, which changes as the synthesis progresses.
  Both input and select signals get mapped by name. Input signals get
  mapped (presumably) in Datapath::generate_interconnect_network, whereas
  select signals in Scheduler::create_fsm. 
  Due to this property, the Array "nets" is of questionable use.
  The problem comes because at some point we need to identify positions
  of the input nets so that select signals can be hooked up to the FSM.
  A possible solution will be to project the list of by-name connections
  to the nets Array, sized by the length of the nets_list. To be investigated.

  SAS 6/18/96
  The ports of a Mux are stored in the following fashion:
  output is in nets[0], accessible by get_output(), initialized by set_output()
  inputs are in nets_list

*****************************************************************************/
class Mux : public Interconnect_unit {
 public:
  Mux(INTCON_LIBELPTR libelem, int bw = 0) : Interconnect_unit(libelem) {
    FRITS_SET_CLASS("Mux");
    assert(bw >= 0);
    bitwidth = bw;
    //ANAND 6/5/96
    //HACK PUT IN BY ANAND TO ENSURE THAT THE NETS ARRAY IS PROPERLY SIZED
    //IN THE _OLD_LIBRARY_ MODE
#ifdef _OLD_LIBRARY_
    nets.resize(1);
    nets.reset((Net *)NULL);
#endif
  }
  ~Mux() {
  }
  virtual const char* get_id(void) {return "Mux";}
  Net* search_input_net(Datapath_element* dpelement);
  virtual int get_number_of_nets(void)
    {return nets_list.get_size() + select_list.get_size() + 1;}
  int get_number_of_inputs(void) {return nets_list.get_size();}
  void connect_output(Net *new_net) 
	{Datapath_element::connect_to_net(new_net, 1);}
  void disconnect_output(Net *old_net) 
	{Datapath_element::disconnect_from_net(old_net, 1);}
  // Finds the position of the net in the nets_list. Returns -1 if not found.
  int find_input_position(Net* net);
#ifdef _SCALP_
  // Creates a component instantiation for mapping to VHDL.
  // The instantiation gets 'label'.
  virtual Component_instantiation* create_component_instantiation(Architecture* arch, const char* label);

// Since Interconnect_units are virtual componets that do not have
// directly associated library elements, thw following two messages
// are necessary:
  virtual Entity* create_entity(const char* entity_name = NULL) {};
  virtual Architecture* create_architecture(const char* architecture_name = "structural") {};
#endif


  // These two messages of a superclass are not supposed to be used.
/*  void connect_to_net(Net *new_net, const int port_num) 
	{FRITS_SET_MESSAGE("connect_to_net"); assert(FALSE);}
  void disconnect_from_net(Net *old_net, const int port_num)
	{FRITS_SET_MESSAGE("disconnect_from_net"); assert(FALSE);} */
};
/********************************************************************/
#endif /* ANAND_DATAPATH_UNIT_H */
