/********************************************************************/
/*              FILE: net.C                                         */
/* Contains an implementation of Net class	 		    */
/********************************************************************/

#include "net.h"

/********************************************************************/
// WARNING: the (dis)connect messages are not resiprocal,
// Datapath_element messages have to be called, too
// Connect the net to a port of a Datapath_element specified by an index port_num
void Net::connect_to_port(Datapath_element *dpelem, int port_num)
{
  PORTCONNPTR new_port_connection;

  FRITS_SET_MESSAGE("connect_to_port");
  assert(dpelem && port_num >= 0);

  //check that this dpelem and port_num are not already connected to this Net
#ifndef NDEBUG
  {
    List_iterator<PORTCONNPTR> netelemscan;
    PORTCONNPTR netelement;
    FOR_EACH_LISTNODE(port_connections, netelemscan) {
      netelement = netelemscan.get_item();
      if(netelement->datapath_element == dpelem && netelement->port_num == port_num) {
	cerr << "ERROR: re-connecting already connected Datapath element " << dpelem
	  << " port #" << port_num << " to Net " << this << endl;
	exit(-1);
      }
    }
  }
#endif
  new_port_connection = new PORTCONN(dpelem, port_num);
  port_connections.append(new_port_connection);
  return;
}
/********************************************************************/
// Connect the net to a port of a Datapath_element specified by a port name 
void Net::connect_to_port(Datapath_element *dpelem, const char* port_name)
{
  PORTCONNPTR new_port_connection;

  FRITS_SET_MESSAGE("connect_to_port");

  //Check that this dpelem and port_name are not already connected to this Net
#ifndef NDEBUG
  {
    if (is_connected_by_name(dpelem, port_name)) {
      cerr << "ERROR: re-connecting already connected Datapath element " << dpelem
	<< " port name:" << port_name << " to Net " << this << endl;
      exit(-1);
    }
  }
#endif
  new_port_connection = new PORTCONN(dpelem, port_name);
  port_connections.append(new_port_connection);
  return;
}
/********************************************************************/
// Disconnect the net from a port of a Datapath_element specified by an index 
void
Net::disconnect_from_port(Datapath_element *dpelem, int port_num)
{
  List_iterator<PORTCONNPTR> netelemscan;
  PORTCONNPTR a_port_connection;
  Boolean found = F;

  FRITS_SET_MESSAGE("disconnect_from_port");
  FOR_EACH_LISTNODE(port_connections, netelemscan) {
    a_port_connection = netelemscan.get_item();
    if(a_port_connection->datapath_element == dpelem && a_port_connection->port_num == port_num) {
      found = T;
      break; //BREAK from the FOR_EACH_LISTNODE loop
    }
  }
  if(!found) {
    cerr << "ERROR: Attempt to delete connection between Datapath element " << dpelem
      << " port #" << port_num << " and net " << this << " that is not present" << endl;
    exit(-1);
  }
  port_connections.remove(a_port_connection);
  delete a_port_connection;
}
/********************************************************************/
//disconnect the net from a port of a Datapath_element specified by a port name 
void
Net::disconnect_from_port(Datapath_element *dpelem, const char* port_name)
{
  List_iterator<PORTCONNPTR> netelemscan;
  PORTCONNPTR a_port_connection;
  Boolean found = F;

  FRITS_SET_MESSAGE("disconnect_from_port");
  FOR_EACH_LISTNODE(port_connections, netelemscan) {
    a_port_connection = netelemscan.get_item();
    if(a_port_connection->datapath_element == dpelem &&
                  !strcmp(a_port_connection->port_name, port_name)) {
      found = T;
      break; //BREAK from the FOR_EACH_LISTNODE loop
    }
  }
  if(!found) {
    cerr << "ERROR: Attempt to delete connection between Datapath element " << dpelem
      << " port name " << port_name << " and net " << this << " that is not present" << endl;
    abort();
    /*exit(-1);*/
  }
  port_connections.remove(a_port_connection);
  delete a_port_connection;
}
/********************************************************************/
// Disconnect this net from ALL ports that it is connected to
void Net::disconnect_from_all_ports()
{
  PORTCONNPTR connection;
  List_iterator<PORTCONNPTR> portconnscan;

  FOR_EACH_LISTNODE(port_connections, portconnscan) {
    connection = portconnscan.get_item();
    assert(connection);
    delete connection;
  }
  port_connections.clear();
  return;
}
/********************************************************************/
// A fairly superficial message. Just checks port_connections for a
// connection BY NAME to the named port of a datapath_element
// However, can be easily modified to be a true "is_connected"
// by looking up a port name of port_connections by order if
// dpelement matches.
Boolean Net::is_connected_by_name(Datapath_element *dpelem, const char* port_name) {
  List_iterator<Port_connection *> pc_iterator;

  assert(dpelem && port_name);
  FOR_EACH_LISTNODE(port_connections, pc_iterator) {
    if (pc_iterator.get_item()->get_datapath_element() == dpelem &&
	!strcmp(pc_iterator.get_item()->get_port_name(), port_name) )
      return T;
  }
  return F;
}
/********************************************************************/
// Returns T if this net is connected to ANY input port of dpelem
Boolean Net::is_connected_to_input_port_by_name(Datapath_element *dpelem) {
  List_iterator<Port_connection *> pc_iterator;

  assert(dpelem);
  FOR_EACH_LISTNODE(port_connections, pc_iterator) {
    if (pc_iterator.get_item()->get_datapath_element() == dpelem &&
	!strncmp(pc_iterator.get_item()->get_port_name(), "in", 2) )
      return T;
  }
  return F;
}
/********************************************************************/
  //Return the datapath element that has its OUT port connected to this net.
  //Asserts that only 1 such datapath element exists
Datapath_element* Net::get_driver(void) {
  PORTCONNPTR connection;
  List_iterator<PORTCONNPTR> portconscan;
  Datapath_element *driver = NULL;

  FOR_EACH_LISTNODE(port_connections, portconscan) {
  connection = portconscan.get_item();
  assert(connection);
  assert(connection->datapath_element);
  // Connections by port names should be only in Muxes, and
  // connections by order chould be only for funcunits/registers
  if ( 
       (!strcmp(connection->datapath_element->get_id(), "Mux") && 
          connection->port_num == MUXOUT) ||
       (!strcmp(connection->datapath_element->get_id(), "Functional_unit") && 
          connection->port_num == OUT1) ||
       (!strcmp(connection->datapath_element->get_id(), "Register") && 
          connection->port_num == REGOUT) ) { 		
    	assert_force(!driver);
    	driver = connection->datapath_element;
    }
  else if(connection->port_name!=NULL) {
	 if((!strcmp(connection->port_name, "out1"))) { 
		  assert_force(!driver);
		  driver = connection->datapath_element;
	 }
  }
  }//end of FOR_EACH_LISTNODE
  return(driver);
}

#ifdef PHYSICAL
void Net::get_load(List_ar<Datapath_element *> & load) const {
  PORTCONNPTR connection;
  List_iterator<PORTCONNPTR> portconscan;
  Datapath_element *driver = NULL;
  assert(load.get_size()==0);	
  FOR_EACH_LISTNODE(port_connections, portconscan) {
     connection = portconscan.get_item();
     assert(connection);
     assert(connection->datapath_element);
     if( (connection->port_name && strcmp( connection->port_name, "out1")) ||
     ( !strcmp(connection->datapath_element->get_id(), "Mux") &&
          connection->port_num != MUXOUT) ||
    (!strcmp(connection->datapath_element->get_id(), "Functional_unit") &&
          connection->port_num != OUT1) ||
    (!strcmp(connection->datapath_element->get_id(), "Register") &&
          connection->port_num != REGOUT) ) {
    load.append( connection->datapath_element);
    }
  }
  return;
}
#endif
/********************************************************************/
