/********************************************************************/
/*              FILE: net.h                                         */
/* Contains a description of Net class				    */
/********************************************************************/
#ifndef ANAND_NET_H
#define ANAND_NET_H

#include "dpelemnt.h"
#include "basic.h"
#ifdef _SCALP_
#include "assuranc.h"
#endif
/********************************************************************/
/*A structure that represents a (Datapath_element, port number) tuple. */
/* port number is 0-based */
class Port_connection 
#ifdef _SCALP_
: public Assurance
#endif
{
  friend class Net;
private:
  Datapath_element *datapath_element;
  int port_num;
  char* port_name;
  Port_connection() {FRITS_SET_CLASS("Port_connection");}
public:
  Port_connection(Datapath_element* dpelem, int number)
  {
    FRITS_SET_CLASS("Port_connection");
    assert(dpelem != NULL && number >= 0);
    datapath_element = dpelem;
    port_num = number;
    port_name = NULL;
  }
  Port_connection(Datapath_element* dpelem, const char* name)
  {
    FRITS_SET_CLASS("Port_connection");
    assert(dpelem != NULL && name != NULL);
    datapath_element = dpelem;
    port_name = strdup(name);
    port_num = -1;
  }
  ~Port_connection() {if (port_name) free(port_name);}
  Datapath_element *get_datapath_element(void) {return datapath_element;}
  int get_port_number(void) {return port_num;}
  const char* get_port_name(void)	{return port_name;}	
};

typedef class Port_connection PORTCONN;
typedef class Port_connection *PORTCONNPTR;

#ifdef _SCALP_
class Net : public Assurance {
#else
class Net {
#endif
private:
  List_ar<PORTCONNPTR> port_connections;
  char* name;
  int bitwidth;
  //can later add data members indicating wire length/capacitance, etc.

public:
  Net() : port_connections() {
    FRITS_SET_CLASS("Net");
    name = NULL;
    bitwidth = 1;
  }
  // A constructor with a name
  Net(const char* n, int bw = 1) : port_connections() {
    FRITS_SET_CLASS("Net");
    name = strdup(n);
    assert(bw > 0);
    bitwidth = bw;
  }
  ~Net() {
    if (name) free(name);
    if(!port_connections.is_empty()) {
      disconnect_from_all_ports();
    }
  }

  char* get_name(void) {return name;}
  int get_bitwidth() {return bitwidth;}
  void set_bitwidth(int bw)  {assert(bw > 0); bitwidth = bw;}
  virtual const char* get_id(void) {return "Net";};
  List_ar<PORTCONNPTR> &get_port_connections(void) {return port_connections;}
  inline int number_of_connections(void) {return port_connections.get_size();}

  Net* set_name(const char* n) {if (name) free(name); name = strdup(n); return this;}

  // WARNING: The connect/disconnect by port number messages DO NOT WORK
  // FOR Datapath_elements with "virtual" ports e.g. Muxes, etc.
  void connect_to_port(Datapath_element *dpelem, int port_num) ;
  void disconnect_from_port(Datapath_element *dpelem, int port_num);
  void connect_to_port(Datapath_element *dpelem, const char* port_name) ;
  void disconnect_from_port(Datapath_element *dpelem, const char* port_name);
  void disconnect_from_all_ports();

  //Whether the net is connected to a given port
  Boolean is_connected_by_name(Datapath_element *dpelem, const char* port_name);
  //Boolean is_connected_by_order(Datapath_element *dpelem, int port_num) {};

  //Whether this net is connected to any input port of a given datapath element
  Boolean is_connected_to_input_port_by_name(Datapath_element *dpelem);

  //Return the datapath element that has its OUT port connected to this net.
  //Asserts that only 1 such datapath element exists
  Datapath_element *get_driver(void);
#ifdef PHYSICAL
  void get_load(List_ar<Datapath_element *> & load) const;  
#endif  

};

/*Iterator to walk thro all the elements connected to a net
 *net is of type class Net, netelemscan is of type List_iterator<PORTCONNPTR>
 */
#define FOR_EACH_NETELEMENT(net, netelemscan) for((netelemscan).start((net).get_port_connections()); (netelemscan).not_done(); (netelemscan).increment())
/********************************************************************/
#endif /* ANAND_NET_H */
