/*  Copyright (c) 2000-2001 by Alternative System Concepts, Inc.  */
/*  All Rights Reserved                                           */
/*                                                                */
/*  THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF                */
/*       Alternative System Concepts, Inc.                        */
/*  The copyright notice above does not evidence any              */
/*  actual or intended publication of such source code.           */

///  <HEADER>
///     <PURPOSE>
///              
///     </PURPOSE>
///              
///  </HEADER>

#include "dfg.h"
#include "stack.h"
#include "dpunit.h"

/********************************************************************/
void Dfg::display(ostream & output)
{
  register int i;
  register NODEPTR node;
  register EDGEPTR edge;

  FRITS_SET_MESSAGE("display");

  output << "CDFG: " << this << endl;
  output << "  name = " << name << endl;
  output << "  number of inputs = " << inputs.get_size() << endl;
  output << "  number of outputs = " << outputs.get_size() << endl;
  output << "  number of loopins = " << loopins.get_size() << endl;
  output << "  number of loopouts = " << loopouts.get_size() << endl;
  output << "  number of constants = " << constants.get_size() << endl;

  //display the names of the inputs
  output << "  inputs: ";
  display_edgelist(inputs, output);

  //display the names of the outputs
  output << "  outputs: ";
  display_edgelist(outputs, output);

  //display the names of the loopins
  output << "  loopins: ";
  display_edgelist(loopins, output);

  //display the names of the loopouts
  output << "  loopouts: ";
  display_edgelist(loopouts, output);

  //display the names of the constants
  output << "  constants: ";
  display_edgelist(constants, output);

  //write out the nodecount;
  output << "  nodecount: " << nodecount << endl;
  //write out the edgecount
  output << "  edgecount: " << edgecount << endl;  
  //write out the array of all nodes
  output << "  nodes: " << nodes;
  //write out the CDFG edges
  output << "  edges: " << edges;

  
  FOR_EACH_NODEINDEX(*this, i) {
    node = get_nthnode(i);
    assert(node);
    node->display(output);
  }

  FOR_EACH_EDGEINDEX(*this, i) {
    edge = get_nthedge(i);
    assert(edge);
    edge->display(output);
  }

  
  if(levellists.get_size() > 0) {
    for(i = 0; i <= maxlevel; i++) {
      cout << "  Levellists[" << i << "]: " << levellists[i] << endl;
    }
  }

  return;
}

/********************************************************************/
void Dfg::display_schedule(ostream &output)
{
  register int i;
  register NODEPTR node;
  register EDGEPTR edge;

  FRITS_SET_MESSAGE("display_schedule");

  output << "CDFG: " << (void *)this << " scheduling info." << endl;
  for(i = 0; i < nodecount; i++) {
    node = nodes[i];
    assert(node);
    output << "  node: " <<node->get_name()<<" "<< node->get_address()
	   << " birth: " << node->get_birth()
	   << " death: " << node->get_death() << endl;
  }
  output << endl;
  for(i = 0; i < edgecount; i++) {
    edge = edges[i];
    assert(edge);
    output << "  edge: " << (*edge)
	   << " birth: " << edge->get_birth()
	   << " death: " << edge->get_death() << endl;
  }
  output << endl;
  return;
}

/********************************************************************/
void Dfg::display_allocation(ostream &output)
{
  register int i;
  register NODEPTR node;
  register EDGEPTR edge;

  FRITS_SET_MESSAGE("display_allocation");

  output << "CDFG: " << (void *)this << " allocation info." << endl;
  for(i = 0; i < nodecount; i++) {
    node = nodes[i];
    assert(node);
    output << "  node: " << node->get_address()
	   << " module type: " << node->get_moduletype()
	   << " fu: " << node->get_functional_unit() << endl;
  }
  output << endl;
 
  for(i = 0; i < edgecount; i++) {
    edge = edges[i];
    assert(edge);
    output << "  edge: " << (*edge)
	   << " storage: " << edge->get_storage_unit() << endl;
  }
  output << endl;

  return;
}
/********************************************************************/

void Dfg::print_vcg(char *vcg_filename)
{
  int i;
  EDGEPTR edge, edge1;
  NODEPTR node;
  char source_node_name[MAXSTRLEN];
  List_iterator<NODEPTR> nodescan;
  List_iterator<EDGEPTR> edgescan;
  ofstream output;

  assert(vcg_filename);

  output.open(vcg_filename, ios::out);
  if(!output) {
    cerr << "ERROR: Could not open file " << vcg_filename << " for output" << endl;
    exit(-1);
  }

  output << "//VCG input file describing a CDFG" << endl;
  output << "//Automatically generated by SCALP" << endl;
  output << endl;
  output << "graph: { title: \"" << name << "\"" << endl;

  output << "layoutalgorithm: dfs" << endl;
  output << "finetuning: no" << endl;
  output << "splines: yes" << endl;
  output << "display_edge_labels: yes" << endl;
  output << "node.borderwidth: 2" << endl;
  output << "node.textcolor: magenta" << endl;
  output << "edge.color: black" << endl;

  //create the operation nodes
  for(i = 0; i <= maxlevel; i++) {
    FOR_EACH_LISTNODE(levellists[i], nodescan) {
      node = nodescan.get_item();
      assert(node);
      //print out the node
      output << "node: { title: \"" << node->get_name()
	     << "\" vertical_order: " << i+1 << " shape: ellipse "
	     << "color: lightblue bordercolor: darkblue }" << endl;
    }
  }

  //Create PI nodes
  FOR_EACH_LISTNODE(inputs, edgescan) {
    edge = edgescan.get_item();
    assert(edge && edge->is_pi());
    output << "node: {title: \"" << edge->get_name() << "_PI"
	     << "\" vertical_order: " << 0 << " shape: triangle "
	     << "color: lightgreen bordercolor: darkgreen }" << endl;
  }

  //Create CONSTANT nodes
  FOR_EACH_LISTNODE(constants, edgescan) {
    edge = edgescan.get_item();
    assert(edge && edge->is_constant());
    output << "node: {title: \"" << edge->get_name() << "_CONST"
	     << "\" vertical_order: " << 0 << " shape: triangle "
	     << "color: lightgreen bordercolor: darkgreen }" << endl;
  }

 
  FOR_EACH_LISTNODE(loopouts, edgescan) {
    edge = edgescan.get_item();
    assert(edge && edge->is_loopout());
    output << "node: {title: \"" << edge->get_name() << "_delay"
	     << "\" vertical_order: " << 0 << " shape: box "
	     << "color: lightgreen bordercolor: darkgreen }" << endl;
  }

  //Create PO nodes
  FOR_EACH_LISTNODE(outputs, edgescan) {
    edge = edgescan.get_item();
    assert(edge && edge->is_po());
    if(!(edge && edge->is_po())) {
      fflush(stdout);
    }
    output << "node: {title: \"" << edge->get_name() << "_PO"
	     << "\" vertical_order: " << maxlevel+2 << " shape: triangle "
	     << "color: lightgreen bordercolor: darkgreen }" << endl;
  }

  //Print out the forward edges (all except delay edges)
  FOR_EACH_EDGEINDEX(*this, i) {
    edge = get_nthedge(i);
    assert(edge);
    //get the source node of the edge.
    if(edge->is_pi()) {
      sprintf(source_node_name, "%s_PI", edge->get_name());
    } else if(edge->is_loopin()) {
      edge1 = edge->get_loopout_link();
      assert(edge1 && edge1->is_loopout() &&
	     edge1->get_loopin_link() == edge);
      sprintf(source_node_name, "%s_delay", edge1->get_name());
    } else if(edge->is_constant()) {
      sprintf(source_node_name, "%s_CONST", edge->get_name());
    } else {
      node = edge->input_node();
      assert(node);
      sprintf(source_node_name, "%s", node->get_name());
    }

    FOR_EACH_FANOUT_NODE(edge, nodescan) {
      node = nodescan.get_item();
      assert(node);
      //print out the arc from the source node to node
      output << "edge: { sourcename: \"" << source_node_name
	     << "\" targetname: \"" << node->get_name()
	     << "\" label: \"" << edge->get_name()
	     << "\" color: black }" << endl;
    }
    if(edge->is_po()) {
      //print out an arc from the source node to the "outer world"
      output << "edge: { sourcename: \"" << source_node_name
	     << "\" targetname: \"" << edge->get_name() << "_PO"
	     << "\" label: \"" << edge->get_name()
	     << "\" color: black }" << endl;
    }
  }

  //Print out the delay edges
  FOR_EACH_LISTNODE(loopouts, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    //get the source node of edge.
    if(edge->is_pi()) {
      sprintf(source_node_name, "%s_PI", edge->get_name());
    } else if(edge->is_loopin()) {
      edge1 = edge->get_loopout_link();
      assert(edge1 && edge1->is_loopout() &&
	     edge1->get_loopin_link() == edge);
      sprintf(source_node_name, "%s_delay", edge1->get_name());
    } else if(edge->is_constant()) {
      sprintf(source_node_name, "%s_CONST", edge->get_name());
    } else {
      node = edge->input_node();
      assert(node);
      sprintf(source_node_name, "%s", node->get_name());
    }
    //print out a backward edge from the node generating the loopin
    //to the delay node
    output << "backedge: { sourcename: \"" << source_node_name
	   << "\" targetname: \"" << edge->get_name() << "_delay"
	   << "\" label: \"" << edge->get_name()
	   << "\" color: black }" << endl;
  }

  output << "}" << endl;
  return;
}
/********************************************************************/
void Dfg::read(char *filename)
{
  int dfg_parse_error;
  extern FILE *dfg_in;

  FRITS_SET_MESSAGE("read");

  assert(filename);
  dfg_in = anand_fopen(filename, "r");
  cout << "*" << endl << "*  Reading CDFG from file \""
    << filename << "\"" << endl;
  dfg_parse_error = dfg_parse();
  if(dfg_parse_error) {
    cerr << "Errors occurred while parsing CDFG from file \""
      << filename << "\"" << endl;
    exit(-1);
  }
  cout << "*" << endl;
  return;
}

/********************************************************************/

NODEPTR Dfg::search_node(const char *nodename)
{
  NODEPTR retptr = NODENULL;
  register int i;

  FRITS_SET_MESSAGE("search_node");

  assert(nodename);
  for(i = 0; i < nodecount; i++) {
    assert(nodes[i]->get_address() == i);
    if(!strcmp(nodename, nodes[i]->get_name())) {
      retptr = nodes[i];
    }
  }
  return(retptr);
}

/********************************************************************/

EDGEPTR Dfg::search_edge(const char *edgename)
{
  EDGEPTR retptr = EDGENULL;
  register int i;

  FRITS_SET_MESSAGE("search_edge");

  assert(edgename);
  FOR_EACH_EDGEINDEX(*this, i) {
    assert(edges[i]->address == i);
    if(!strcmp(edgename, edges[i]->get_name())) {
      retptr = edges[i];
    }
  }
  return(retptr);
}

EDGEPTR Dfg::search_edge(EDGEPTR edgeptr)
{
  register int i;

  FRITS_SET_MESSAGE("search_edge");

  assert(edgeptr);
  FOR_EACH_EDGEINDEX(*this, i) {
    assert(edges[i]->address == i);
    if(edgeptr == edges[i]) {
      return edgeptr;
    }
  }
  return(NULL);
}

/********************************************************************/

EDGEPTR Dfg::search_input(const char *edgename)
{
  EDGEPTR edge, retptr = EDGENULL;
  List_iterator<EDGEPTR> edgescan;

  FRITS_SET_MESSAGE("search_input");

  assert(edgename);
  FOR_EACH_LISTNODE(inputs, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    if(!strcmp(edgename, edge->get_name())) {
      retptr = edge;
    }
  }
  return(retptr);
}

/********************************************************************/

EDGEPTR Dfg::search_output(const char *edgename)
{
  EDGEPTR edge, retptr = EDGENULL;
  List_iterator<EDGEPTR> edgescan;

  FRITS_SET_MESSAGE("search_output");

  assert(edgename);
  FOR_EACH_LISTNODE(outputs, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    if(!strcmp(edgename, edge->get_name())) {
      retptr = edge;
    }
  }
  return(retptr);
}

/********************************************************************/

EDGEPTR Dfg::search_loopin(const char *edgename)
{
  EDGEPTR edge, retptr = EDGENULL;
  List_iterator<EDGEPTR> edgescan;

  FRITS_SET_MESSAGE("search_loopin");

  assert(edgename);
  FOR_EACH_LISTNODE(loopins, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    if(!strcmp(edgename, edge->get_name())) {
      retptr = edge;
    }
  }
  return(retptr);
}

/********************************************************************/

EDGEPTR Dfg::search_constant(const char *edgename)
{
  EDGEPTR edge, retptr = EDGENULL;
  List_iterator<EDGEPTR> edgescan;

  FRITS_SET_MESSAGE("search_constant");

  assert(edgename);
  FOR_EACH_LISTNODE(constants, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    if(!strcmp(edgename, edge->get_name())) {
      retptr = edge;
    }
  }
  return(retptr);
}

/********************************************************************/
Dfg *Dfg::add_node(NODEPTR node)
{
  register int i;

  FRITS_SET_MESSAGE("add_node");

  assert(node);
 
  assert(node->get_name());
  assert(node->get_func() != NONE);
  assert(nodecount >= 0);
  assert(!nodes[nodecount]);
#ifndef NDEBUG
  for(i = 0; i < nodecount; i++) {
    assert(nodes[nodecount] != node);
  }
#endif

  node->address = nodecount;
  nodes[nodecount] = node;
  nodecount++;

  return(this);
}

/********************************************************************/
Dfg *Dfg::add_edge(EDGEPTR edge)
{
  register int i;

  FRITS_SET_MESSAGE("add_edge");

  assert(edge);
 
  assert(edge->get_name());
  assert(!edges[edgecount]);


  for(i = 0; i < edgecount; i++) {
    assert(edges[i] != edge);
  }

  edge->address = edgecount;
  edges[edgecount++] = edge;

  return(this);
}

/********************************************************************/

Dfg *Dfg::add_input(EDGEPTR inedge)
{
  FRITS_SET_MESSAGE("add_input");
  assert(inedge);
  //an edge cant be added multiple times to inputs list
  assert(!inputs.find(inedge)); 
  assert(inedge->is_pi());
  add_edge(inedge);
  inputs.append(inedge);

#ifdef _SCALP_
  defined_operands->st_add(inedge);
#endif

  return(this);
}
/********************************************************************/

Dfg *Dfg::add_output(EDGEPTR outedge)
{
  FRITS_SET_MESSAGE("add_output");
  assert(outedge);
  EDGEPTR edge_found = NULL;
  //an edge cant be added multiple times to outputs list
  assert(!outputs.find(outedge)); 
  assert(outedge->is_dfgoutput());
  if ( !(edge_found = search_edge(outedge)) )
    add_edge(outedge);
  else
    edge_found->set_po();
  outputs.append(outedge);
  return(this);
}
/********************************************************************/

Dfg *Dfg::add_constant(EDGEPTR constedge)
{
  FRITS_SET_MESSAGE("add_constant");
  assert(constedge);
  //check for constants multiply defined
  assert(!constants.find(constedge));

  assert(constedge->is_constant());
  add_edge(constedge);
  constants.append(constedge);

#ifdef _SCALP_
  defined_operands->st_add(constedge);
#endif

  return(this);
}

/********************************************************************/

Dfg *Dfg::add_loopin(EDGEPTR edge)
{
  FRITS_SET_MESSAGE("add_loopin");
  assert(edge);
  //check for loopins multiply defined
  assert(!loopins.find(edge));

  assert(edge->is_loopin());
  add_edge(edge);
  loopins.append(edge);
#ifdef _SCALP_
  defined_operands->st_add(edge);
#endif

  return(this);
}

/********************************************************************/
#ifdef _SCALP_
  
  Dfg* Dfg::add_symbol_substitution(const char* name, EDGEPTR edge) {
    defined_operands->st_add(edge, name);
    return this;
  }
#endif
/********************************************************************/

void Dfg::reset_scheduling_info() {
  register int i;
  register NODEPTR node;
  register EDGEPTR edge;

  FRITS_SET_MESSAGE("reset_scheduling_info");

  //initialize all node and edge birth and death times to -1
  for(i = 0; i < nodecount; i++) {
    node = nodes[i];
    assert(node);
    node->set_birth(-1);
    node->set_death(-1);
  }
  FOR_EACH_EDGEINDEX(*this, i) {
    edge = edges[i];
    assert(edge);
    edge->set_birth(-1);
    edge->set_death(-1);
  }
  return;
}

/********************************************************************/

void Dfg::reset_allocation_info() {
  register int i;
  register NODEPTR node;
  register EDGEPTR edge;

  FRITS_SET_MESSAGE("reset_allocation_info");

  //initialize all node moduletype to -1, and
  //all node fu and register reg fields to NULL
  for(i = 0; i < nodecount; i++) {
    node = nodes[i];
    assert(node);
    node->set_moduletype(-1);
    node->set_functional_unit(NULL);
  }
  FOR_EACH_EDGEINDEX(*this, i) {
    edge = edges[i];
    edge->set_storage_unit(NULL);
  }
  return;
}

/********************************************************************/

void Dfg::levelize() {
  int max;
  register int i;
  int num_pi_fanins;
  NODEPTR node, innode, outnode;
  EDGEPTR edge, outedge, ctrledge;
  List_iterator<NODEPTR> nodescan;
  List_iterator<EDGEPTR> edgescan;
  Stack<NODEPTR> dfsstack(nodecount);
  Array<int> indegrees(nodecount);

  FRITS_SET_MESSAGE("levelize");

  //if necessary, allocate the levellists, and create them
  if(levellists.get_size() != nodecount) {
    levellists.resize(nodecount);
  }
  for(i = 0; i < nodecount; i++) {
    levellists[i].clear();
  }

  //set the indegrees of each node to the number of non-{pi,loopin,constant}
  //fanins of the node. Also push level 0 nodes onto the Stack
  for(i = 0; i < nodecount; i++) {
    node = nodes[i];
    assert(node);
    num_pi_fanins = 0;
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      if(edge->is_dfginput()) {
	num_pi_fanins++;
      }
    }
    ctrledge = node->get_control_edge();
    if ((ctrledge) && (ctrledge->is_dfginput())) num_pi_fanins++;
    if(num_pi_fanins == node->number_all_input_edges()) {
      node->set_level(0);
      indegrees[i] = 0;
      levellists[0].append(node);
      dfsstack.push(node);
    } else {
      indegrees[i] = node->number_all_input_edges() - num_pi_fanins;
      node->set_level(-1);
    }
  }
  
  while(dfsstack.is_empty() == F) {
    node = dfsstack.pop();
    assert(node->get_level() >= 0);
    outedge = node->get_output_edge();
    assert(outedge);
    FOR_EACH_FANOUT_NODE(outedge, nodescan) {
      outnode = nodescan.get_item();
      assert(outnode->get_level() == -1);
      indegrees[outnode->get_address()]--;
      if(indegrees[outnode->get_address()] == 0) {
	//mark the level of outnode
	max = 0;
	FOR_EACH_FANIN_EDGE(outnode, edgescan) {
	  edge = edgescan.get_item();
	  innode = edge->input_node();
	  if(innode) {
#ifdef Isad_debug      
      if(indegrees[innode->get_address()] != 0) {
        cout <<" The start node is: " << node->get_address() << endl;
        cout <<" The innode is:  " << innode->get_address() << endl;
        cout <<" The output edge of this innode is: "<< edge->get_address() <<endl;
        cout <<" The indegree of this innode is:  " 
             << indegrees[innode->get_address()] << endl;
        cout <<" The outnode of this innode is: " << outnode->get_address() << endl;
      }
#endif
	    assert(indegrees[innode->get_address()] == 0);
            assert(innode->get_level() >= 0);
	    max = MAX(max, innode->get_level());
	  }
	}
	ctrledge = outnode->get_control_edge();
	if(ctrledge) {
	  innode = ctrledge->input_node();
	  if(innode) {
	    assert(indegrees[innode->get_address()] == 0 && innode->get_level() >= 0);
	    max = MAX(max, innode->get_level());
	  }
	}
   	outnode->set_level(max+1);
	levellists[max+1].append(outnode);
#ifdef Isad_debug
   cout << " The grouped node is:  " << outnode->get_address() << endl;
   cout << " Now the current level is:  " << max+1 << endl;
#endif
	dfsstack.push(outnode);
      }
    }
  }
  for(i = 0; i < nodecount; i++) {
    node = nodes[i];
    assert(node);
    maxlevel = MAX(maxlevel, node->get_level());
  }
  //this->display();
#ifndef NDEBUG
  {
    int levelcount = 0;
    assert(dfsstack.is_empty());
    assert(levellists.get_size() == nodecount);
    for(i = 0; i < nodecount; i++) {
      assert((i <= maxlevel && levellists[i].get_size() > 0) ||
	     (levellists[i].get_size() == 0));
      //assert that each node in levellists[i] has a level of i
      levelcount += levellists[i].get_size();
    }
#ifdef ISad_debug 
  for( i = 0; i < nodecount; i++)
    cout <<"The " << i << "th level has " <<
           levellists[i].get_size() << "nodes."<<endl;
#endif    
  assert(levelcount == nodecount);
  }
#endif
  
  return;
}

/********************************************************************/

void Dfg::simulate(List_ar<unsigned int> &vector, List_ar<unsigned int> &feedback_values,
		   Array<unsigned int> &values, const Boolean reset_loopins)
{
  register EDGEPTR edge;
  register NODEPTR node;
  unsigned int val;
  register int level;
  List_iterator<unsigned int> uintscan;
  List_iterator<EDGEPTR> edgescan;
  List_iterator<NODEPTR> nodescan;

  FRITS_SET_MESSAGE("simulate");

  
  assert(levellists.get_size() == nodecount);
  if(vector.get_size() != inputs.get_size()) {
    cerr << "ERROR: too few/too many values supplied in given vector"
      << endl;
    exit(-1);
  }
  assert_force(feedback_values.get_size() == loopins.get_size());
  assert_force(values.get_size() == edgecount);

  //setup the input values from the vector
  for(edgescan.start(inputs),uintscan.start(vector);
      edgescan.not_done() && uintscan.not_done();
      uintscan.increment(),edgescan.increment()) {
    val = uintscan.get_item();
    edge = edgescan.get_item();
    assert(edge);
    assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
    assert(edge->is_pi());
    values[edge->get_address()] = val & ((1 << bitwidth)-1);
  }
  //setup the constant values
  FOR_EACH_LISTNODE(constants, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
    assert(edges[edge->get_address()] == edge);
    assert(edge->is_constant());
    values[edge->get_address()] = edge->get_value() & ((1 << bitwidth)-1);
  }
  //setup loopin values
  if(reset_loopins) {
    FOR_EACH_LISTNODE(loopins, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(edges[edge->get_address()] == edge);
      assert(edge->is_loopin());
      values[edge->get_address()] = edge->get_value() & ((1 << bitwidth)-1);
    }
  } else {
    //setup the loopin values from the stored loopout values
    for(edgescan.start(loopins),uintscan.start(feedback_values);
	edgescan.not_done() && uintscan.not_done();
	edgescan.increment(),uintscan.increment()) {
      val = uintscan.get_item();
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(edge->is_loopin());
      values[edge->get_address()] = val & ((1 << bitwidth)-1);
    }
  }

  //Perform a level-by-level evaluation of the operations
  for(level = 0; level <= maxlevel; level++) {
    FOR_EACH_LISTNODE(levellists[level], nodescan) {
      node = nodescan.get_item();
      assert(node);
      assert(node->get_address() >= 0 && node->get_address() < nodecount);
      assert(nodes[node->get_address()] == node);
      evaluate_node(node,values);
    }
  }

  //Update the feedback_values list
  for(edgescan.start(loopouts),uintscan.start(feedback_values);
      edgescan.not_done() && uintscan.not_done();
      edgescan.increment(),uintscan.increment()) {
    edge = edgescan.get_item();
    assert(edge);
    assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
    //assert that the edge is a loopout
    val = values[edge->get_address()];
    assert(!(values[edge->get_address()] & ~((1 << bitwidth) - 1)));
    uintscan.set_item(val);
  }
  
  return;
}
/********************************************************************/

void Dfg::evaluate_node(NODEPTR node, Array<unsigned int> &values)
{
  register EDGEPTR edge;
  List_iterator<EDGEPTR> edgescan;
  unsigned int result;
  int i, temp;

  FRITS_SET_MESSAGE("evaluate_node");

  assert(node);
  assert(node->get_address() >= 0 && node->get_address() < nodecount);
  assert(node->number_output_edges() == 1);
  assert(node->number_input_edges() >= 1 && node->number_input_edges() <= 2);

  switch(node->get_func()) {
  case PLUS_:
    result = 0;
    assert_force(node->number_input_edges() == 2);
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(!(values[edge->get_address()] & ~((1 << bitwidth) - 1)));
      result += values[edge->get_address()];
    }
    break;
  case MULT_:
    result = 1;
    assert_force(node->number_input_edges() == 2);
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(!(values[edge->get_address()] & ~((1 << bitwidth) - 1)));
      result *= values[edge->get_address()];
    }
    break;
  case MINUS_:
    result = 0;
    i = 0;
    assert_force(node->number_input_edges() == 2);
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(!(values[edge->get_address()] & ~((1 << bitwidth) - 1)));
      if(i) {
	result -= values[edge->get_address()];
      } else {
	result += values[edge->get_address()];
      }
       i++;
    }
    break;
  case DEV_:
    result = 1;
    i = 0;
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(!(values[edge->get_address()] & ~((1 << bitwidth) - 1)));
      if(i) {
	result /= values[edge->get_address()];
      } else {
	result *= values[edge->get_address()];
      }
      i++;
    }
    break;
  case AND_:
    assert_force(node->number_input_edges() == 2);
    result = (1 << bitwidth) - 1;
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(!(values[edge->get_address()] & ~((1 << bitwidth) - 1)));
      result &= values[edge->get_address()];
    }
    break;
  case OR_:
    assert_force(node->number_input_edges() == 2);
    result = 0;
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(!(values[edge->get_address()] & ~((1 << bitwidth) - 1)));
      result |= values[edge->get_address()];
    }
    break;
  case XOR_:
    assert_force(node->number_input_edges() == 2);
    result = 0;
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(!(values[edge->get_address()] & ~((1 << bitwidth) - 1)));
      result ^= values[edge->get_address()];
    }
    break;
  case NOT_:
    assert_force(node->number_input_edges() == 1);
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(!(values[edge->get_address()] & ~((1 << bitwidth) - 1)));
      result = ~(values[edge->get_address()]);
    }
    break;
  case GREAT_:
    temp = 0;
    i = 0;
    assert_force(node->number_input_edges() == 2);
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(!(values[edge->get_address()] & ~((1 << bitwidth) - 1)));
      if(i) {
	temp -= values[edge->get_address()];
      } else {
	temp += values[edge->get_address()];
      }
      i++;
    }
    result = (temp > 0);
    break;

  case LESS_:
    temp = 0;
    i = 0;
    assert_force(node->number_input_edges() == 2);
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(!(values[edge->get_address()] & ~((1 << bitwidth) - 1)));
      if(i) {
	temp -= values[edge->get_address()];
      } else {
	temp += values[edge->get_address()];
      }
      i++;
    }
    result = (temp < 0);
    break;

  case EQUAL_:
    temp = 0;
    i = 0;
    assert_force(node->number_input_edges() == 2);
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(!(values[edge->get_address()] & ~((1 << bitwidth) - 1)));
      if(i) {
	temp -= values[edge->get_address()];
      } else {
	temp += values[edge->get_address()];
      }
      i++;
    }
    result = (temp == 0);
    break;

  case LW_:
    temp = 0;
    i = 0;
    assert_force(node->number_input_edges() == 2);
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(!(values[edge->get_address()] & ~((1 << bitwidth) - 1)));
      if(!i)
	temp += values[edge->get_address()];
      i++;
    }
    result = (temp < 0);
    break;
    
  case SW_:
    temp = 0;
    i = 0;
    assert_force(node->number_input_edges() == 2);
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
      assert(!(values[edge->get_address()] & ~((1 << bitwidth) - 1)));
      if(!i)
	temp += values[edge->get_address()];
      i++;
    }
    result = (temp < 0);
    break;



  default:
    cerr << "ERROR: unknown/unsupported node type encountered" << endl;
    exit(-1);
    break;
  }



  result &= (1 << bitwidth) - 1;
  edge = node->get_output_edge();
  assert(edge);
  assert(edge->get_address() >= 0 && edge->get_address() < edgecount);
  values[edge->get_address()] = result;
}
/********************************************************************/
void Dfg::display_values(Array<unsigned int> &values, ofstream &output)
{
  register int i;
  register EDGEPTR edge;

  FRITS_SET_MESSAGE("display_values");

  assert(values.get_size() == edgecount);

  output << endl << "CDFG Values:" << endl;
  for(i = 0; i < edgecount; i++) {
    edge = edges[i];
    assert(edge && edge->get_address() == i);
    output << " " << (*edge) << ": " << values[i] << endl;
  }
  output << "------------------------------------------" << endl;

  return;
}
/********************************************************************/
void Dfg::copy_schalloc_info(Schalloc_info &info_source)
{
  register int i;
  register NODEPTR node;
  register EDGEPTR edge;

  FRITS_SET_MESSAGE("reset_schalloc_info");

  assert(info_source.nodebirths.get_size() == nodecount);
  assert(info_source.nodedeaths.get_size() == nodecount);
  assert(info_source.moduletypes.get_size() == nodecount);
  assert(info_source.functional_units.get_size() == nodecount);
  assert(info_source.edgebirths.get_size() == edgecount);
  assert(info_source.edgedeaths.get_size() == edgecount);
  assert(info_source.storage_units.get_size() == edgecount);

  for(i = 0; i < nodecount; i++) {
    node = nodes[i];
    assert(node && node->get_address() == i);
    node->set_birth(info_source.nodebirths[i]);
    node->set_death(info_source.nodedeaths[i]);
    node->set_moduletype(info_source.moduletypes[i]);
    node->set_functional_unit(info_source.functional_units[i]);
  }

  for(i = 0; i < edgecount; i++) {
    edge = edges[i];
    assert(edge && edge->get_address() == i);
    edge->set_birth(info_source.edgebirths[i]);
    edge->set_death(info_source.edgedeaths[i]);
    edge->set_storage_unit(info_source.storage_units[i]);
  }
  return;
}
/********************************************************************/
#ifdef _SCALP_

Dfg* Dfg::add_lhs_edge(Dfgedge* lhs_edge)
{
  FRITS_SET_MESSAGE("add_lhs_edge");
  add_edge(lhs_edge);
  defined_operands->st_add(lhs_edge);
  return(this);
}
#endif
/********************************************************************/
#ifdef _SCALP_
void Dfg::create_loopouts(void)
{
  EDGEPTR loopout_edge = NULL;
  List_iterator<EDGEPTR> edgescan;
  EDGEPTR loopin_edge;

  FOR_EACH_LISTNODE(loopins, edgescan) {
    loopin_edge = edgescan.get_item();
    assert(loopin_edge);
    //Search for the last assignment to the same variable - the loopout
    
    defined_operands->st_search_local(loopin_edge->get_name(), (FRITS_object*&)loopout_edge);
    if (loopout_edge) {
      
      loopin_edge->set_loopout_link(loopout_edge);
      loopout_edge->set_loopin_link(loopin_edge);
      assert(!loopouts.find(loopout_edge));
      loopouts.append(loopout_edge);
    } else if ( loopin_edge->get_value() != -1 ) {
      
      loopins.remove(loopin_edge);
      loopin_edge->inputtype = CONSTANT_;
      assert(!constants.find(loopin_edge));
      constants.append(loopin_edge);
    } else {
      // undefined variable on the LHS
      error("Undefined primary %s found", loopin_edge->get_name());
    }
  }
  return;
}
#endif
/********************************************************************/
void Schalloc_info::extract_info(Dfg &source)
{
  register int i;
  register NODEPTR node;
  register EDGEPTR edge;

  FRITS_SET_MESSAGE("extract_info");
  for(i = 0; i < source.numnodes(); i++) {
    node = source.get_nthnode(i);
    assert(node);
    nodebirths[i] = node->get_birth();
    nodedeaths[i] = node->get_death();
    moduletypes[i] = node->get_moduletype();
    functional_units[i] = node->get_functional_unit();
  }

  for(i = 0; i < source.numedges(); i++) {
    edge = source.get_nthedge(i);
    assert(edge);
    edgebirths[i] = edge->get_birth();
    edgedeaths[i] = edge->get_death();
    storage_units[i] = edge->get_storage_unit();
  }
  return;
}
/********************************************************************/
void Schalloc_info::display(ostream &output)
{
  FRITS_SET_MESSAGE("display");
  output << "Scheduling & allocation info: " << (void *) this << endl;
  output << "nodebirths: " << nodebirths;
  output << "nodedeaths: " << nodedeaths;
  output << "moduletypes: " << moduletypes;
  output << "functional_units: " << functional_units;
  output << "edgebirths: " << edgebirths;
  output << "edgedeaths: " << edgedeaths;
  output << "registers: " << storage_units;
  output << endl;
  return;
}
/********************************************************************/
void Dfg::clear()
{
  int i;

  inputs.clear();
  outputs.clear();
  loopins.clear();
  loopouts.clear();
  constants.clear();
  for(i = 0; i < nodecount; i++) {
    if(nodes[i]) {
      delete nodes[i];
      nodes[i] = (NODEPTR)NULL;
    }
  }
  for(i = 0; i < edgecount; i++) {
    if(edges[i]) {
      delete edges[i];
      edges[i] = (EDGEPTR)NULL;
    }
  }

  bitwidth = 0;

  //clear the levellists, and set maxlevel to -1
  for(i = 0; i < levellists.get_size(); i++) {
    levellists[i].clear();
  }

  maxlevel = -1;

}
/********************************************************************/
void Dfg::copy(Dfg &source_dfg)
{
  int i;
  List_iterator<NODEPTR> nodescan;
  List_iterator<EDGEPTR> edgescan;
  NODEPTR node, copynode;
  EDGEPTR edge, copyedge;

  //clear this CDFG first;
  clear();

  strcpy(name, source_dfg.name);
  nodecount = source_dfg.numnodes();
  edgecount = source_dfg.numedges();
  bitwidth = source_dfg.get_bitwidth();
  clock_period = source_dfg.get_clock_period();

#ifdef _SCALP_
// Copy the defined_operands symbol table
#endif

#ifdef _FAULT_TOLERANCE_
  num_orig_nodes = source_dfg.num_orig_nodes;
  num_orig_edges = source_dfg.num_orig_edges;
#endif

  //create nodes corresponding to those in source_dfg
  for(i = 0; i < source_dfg.numnodes(); i++) {
    node = source_dfg.get_nthnode(i);
    assert(node);
    copynode = new NODE;;
    assert(copynode);
    copynode->copy(*node);
    nodes[i] = copynode;
  }

  //create arrays corresponding to those in source_dfg
  /*for(i = 0; i < source_dfg.numarrays(); i++) {
    array = source_dfg.get_ntharray(i);
    assert(array);
    copyarray = new MEM;;
    assert(copyarray);
    copyarray->copy(*array);
    arrays[i] = copyarray;
  }*/


  //create edges corresponding to those in source_dfg
  for(i = 0; i < source_dfg.numedges(); i++) {
    edge = source_dfg.get_nthedge(i);
    assert(edge);
    copyedge = new EDGE;
    assert(copyedge);
    copyedge->copy(*edge);
    edges[i] = copyedge;
  }
  for(i = 0; i < source_dfg.numnodes(); i++) {
    node = source_dfg.get_nthnode(i);
    assert(node);
    copynode = get_nthnode(i);
    assert(copynode);

    copynode->get_input_edges().clear();
    copynode->get_output_edges().clear();
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      copyedge = get_nthedge(edge->get_address());
      copynode->add_input_edge(copyedge);
    }
    edge = node->get_control_edge();
    if(edge){
      copyedge = get_nthedge(edge->get_address());
      copynode->set_control_edge(copyedge);
    }
    edge = node->get_output_edge();
    copyedge = get_nthedge(edge->get_address());
    copynode->set_output_edge(copyedge);
  }
  for(i = 0; i < source_dfg.numedges(); i++) {
    edge = source_dfg.get_nthedge(i);
    copyedge = get_nthedge(i);
    assert(copyedge);
    assert(edge);

    copyedge->get_sink_nodes().clear();
    copyedge->get_source_nodes().clear();
    FOR_EACH_FANOUT_NODE(edge, nodescan) {
      node = nodescan.get_item();
      copynode = get_nthnode(node->get_address());
      copyedge->add_sink_node(copynode);
    }

    node = edge->input_node();
    assert(node || edge->is_pi() || edge->is_loopin() || edge->is_constant());
    if(node) {
      copynode = get_nthnode(node->get_address());
      copyedge->set_source_node(copynode);
    }
  }
  FOR_EACH_LISTNODE(source_dfg.inputs, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    copyedge = get_nthedge(edge->get_address());
    assert(copyedge);
    inputs.append(copyedge);
  }
  FOR_EACH_LISTNODE(source_dfg.outputs, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    copyedge = get_nthedge(edge->get_address());
    assert(copyedge);
    outputs.append(copyedge);
  }
  FOR_EACH_LISTNODE(source_dfg.loopins, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    copyedge = get_nthedge(edge->get_address());
    assert(copyedge && copyedge->get_loopout_link());
    copyedge->set_loopout_link(get_nthedge(edge->get_loopout_link()->get_address()));
    loopins.append(copyedge);
  }
  FOR_EACH_LISTNODE(source_dfg.constants, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    copyedge = get_nthedge(edge->get_address());
    assert(copyedge);
    constants.append(copyedge);
  }
  FOR_EACH_LISTNODE(source_dfg.loopouts, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    copyedge = get_nthedge(edge->get_address());
    assert(copyedge && copyedge->get_loopin_link());
    copyedge->set_loopin_link(get_nthedge(edge->get_loopin_link()->get_address()));
    loopouts.append(copyedge);
  }
  maxlevel = source_dfg.maxlevel;
  if(levellists.get_size() != nodecount) {
    levellists.resize(nodecount);
  }
  for(i = 0; i < nodecount; i++) {
    levellists[i].clear();
    if (source_dfg.levellists.get_size())	
    FOR_EACH_LISTNODE(source_dfg.levellists[i], nodescan) {
      node = nodescan.get_item();
      assert(node && node->get_address() >= 0 &&
	     node->get_address() < source_dfg.numnodes());
      assert(source_dfg.get_nthnode(node->get_address()) == node);

      copynode = get_nthnode(node->get_address());
      assert(copynode && copynode->get_address() >= 0 &&
	     copynode->get_address() < numnodes());
      assert(get_nthnode(copynode->get_address()) == copynode);
      levellists[i].append(copynode);
    }
  }

  return;
}
/********************************************************************/
void Dfg::remove_delays()
{
  List_iterator<EDGEPTR> edgescan;
  List_iterator<NODEPTR> nodescan;
  NODEPTR outnode;
  EDGEPTR edge, loopout_edge, tmpedge;
  int i, j, old_numedges, earliest_index, tmpcount;

  FOR_EACH_LISTNODE(loopins, edgescan) {
    edge = edgescan.get_item();
    assert(edge && edge->is_loopin());
    loopout_edge = edge->get_loopout_link();
    assert(loopout_edge && loopout_edge->is_loopout() &&
	   loopout_edge->get_loopin_link() == edge);
    //replace all fanouts of edge by loopout_edge;
    FOR_EACH_FANOUT_NODE(edge, nodescan) {
      outnode = nodescan.get_item();
      assert(outnode);
      tmpcount = outnode->get_input_edges().replace_all(edge, loopout_edge);
      assert(tmpcount >= 1);
      loopout_edge->add_sink_node(outnode);
    }
    if(edge->is_loopout()) {
      tmpedge = edge->get_loopin_link();
      assert(tmpedge && tmpedge->is_loopin() &&
	     tmpedge->get_loopout_link() == edge);
      tmpedge->set_loopout_link(loopout_edge);
      loopout_edge->set_loopin_link(tmpedge);
    } else {
      //loopout_edge is no longer a loopout
      loopout_edge->loopout = F;
      loopout_edge->set_loopin_link(NULL);
    }
  }

  old_numedges = numedges();
  //The flowgraph now has no delays, so we can blow away the loopins
  //and loopouts
  FOR_EACH_LISTNODE(loopins, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    i = edge->get_address();
    delete edge;
    edges[i] = NULL;
    edgecount--;
  }
  loopins.clear();

#ifndef NDEBUG
  FOR_EACH_LISTNODE(loopouts, edgescan) {
    edge = edgescan.get_item();
    assert(!edge->is_loopout());
  }
#endif
  loopouts.clear();

  //re-organize the edges array
  for(i = numedges(); i < old_numedges; i++) {
    if(edges[i] != NULL) {
      edge = edges[i];

      earliest_index = -1;
      for(j = 0; j < numedges(); j++) {
	if(edges[j] == NULL) {
	  earliest_index = j;
	  break;
	}
      }
      assert_force(earliest_index >= 0 && earliest_index < numedges());
      edges[earliest_index] = edge;
      edge->address = earliest_index;
      edges[i] = NULL;
    }
  }

  return;
}

//The following routine does a cdfg check to make sure
// the cdfg struct is legal

void Dfg::check()
{
  register int i;
  register NODEPTR node;
  register EDGEPTR edge;

  FRITS_SET_MESSAGE("check");

  FOR_EACH_EDGEINDEX(*this, i) {
    assert(edges[i]->address == i);
    edge = edges[i];
    if(!edge->number_source_nodes()) {
      if(!edge->is_dfginput()) {
        cerr << "The "<<edge->get_address()<<"th edge
             has no source nodes and is not inputs!"
             << endl;
       //  exit(-1);
      } //end if
    } // end if
    if(!edge->number_sink_nodes()) {
      if(!edge->is_dfgoutput()) {
        cerr << "The "<<edge->get_address()<<"th edge
             has no sink nodes and is not outputs!"
             << endl;
        if(edge->number_source_nodes()) {
          cout << "the source node of this edge is: " <<
                  edge->input_node()->get_address() << endl;
          cout <<" The source node perform "<< edge->input_node()->get_func()
               << "operation " << endl;
        } 
        else {
          cout <<"this edge is primary input edge. " << endl;
        }
       // exit(-1);
      } //end if
    } // end if
  } // end FOR_EACH
}



/********************************************************************/
/*Dfg *Dfg::add_array(MEMPTR array)
{
  register int i;

  

  assert(array);
 
  assert(array->get_name());
  assert(arraycount >= 0);
  assert(!arrays[arraycount]);
  
  arrays[arraycount] = array;
  arraycount++;

  return(this);
}

MEMPTR Dfg::get_array(const char *arrayname)
{
  MEMPTR retptr = MEMNULL;
  register int i;

  assert(nodename);
  for(i = 0; i < arraycount; i++) {
    if(!strcmp(arrayname, arrays[i]->get_name())) {
      retptr = arrays[i];
    }
  }
  return(retptr);
}*/
