/*  Copyright (c) 2000-2001 by Alternative System Concepts, Inc.  */
/*  All Rights Reserved                                           */
/*                                                                */
/*  THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF                */
/*       Alternative System Concepts, Inc.                        */
/*  The copyright notice above does not evidence any              */
/*  actual or intended publication of such source code.           */

///  <HEADER>
///     <PURPOSE>
///              
///     </PURPOSE>
///              
///  </HEADER>

/********************************************************************/
/*                FILE: cdfg.h                                      */
/********************************************************************/
#ifndef KAMAL_CDFG_H
#define KAMAL_CDFG_H

#include "dfgnode.h"
#include "dfgedge.h"
#include "matrix.h"
#include "parameter.h"

#ifdef _SCALP_
#include "symtab.h"
#endif

#define FOR_EACH_NODETYPE(graph, i) for((i) = 1; (i) < NUMNODETYPES; (i)++)
#define FOR_EACH_NODEOF_TYPE(graph, type,j) for((j) = 0; (j) < ((graph).typecounts[(type)]); (j)++)

/*Macro to loop through nodes*/
#define FOR_EACH_NODEINDEX(graph, i) for((i) = 0; (i) < ((graph).numnodes()); (i)++)

/*Macros to loop through edges*/
#define FOR_EACH_EDGEINDEX(graph, i) for((i) = 0; (i) < ((graph).numedges()); (i)++)

class Schalloc_info;
typedef class List_iterator<NODEPTR> node_iterator;
typedef class List_iterator<EDGEPTR> edge_iterator;
/********************************************************************/
#ifdef _SCALP_
class Dfg : public Assurance {
#else
class Dfg {
#endif
  friend class Scheduler;
#ifdef PHYSICAL  
  friend class iScheduler;
#endif  
  friend class Retpipe;
  friend class Scm;
  friend class ISAD;

private:
  char name[MAXSTRLEN];                     //name of this Dfg
  Array<NODEPTR> nodes;                     //array of all nodes
  Array<EDGEPTR> edges;                     //all edges in the Dfg
  List_ar<EDGEPTR> inputs;                  //list of primary inputs
  List_ar<EDGEPTR> outputs;                 //list of primary outputs
  List_ar<EDGEPTR> loopins;                 //list of loopins
  List_ar<EDGEPTR> loopouts;                //list of loopouts
  List_ar<EDGEPTR> constants;               //list of constants
  int nodecount;                         //number of nodes
  int edgecount;                         //number of edges
  int bitwidth;                          //bit width of the datapath

  //Additional data structures used for speeding up computation
  Matrix<NODEPTR> nodetable;             //nodes classified by type
  Array<int> typecounts;                 //node counts classified by type
  Array< List_ar<NODEPTR> > levellists;      //levelized array of lists
  int maxlevel;                          //max. level of any node
  float clock_period;
  /*List_ar<MEMPTR> arrays;               //list of memory arrays
  int arraycount;*/	 				    
#ifdef _SCALP_
  Symbol_table* defined_operands;
#endif

#ifdef _FAULT_TOLERANCE_
  int num_orig_nodes;
  int num_orig_edges;
#endif

  int dfg_parse();
  void dfgparse_checknode(NODEPTR, Operator, Boolean);
  void dfgparse_checkedge(EDGEPTR, Boolean);
  void dfgparse_add_inputedge(EDGEPTR, PITYPE);

  void evaluate_node(NODEPTR, Array<unsigned int> &);
  void set_name(const char *newname) {
    assert(newname);
    assert(strlen(newname) < MAXSTRLEN);
    strcpy(name, newname);
  }
  void remove_delays();

public:
  Dfg() :nodes(MAXNUMNODES), edges(MAXNUMEDGES), inputs(), outputs(), loopins(), loopouts(), constants(), nodetable(NUMNODETYPES,MAXSAMETYPENODES), typecounts(NUMNODETYPES), levellists(0) {

    FRITS_SET_CLASS("Dfg");
#ifdef _SCALP_
    mem_ok(defined_operands = new Symbol_table);
#endif
    //make the name the NULL string
    name[0] = '\0';

    //initialize the array of all nodes to all NULL
    nodes.reset((NODEPTR)0);
    //initialize the edge array to all NULL
    edges.reset((EDGEPTR)0);
    //reset node and edge counts
    nodecount = 0;
    edgecount = 0;
    // arraycount = 0;
    //initialize the nodetable to all NULL
    nodetable.reset((NODEPTR)0);
    //reset the typecounts array
    typecounts.reset(0);

    bitwidth = 0;
    maxlevel = -1;
    clock_period = 0.0;
    /*arrays = new List_ar<MEMPTR>;*/
  }

  ~Dfg () {
    register int i, j;

    //free all the nodes
    for(i = 0; i < nodecount; i++) {
      delete nodes[i];
    }
    //free all the edges
    for(i = 0; i < edgecount; i++) {
      delete edges[i];
    }

    //destroy the levellist
    j = levellists.get_size();
    for(i = 0; i < j; i++) {
      levellists[i].clear();
    }
    //delete arrays;

#ifdef _SCALP_
  if (defined_operands)  delete defined_operands;
#endif

  return;
  }

  const char *get_name() {return name;}

  //read the Dfg from a file
  void read(char *);

  //add a NODE to the Dfg
  Dfg *add_node(NODEPTR);

  //add an EDGE to the Dfg
  Dfg *add_edge(EDGEPTR);
  
  /*  Dfg *add_array(MEMPTR array);*/


  //add an input EDGE to the Dfg
  //ASSERTS that the inputtype field of the EDGE is properly set
  Dfg *add_input(EDGEPTR);

  //add an output EDGE to the Dfg
  //ASSERTS that the dfgoutput field of the EDGE is set
  Dfg *add_output(EDGEPTR);

  //add a constant EDGE to the Dfg
  //ASSERTS that the inputtype field of the EDGE is properly set
  Dfg *add_constant(EDGEPTR);

  //ASSERTS that the inputtype field of the EDGE is properly set
  Dfg *add_loopin(EDGEPTR);

#ifdef _SCALP_
  //create loopouts for all loopins
  void create_loopouts(void);
#endif

  //search for a node by name -- linear in the # of nodes
  NODEPTR search_node(const char *);
  
  /*MEMPTR get_array(const char *);*/

  //search for an edge by name -- linear in the # of edges
  EDGEPTR search_edge(const char *);
  //search for an edge by pointer matching -- linear in the # of edges
  EDGEPTR search_edge(EDGEPTR);
  //search for an input edge by name -- linear in the # of inputs
  EDGEPTR search_input(const char *);
  //search for an output by name -- linear in the # of outputs
  EDGEPTR search_output(const char *);
  //search for a loopin by name -- linear in the # of delays
  EDGEPTR search_loopin(const char *);
  //search for a constant by name -- linear in the # of constants
  EDGEPTR search_constant(const char *);

  //search for a node by index -- constant time
  inline const NODEPTR get_nthnode(int index) {
    FRITS_SET_MESSAGE("get_nthnode");
    assert(index >= 0 && index < nodecount);
    assert(nodes[index] && nodes[index]->get_address() == index);
    return nodes[index];
  }
  //search for an edge by index -- constant time
  inline const EDGEPTR get_nthedge(int index) {
    FRITS_SET_MESSAGE("get_nthedge");
    assert(index >= 0 && index < edgecount);
    return edges[index];
  }

  /*inline const MEMPTR get_ntharray(int index) {
    assert(index >= 0 && index < arraycount);
    return arrays[index];
  }*/

  inline const int numnodes() {
    FRITS_SET_MESSAGE("numnodes");
    return nodecount;
  }
  /*inline const int numarrays() {
    return arraycount;
  }*/
  

  inline const int numedges() {
    FRITS_SET_MESSAGE("numedges");
    return edgecount;
  }

  inline const int get_bitwidth() {
    FRITS_SET_MESSAGE("get_bitwidth");
    return bitwidth;
  }
  inline void set_clock_period(float clock) {
    clock_period = clock;
  }
  inline const float get_clock_period() {
    return clock_period;
  }

  // added by wwang in the following
  inline void disable_output(EDGEPTR edge) {
    assert(outputs.find(edge));
    outputs.remove(edge);
  }

  // added by wwang
  inline void disable_loopout(EDGEPTR edge) {
    assert(loopouts.find(edge));
    loopouts.remove(edge);
  }

  // added by wwang
  inline void add_loopout(EDGEPTR edge) {
    if(!edge->is_loopout())
      edge->set_po();
    loopouts.append(edge);
  }

  inline const int get_numcsteps() {
    register int i;
    int maxdeath = 0;
    register EDGEPTR edge;

    FRITS_SET_MESSAGE("get_numcsteps");
    FOR_EACH_EDGEINDEX(*this, i) {
      edge = edges[i];
      assert(edge);
      assert(edge->get_birth() >= 0 && edge->get_death() > edge->get_birth());
      maxdeath = MAX(maxdeath, edge->get_death());
    }
      return(maxdeath);
  }

  //reset the scheduling info associated with each node and edge
  void reset_scheduling_info();

  //reset the module allocation and type info of each node and edge
  void reset_allocation_info();

  //copy scheduling and allocation information from a schalloc_info object
  void copy_schalloc_info(Schalloc_info &);

  //initialize the Cdfg
  void levelize();

  //display the Cdfg
  void display(ostream & = cout);

  //display the scheduling information
  void display_schedule(ostream & = cout);

  //display the module selection and assignment information
  void display_allocation(ostream & = cout);

  //print out the cdfg to a file VCG input format
  void print_vcg(char *);

  //Given an input vector, simulate. The values taken on by
  //variables are stored in the values array
  
  void simulate(List_ar<unsigned int> &, List_ar<unsigned int> &,
		      Array<unsigned int> &, const Boolean);

  //Given an array of values in the CDFG, output the values
  //in proper format - used for DEBUGGING
  void display_values(Array<unsigned int> &, ofstream &);

  //Empty this CDFG
  void clear();

  //Copy this CDFG from another CDFG
  void copy(Dfg &);

  // check whether or not the cdfg is legal
  void check();

#ifdef _FAULT_TOLERANCE_
  //Create a duplicate (copy) of the DFG - signals from 
  //the original and copy are compared for FAULT TOLERANCE
  //If the original set of nodes and edges were numbered 0..n1-1 and 0..n2-1
  //respectively, the set of nodes and edges in the copy are numbered
  //n1..((2*n1)-1) and n2...((2*n2)-1), respectively.
  //NOTE: this function does NOT create any "compare" or "checkpoint"
  //operations - these are done separately.
  void ft_duplicate();

  //Add comparison operations at the POs and LOOPOUTs of a duplicated DFG
  void ft_create_dfgoutput_comparisons();

  //Add comparison operations to a given set of operations
  void ft_add_comparisons(List_ar<int> &);

  //Delete the comparisons at a given set of operations
  void ft_delete_comparisons(List_ar<int> &);

  //Delete all comparisons added to the DFG
  void ft_delete_all_comparisons();

  //Given an operation, retrieve its checkpoint, if any
  const NODEPTR ft_get_checkpoint(NODEPTR);

  //Given a list of operations, separate it into operations from the original and
  //operations from the copy
  void separate_orig_and_copy_operations(const List_ar<NODEPTR> &, List_ar<NODEPTR> &,
					 List_ar<NODEPTR> &);
  //copy another DFG into this DFG
  //void copy(Dfg &);


#endif

#ifdef _SCALP_
  Dfg* add_symbol_substitution(const char* name, EDGEPTR edge);

  // Returns the latest dfgedge associated with a name
  inline Boolean get_defined_operand(const char* name, FRITS_object*& obj) {
    return(defined_operands->st_search_local(name, obj));
  }

  // Used ONLY for LHSides
  Dfg *add_lhs_edge(EDGEPTR lhs_edge);

  void print_defined_operands(void) {defined_operands->print();}

#endif
};
/********************************************************************/
/*Forward Declarations*/
#ifdef _SCALP_
class Schalloc_info : public Assurance {
#else
class Schalloc_info {
#endif
  friend class Dfg;
  friend class Scheduler;
#ifdef PHYSICAL  
  friend class iScheduler;
#endif  
private:
  Array<int> nodebirths;
  Array<int> nodedeaths;
  Array<int> moduletypes;
  Array<FUPTR> functional_units;
  Array<int> edgebirths;
  Array<int> edgedeaths;
  Array<STORPTR> storage_units;

public:
  Schalloc_info(const int nodecount, const int edgecount) : nodebirths(),
  nodedeaths(), moduletypes(), functional_units(), edgebirths(), edgedeaths(), storage_units() {

    FRITS_SET_CLASS("Schalloc_info");
    assert(nodecount >= 0);
    nodebirths.resize(nodecount);
    nodedeaths.resize(nodecount);
    moduletypes.resize(nodecount);
    functional_units.resize(nodecount);
    edgebirths.resize(edgecount);
    edgedeaths.resize(edgecount);
    storage_units.resize(edgecount);
    return;
  }

  ~Schalloc_info() {
    return;
  }

  void resize(int new_nodecount, int new_edgecount) {
    assert(new_nodecount > 0 && new_edgecount > 0);
    if(new_nodecount != nodebirths.get_size()) {
      nodebirths.resize(new_nodecount);
      nodedeaths.resize(new_nodecount);
      moduletypes.resize(new_nodecount);
      functional_units.resize(new_nodecount);
    }
    if(new_edgecount != edgebirths.get_size()) {
      edgebirths.resize(new_edgecount);
      edgedeaths.resize(new_edgecount);
      storage_units.resize(new_edgecount);
    }
    return;
  }
  void extract_info(Dfg &source);

  inline const int get_numcsteps() {
    register int i;
    int maxdeath = 0;
    FRITS_SET_MESSAGE("get_numcsteps");
    for(i = 0; i < edgebirths.get_size(); i++) {
      assert(edgebirths[i] >= 0 && edgedeaths[i] > edgebirths[i]);
      maxdeath = MAX(maxdeath, edgedeaths[i]);
    }
    return(maxdeath);
  }

  //display the info
  void display(ostream & = cout);
};
/********************************************************************/
//Syntactic sugar for displaying the Dfg
inline ostream &operator <<(ostream &output, Dfg &gr)
{gr.display(output);return(output);}
/********************************************************************/
#endif


