/********************************************************************/
/*          FILE: ft_cdfg.C                                         */
/********************************************************************/

#include "dfg.h"

extern void ft_set_checkpoint(NODEPTR);
extern void ft_reset_all_checkpoints();
void display_op(ostream &, Operator);
/********************************************************************/
void Dfg::ft_duplicate()
{
  register int i;
  NODEPTR node, copy_node;
  EDGEPTR edge, copy_edge, tmpedge, copy_tmpedge;
  List_iterator<NODEPTR> nodescan;
  List_iterator<EDGEPTR> edgescan;
  char tmpname[MAXSTRLEN];

  num_orig_nodes = nodecount;
  num_orig_edges = edgecount;

  //create the nodes in the copy
  for(i = 0; i < num_orig_nodes; i++) {
    node = get_nthnode(i);
    assert(node);
    copy_node = new Dfgnode;
    assert(copy_node);

    strcpy(tmpname, node->get_name());
    strcat(tmpname, "_copy");
    copy_node->set_name(tmpname);
    copy_node->set_func(node->get_func());
    add_node(copy_node);
  }

  //create the edges in the copy, and add respective edges
  //to the inputs, loopins, constants, and loopouts lists
  for(i = 0; i < num_orig_edges; i++) {
    edge = get_nthedge(i);
    assert(edge);
    //Note: there are NO PRIMARY OUTPUTS IN THE COPY
    copy_edge = new Dfgedge(edge->get_inputtype(), F);
    assert(copy_edge);

    strcpy(tmpname, edge->get_name());
    strcat(tmpname, "_copy");
    copy_edge->set_name(tmpname);

    if(edge->is_pi()) {
      add_input(copy_edge);
    } else if(edge->is_constant()) {
      copy_edge->set_value(edge->get_value());
      add_constant(copy_edge);
    } else if (edge->is_loopin()) {
      copy_edge->set_value(edge->get_value());
      add_loopin(copy_edge);
    } else {
      add_edge(copy_edge);
    }

    //if this edge is a loopout, add copy_edge to the loopouts list
    if(edge->is_loopout()) {
      assert(!loopouts.find(copy_edge));
      loopouts.append(copy_edge);
      copy_edge->loopout = T;
    }
  }

  assert_force(nodecount%2 == 0 && nodecount/2 == num_orig_nodes);
  assert_force(edgecount%2 == 0 && edgecount/2 == num_orig_edges);

  //create the input and output edge lists for each node
  for(i = 0; i < num_orig_nodes; i++) {
    node = get_nthnode(i);
    copy_node = get_nthnode(i + num_orig_nodes);
    assert(node && copy_node);
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      copy_edge = get_nthedge(edge->get_address() + num_orig_edges);
      assert(copy_edge);
      copy_node->add_input_edge(copy_edge);
    }
    FOR_EACH_FANOUT_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      copy_edge = get_nthedge(edge->get_address() + num_orig_edges);
      copy_node->add_output_edge(copy_edge);
    }
  }

  //create the source and sink node lists for each edge
  for(i = 0; i < num_orig_edges; i++) {
    edge = get_nthedge(i);
    copy_edge = get_nthedge(i + num_orig_edges);
    assert(edge && copy_edge);
    //an EDGE for now can have only ONE source node
    node = edge->input_node();
    assert(node || edge->is_dfginput());
    if(node) {
      copy_node = get_nthnode(node->get_address() + num_orig_nodes);
      assert(copy_node);
      copy_edge->set_source_node(copy_node);
    }

    FOR_EACH_FANOUT_NODE(edge, nodescan) {
      node = nodescan.get_item();
      assert(node);
      copy_node = get_nthnode(node->get_address() + num_orig_nodes);
      assert(copy_node);
      copy_edge->add_sink_node(copy_node);
    }
    if((tmpedge = edge->get_loopin_link()) != NULL) {
      copy_tmpedge = get_nthedge(tmpedge->get_address() + num_orig_edges);
      assert(copy_tmpedge);
      copy_edge->set_loopin_link(copy_tmpedge);
    }
    if((tmpedge = edge->get_loopout_link()) != NULL) {
      copy_tmpedge = get_nthedge(tmpedge->get_address() + num_orig_edges);
      assert(copy_tmpedge);
      copy_edge->set_loopout_link(copy_tmpedge);
    }
  }

  return;
}
/********************************************************************/
//Add a comparison operation at each PO and LOOPOUT in the duplicated CDFG
void Dfg::ft_create_dfgoutput_comparisons()
{
  int old_nodecount, old_edgecount;
  EDGEPTR tmp_edge, copy_edge, error_edge;
  NODEPTR compare_node;
  List_iterator<EDGEPTR> edgescan;
  char tmpname[MAXSTRLEN];

  old_nodecount = nodecount;
  old_edgecount = edgecount;
  FOR_EACH_LISTNODE(loopouts, edgescan) {
    tmp_edge = edgescan.get_item();
    if(tmp_edge->get_address() < num_orig_edges) {
      copy_edge = get_nthedge(tmp_edge->get_address() + num_orig_edges);
      assert(copy_edge);

      compare_node = new NODE(NOTEQ_);
      strcpy(tmpname, "comparision_");
      strcat(tmpname, tmp_edge->get_name());
      compare_node->set_name(tmpname);
      compare_node->add_input_edge(tmp_edge);
      compare_node->add_input_edge(copy_edge);
      tmp_edge->add_sink_node(compare_node);
      copy_edge->add_sink_node(compare_node);

      add_node(compare_node);

      error_edge = new EDGE(NOTPI, T);
      strcpy(tmpname, "error_");
      strcat(tmpname, tmp_edge->get_name());
      error_edge->set_name(tmpname);
      compare_node->add_output_edge(error_edge);
      error_edge->set_source_node(compare_node);
      add_output(error_edge);

      //mark the appropriate operations as checkpointed in Ganesh's data structures
      ft_set_checkpoint(tmp_edge->input_node());
      ft_set_checkpoint(copy_edge->input_node());
    }
  }

  FOR_EACH_LISTNODE(outputs, edgescan) {  
    tmp_edge = edgescan.get_item();
    assert(tmp_edge->get_address() < num_orig_edges ||
	   tmp_edge->get_address() >= 2*num_orig_edges);
    if(tmp_edge->get_address() < num_orig_edges && !tmp_edge->is_loopout()) {
      copy_edge = get_nthedge(tmp_edge->get_address() + num_orig_edges);
      assert(copy_edge);

      compare_node = new NODE(NOTEQ_);
      strcpy(tmpname, "comparison_");
      strcat(tmpname, tmp_edge->get_name());
      compare_node->set_name(tmpname);
      compare_node->add_input_edge(tmp_edge);
      compare_node->add_input_edge(copy_edge);
      tmp_edge->add_sink_node(compare_node);
      copy_edge->add_sink_node(compare_node);

      add_node(compare_node);

      error_edge = new EDGE(NOTPI, T);
      strcpy(tmpname, "error_");
      strcat(tmpname, tmp_edge->get_name());
      error_edge->set_name(tmpname);
      compare_node->add_output_edge(error_edge);
      error_edge->set_source_node(compare_node);
      add_output(error_edge);

      //mark the appropriate operations as checkpointed in Ganesh's data structures
      ft_set_checkpoint(tmp_edge->input_node());
      ft_set_checkpoint(copy_edge->input_node());
    }
  }

  return;
}
/********************************************************************/
/*Given a list of DFG nodes that represent operations in the original,
 *this function updates the DFG by adding comparisons at the output of
 *each operation in the given list.
 */
void Dfg::ft_add_comparisons(List_ar<int> &nodes_to_secure)
{
  int index;
  char tmpname[MAXSTRLEN];
  NODEPTR node, compare_node;
  EDGEPTR edge, copy_edge, error_edge;
  List_iterator<int> intscan;

  FOR_EACH_LISTNODE(nodes_to_secure, intscan) {
    index = intscan.get_item();
    node = get_nthnode(index);
    assert(node && node->get_address() == index);
    //this must be an operation in the original
    assert(node->get_address() < num_orig_nodes);
    edge = node->get_output_edge();
    assert(edge && edge->get_address() < num_orig_edges);
#ifndef NDEBUG
    //this node should not be already checkpointed
    {
      List_iterator<NODEPTR> nodescan;
      NODEPTR outnode;
      FOR_EACH_FANOUT_NODE(edge, nodescan) {
	outnode = nodescan.get_item();
	assert(outnode->get_address() < 2*num_orig_nodes);
      }
    }
#endif
    copy_edge = get_nthedge(edge->get_address() + num_orig_edges);

    compare_node = new NODE(NOTEQ_);
    strcpy(tmpname, "comparision_");
    strcat(tmpname, edge->get_name());
    compare_node->set_name(tmpname);
    compare_node->add_input_edge(edge);
    compare_node->add_input_edge(copy_edge);
    edge->add_sink_node(compare_node);
    copy_edge->add_sink_node(compare_node);

    add_node(compare_node);

    error_edge = new EDGE(NOTPI, T);
    strcpy(tmpname, "error_");
    strcat(tmpname, edge->get_name());
    error_edge->set_name(tmpname);
    compare_node->add_output_edge(error_edge);
    error_edge->set_source_node(compare_node);
    add_output(error_edge);

    //set the nodes as checkpointed in Ganeshs data structures
    ft_set_checkpoint(edge->input_node());
    ft_set_checkpoint(copy_edge->input_node());
  }

  return;
}
/********************************************************************/
/*Given a list of operations in the original whose outputs have been
 *checkpointed, this function removes the checkpoints from the DFG.
 *This assumes that the added comparison operations and error output
 *edges are the LAST consecutive entries in the nodes and edges arrays.
 */
void Dfg::ft_delete_comparisons(List_ar<int> &secured_operations)
{
  int i, nodeindex, edgeindex, count;
  NODEPTR node, outnode;
  EDGEPTR edge, copy_edge;
  List_ar<NODEPTR> comparisons;
  List_iterator<NODEPTR> nodescan;
  List_iterator<int> intscan;

  //First, remove the connections to the added comparisons
  //from the original and the copy 
  comparisons.clear();
  FOR_EACH_LISTNODE(secured_operations, intscan) {
    nodeindex = intscan.get_item();
    node = get_nthnode(nodeindex);
    assert(node && node->get_address() == nodeindex);

    assert(node->get_address() < num_orig_nodes);
    edge = node->get_output_edge();
    assert(edge && edge->get_address() < num_orig_edges);
    assert(!edge->is_dfgoutput());
    copy_edge = get_nthedge(edge->get_address() + num_orig_edges);
    assert(copy_edge);

    count = 0;
    FOR_EACH_FANOUT_NODE(edge, nodescan) {
      outnode = nodescan.get_item();
      if(outnode->get_address() >= 2*num_orig_nodes) {
	assert(outnode->get_func() == NOTEQ_);
	count++;
	edge->delete_sink_node(outnode);
	comparisons.append(outnode);
	copy_edge->delete_sink_node(outnode);
      }
    }
    assert_force(count == 1);
  }

  //next, delete the comparisons and their output edges from the
  //nodes and edges Arrays, as well as from the dfg outputs list
  nodeindex = nodecount - comparisons.get_size();
  edgeindex = edgecount - comparisons.get_size();
  FOR_EACH_LISTNODE(comparisons, nodescan) {
    node = nodescan.get_item();
    assert(node->get_address() >= 2*num_orig_nodes);
    assert_force(node->get_address() >= nodeindex);
    edge = node->get_output_edge();
    assert(edge && edge->get_address() >= 2*num_orig_edges);
    assert_force(edge->get_address() >= edgeindex);
    assert_force(node->get_func() == NOTEQ_ && edge->is_dfgoutput());

    nodes[node->get_address()] = NULL;
    delete node;

    count = outputs.remove(edge);
    assert_force(count == 1);
    edges[edge->get_address()] = NULL;
    delete edge;
  }

  for(i = nodeindex+1; i < nodecount; i++) {
    assert(nodes[i] == NULL);
  }
  for(i = edgeindex+1; i < edgecount; i++) {
    assert(edges[i] == NULL);
  }

  nodecount = nodeindex;
  edgecount = edgeindex;
  return;
}
/********************************************************************/
/*Delete all the comparison operations added to the duplicated CDFG.
 */
void Dfg::ft_delete_all_comparisons()
{
  int i;
  NODEPTR node;
  EDGEPTR edge;
  List_iterator<EDGEPTR> edgescan;

  assert(2*num_orig_nodes <= nodecount); 
  for(i = 2*num_orig_nodes; i < nodecount; i++) {
    node = get_nthnode(i);
    assert(node->get_func() == NOTEQ_);
    //disconnect all input edges to this comparison operation
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge); 
      edge->delete_sink_node(node);
      /*This node has sink nodes left, or it is a dfg output
       *or it is the copy of a dfgoutput
       */
      //assert(edge->number_sink_nodes() > 0 || edge->is_dfgoutput() ||
      //(edge->get_address() >= num_orig_edges &&
      //edge->get_address() < 2*num_orig_edges &&
      //get_nthedge(edge->get_address() - num_orig_edges)->is_dfgoutput()) );
    }

    //delete the output edge of this comparison from the CDFG, and free it
    edge = node->get_output_edge();
    outputs.remove(edge);
    delete edge;

    //remove the comparison operation from the CDFG, and then free it
    delete node;
    nodes[i] = (NODEPTR)NULL;
  }

  for(i = 2*num_orig_edges; i < edgecount; i++) {
    edges[i] = (EDGEPTR)NULL;
  }

  nodecount = 2*num_orig_nodes;
  edgecount = 2*num_orig_edges;

  ft_reset_all_checkpoints();

  return;
}
/********************************************************************/
const NODEPTR Dfg::ft_get_checkpoint(NODEPTR node)
{
  NODEPTR tmpnode, checkpoint_node = NULL;
  EDGEPTR edge;
  List_iterator<NODEPTR> nodescan;

  assert(node && node->get_address() >= 0 && node->get_address() < 2*num_orig_nodes);
  assert(get_nthnode(node->get_address()) == node);

  edge = node->get_output_edge();
  assert(edge);
  FOR_EACH_FANOUT_NODE(edge, nodescan) {
    tmpnode = nodescan.get_item();
    assert(tmpnode);
    if(tmpnode->get_address() >= 2*num_orig_nodes) {
      assert(tmpnode->get_func() == NOTEQ_);
      assert(!checkpoint_node);
      checkpoint_node = tmpnode;
    }
  }

  return(checkpoint_node);
}
/********************************************************************/
void Dfg::separate_orig_and_copy_operations(const List_ar<NODEPTR> &operlist,
		    List_ar<NODEPTR> &orig_operations, List_ar<NODEPTR> &copy_operations)
{
  List_iterator<NODEPTR> nodescan;
  NODEPTR node;

  FOR_EACH_LISTNODE(operlist, nodescan) {
    node = nodescan.get_item();
    assert(node);
    if(node->get_address() < num_orig_nodes) {
      orig_operations.append(node);
    } else if(node->get_address() < 2*num_orig_nodes) {
      copy_operations.append(node);
    } else {
      //comparison operations do not require aliasing analysis since they
      //are implemented by self-checking units
      assert(node->get_func() == NOTEQ_);
      continue;
    }
  }

  return;
}
/********************************************************************/
