/*  Copyright (c) 2000-2001 by Alternative System Concepts, Inc.  */
/*  All Rights Reserved                                           */
/*                                                                */
/*  THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF                */
/*       Alternative System Concepts, Inc.                        */
/*  The copyright notice above does not evidence any              */
/*  actual or intended publication of such source code.           */

///  <HEADER>
///     <PURPOSE>
///              
///     </PURPOSE>
///              
///  </HEADER>

/********************************************************************/
/*        FILE: cdfgedge.h                                          */
/********************************************************************/

#ifndef KAMAL_DFGEDGE_H
#define KAMAL_DFGEDGE_H

#include "basic.h"
#include "list_ar.h"
#include "parameter.h"
#include "matrix.h"

#ifdef _SCALP_
#include "fritso.h"
#endif

typedef enum pi_type {
  NOTPI = 0,
  PI = 1,
  LOOPIN = 2,
  CONSTANT_ = 3
} PITYPE;

//forward declarations
class Dfgnode;
typedef class Dfgnode * NODEPTR;
class Storage_unit;
typedef class Register STOR;
typedef class Register *STORPTR;

typedef class Dfgedge *EDGEPTR;
typedef class Dfgedge EDGE;
#define EDGENULL ((EDGEPTR)NULL);

/********************************************************************/
#ifdef _SCALP_
class Dfgedge : public FRITS_object, public Assurance {
#else
class Dfgedge {
#endif
  friend class Dfg;
  //friend class Transform;
  friend class Retpipe;
  friend ostream &operator <<(ostream &, const Dfgedge &);
  friend ostream &operator <<(ostream &, const EDGEPTR);
private:
  char name[MAXSTRLEN];
  int address;                    //address in the edges array
  PITYPE inputtype;               //whether this edge is a PI/LOOPIN/CONSTANT
  Boolean po;                     //whether this edge is a PO
  Boolean loopout;                //whether this edge is a LOOPOUT
  List_ar<NODEPTR> *source_nodes;
  List_ar<NODEPTR> *sink_nodes;
  EDGEPTR loopin_link;            //link from a loopout to corresponding loopin
  EDGEPTR loopout_link;           //link from a loopin to corresponding loopout
  int value;                      //constant value if edge is a CONSTANT_, initial value if LOOPIN
  //scheduling information
  int birth;
  int death;
  //register assignment information
  STORPTR storage_unit;
  bool ce;
public:
  Dfgedge(PITYPE tp = NOTPI, Boolean is_po = F, Boolean is_ce = F) {
    FRITS_SET_CLASS("Dfgedge");
    name[0] = '\0';
    address = 0;
    inputtype = tp;
    po = is_po;
    loopout = F;
    birth = death = -1;
   loopin_link = loopout_link = NULL;
    value = -1;
    source_nodes = new List_ar<NODEPTR>;
    sink_nodes = new List_ar<NODEPTR>;
    storage_unit = NULL;
    ce = is_ce;
  }

  ~Dfgedge() {
    delete source_nodes;
    delete sink_nodes;
  }

  //messages to get the fields of an EDGE
  inline const char *get_name() {
    FRITS_SET_MESSAGE("get_name");
    return name;
  }
  inline List_ar<NODEPTR> &get_source_nodes() {
    FRITS_SET_MESSAGE("get_source_nodes");
    return(*source_nodes);
  }
  inline List_ar<NODEPTR> &get_sink_nodes() {
    FRITS_SET_MESSAGE("get_sink_nodes");
    return(*sink_nodes);
  }
  inline const int number_source_nodes() {
    FRITS_SET_MESSAGE("number_source_nodes");
    return source_nodes->get_size();
  }
  inline const int number_sink_nodes() {
    FRITS_SET_MESSAGE("number_sink_nodes");
    return sink_nodes->get_size();
  }
  inline const int get_address() {
    FRITS_SET_MESSAGE("get_address");
    return address;
  }
  inline const PITYPE get_inputtype() {
    FRITS_SET_MESSAGE("get_inputtype");
    return inputtype;
  }
  inline const int get_birth() {
    FRITS_SET_MESSAGE("get_birth");
    return birth;
  }
  inline const int get_death() {
    FRITS_SET_MESSAGE("get_death");
    return death;
  }
  inline const bool is_ce() {
    return ce;
  }
  inline const STORPTR get_storage_unit() {
    FRITS_SET_MESSAGE("get_storage_unit");
    return storage_unit;
  }
  inline const int get_value() {
    FRITS_SET_MESSAGE("get_value");
    assert(inputtype == CONSTANT_ || inputtype == LOOPIN);
    return value;
  }
  inline const EDGEPTR get_loopin_link() {
    FRITS_SET_MESSAGE("get_loopin_link");
    return(loopin_link);
  }
  inline const EDGEPTR get_loopout_link() {
    FRITS_SET_MESSAGE("get_loopout_link");
    return(loopout_link);
  }

  //get the node driving this Dfgedge
  //asserts the fact that a Dfgedge can be driven by at most one Dfgnode
  inline NODEPTR input_node() {
    FRITS_SET_MESSAGE("input_node");
    assert(source_nodes->get_size() == 0 || source_nodes->get_size() == 1);
    return( (source_nodes->get_size()) ? ((*source_nodes)[0]) : (NODEPTR)NULL );
  }

  //returns T if edge is a PI, LOOPIN, or CONSTANT_
  const Boolean is_dfginput() {
    FRITS_SET_MESSAGE("is_dfginput");
    if(!((source_nodes->get_size() && inputtype == NOTPI) || inputtype == PI ||
		           inputtype == LOOPIN || inputtype == CONSTANT_))
	    cout<<get_name();

    assert((source_nodes->get_size() && inputtype == NOTPI) || inputtype == PI ||
	   inputtype == LOOPIN || inputtype == CONSTANT_);
    return( (inputtype == PI || inputtype == CONSTANT_ || inputtype == LOOPIN)?T:F);
  }

  //returns T if edge is a PO, or LOOPOUT
  Boolean is_dfgoutput() {
    FRITS_SET_MESSAGE("is_dfgoutput");
    if(po == T || loopout == T) {
      return T;
    } else {
      return F;
    }
  }

  inline const Boolean is_pi() {
    FRITS_SET_MESSAGE("is_pi");
    return(inputtype == PI?T:F);
  }

  inline const Boolean is_po() {
    FRITS_SET_MESSAGE("is_po");
    return(po);
  }

  inline const Boolean is_loopout() {
    FRITS_SET_MESSAGE("is_loopout");
    return(loopout);
  }

  inline const Boolean is_constant() {
    FRITS_SET_MESSAGE("is_constant");
    return(inputtype == CONSTANT_?T:F);
  }
  inline const Boolean is_loopin() {
    FRITS_SET_MESSAGE("is_loopin");
    return(inputtype == LOOPIN?T:F);
  }

  //inline functions for setting fields of a Dfgedge
  inline EDGEPTR set_name(const char *newname) {
    FRITS_SET_MESSAGE("set_name");
    assert(newname);
    assert(strlen(newname) < MAXSTRLEN);
    strcpy(name, newname);
    return(this);
  }

  // Sets the primary output status to T
  EDGEPTR set_po(void)	{po = T; return this;}
  
  // sets the loop output status to T, added by wwang
  EDGEPTR set_loopout(void) {loopout = T; return this;}

  //sets the edge to be CONSTANT edge
  EDGEPTR set_constant(const int i) {
    inputtype = CONSTANT_;
    value = i;
    return this;
  }

  inline EDGEPTR set_value(const int newvalue) {
    FRITS_SET_MESSAGE("set_value");
    value = newvalue;
    return(this);
  }

  inline EDGEPTR set_birth(const int newbirth) {
    FRITS_SET_MESSAGE("set_birth");
    birth = newbirth;
    return(this);
  }

  inline EDGEPTR set_death(const int newdeath) {
    FRITS_SET_MESSAGE("set_death");
    death = newdeath;
    return(this);
  }

  inline EDGEPTR set_storage_unit(const STORPTR newreg) {
    FRITS_SET_MESSAGE("set_storage_unit");
    storage_unit = newreg;
    return(this);
  }

  inline void set_loopin_link(EDGEPTR link) {
    FRITS_SET_MESSAGE("set_loopin_link");
    loopin_link = link;
    return;
  }

  inline void set_loopout_link(EDGEPTR link) {
    FRITS_SET_MESSAGE("set_loopout_link");
    loopout_link = link;
    return;
  }

  inline EDGEPTR set_source_node(const NODEPTR node) {
    FRITS_SET_MESSAGE("set_source_node");
    assert(source_nodes->get_size() == 0);
    return add_source_node(node);
  }

  EDGEPTR add_source_node(const NODEPTR);

  //Add a NODE to the list of nodes fed by this edge
  EDGEPTR add_sink_node(const NODEPTR);

  //Remove a NODE from the list of nodes fed by this edge
  EDGEPTR delete_sink_node(const NODEPTR);

  void display(ostream & = cout);
  
  void copy(Dfgedge &source_edge);
};

//iteration macro for each input or output node of a edge
//edge is a EDGEPTR, node_it is an node_iterator
#define FOR_EACH_FANIN_NODE(edge, node_it) for((node_it).start((edge)->get_source_nodes()); (node_it).not_done(); (node_it).increment())
#define FOR_EACH_FANOUT_NODE(edge, node_it) for((node_it).start((edge)->get_sink_nodes()); (node_it).not_done(); (node_it).increment())


void display_edgelist(const List_ar<EDGEPTR> &, ostream & = cout);
/********************************************************************/
#endif
