#include "FloorPlan.h"
#include "rcmath.h"
#include "RecVector.h"

#include <numeric>
#include <map>

/*###########################################################################*/
// Public
/*###########################################################################*/
#ifdef PHYSICAL
FloorPlan::FloorPlan():
	super(0.0, 0.0), 
	cores_(0, Rect<2, double>(0.0, 0.0)),
	max_aspect_ratio_(1.0),
	tie_(0),
        pos_(0,
	     PRect<2, double>(make_pos(0.0, 0.0), make_rect(0.0, 0.0))),
        bounding_box_(make_pos(0.0, 0.0), make_rect(0.0, 0.0)),
        raw_core_area_(0.0)
{
}
#endif

FloorPlan::FloorPlan(const RVector<Rect<2, double> > & cores,
double max_aspect_ratio, const AssocVec<double> & tie) :
	super(0.0, 0.0),
	cores_(cores),
	max_aspect_ratio_(max_aspect_ratio),
	tie_(tie),
	pos_(cores.size(),
		PRect<2, double>(make_pos(0.0, 0.0), make_rect(0.0, 0.0))),
	bounding_box_(make_pos(0.0, 0.0), make_rect(0.0, 0.0)),
	raw_core_area_(0.0)
{
#ifndef PHYSICAL	
	solve();
#endif	
}


/*===========================================================================*/
FloorPlan::FloorPlan(const RVector<Rect<2, double> > & cores,
double max_aspect_ratio) :
	super(0.0, 0.0),
	cores_(cores),
	max_aspect_ratio_(max_aspect_ratio),
	tie_(cores_.size(), 0),
	pos_(cores.size(),
		PRect<2, double>(make_pos(0.0, 0.0), make_rect(0.0, 0.0))),
	bounding_box_(make_pos(0.0, 0.0), make_rect(0.0, 0.0)),
	raw_core_area_(0.0)
{
#ifndef PHYSICAL	
	solve();
#endif	
}

/*===========================================================================*/
void FloorPlan::print_to(ostream & os) const {
// Build the display screen.
	const long scr_width = SCREEN_WIDTH;

	const long scr_height =
		static_cast<long>(ceil(scr_width / (*this)[0] * (*this)[1]));

	RecVector2<char> screen(scr_width, scr_height, ' ');

	const double x_offset = -bounding_box_.min_pos(0);
	const double y_offset = -bounding_box_.min_pos(1);
	const double scr_x_mul = scr_width / (*this)[0];
	const double scr_y_mul = scr_height / (*this)[1];

// Write each core into the screen.
	MAP(pos_i, pos_.size()) {
		const PRect<2, double> & r = pos_[pos_i];

		const double min_x_d = (r.min_pos(0) + x_offset) * scr_x_mul;
		const double max_x_d = (r.max_pos(0) + x_offset) * scr_x_mul;

		const double min_y_d = (r.min_pos(1) + y_offset) * scr_y_mul;
		const double max_y_d = (r.max_pos(1) + y_offset) * scr_y_mul;

		const long min_x = static_cast<long>(min_x_d);
		const long max_x = static_cast<long>(max_x_d);
		const long min_y = static_cast<long>(min_y_d);
		const long max_y = static_cast<long>(max_y_d);

		for (long x = min_x; x < max_x; ++x) {
			for (long y = min_y; y < max_y; ++y) {
				RASSERT(screen(x, y) == ' ');
				screen(x, y) = index_to_char(pos_i);
			}
		}
	}
	
// Print it.
	os << "Shape: ";
	super::print_to(os);
	os << "\n";

	os << "Area: " << area() << "\n";
	os << "Area Efficiency: " << area_efficiency() * 100.0 << "%\n";

// Show the keys.
	MAP(x, pos_.size()) {
		os << x << ":" << index_to_char(x);
		if ((x + 1) % 10) {
			os << " ";
		} else {
			os << "\n";
		}
	}
	os << "\n\n";

// Show the screen.
	MAP(x, screen.size()[1]) {
		MAP(y, screen.size()[0]) {
			os << screen(y, x);
		}
		os << "\n";
	}
}

/*===========================================================================*/
void FloorPlan::self_check() const {
	RASSERT(! cores_.empty());
	RASSERT(max_aspect_ratio_ > 1.0);
	RASSERT(tie_.size() == cores_.size());

	MAP(x, tie_.size()) {
		RASSERT(! tie_(x, x));
	}
}

/*###########################################################################*/
// Private
/*###########################################################################*/
char
FloorPlan::index_to_char(long i) const {
	return 'A' + i % 52;
}

/*===========================================================================*/
void FloorPlan::solve() {
	self_check();
	IndexVec ivec(cores_.size());
	iota(ivec.begin(), ivec.end(), 0);

	IndexTree itree = partition_recurse(ivec);
	RASSERT(itree.header());

	ExploreTree explore_tree = merge_and_rotate_recurse(itree.header());

// Find best solution allowing desired aspect ratio.
	const long best_exp_tree_index = find_best_top_soln(explore_tree);

// Prune away everything but the information for the best solution.
	ResultTree result_tree = prune_recurse(explore_tree.header(),
		best_exp_tree_index);

// Set the area.
	super::operator=(plump(*result_tree.header()->data()));

// Find the raw area for efficiency computation.
	MAP(x, cores_.size()) {
		raw_core_area_ += cores_[x].area();
	}

// Generate positions.
	gen_pos_recurse(result_tree.header(),
		make_pos(-(*this)[0] / 2.0, -(*this)[1] / 2.0));

// Find the bounding box.
	bounding_box_ = make_union(pos_.begin(), pos_.end());
	bounding_box_.rect() = plump(bounding_box_);

	RASSERT(bounding_box_.rect() == *this);
}

/*===========================================================================*/
FloorPlan::IndexTree
FloorPlan::partition_recurse( IndexVec & ivec) const {
	RASSERT(! ivec.empty());

	IndexTree ret_val;

	if (ivec.size() == 1) {
		ret_val.add_left(0, ivec.front());
	} else {
		pair<IndexVec, IndexVec> bal = balance_ivec(ivec);

		find_best_tie_cut(bal.first, bal.second);

		IndexTree left = partition_recurse(bal.first);
		IndexTree right = partition_recurse(bal.second);

		ret_val.add_left(0, -1);
		ret_val.consume_left(ret_val.header(), left);
		ret_val.consume_right(ret_val.header(), right);
	}

	return ret_val;
}

/*===========================================================================*/
struct area_less : public rbinary_function <long, long, bool> {
	area_less(const RVector<Rect<2, double> > & core) : core_(core) {}

	bool operator()(long a, long b) const {
		return core_[a].area() < core_[b].area();
	}

private:
		const RVector<Rect<2, double> > & core_;
};

/*===========================================================================*/
struct aspect_less : public rbinary_function <long, long, bool> {
	aspect_less(const RVector<Rect<2, double> > & core) : core_(core) {}

	bool operator()(long a, long b) const {
		return core_[a][0] / core_[a][1] < core_[b][0] / core_[a][1];
	}

private:
		const RVector<Rect<2, double> > & core_;
};

/*===========================================================================*/
pair<FloorPlan::IndexVec, FloorPlan::IndexVec>
FloorPlan::balance_ivec(const IndexVec & ivec) const {
	RASSERT(ivec.size() >= 2);

// Sort vec in order of increasing area.
	IndexVec ivec_tmp = ivec;

	area_less lt(cores_);
	sort(ivec_tmp.begin(), ivec_tmp.end(), lt);

/* Starting from the highest area, place cores in the least weighted pair
half.  Basically, bin packing. */
	pair<IndexVec, IndexVec> ret;

	double area_a = 0.0;
	double area_b = 0.0;
	//<lzhong> 10222001
	//The following codes could be improved.They are fast but not 
	//close to optimal.
	for (long x = ivec_tmp.size() - 1; x >= 0; --x) {
		if (area_a < area_b) {
			ret.first.push_back(ivec_tmp[x]);
			area_a += cores_[ivec_tmp[x]].area();
		} else {
			ret.second.push_back(ivec_tmp[x]);
			area_b += cores_[ivec_tmp[x]].area();
		}
	}

	RASSERT(! ret.first.empty());
	RASSERT(! ret.second.empty());

	return ret;
}

/*===========================================================================*/
void FloorPlan::find_best_tie_cut(IndexVec & ivec_a,
IndexVec & ivec_b) const { 
	RASSERT(! ivec_a.empty());
	RASSERT(! ivec_b.empty());

	IndexVec locked_a;
	IndexVec locked_b;

	IndexVec unlocked_a = ivec_a;
	IndexVec unlocked_b = ivec_b;

	IndexVec best_ivec_a = ivec_a;
	IndexVec best_ivec_b = ivec_b;

	double cost = calc_cost(ivec_a, ivec_b);
	double best_cost = cost;
	pair<double, double> area_p = calc_area(ivec_a, ivec_b);

	RASSERT(cost >= 0.0);
	RASSERT(area_p.first >= 0.0);
	RASSERT(area_p.second >= 0.0);

	MAP(x, ivec_a.size() + ivec_b.size()) {
		migrate(unlocked_a, locked_a, area_p.first,
			unlocked_b, locked_b, area_p.second, cost);

		if (cost < best_cost) {
			best_cost = cost;

			ivec_a = unlocked_a;
			ivec_a.insert(ivec_a.end(), locked_a.begin(), locked_a.end());

			ivec_b = unlocked_b;
			ivec_b.insert(ivec_b.end(), locked_b.begin(), locked_b.end());
		}
	}
}

/*===========================================================================*/
void FloorPlan::migrate(IndexVec & free_a, IndexVec & locked_a,
double & area_a, IndexVec & free_b, IndexVec & locked_b, double & area_b,
double & cost) const {
/* Find the best migration on the side with the least area but at least
two cores remaining. */
	bool do_a = false;
	bool do_b = false;

	const long total_size_a = free_a.size() + locked_a.size();
	const long total_size_b = free_b.size() + locked_b.size();

	RASSERT(total_size_a);
	RASSERT(total_size_b);

	if (total_size_a == 1 && total_size_b == 1) {
// Migration won't matter.  Just a single pair.
		return;
	}

	if (total_size_a < 2) do_b = true;
	if (total_size_b < 2) do_a = true;

	if (free_a.empty()) do_b = true;
	if (free_b.empty()) do_a = true;

	RASSERT(! do_a || ! do_b);

	if (! do_a && ! do_b) {
		if (area_a < area_b) {
			do_b = true;
		} else {
			do_a = true;
		}
	}

	RASSERT(do_a || do_b);
	const IndexVec & mig_ivec = do_a ? free_a : free_b;

	long best_candidate = -1;
	double best_cost_change = HUGE_VAL;

	IndexVec whole_ivec_a = free_a;
	whole_ivec_a.insert(whole_ivec_a.end(), locked_a.begin(), locked_a.end());

	IndexVec whole_ivec_b = free_b;
	whole_ivec_b.insert(whole_ivec_b.end(), locked_b.begin(), locked_b.end());

// Find best candidate.
	MAP(x, mig_ivec.size()) {
/* Indexing by x only works because the free indices are guaranteed to be at
the start of the whole ivec. */
		const double cost_change = calc_cost_inc(whole_ivec_a, whole_ivec_b,
			do_a, x);

		if (cost_change < best_cost_change) {
			best_cost_change = cost_change;
			best_candidate = x;
		}
	}

// Adjust cost.
	cost += best_cost_change;

// Adjust areas.
	pair<double, double> area_change =
		calc_area_inc(whole_ivec_a, whole_ivec_b, do_a, best_candidate);

	area_a += area_change.first;
	area_b += area_change.second;

// Migrate it.
	if (do_a) {
		locked_b.push_back(free_a[best_candidate]);
		free_a.erase(&free_a[best_candidate]);
	} else {
		locked_a.push_back(free_b[best_candidate]);
		free_b.erase(&free_b[best_candidate]);
	}
}

/*===========================================================================*/
double
FloorPlan::calc_cost(const IndexVec & ivec_a, const IndexVec & ivec_b) const {
	double cost = 0.0;
	MAP2(x, ivec_a.size(), y, ivec_b.size()) {
		cost += tie_(ivec_a[x], ivec_b[y]);
	}

	return cost;
}

/*===========================================================================*/
pair<double, double>
FloorPlan::calc_area(const IndexVec & ivec_a, const IndexVec & ivec_b) const {
	pair<double, double> ret(0.0, 0.0);

	MAP(x, ivec_a.size()) {
		ret.first += cores_[ivec_a[x]].area();
	}

	MAP(x, ivec_b.size()) {
		ret.second += cores_[ivec_b[x]].area();
	}

	RASSERT(ret.first > 0.0);
	RASSERT(ret.second > 0.0);

	return ret;
}

/*===========================================================================*/
double
FloorPlan::calc_cost_inc(const IndexVec & ivec_a,
const IndexVec & ivec_b, bool do_migrat_a, long migrat_index) const {
	double new_cut_cost = 0.0;

	const IndexVec & src = do_migrat_a ? ivec_a : ivec_b;
	const IndexVec & dest = do_migrat_a ? ivec_b : ivec_a;

	MAP(x, src.size()) {
		new_cut_cost += tie_(src[migrat_index], src[x]);
	}

	double old_cut_cost_reduction = 0.0;
	MAP(x, dest.size()) {
		old_cut_cost_reduction += tie_(src[migrat_index], dest[x]);
	}

	return new_cut_cost - old_cut_cost_reduction;
}

/*===========================================================================*/
pair<double, double>
FloorPlan::calc_area_inc(const IndexVec & ivec_a, const IndexVec & ivec_b,
bool do_migrat_a, long migrat_index) const {
	pair<double, double> ret(0.0, 0.0);

	if (do_migrat_a) {
		ret.first -= cores_[ivec_a[migrat_index]].area();
		ret.second += cores_[ivec_a[migrat_index]].area();
	} else {
		ret.first += cores_[ivec_b[migrat_index]].area();
		ret.second -= cores_[ivec_b[migrat_index]].area();
	}

	return ret;
}

/*===========================================================================*/
FloorPlan::ExploreTree
FloorPlan::merge_and_rotate_recurse(const IndexTree::Node * nd) const {
	if (nd->terminal()) {
		RASSERT(nd->data() >= 0);
		RotVec vec;

		HolderPtr<Rotation>
			h_rot(new LeafRot(nd->data(), cores_[nd->data()]));

		vec.push_back(h_rot);
		ExploreTree ret;
		ret.add_left(0, vec);
		return ret;
	}

// Merge the child trees at this level.
	RASSERT(nd->left());
	RASSERT(nd->right());

// Recurse.
	ExploreTree left = merge_and_rotate_recurse(nd->left());
	const RotVec & left_vec = left.header()->data();

	ExploreTree right = merge_and_rotate_recurse(nd->right());
	const RotVec & right_vec = right.header()->data();

// Find all combinations except those with widths less than heights.
	map<InnerRot, bool> rot;
	MAP4(l, left_vec.size(), l_rot, 1, r, right_vec.size(), r_rot, 1) {
		InnerRot ir_horiz(InnerRot::HORIZ, left_vec, l, l_rot,
			right_vec, r, r_rot);

		if (ir_horiz[0] >= ir_horiz[1]) rot.insert(make_pair(ir_horiz, false));

		InnerRot ir_vert(InnerRot::VERT, left_vec, l, l_rot,
			right_vec, r, r_rot);

		if (ir_vert[0] >= ir_vert[1]) rot.insert(make_pair(ir_vert, false));
	}

// Duplicates can't exist, it's a set.

// Mark the dominated entries.
	for (map<InnerRot, bool>::iterator i = rot.begin(); i != rot.end(); ++i) {
		map<InnerRot, bool>::iterator j = i;
		++j;

		for ( ; j != rot.end(); ++j) {
			if (i->first.dominates(j->first)) {
				j->second = true;
			} else if (j->first.dominates(i->first)) {
				i->second = true;
			}
		}
	}

// Erase all the dominated entries.
	for (map<InnerRot, bool>::iterator i = rot.begin(); i != rot.end(); ) {
		if (i->second) {
			map<InnerRot, bool>::iterator new_i = i;
			++new_i;
			rot.erase(i);
			i = new_i;
		} else {
			++i;
		}
	}

// Build a conventional rotation vector.
	RotVec rvec;
	for (map<InnerRot, bool>::const_iterator i = rot.begin();
		i != rot.end(); ++i)
	{
		HolderPtr<Rotation>
			holder(new InnerRot(i->first));

		rvec.push_back(holder);
	}

// Jam them into a new node.
	ExploreTree ret;
	ret.add_left(0, rvec);

// Absorb the child nodes.
	ret.consume_left(ret.header(), left);
	ret.consume_right(ret.header(), right);

	return ret;
}

/*===========================================================================*/
long
FloorPlan::find_best_top_soln(const ExploreTree & et) const {
	const RVector<HolderPtr<Rotation> > & rv = et.header()->data();

// Select the index of the best solution.
	double best_area = HUGE_VAL;
	long best_index = -1;

	MAP(x, rv.size()) {
// Plump up the solution so it doesn't exceed the aspect ratio bound.
		double this_area = plump(*rv[x]).area();
		if (this_area < best_area) {
			best_area = this_area;
			best_index = x;
		}
	}

	return best_index;
}

/*===========================================================================*/
Rect<2, double>
FloorPlan::plump(const Rect<2, double> & r) const {
	Rect<2, double> ret = r;
	const double min_height = ret[0] / max_aspect_ratio_;
	ret[1] = max(min_height, ret[1]);

	return ret;
}

/*===========================================================================*/
FloorPlan::ResultTree
FloorPlan::prune_recurse(ExploreTree::Node * nd,
const long best_exp_tree_index) const {
	ResultTree ret;

	if (nd->terminal()) {
		RASSERT(nd->data().size() == 1);
		RASSERT(best_exp_tree_index == 0);

		HolderPtr<Rotation> holder = nd->data()[best_exp_tree_index];
		ret.add_left(0, holder);
		return ret;
	}

	const InnerRot & rot =
		dynamic_cast<const InnerRot &>(*nd->data()[best_exp_tree_index]);

	RASSERT(nd->left());
	RASSERT(nd->right());

	ResultTree left = prune_recurse(nd->left(), rot.a_index_);
	ResultTree right = prune_recurse(nd->right(), rot.b_index_);

	HolderPtr<Rotation> holder(rot.clone());
	ret.add_left(0, holder);
	ret.consume_left(ret.header(), left);
	ret.consume_right(ret.header(), right);

	return ret;
}

/*===========================================================================*/
void FloorPlan::gen_pos_recurse(const ResultTree::Node * nd,
const Pos<2, double> & lower_left) {
	if (nd->terminal()) {
		const LeafRot & rot = dynamic_cast<const LeafRot &>(*nd->data());
		Pos<2, double> p = lower_left;
		p[0] += rot[0] / 2.0;
		p[1] += rot[1] / 2.0;
		pos_[rot.size_index_] = PRect<2, double>(p, rot);
		return;
	}

	RASSERT(nd->left());
	RASSERT(nd->right());

	const InnerRot & rot = dynamic_cast<const InnerRot &>(*nd->data());
	switch(rot.compos_) {
	case InnerRot::HORIZ: {
		Pos<2, double> right_ll = lower_left;
		right_ll[0] += (*nd->left()->data())[0];

		gen_pos_recurse(nd->left(), lower_left);
		gen_pos_recurse(nd->right(), right_ll);
		break;
	}

	case InnerRot::VERT: {
		Pos<2, double> top_ll = lower_left;
		top_ll[1] += (*nd->left()->data())[1];

		gen_pos_recurse(nd->left(), lower_left);
		gen_pos_recurse(nd->right(), top_ll);
		break;
	}

	default:
		Rabort();
	}
}

/*###########################################################################*/
FloorPlan::LeafRot::LeafRot(long size_index, const EpsRect & r) :
	super(r),
	size_index_(size_index),
	rotated_(false)
{
	if ((*this)[0] < (*this)[1]) {
		swap_dim(0, 1);
		rotated_ = true;
	}
}

/*===========================================================================*/
void FloorPlan::LeafRot::print_to(ostream & os) const {
	super::print_to(os);

	os << " I: " << size_index_ << ", R: " << rotated_;
}

/*###########################################################################*/
FloorPlan::InnerRot::InnerRot(Composition compos,
const RotVec & a_vec, long a_index, bool a_rot, const RotVec & b_vec,
long b_index, bool b_rot) :
	super(EpsRect(0.0, 0.0)),
	compos_(compos),
	a_index_(a_index),
	b_index_(b_index),
	a_rot_(a_rot),
	b_rot_(b_rot)
{
	EpsRect a_proxy = *a_vec[a_index];
	if (a_rot) a_proxy.swap_dim(0, 1);

	EpsRect b_proxy = *b_vec[b_index];
	if (b_rot) b_proxy.swap_dim(0, 1);

	switch (compos) {
	case HORIZ:
		(*this)[0] = a_proxy[0] + b_proxy[0];
		(*this)[1] = max(a_proxy[1], b_proxy[1]);
		break;
	case VERT:
		(*this)[0] = max(a_proxy[0], b_proxy[0]);
		(*this)[1] = a_proxy[1] + b_proxy[1];
		break;
	default:
		Rabort();
	}

	RASSERT((*this)[0] > 0.0);
	RASSERT((*this)[1] > 0.0);
}

/*===========================================================================*/
void FloorPlan::InnerRot::print_to(ostream & os) const {
	super::print_to(os);

	os << " Compos: " << compos_ << ", AI: " << a_index_ <<
		", BI: " << b_index_ << ", AR: " << a_rot_ << ", BR: " << b_rot_;
}

/*===========================================================================*/
bool
FloorPlan::InnerRot::dominates(const self & a) const {
	if ((*this)[0] <= a[0] && (*this)[1] <= a[1] && *this != a) {
		return true;
	}

	return false;
}

/*###########################################################################*/
ostream &
operator<<(ostream & os, FloorPlan::InnerRot::Composition compos) {
	switch (compos) {
	case FloorPlan::InnerRot::HORIZ:
		os << "H";
		break;
	case FloorPlan::InnerRot::VERT:
		os << "V";
		break;
	default:
		Rabort();
	}

	return os;
}
