#ifndef FLOOR_PLAN_H_
#define FLOOR_PLAN_H_

/*###########################################################################*/
#include "RVector.h"
#include "BinTree.h"
#include "Geom.h"
#include "AssocVec.h"
#include "HolderPtr.h"
#include "Epsilon.h"

/*###########################################################################*/
// Uses Rectilinear distance measure.

class FloorPlan :
	public Rect<2, double>,
	public Prints<FloorPlan>,
	public SChecks<FloorPlan>
{
	typedef Rect<2, double> super;

public:
#ifdef PHYSICAL
	FloorPlan(void);
#endif	
	FloorPlan(const RVector<Rect<2, double> > & cores,
		double max_aspect_ratio, const AssocVec<double> & tie);

/* Pass in an RVector of core geometries and a maximum aspect
ratio. */
	FloorPlan(const RVector<Rect<2, double> > & cores,
		double max_aspect_ratio = 2.0);

	const RVector<PRect<2, double> > & pos() const { return pos_; }
	double area_efficiency() const { return raw_core_area_ / area(); }

// Interface
	virtual void print_to(ostream & os) const;
	virtual void self_check() const;

protected:
//was private :
	class Rotation;
	typedef RVector<long> IndexVec;
	typedef BinTree<long> IndexTree;
	typedef RVector<HolderPtr<Rotation> > RotVec;
	typedef BinTree<RotVec> ExploreTree;
	typedef BinTree<HolderPtr<Rotation> > ResultTree;
	typedef Rect<2, double, eps_comp_obj<double> > EpsRect;

	char index_to_char(long i) const;
	PRect<2, double> bounding_box() const { return bounding_box_; }

 	virtual void solve();

//-----
/* Recursively divide cores such that tie cut weight is minimized.  Takes
range, constructs partitioned tree in which each node holds the index of the
corresponding core or -1. */
	virtual IndexTree partition_recurse( IndexVec & ivec) const;

/* Returns the end of the first portion.  Balances sub-ivec areas.  Destroys
argument. */
	//partition_recurse();
	//Generate initial for find_best_tie_cut();
	virtual pair<IndexVec, IndexVec> balance_ivec(const IndexVec & ivec) const;

/* Optimizes the partition of cores. */
	//partition_recurse();
	virtual void find_best_tie_cut(IndexVec & ivec_a, IndexVec & ivec_b) const;

/* Moves the best cost gain core in the largest area partition to the other
from one partition to the other. */
	//find_best_tie_cut();
	virtual void migrate(IndexVec & free_a, IndexVec & locked_a, double & area_a,
		IndexVec & free_b, IndexVec & locked_b, double & area_b,
		double & cost) const;

// Computes the cost of the tie cuts in the range.
	virtual double calc_cost(const IndexVec & ivec_a, const IndexVec & ivec_b) const;

	virtual pair<double, double> calc_area(const IndexVec & ivec_a,
		const IndexVec & ivec_b) const;

// Computes the cost change of migrating the migrator.
	virtual double calc_cost_inc(const IndexVec & ivec_a, const IndexVec & ivec_b,
		bool do_migrat_a, long migrat_index) const;

	virtual pair<double, double> calc_area_inc(const IndexVec & ivec_a,
		const IndexVec & ivec_b, bool do_migrat_a, long migrat_index) const;

//-----
/* Merges and rotate cores, gathering all non-dominated (area) solutions at
each level.  All stored Rotations are wider than tall. */
	virtual ExploreTree merge_and_rotate_recurse(const IndexTree::Node * nd) const;

	virtual long find_best_top_soln(const ExploreTree & et) const;
	Rect<2, double> plump(const Rect<2, double> & r) const;

/* Prunes all the unused rotations from the explore tree to produce a result
tree. */
	virtual ResultTree prune_recurse(ExploreTree::Node * nd,
		long best_exp_tree_index) const;

// Generates the positions (pos_) by traversing the result tree.
	virtual void gen_pos_recurse(const ResultTree::Node * nd,
		const Pos<2, double> & lower_left);
#ifdef PHYSICAL
public:	
	class Rotation : public EpsRect {
	public:
		Rotation(const EpsRect & r) : EpsRect(r) {}
	};

	class LeafRot : public Rotation {
		typedef LeafRot self;
		typedef Rotation super;
	public:
		LeafRot(long size_index, const EpsRect & r);
		virtual self * clone() const { return new self(*this); }
		virtual void print_to(ostream & os) const;
	private:
			long size_index_;
			bool rotated_;

		friend FloorPlan;
	};

	class InnerRot : public Rotation {
		typedef Rotation super;
		typedef InnerRot self;
	public:
		enum Composition { HORIZ=0, VERT=1 };

		InnerRot(Composition compos, const RotVec & a_vec,
			long a_index, bool a_rot, const RotVec & b_vec,
			long b_index, bool b_rot);

		virtual self * clone() const { return new self(*this); }
		virtual void print_to(ostream & os) const;

		bool dominates(const self & a) const;

	private:
			Composition compos_;
			long a_index_;
			long b_index_;
			bool a_rot_;
			bool b_rot_;

		friend FloorPlan;
	};
protected:
#endif	
#ifdef PHYSICAL
		RVector<Rect<2, double> > cores_;
		double max_aspect_ratio_;
		AssocVec<double> tie_;

#else	
		const RVector<Rect<2, double> > cores_;
		const double max_aspect_ratio_;
		const AssocVec<double> tie_;
#endif		

		RVector<PRect<2, double> > pos_;
		PRect<2, double, eps_comp_obj<double> > bounding_box_;
		double raw_core_area_;
		static const int SCREEN_WIDTH = 78;
};

ostream & operator<<(ostream & os, FloorPlan::InnerRot::Composition compos);

/*###########################################################################*/
#endif
