#include "FloorPlan2.h"
#include "wire.h"
#include "rcmath.h"
#include "RecVector.h"
#include "RStd.h"

#include <numeric>
#include <map>

//Manhattan Distance is used 
//Dedicated link
void FloorPlan2::estimate_link_length(void) {
	for(long i = 0; i<cores_.size();i++){
	  for(long j = i+1;j<cores_.size();j++){
	  	const PRect<2, double> &r1 = pos_[i];
		const PRect<2, double> &r2 = pos_[j];		
		const double x1_center = ((Pos<2, double>)r1)[0];
		const double x2_center = ((Pos<2, double>)r2)[0];
		const double y1_center = ((Pos<2, double>)r1)[1];
		const double y2_center = ((Pos<2, double>)r2)[1];
		link_length_(i,j)= fabs(x1_center - x2_center);
		link_length_(i,j)+= fabs(y1_center- y2_center);
	  }
	}
	return;
}
// Minimum spanning tree is constructed for each output
// 07212002 Note: the interconnect is not shared.
float FloorPlan2::estimate_output_network_length(
		RVector<unsigned long> & connected_indice ) {

        RVector< PRect<2, double> > connected_cores;
        for(unsigned long i=0; i < connected_indice.size(); i++) {
	        connected_cores.push_back(pos_[connected_indice[i]]);
	}
	//Using the Kruskal algorithm implemneted by lzhong
        MST<2, double> mst(&dist_manhattan<2, double>, connected_cores);
        return mst.total_len();
}
//For a non-optimal Steiner Tree style shared interconnect.
//vertical indicates whether the trunk is vertical or horizontal
float FloorPlan2::estimate_output_network_length(
	        const RVector<unsigned long> & connected_indice,
	        RVector<float> & trunk_branches,
	        bool vertical) {
	short x, y;
	float total;
	unsigned long core_num = connected_indice.size();
	assert(core_num>=0);
	if(vertical) {
		x = 0; y = 1;
	} else {
		x = 1; y = 0;
	}
	
	trunk_branches.clear();
	unsigned long index;
	float max_y(-HUGE_VAL), min_y(HUGE_VAL);
	float center_x, center_y;
	MAP(index, core_num) {
		center_y = (float)(Pos<2,double>(pos_[connected_indice[index]]))[y];
		if(center_y<min_y)  min_y = center_y;
		if(center_y>max_y)  max_y = center_y;
	}	
	//This is the length of trunk
	trunk_branches.push_back(max_y-min_y);
	total = max_y - min_y;
	float x_center = (float)(Pos<2, double>(pos_[connected_indice[0]]))[x];
	for(index = 1; index<core_num; index ++ ) {
		//This is the length of branch;
		center_x =  (float)(Pos<2, double>(pos_[connected_indice[index]]))[x];
		trunk_branches.push_back(fabs(x_center -center_x));
		total +=fabs(x_center - center_x);
	}
	return total;	
}

//Print out the links and their priorities and lengths
void FloorPlan2::print_links(ostream &os) const {
	unsigned long module_num = cores_.size();
	for(long i = 0;i<module_num;i++){
	  for(long j = i+1;j<module_num;j++){
		if(tie_(i,j)>0){
	  		os<<"("<<(char)('A'+i%52)<<", "<<(char)('A'+j%52)<<"): "<<" SW = "<<tie_(i,j)<<" Len = "<<link_length_(i,j)<<endl;	  
		  }
	  }
	}
	os<<"Total data transfer cost = "<<wire_sc_<<endl;
	os<<"Total clock tree cost = "<<clock_tree_sc_<<endl;
	os<<"Total interconnect cost = "<<wire_sc_+clock_tree_sc_<<endl;
	return;
}

//Generate the total interconnection switching capacitance assuming unit capacitance 
//per length unit
void FloorPlan2::generate_total_wire_sc(void) {
	unsigned long module_num(tie_.size());	
	//assert(tie_.size()==cores_.size());
	double total = 0.0;
	for(long i = 0; i<module_num;i++){
	  for(long j = i+1; j< module_num; j++) {
		  total += (tie_(i,j)*link_length_(i,j));
	  }
	}
	//cout<<"Before = "<<wire_sc_<<" After = "<<total<<endl;
	wire_sc_ = total;//in fJoule
	return;
}
////////////////////////////////////////////////////
//
void FloorPlan2::add_clock_wires() {

	wire_energy wire_en;
	for(unsigned long i=0;i<tie_.size();i++) {
		for(unsigned long j = i+1; j<tie_.size();j++) {
			tie_(i,j) += (wire_en.get_pattern_energy(1));//sts
		}
	}
	return;
}
void FloorPlan2::remove_clock_wires() {
	wire_energy wire_en;
        for(unsigned long i=0;i<tie_.size();i++) {
                for(unsigned long j = i+1; j<tie_.size();j++) {
                        tie_(i,j) -= (wire_en.get_pattern_energy(1));//sts
                }
        }
        return; 
}
void FloorPlan2::generate_clock_tree_sc(void) 
{
	wire_energy wire_en;
	RVector< PRect<2, double> > clocked_cores;
	for(unsigned long x=0; x< clocked_cores_.size(); x++) {
		clocked_cores.push_back(pos_[x]);
	}
	MST<2, double> mst(&dist_manhattan<2, double>, clocked_cores.begin(),
				clocked_cores.end());
	clock_tree_sc_ = mst.total_len() * wire_en.get_pattern_energy(1);

}
void FloorPlan2::solve() {
        self_check();
        IndexVec ivec(cores_.size());
        iota(ivec.begin(), ivec.end(), 0);

	//Different with class FloorPlan
        itree_ = partition_recurse(ivec);
        RASSERT(itree_.header());

	//Different with class FloorPlan
        explore_tree_ = merge_and_rotate_recurse(itree_.header());

// Find best solution allowing desired aspect ratio.
        const long best_exp_tree_index = find_best_top_soln(explore_tree_);

// Prune away everything but the information for the best solution.
        ResultTree result_tree = prune_recurse(explore_tree_.header(),
                best_exp_tree_index);

// Set the area.
        super::operator=(plump(*result_tree.header()->data()));

// Find the raw area for efficiency computation.
        MAP(x, cores_.size()) {
                raw_core_area_ += cores_[x].area();
        }

// Generate positions.
        gen_pos_recurse(result_tree.header(),
                make_pos(-(*this)[0] / 2.0, -(*this)[1] / 2.0));

// Find the bounding box.
        bounding_box_ = make_union(pos_.begin(), pos_.end());
        bounding_box_.rect() = plump(bounding_box_);

        RASSERT(bounding_box_.rect() == *this);
}

void FloorPlan2::solve(const RVector<Rect<2, double> > & cores,
                        const RVector<unsigned long> & clocked_cores,
                        double max_aspect_ratio, const AssocVec<double> & tie )
        {
		bounding_box_ =  PRect<2, double, eps_comp_obj<double> > (make_pos(0.0, 0.0), make_rect(0.0, 0.0));
		raw_core_area_ = 0.0;	
                cores_.resize(cores.size(), cores[0]);
                for (long i = 0; i<cores.size(); i++) {
                        cores_[i] = cores[i];
                }
                clocked_cores_.resize(clocked_cores.size(), clocked_cores[0]);
                for ( long i = 0; i<clocked_cores.size(); i++) {
                        clocked_cores_[i] = clocked_cores[i];
                }

                max_aspect_ratio_ = max_aspect_ratio;

		tie_.clear();
                tie_.resize(tie.size(), 0.0);
#ifdef B_P              
                for(long i = 0; i< tie.size();i++) {
                   for(long j = i+1; j <tie.size(); j++) {
                        tie_(i,j) = tie(i,j);
                   }
                }
#else
                for(long i =0;i<tie.size();i++) {
                   for(long j = i+1;j<tie.size();j++){
                         if(tie_(i,j)>0)
                               tie_(i,j)=1.0;
                   }
                }
#endif
		link_length_.clear();
                link_length_.resize(cores.size(), 0.0);

		pos_.clear();
		pos_.resize(cores.size(), 
			 PRect<2, double>(make_pos(0.0, 0.0), make_rect(0.0, 0.0)));

                add_clock_wires();
                solve();
                remove_clock_wires();
                estimate_link_length();
#ifndef B_P
                for(long i =0;i<tie.size();i++) {
                   for(long j = i+1;j<tie.size();j++){
                         tie_(i,j)=tie(i,j);
                   }
                }
#endif
                generate_total_wire_sc();
                generate_clock_tree_sc();

                return;
        }

void FloorPlan2::module_size_change( const Rect<2, double> &new_size, long index){
	cores_[index] = new_size;

        self_check();

        //Different with class FloorPlan
        //itree_ = partition_recurse(ivec);
        RASSERT(itree_.header());

        //Different with class FloorPlan
	explore_tree_.clear();
        explore_tree_ = merge_and_rotate_recurse(itree_.header());

// Find best solution allowing desired aspect ratio.
        const long best_exp_tree_index = find_best_top_soln(explore_tree_);

// Prune away everything but the information for the best solution.
        ResultTree result_tree = prune_recurse(explore_tree_.header(),
                best_exp_tree_index);

// Set the area.
        super::operator=(plump(*result_tree.header()->data()));

// Find the raw area for efficiency computation.
	raw_core_area_ = 0.0;
        MAP(x, cores_.size()) {
                raw_core_area_ += cores_[x].area();
        }

// Generate positions.
        gen_pos_recurse(result_tree.header(),
                make_pos(-(*this)[0] / 2.0, -(*this)[1] / 2.0));

// Find the bounding box.
        bounding_box_ = make_union(pos_.begin(), pos_.end());
        bounding_box_.rect() = plump(bounding_box_);
	
	estimate_link_length();
	generate_total_wire_sc();
        RASSERT(bounding_box_.rect() == *this);
	return;
}
