#include "SAFloorPlan.h"
#include "GDFloorPlan.h"
#include <iostream>
#include <cfloat>

#include "RStd.h"
#include "RGen.h"
#include "MST.h"
#include "Epsilon.h"

/*###########################################################################*/
enum Action {
	SWAP_POS, ROTATE, JIGGLE, MAX_ACTION = JIGGLE
};

/*###########################################################################*/
SAFloorPlan::SAFloorPlan(const RVector<Rect<2, double> > & cores,
double max_aspect_ratio, const AssocVec<double> & tie,
double heat_init, double heat_mul,
double heat_add,
int change_steps, int halt_steps) :
	super(Pos<2, double>(0.0, 0.0), Rect<2, double>(0.0, 0.0)),
	core_(),
	raw_core_area_(0.0),
	max_aspect_ratio_(max_aspect_ratio),
	tie_(tie),

	overlap_cost_(DBL_MAX),
	wire_cost_(DBL_MAX),
	dist_(cores.size(), 0.0),

	heat_(heat_init),
	heat_mul_(heat_mul),
	heat_add_(heat_add),
	change_steps_(change_steps),
	halt_steps_(halt_steps),

	try_core_(),
	try_wire_cost_(-1.0),
	try_overlap_cost_(-1.0),
	try_dist_(-1.0),
	bounding_box_(Pos<2, double>(0.0, 0.0), Rect<2, double>(0.0, 0.0)),
	GDFloorPlan(0.0, 0, 0)
{
	MAP(x, cores.size()) {
		core_.push_back(PRect<2, double>(Pos<2, double>(0.0, 0.0),
			cores[x]));

		raw_core_area_ += cores[x].area();
	}

	//optimize();
	try_core_ = core_;
	calc_costs();
	gradient_decent();
}
/*###########################################################################*/
/* Initialized with a floorplan*/
SAFloorPlan::SAFloorPlan(const RVector<Rect<2, double> > & cores,
const RVector<Pos<2, double> > & position,		
double max_aspect_ratio, const AssocVec<double> & tie,
double heat_init, double heat_mul,
double heat_add,
int change_steps, int halt_steps) :
        super(Pos<2, double>(0.0, 0.0), Rect<2, double>(0.0, 0.0)),
        core_(),
        raw_core_area_(0.0),
        max_aspect_ratio_(max_aspect_ratio),
        tie_(tie),

        overlap_cost_(DBL_MAX),
        wire_cost_(DBL_MAX),
        dist_(cores.size(), 0.0),

        heat_(heat_init),
        heat_mul_(heat_mul),
        heat_add_(heat_add),
        change_steps_(change_steps),
        halt_steps_(halt_steps),

        try_core_(),
        try_wire_cost_(-1.0),
        try_overlap_cost_(-1.0),
        try_dist_(-1.0),
        bounding_box_(Pos<2, double>(0.0, 0.0), Rect<2, double>(0.0, 0.0)),
	GDFloorPlan(0.0, 0, 0)
{
        MAP(x, cores.size()) {
        	core_.push_back(PRect<2, double> (position[x],
	                cores[x]));

	        raw_core_area_ += cores[x].area();
	}
	MAP(x, core_.size()) {
		bounding_box_.make_union(core_[x]);
	}
	overlap_cost_ = overlap_cost(core_);
	bounding_box_.rect() = plump(bounding_box_.rect());
	normalize_bounding_box(bounding_box_, core_);
        super::rect()[0] = bounding_box_.rect()[0];
        super::rect()[1] = bounding_box_.rect()[1];
        super::pos()[0] = bounding_box_.pos()[0];
        super::pos()[1] = bounding_box_.pos()[1];

	try_core_ = core_ ; 

	calc_costs();

        //optimize();
	gradient_decent();
}
/*===========================================================================*/
SAFloorPlan::SAFloorPlan(const RVector<Rect<2, double> > & cores,
double max_aspect_ratio,
double heat_init, double heat_mul,
double heat_add,
int change_steps, int halt_steps) :
	super(Pos<2, double>(0.0, 0.0), Rect<2, double>(0.0, 0.0)),
	core_(),
	raw_core_area_(0.0),
	max_aspect_ratio_(max_aspect_ratio),
	tie_(AssocVec<double>(cores.size(), 0.0)),

	overlap_cost_(DBL_MAX),
	wire_cost_(DBL_MAX),
	dist_(cores.size(), 0.0),

	heat_(heat_init),
	heat_mul_(heat_mul),
	heat_add_(heat_add),
	change_steps_(change_steps),
	halt_steps_(halt_steps),

	try_core_(),
	try_wire_cost_(-1.0),
	try_overlap_cost_(-1.0),
	try_dist_(-1.0),
	bounding_box_(Pos<2, double>(0.0, 0.0), Rect<2, double>(0.0, 0.0)),	
	GDFloorPlan(0.0, 0, 0)
{
	MAP(x, cores.size()) {
		core_.push_back(PRect<2, double>(Pos<2, double>(0.0, 0.0),
			cores[x]));

		raw_core_area_ += cores[x].area();
	}

	optimize();
}
//===========================================================================
void SAFloorPlan::gradient_decent() {

	set_alpha(0.1);
	MAP(i, 100) {
		RVector< Rect<2, double> > delta = GDFloorPlan::compute_gradient(try_core_, tie_);
		MAP(x, try_core_.size()) {
			try_core_[x].pos()[0] -=(alpha() * delta[x][0]);
			try_core_[x].pos()[1] -=(alpha() * delta[x][1]);
			//set_alpha(GDFP.alpha()*(1.0-1.0/(i+2.0)));		
			set_wire_overlap_weight(i, 100);
		}		
		calc_costs();
	}
        core_ = try_core_;
        wire_cost_ = try_wire_cost_;
        overlap_cost_ = try_overlap_cost_;
        dist_ = try_dist_;
        super::rect()[0] = bounding_box_.rect()[0];
        super::rect()[1] = bounding_box_.rect()[1];
        super::pos()[0] = bounding_box_.pos()[0];
        super::pos()[1] = bounding_box_.pos()[1];

	return;
}
/*===========================================================================*/
void SAFloorPlan::optimize() {
	int no_improve = 0;
	double best_cost = DBL_MAX;
	while (no_improve < halt_steps_) {
		int attempt = 0;
		bool improve = false;
		while (attempt < change_steps_) {
			bool accept = anneal();

			attempt++;
			if (accept && cost() < best_cost) {
				attempt = 0;
				best_cost = cost();
				improve = true;
			}

//cout << heat_ << "    " << overlap_cost_ << "    " << wire_cost_ << "\n";
		}

		heat_ = (heat_ * heat_mul_) + heat_add_;
		heat_ = max(heat_, 0.0);

		if (improve) {
			no_improve = 0;
		} else {
			++no_improve;
		}
	}
}

/*===========================================================================*/
double SAFloorPlan::
wire_cost() const
{
	return wire_cost_;
}

/*===========================================================================*/
double SAFloorPlan::
overlap_cost() const
{
	return overlap_cost_;
}

/*===========================================================================*/
double
SAFloorPlan::dist(int x, int y) const {
	return dist_(x, y);
}

/*===========================================================================*/
/*
void SAFloorPlan::print_to(ostream & os) const {
	os << overlap_cost_ << "\t\t" << wire_cost_ << "\n\n";
	print_cont(core_, os, "\n");
	os << "\n";
}
*/
//===========================================================================
void SAFloorPlan::print_to(ostream & os) const {
// Build the display screen.
        const long scr_width = SCREEN_WIDTH;

        const long scr_height =
                static_cast<long>(ceil(scr_width / super::rect()[0] * super::rect()[1]));
        RecVector2<char> screen(scr_width, scr_height, ' ');
        const double scr_x_mul = scr_width / super::rect()[0];
        const double scr_y_mul = scr_height / super::rect()[1];

// Write each core into the screen.
        MAP(core_i, core_.size()) {
                const PRect<2, double> & r = core_[core_i];

                const double min_x_d = r.min_pos(0) * scr_x_mul;
                const double max_x_d = r.max_pos(0) * scr_x_mul;

                const double min_y_d = r.min_pos(1)  * scr_y_mul;
                const double max_y_d = r.max_pos(1) * scr_y_mul;

                const long min_x = static_cast<long>(min_x_d);
                const long max_x = static_cast<long>(max_x_d);
                const long min_y = static_cast<long>(min_y_d);
                const long max_y = static_cast<long>(max_y_d);

                for (long x = min_x; x < max_x; ++x) {
                        for (long y = min_y; y < max_y; ++y) {
				//We have overlap.
                                //RASSERT(screen(x, y) == ' ');
                                screen(x, y) = index_to_char(core_i);
                        }
                }
        }

// Print it.
        os << "Shape: ";
        super::print_to(os);
        os << "\n";

        os << "Area: " << area() << "\n";
        os << "Area Efficiency: " << area_efficiency() * 100.0 << "%\n";

// Show the keys.
        MAP(x, core_.size()) {
                os << x << ":" << index_to_char(x);
                if ((x + 1) % 10) {
                        os << " ";
                } else {
                        os << "\n";
                }
        }
        os << "\n\n";

// Show the screen.
        MAP(x, screen.size()[1]) {
                MAP(y, screen.size()[0]) {
                        os << screen(y, x);
                }
                os << "\n";
        }
}

/*===========================================================================*/
bool
SAFloorPlan::anneal() {
	generate();
	calc_costs();

	bool accept = evaluate();

	if (accept) {
		//cout<<"Accepted "<<endl;
		core_ = try_core_;
		wire_cost_ = try_wire_cost_;
		overlap_cost_ = try_overlap_cost_;
		dist_ = try_dist_;
		super::rect()[0] = bounding_box_.rect()[0];
		super::rect()[1] = bounding_box_.rect()[1];
		super::pos()[0] = bounding_box_.pos()[0];
		super::pos()[1] = bounding_box_.pos()[1];
	}

	return accept;
}

/*===========================================================================*/
void SAFloorPlan::
generate()
{
	try_core_ = core_;
	Action act =
		static_cast<Action>(RGen::gen().flat_range_l(0, MAX_ACTION + 1));
//The probability for each move should not be the same. JIGGLE may be more desirable at the beginning
	if(overlap_cost_>0.0)
		act = JIGGLE;
	switch (act) {
	case SWAP_POS:
		//cout<<"SWAP_POS ";
		swap_pos();
		break;

	case ROTATE:
		//cout<<"ROTATE ";
		rotate();
		break;

	case JIGGLE:
		//cout<<"JIGGLE ";
		jiggle();
		break;

	default:
		Rabort();
	}
	mutual_repel();
}

/*===========================================================================*/
void SAFloorPlan::swap_pos() {
// Pick two cores at random.
	int a = RGen::gen().flat_range_l(0, core_.size());
	int b = RGen::gen().flat_range_l(0, core_.size());

	if (a == b) return;

	swap(try_core_[a].pos(), try_core_[b].pos());
}

/*===========================================================================*/
void SAFloorPlan::rotate() {
// Pick a core at random.
	int x = RGen::gen().flat_range_l(0, core_.size());
	try_core_[x].swap_dim(0, 1);
}

/*===========================================================================*/
void SAFloorPlan::jiggle() {
	const double scalar = (pos()[0] + pos()[1]) * heat_ * 4.0;

// For all cores.
	MAP(i, try_core_.size()) {
		double x = RGen::gen().gauss_mean_d(0.0, 1.0);
		double y = RGen::gen().gauss_mean_d(0.0, 1.0);
		try_core_[i].pos()[0] += scalar * x;
		try_core_[i].pos()[1] += scalar * y;
	}
}

/*===========================================================================*/
void SAFloorPlan::mutual_repel() {


}
/*===========================================================================*/
void SAFloorPlan::calc_costs() {
// Push cores away from each other relative to temperature.
/*
cout << "A:\n";
cout << try_core_;
	for (int i = 0; i < core_.size(); ++i) {
		for (int j = i + 1; j < core_.size(); ++j) {
			mutual_repel(try_core_[i], try_core_[j], 1.0);

cout << "B:\n";
cout << try_core_;

// Push cores within IC borders.
			MAP(x, try_core_.size()) {
				conform(try_core_[x], (*this), 1.0);
			}

cout << "C:\n";
cout << try_core_;
		}
	}
*/

// Find the bounding box 
	bounding_box_ = PRect<2, double>(Pos<2, double>(0.0, 0.0), Rect<2, double>(0.0, 0.0));
	MAP(x, try_core_.size()) {
		bounding_box_.make_union(try_core_[x]);
	}
	bounding_box_.rect() = plump(bounding_box_.rect());
	normalize_bounding_box(bounding_box_, try_core_);
// Find overlap.
	try_overlap_cost_ = overlap_cost(try_core_);

/*	
// Find MST.
	MST<2, double> mst(&dist_rect<2, double>, try_core_.begin(),
				try_core_.end());
// Find communication costs.
			try_wire_cost_ = mst.total_len();
*/

//Get the wire cost
	try_dist_ = estimate_dist(try_core_);
	try_wire_cost_ =  compute_wire_cost(tie_, try_dist_);	
	cout<<"Total area = "<<bounding_box_.area()<<"  Overlap cost = "<<try_overlap_cost_
			<<"   Wire cost = "<<try_wire_cost_<<endl;
}
//===========================================================================
double 
SAFloorPlan::overlap_cost(const RVector< PRect<2, double> > & cores) const {
	double _cost(0.0);
	MAP(i, cores.size()) {
		PRect<2, double> a = cores[i];
		for( int j = i; j < cores.size(); j++) {
			/*
			if(a.intersect(cores[j])) {
				_cost += a.area();
			}
			*/
			_cost +=overlap_cost( cores[i], cores[j]);
		}
	}
	cout<<_cost<<endl;
	return _cost;
}	
//==========================================================================
double SAFloorPlan::overlap_cost(const PRect<2, double> & a, 
		const PRect<2, double> & b) const
{
	double x = GDFloorPlan::line_overlap(a.min_pos(0),a.max_pos(0),
		       			b.min_pos(0), b.max_pos(0));
        double y = GDFloorPlan::line_overlap(a.min_pos(1),a.max_pos(1),
	                                b.min_pos(1), b.max_pos(1));
	return x * y;
}
//===========================================================================
Rect<2, double>
SAFloorPlan::plump(const Rect<2, double> & r) const {
        Rect<2, double> ret = r;
        const double min_height = ret[0] / max_aspect_ratio_;
        ret[1] = max(min_height, ret[1]);

        return ret;
}
void SAFloorPlan::normalize_bounding_box(PRect<2, double> & bb, RVector<PRect<2, double> > & blocks) {
	double x(bb.pos()[0]), y(bb.pos()[1]), w(bb.rect()[0]/2.0), h(bb.rect()[1]/2.0);
	MAP(i, blocks.size()) {
		blocks[i].pos()[0] += (w - x);	
		blocks[i].pos()[1] += (h - y);
	}
	bb.pos()[0] = bb.rect()[0]/2.0;	
	bb.pos()[1] = bb.rect()[1]/2.0;
}
/*===========================================================================*/
bool SAFloorPlan::evaluate() {
	double wire_cost_improve = wire_cost_ - try_wire_cost_;
	double overlap_cost_improve = overlap_cost_ - try_overlap_cost_;

	double improve = cost() - try_cost();
	if(overlap_cost_improve<0.0) {
		return false;
	}
	if (improve > 0.0) {
		return true;
	}

	double goodness = exp(improve / heat_);

	double rnd = RGen::gen().flat_range_d(0.0, 1.0);
	eps_greater<double> gr;
	return gr(goodness, rnd);
}

/*===========================================================================*/
double
SAFloorPlan::cost() const{
	return overlap_cost_ * (1.0 - heat_) + wire_cost_ * heat_;
}

/*===========================================================================*/
double SAFloorPlan::try_cost() const {
	return try_overlap_cost_ * (1.0 - heat_) + try_wire_cost_ * heat_;
}

/*===========================================================================*/
void SAFloorPlan::self_check() const {
	Rabort();
}
/*=========================================================================*/
char
SAFloorPlan::index_to_char(long i) const {
        return 'A' + i % 52;
}


/*###########################################################################*/
void mutual_repel(PRect<2, double> & a, PRect<2, double> & b,
double pressure) {
/* Move position rectangles away from each other such that they will
touch but not overlap if pressure == 1.0. */
	Rabort();
}

/*===========================================================================*/
void conform(PRect<2, double> & a, PRect<2, double> & b,
double pressure) {
/* Move position rectangle a within position rectangle b such that
it will touch the border but not overlap if pressure == 1.0. */
	Rabort();
}
////////////////////////////////////////////////////////////////////////////////
//Added for interconnection borrowed from FloorPlan2
#define WIRE_LENGTH_SCALE (1.0) 

//Manhattan Distance is used 
AssocVec<double> SAFloorPlan::estimate_dist(const RVector<PRect<2, double>  > & cores) const {
	AssocVec<double> _dist(cores.size(), 0.0);
        for(long i = 0; i<cores.size(); i++){
          for(long j = i + 1; j<cores.size(); j++){
                double x1_center = cores[i].pos()[0] - cores[j].pos()[0];
                double y1_center = cores[i].pos()[1] - cores[j].pos()[1];
                _dist(i,j)= (x1_center>0) ? x1_center:(-x1_center);
                _dist(i,j)+= (y1_center>0) ? y1_center:(-y1_center);
          }
        }
        return _dist;
}

//Print out the links and their priorities and lengths
void SAFloorPlan::print_links(ostream &os) const {
        for(long i = 0;i<core_.size();i++){
          for(long j = i+1;j<core_.size();j++){
                if(tie_(i,j)>0){
                        os<<"("<<(char)('A'+i%52)<<", "<<(char)('A'+j%52)<<"): "<<" SW = "<<tie_(i,j)<<" Len = "<<dist_(i,j)<<endl;
                  }
          }
        }
	os<<"Total wire cost = "<<wire_cost_<<endl;
        return;
}

//Generate the total interconnection switching capacitance assuming unit capacitance 
//per length unit
double SAFloorPlan::compute_wire_cost(const AssocVec<double> & links,
	       	const AssocVec<double> & distance ) const {

        double total = 0.0;
        for(long i = 0; i<links.size();i++){
          for(long j = i + 1; j< links.size(); j++) {
                  total += (links(i,j)*distance(i,j));
          }
        }
        return total;
}

