#ifndef I_SET_H_
#define I_SET_H_

/*###########################################################################*/
#ifdef ROB_DEBUG
#	include "RStd.h"
#	include <typeinfo>
#endif

#include "RMath.h"
#include "Interface.h"

#include <set>
#include <iosfwd>
#include <utility>
#include <climits>

template <typename, typename, typename> class Triple;

/*###########################################################################*/
// Time or space interval to be used in interval set.

template <typename T>
class Interval :
	public Prints<Interval<T> >,
	public Comps<Interval<T> >,
	public SChecks<Interval<T> >,
	public Clones<Interval<T> >,
	public Swaps<Interval<T> >
{
	typedef Interval<T> self;

public:
	typedef long time_type;
	static const long TTYPE_MAX = LONG_MAX;

	Interval(time_type start, time_type duration, time_type period,
	  const T & data);

	~Interval() throw() {}

// Interface
	void print_to(ostream & os) const;
	self * clone() const { return new self(*this); }
	void rswap(self & a) { self tmp = *this; *this = a; a = tmp; }
	void self_check() const;
	void self_check_deep() const { self_check(); try_scheck_deep(data_); }	
// This doesn't pay attention to data.
	comp_type comp(const self & a) const;

	time_type raw_start() const { return raw_start_; }
	time_type start() const { return start_; }
	time_type duration() const { return duration_; }
	time_type finish() const { return start_ + duration() - 1; }
	const T & data() const { return data_; }

private:
		time_type period_;
		time_type raw_start_;
		time_type start_;
		time_type duration_;
		T data_;
};

/*###########################################################################*/
// Set of intervals for use in scheduling.

template <typename T, typename ITV = Interval<T> >
class ISet :
	public Clones<ISet<T, ITV> >,
	public Swaps<ISet<T, ITV> >,
	public SChecks<ISet<T, ITV> >
{
	typedef set<ITV> impl;
	typedef ISet self;

public:
// Typedefs
	typedef typename ITV::time_type time_type;
	static const long TTYPE_MAX = ITV::TTYPE_MAX;

	typedef typename impl::key_type key_type;
	typedef typename impl::key_type value_type;
	typedef typename impl::key_compare key_compare;
	typedef typename impl::value_compare value_compare;
	typedef typename impl::pointer pointer;
	typedef typename impl::const_pointer const_pointer;
	typedef typename impl::reference reference;
	typedef typename impl::const_reference const_reference;
	typedef typename impl::iterator iterator;
	typedef typename impl::const_iterator const_iterator;
	typedef typename impl::reverse_iterator reverse_iterator;
	typedef typename impl::const_reverse_iterator const_reverse_iterator;
	typedef typename impl::size_type size_type;
	typedef typename impl::difference_type difference_type;

// Construction
	virtual ~ISet() throw() {}

	ISet(time_type period);

	template <typename InputIterator>
		ISet(time_type period, InputIterator first, InputIterator last);

	virtual self & operator=(const self & a);

// Interface
	virtual void self_check() const;
	virtual void self_check_deep() const;
	virtual void rswap(self & x);
	virtual self * clone() const { return new self(*this); }

// Modifiable
	virtual void gap_print_to(ostream & os, const char * sep = "\n") const;

	virtual const pair<iterator, bool> insert(const value_type & x);
	virtual iterator insert(iterator position, const value_type & x);
	virtual void erase(iterator position);
	virtual size_type erase(const key_type & x);
	virtual void erase(iterator first, iterator last);
	virtual void clear();

// Final
	void print_to_default(ostream & os) const;
	time_type period() const { return period_; }

// Returns iterator for scheduled event or end() if scheduling not possible.
	virtual iterator sched(time_type earliest, time_type duration,
		const T & data);

	void wrap_inc(iterator & i) const { if (++i == end()) i = begin(); }
	void wrap_dec(iterator & i) const { if (i == begin()) i = end(); i--; }
	time_type time_gap_duration(time_type from, time_type to) const;
	time_type gap_duration(const_iterator from, const_iterator to) const;
	time_type prev_gap_duration(const_iterator a) const;
	time_type next_gap_duration(const_iterator a) const;
	time_type gap_duration(time_type start, time_type end) const;

	template <typename InputIterator>
		void insert(InputIterator first, InputIterator last);

	iterator begin() { return impl_.begin(); }
	const_iterator begin() const { return impl_.begin(); }
	iterator end() { return impl_.end(); }
	const_iterator end() const { return impl_.end(); }
	iterator rbegin() { return impl_.rbegin(); }
	const_iterator rbegin() const { return impl_.rbegin(); }
	iterator rend() { return impl_.rend(); }
	const_iterator rend() const { return impl_.rend(); }
	bool empty() const { return impl_.empty(); }
	size_type size() const { return impl_.size(); }
	size_type max_size() const { return impl_.max_size(); }

	iterator find(const key_type & x) const { return impl_.find(x); }
	size_type count(const key_type & x) const { return impl_.count(x); }
	iterator lower_bound(const key_type & x) const;
	iterator upper_bound(const key_type & x) const;
	pair<iterator, iterator> equal_range(const key_type & x) const;

protected:
	virtual void fix_gap_insert(iterator a);
	virtual void fix_gap_erase(iterator a);

	typedef multiset<time_type> GapSet;

private:
		impl impl_;
		time_type period_;
		GapSet gap_;
};

/*===========================================================================*/
template <typename T1, typename ITV1, typename T2, typename ITV2>
	const pair<ISet<T1, ITV1>::iterator, typename ISet<T2, ITV2>::iterator>
	sched2(ISet<T1, ITV1>::time_type earliest,
	ISet<T1, ITV1>::time_type duration,
	const T1 & data_a, const T2 & data_b, ISet<T1, ITV1> & a, ISet<T2, ITV2> & b);

template <typename T1, typename ITV1, typename T2, typename ITV2,
	typename T3, typename ITV3>
	const Triple<ISet<T1, ITV1>::iterator, typename ISet<T2, ITV2>::iterator,
	typename ISet<T3, ITV3>::iterator>
	sched3(ISet<T1, ITV1>::time_type earliest, ISet<T1, ITV1>::time_type duration,
	const T1 & data_a, const T2 & data_b, const T3 & data_c,
	ISet<T1, ITV1> & a, ISet<T2, ITV2> & b, ISet<T3, ITV3> & c);

/*###########################################################################*/
void ISet_test();

/*###########################################################################*/
#include "ISet.cct"
#endif
