#ifndef REF_ADAPTER_H_
#define REF_ADAPTER_H_

/*###########################################################################*/
#ifdef ROB_DEBUG
#	include "RStd.h"
#endif

#include "Interface.h"

#include <iterator>
#include <algorithm>

/*===========================================================================*/
template <typename> class ref_iterator;
template <typename> class iterator_traits;

/*===========================================================================*/
// Iterator that double-dereferences.  Useful for containers of pointers.

template <> template <typename I>
struct iterator_traits<ref_iterator<I> > {
	typedef typename iterator_traits<I>::iterator_category iterator_category;

	typedef typename iterator_traits<
		typename iterator_traits<I>::value_type>::value_type value_type;

	typedef typename iterator_traits<I>::difference_type difference_type;

	typedef typename iterator_traits<
		typename iterator_traits<I>::value_type>::pointer pointer;

	typedef typename iterator_traits<
		typename iterator_traits<I>::value_type>::reference reference;
};

/* Note that this is not a STL-conforming iterator, as operator* returns
something other than a T &.  However, it is useful for its intended
purpose. */

template <typename I>
class ref_iterator {
	typedef ref_iterator<I> self;

public:
	typedef typename iterator_traits<self>::iterator_category iterator_category;
	typedef typename iterator_traits<self>::value_type value_type;
	typedef typename iterator_traits<self>::difference_type difference_type;
	typedef typename iterator_traits<self>::pointer pointer;
	typedef typename iterator_traits<self>::reference reference;

	~ref_iterator() throw() {}
	ref_iterator(I i) : iter_(i) {}

	I base() const { return iter_; }

 	reference operator*() const { return ** iter_; }
	pointer operator->() const { return * iter_; }

	self & operator++() { iter_++; return * this; }
	const self operator++(int) { self tmp = * this; iter_++; return tmp; }
	self & operator--() { iter_--; return * this; }
	const self operator--(int) { self tmp = * this; iter_--; return tmp; }

	const self operator-(difference_type n) { return self(iter_ - n); }
	const self operator+(difference_type n) { return self(iter_ + n); }
	reference operator[](difference_type n) { return **(iter_ + n); }
	const reference operator[](difference_type n) const;

// Interface
	bool operator==(const self & a) const { return iter_ == a.iter_; }
	difference_type operator-(const self & a) const { return iter_ - a.iter_; }
	self operator+(difference_type & a) const { return iter_ + a.iter_; }

private:
		I iter_;
};

/*===========================================================================*/
// Adapter that gives a reference-based view of a container of pointers.

template <typename C>
class RefAdapter :
	public Clones<RefAdapter<C> >,
	public Swaps<RefAdapter<C> >
{
	typedef RefAdapter self;

public:
// Typedefs
	typedef ref_iterator<typename C::iterator> iterator;
	typedef ref_iterator<typename C::const_iterator> const_iterator;
	typedef typename iterator::reference reference;
	typedef const typename iterator::reference const_reference;
	typedef typename C::size_type size_type;
	typedef reverse_iterator<const_iterator> const_reverse_iterator;
	typedef reverse_iterator<iterator> reverse_iterator;

// Construction
	virtual ~RefAdapter() throw() {}
	RefAdapter(C & cont) : cont_(& cont) {}

// Interface
	virtual void rswap(self & a) { ::swap(cont_, a.cont_); }
	virtual self * clone() const { return new self(*this); }

// Final
	void attach(C & cont) { cont_ = & cont; }

	C & container() { return * cont_; }
	const C & container() const { return * cont_; }

	iterator begin() { return iterator(cont_->begin()); }
	const_iterator begin() const { return const_iterator(cont_->begin()); }
	iterator end() { return iterator(cont_->end()); }
	const_iterator end() const { return const_iterator(cont_->end()); }

	iterator rbegin() { return reverse_iterator(cont_->end()); }
	const_iterator rbegin() const;
	iterator rend() { return reverse_iterator(cont_->begin()); }
	const_iterator rend() const;
	size_type size() const { return cont_->size(); }
	bool empty() const { return cont_->empty(); }
	reference front() { return * begin(); }
	const_reference front() const { return * begin(); }
	reference back() { return *(end() - 1); }
	const_reference back() const { return *(end() - 1); }

	reference operator[](size_type a) { return * cont_->operator[](a); }
	const_reference operator[](size_type a) const { return runconst(* this)[a]; }

private:
		C * cont_;
};

/*===========================================================================*/
void RefAdapter_test();

/*###########################################################################*/
#include "RefAdapter.cct"
#endif
