/********************************************************************/
/*******		File: fsm.h				*****/
/*******  Contains definition of the FSM class			*****/
/********************************************************************/

// The FSM design currently supports only Moore-type machines
// which is sufficient to get by in PhaseI

#ifndef FSM_H
#define FSM_H

#include "matrix.h"
#include "list_ar.h"
#include "dpunit.h"

#ifdef _SCALP_
#include "entity.h"
#include "compinst.h"
#endif

class FSM : public Datapath_element
{
protected:
// This structure stores a column of the action matrix as a Boolean array.
struct Output_column {
char* output_name;
Array<Boolean>* action_table_column;
Output_column() {output_name=NULL; action_table_column=NULL;}
~Output_column() {if (output_name) delete [] output_name;
	   if (action_table_column) delete action_table_column;
	  }
};
int number_of_states;

// Action table is a states (rows) vs. output (columns) matrix which
// stores information about in which states a certain output goes high.
// In the action table, only TRUE values have to be set
List_ar<Output_column*> action_table;	// Encapsulates output generation

// Decision table is a transitions (rows) vs. intputs (columns) matrix which
// stores information about conditions of a particular transition.
// Not used presently.
Matrix<Boolean> *decision_table;  	// Encapsulates next_state generation

Output_column* current_output;		// A pointer to the current column

int number_of_control;			// index for control logic entity names
// A temporary storage for VHDL views of the FSM.
// Since there can be more than one control in general (distributed), 
// 'control' should probably be made a list.
// These two members are initialized by get_counter_entity and 
// get_control_entity, respectively.
#ifdef _SCALP_
Entity* counter;
Entity* control;
#endif

struct FSM_transition
{
// It seems sufficient to represent a state as an integer.
// 0 will correspond to the initial/reset state.
int from_state;
int to_state;
FSM_transition(void) {from_state = to_state = -1;}
FSM_transition(int fst, int tst)
	{from_state = fst; to_state = tst;}
FSM_transition(const FSM_transition &fsmt)
	{from_state = fsmt.from_state; to_state = fsmt.to_state;}
~FSM_transition(void) {}
};

List_ar<FSM_transition*> transitions;
List_ar<const char*> inputs;

public:
FSM(int n_of_states) : transitions(), inputs(), action_table() {
  FRITS_SET_CLASS("FSM");
  number_of_states = n_of_states;
  decision_table = NULL;
  current_output = NULL;
  number_of_control = 0;
#ifdef _SCALP_
  counter = NULL;
  control = NULL;
#endif
}
~FSM(void) {}
int get_number_of_states(void)		{return number_of_states;}
FSM* add_transition(FSM_transition* ft)	{transitions.append(ft); return this;}
int get_number_of_transitions(void)	{return transitions.get_size();}
FSM* add_input(const char* name)  
	{inputs.append(strdup(name)); return this;}
int get_number_of_inputs(void)		{return inputs.get_size();}	
FSM* add_output(const char* name);	
int get_number_of_outputs(void)		{return action_table.get_size();}
void initialize(void);  		// Not presently used

// This message selects a row in the action table corresponding to the output
// with output_name. See optimization note in list_ar.h for get_position()
Boolean select_output(const char* output_name);

// Note: state_count is 0-based: [0, number_of_states).
FSM* set_active_state(int state_count, const char* output_name = NULL);

// Assigns binary vector_value to vector_size cells (top-down) in the 
// state_count state. Starts from current_output_index or the output with
// output_name. Stops if hits the last row before vector_size is up.
// Used primarily for specifying fsm output for mux-tree select signals.
FSM* set_vector_active_state(int state_count, int vector_value, int vector_size, const char* output_name = NULL);

// A helper message for Scheduler::create_fsm().
// Based on the select_index value, creates an output of the FSM, 
// a select signal of the interconnect_unit, and connects them together
// with a new net which is returned.
Net* connect_to_mux_select(int select_index, Interconnect_unit*& iu);

#ifdef _SCALP_
// Gets global FSM
Entity* get_entity(void);
// Gets the counter. Counter has a one-hot vector output by default.
Entity* get_counter_entity(void);
// Gets the control glue logic lump generating outputs with 'output_names'
// Inactive_low defines an inactive signal for outputs without entries
// in the action_table. Default is '0', to get '1' specify FALSE.
// NULL for a list gets a control entity with all FSM output ports.
Entity* get_control_entity(List_ar<const char*> *output_names = NULL, Boolean inactive_low = TRUE);
virtual Component_instantiation* create_component_instantiation(Architecture* arch, const char* label);
#endif /* _SCALP_ */

#ifndef _SCALP_
inline char *get_output_name(int index) {
  return(action_table[index]->output_name);
}

// Print out a KISS format description of the FSM into
// a file with the given name
void write_kiss(char *);
#endif

};
#endif /* FSM_H */
