/**********************************************************/
/* This file contains the interface between the datapath
 * and floorplanner etc. Ideally, the floorplan and physical 
 * package should be incorporated into the high level synthesis
 * tool easily with minimal code changes        
 * Lin Jan. 26 2002  					*/
#include "scm.h"
#include"datapath.h"
#include <utility>
#include "i_macro.h"

////////////////////////////////////////////////////////////////////////
void Datapath::do_floorplan(library *lib, Scm *scm) {
	generate_module_size(lib);
	generate_module_link(lib, scm);
	unsigned long reg_num = storage_units.get_size();
	unsigned long fu_num = functional_units.get_size();
	RVector<unsigned long> clocked_cores(reg_num);
	for(unsigned long i = 0; i< reg_num; i++) {
		clocked_cores[i] = i + fu_num;
	}
	assert(module_size.size()==module_link.size());
	floorplan.solve(module_size, clocked_cores, 2.0, module_link);
	floorplan.set_ready(true);
	area_after_floorplan = floorplan.area();
#ifdef PHYSICAL_DEBUG
	cout<<"Datapath::do_floorplan() ";
	cout<<" FU # = "<<functional_units.get_size();
	cout<<" Reg # = "<<storage_units.get_size();
	cout<<" Mux # = "<<interconnect_units.get_size()<<endl;
	cout <<" Datapath area efficiency: "<<floorplan.area_efficiency();
	cout <<" Total datapath area is "<<area_after_floorplan<<endl;
#endif
	//estimate of wiring area

	//estimate of clock network area

	return;
}

/////////////////////////////////////////////////////////////////////////
//Display the floorplan
void Datapath::display_floorplan(ostream & os) const
{
	os<<endl<<"*************************************************************************"<<endl;
        floorplan.print_to(os);
        floorplan.print_links(os);
	/*
        delete floorplan;
	*/
        return;
}

/////////////////////////////////////////////////////////////////////////
//Display the map between modules on the FP and the datapath units.
void Datapath::display_module_map(ostream &os) const{
        FUPTR fu;
        List_iterator<FUPTR> fuscan;
        NODEPTR node;
        List_iterator<NODEPTR> nodescan;
        Storage_unit *reg;
        List_iterator<Storage_unit *> regscan;
        EDGEPTR var;
        List_iterator<EDGEPTR> varscan;
	Interconnect_unit *mux;
	List_iterator<Interconnect_unit *> muxscan;
	os<<endl<<"*************************************************************************"<<endl;
        unsigned long index = 0;
        FOR_EACH_LISTNODE(functional_units, fuscan){
                fu = fuscan.get_item();
                assert(fu);
                os<<"#"<<index<<" < ";
                FOR_EACH_LISTNODE(fu->get_operations(),nodescan){
                        node = nodescan.get_item();
                        assert(node);
                        os<<node->get_name()<<" ";
                }
                os<<">"<<endl;
                index++;
        }
        FOR_EACH_LISTNODE(storage_units, regscan){
                reg = regscan.get_item();
                assert(reg);
                os<<"@"<<index<<" < ";
                FOR_EACH_LISTNODE(((STORPTR)reg)->get_variables(),varscan){
                        var = varscan.get_item();
                        assert(var);
                        os<<var->get_name()<<" ";
                }
                os<<">"<<endl;
                index++;
        }
	index = 0;
	FOR_EACH_LISTNODE(interconnect_units, muxscan) {
		mux = muxscan.get_item();
		assert(mux);
		if(mux->get_number_of_nets()>2) {
			os<<"MUX"<<(mux->get_number_of_nets()-1)<<"_to_1#"<<index<<endl;
			index++;
		}
	}
        return;
}

/////////////////////////////////////////////////////////////////////////
//Display the map betweeen links on the FP and the datapath interconnect
void Datapath::display_link_map(ostream &os) const {

        List_iterator<FUPTR> fuscan;
        FUPTR fu;
        EDGEPTR output_edge, edge;
        List_iterator<EDGEPTR> output_edgescan, edgescan;
        Storage_unit * reg;
        List_iterator<Storage_unit *> regscan;
        NODEPTR node;
        List_iterator<NODEPTR> nodescan;

	os<<endl<<"*************************************************************************"<<endl;
//////////The following codes could be simplified using get_fu_output() and get_reg_output()///////////
	int fu_num = functional_units.get_size();
	int fu_index = 0;
	FOR_EACH_LISTNODE(functional_units, fuscan){
          fu = fuscan.get_item();
          assert(fu);
          List_ar<EDGEPTR> output_list;
	  list< pair<int, Storage_unit *> > reg_list;
          FOR_EACH_LISTNODE(fu->get_operations(),nodescan){
            node = nodescan.get_item();
            assert(node);
            FOR_EACH_LISTNODE(node->get_output_edges(),output_edgescan){
              output_edge =output_edgescan.get_item();
              assert(output_edge);
              //Different operations do not have same output edge;
              assert(output_list.find(output_edge)==NULL);
              output_list.append(output_edge);
              int reg_index = 0;
              FOR_EACH_LISTNODE(storage_units,regscan){
                reg = regscan.get_item();
                assert(reg);
                if((((STORPTR)reg)->get_variables()).find(output_edge)!=NULL){
		  //FU feed into this reg
		  pair<int, Storage_unit *> tmp(reg_index, reg);	
                  list< pair<int, Storage_unit *> >::iterator result = 
			find(reg_list.begin(), reg_list.end(),tmp);
                  if(result==reg_list.end()){//not found
                    reg_list.push_back(tmp);
                  }
                }//The output of fu feeds into reg
                 reg_index++;
              }//For each register
            }//for each node output edge
          }//for each operation bound to fu
	  os<<"FU "<<fu_index<<" outputs < ";
	  FOR_EACH_LISTNODE(output_list, output_edgescan){
	    output_edge = output_edgescan.get_item();
	    assert(output_edge);
	    os<<output_edge->get_name()<<"  ";
	  }//for each output
	  os<<"> ("<<output_sw[fu_index]<<") to REG"<<endl; 
          for(list< pair<int, Storage_unit *> >::iterator i=reg_list.begin();
						i!=reg_list.end();i++){
	    os<<((*i).first+fu_num)<<"< ";
	    FOR_EACH_LISTNODE(((STORPTR)((*i).second))->get_variables(), edgescan){
	      edge = edgescan.get_item();
	      assert(edge);
	      os<<edge->get_name()<<"  ";
	    }
	    os<<">"<<endl;		    
          }
          fu_index++;
	}//for each fu;

	//Second, check the output of a register
        int reg_index = 0;
        FOR_EACH_LISTNODE(storage_units, regscan){
          reg = regscan.get_item();
          assert(reg);
          List_ar<EDGEPTR> output_list;
          output_list.clear();
	  list< pair<int, FUPTR> > fu_list; 
          //list<int> reg_output;
          //list<int> fu_list;
          FOR_EACH_LISTNODE( ((STORPTR)reg)->get_variables(),output_edgescan){
            output_edge = output_edgescan.get_item();
            assert(output_edge);
            //Different registers do not have same variable;
            assert(output_list.find(output_edge)==NULL);
            output_list.append(output_edge);
            //reg_output.push_back(output_edge->get_address());
            int fu_index = 0;
            FOR_EACH_LISTNODE(functional_units,fuscan){
              fu = fuscan.get_item();
              assert(fu);
              FOR_EACH_LISTNODE(fu->get_operations(), nodescan){
                node = nodescan.get_item();
                assert(node);
		pair<int, FUPTR> tmp(fu_index,fu);
                if((node->get_input_edges()).find(output_edge)!=NULL){
                  list< pair<int, FUPTR> >::iterator result = 
			find(fu_list.begin(), fu_list.end(),tmp);
                  if(result==fu_list.end()){//not found
                    fu_list.push_back(tmp);
                  }
                }//The output of reg feeds into fu
              }//each node bound to this fu           
              fu_index++;
            }//For each fu 
          }  //for each variable bound to reg
	  os<<"REG "<<fu_num+reg_index<<" outputs <";
	  FOR_EACH_LISTNODE(output_list, output_edgescan){
	    output_edge = output_edgescan.get_item();
	    assert(output_edge);
	    os<<output_edge->get_name()<<"  ";
	  }//for each output
	  os<<"> ("<<output_sw[fu_num+reg_index]<<") to FU"<<endl;
          for(list< pair<int, FUPTR> >::iterator i=fu_list.begin();i!=fu_list.end();i++){
	    os<<(*i).first<<" <";	  
	    FOR_EACH_LISTNODE((*i).second->get_operations(), nodescan){
	      node = nodescan.get_item();
	      assert(node);
	      os<<node->get_name()<<"  ";
	    }
	    os<<">"<<endl;
	  }
          reg_index++;
        }//for each reg;
        return;

}

/*****************************************************/
/* Put the Class Datapath specific physical information
 * into floorplanner.		*/
/////////////////////////////////////////////////////////////////////////
//Create the dimension information about modules from the Datapath
void Datapath::generate_module_size(library *lib)
{
  register FUPTR fu;
  register STORPTR reg;
  register Mux *mux;
  int num_inputs;
  float total = 0.0;
  List_iterator<FUPTR> fuscan;
  List_iterator<Storage_unit *> regscan;


  double module_area;
  double x, y;
  module_size.clear();

  assert(lib);
  //functional unit area
  FOR_EACH_LISTNODE(functional_units, fuscan) {
    fu = fuscan.get_item();
    assert(fu);
    //Temporarily we assume all modules are square(hard modules).
    module_area = fu->get_libelement()->get_area();
    x = y = sqrt(module_area)*TECH_SCALE * lib->get_lamda_scale_down();
    module_size.push_back(Rect<2, double>(x, y));
    //Get the interconnection of this module;
  }
  int reg_index_head(module_size.size() - 1);
  //register area
  FOR_EACH_LISTNODE(storage_units, regscan) {
    assert(!strcmp(regscan.get_item()->get_id(),"Register"));
    reg = (STORPTR) regscan.get_item();
    assert(reg);
   //Temporarily we assume all modules are square(hard modules).
    module_area =lib->get_reg_bit_area()*get_bitwidth();
    x = y = sqrt(module_area)*TECH_SCALE * lib->get_lamda_scale_down();
    module_size.push_back(Rect<2, double>(x, y));
  }
  assert(module_size.size()==(functional_units.get_size()+storage_units.get_size()));
  return;
}

///////////////////////////////////////////////////////////////////////
//Create the module interconnection information from the Datapath
void Datapath::generate_module_link(library *lib, Scm *scm)
{
        List_iterator<FUPTR> fuscan;
        FUPTR fu;
        Storage_unit * reg;
        List_iterator<Storage_unit *> regscan;
        const long fu_num = functional_units.get_size();
        const long reg_num = storage_units.get_size();

	output_sw.clear();
        module_link.clear();
        module_link.resize(fu_num+reg_num,0.0);
        output_sw.resize(fu_num+reg_num,0.0);
        
        //First, check the output of a functional units
        int fu_index = 0;
        FOR_EACH_LISTNODE(functional_units, fuscan){
          fu = fuscan.get_item();
          assert(fu);
	  Register reg_temp(NULL);
          vector<int> reg_list;
	  get_fu_output(reg_temp,reg_list, fu);
          float sw_cost = scm->get_total_wiresw(reg_temp.get_variables());
          for( int i= 0;i<reg_list.size(); i++ )  {
            module_link(fu_index,reg_list[i]+fu_num) = sw_cost;
          }
	  output_sw[fu_index] = sw_cost;
         fu_index++;
        }//for each fu;
	 //Second, check the output of a register
        int reg_index = 0;
        FOR_EACH_LISTNODE(storage_units, regscan )   {
          reg = regscan.get_item();
          assert( reg );
	  Register reg_temp(NULL);
          vector<int> fu_list;

	  get_reg_output(reg_temp, fu_list, reg);

          float sw_cost = scm->get_total_wiresw(reg_temp.get_variables());
          for(int i=0; i<fu_list.size(); i++){
            module_link(fu_list[i],reg_index+fu_num) +=sw_cost;
          }
	  output_sw[fu_num+reg_index] = sw_cost;
          reg_index++;
        }//for each reg;
	assert(module_link.size()==(fu_num+reg_num));
        return;
}

///////////////////////////////////////////////////////////////////////
//Create the output network information for a fu
//For estimation of the interconnect energy in the case of dedicated buses
float Datapath::generate_fu_output_network_energy(library *lib, Scm *scm, FUPTR fu, 
	unsigned long index)
{
        const long fu_num = functional_units.get_size();

        Register reg_temp(NULL);
        vector<int> reg_list;
        get_fu_output(reg_temp,reg_list, fu);
        float sw_cost = scm->get_total_wiresw(reg_temp.get_variables());
	assert(sw_cost==output_sw[index]);
	RVector<unsigned long>  connected_indice;
	connected_indice.push_back(index);
	for(int i=0; i<reg_list.size();i++){
		connected_indice.push_back((unsigned long)reg_list[i]+(unsigned long)fu_num);
	}		
	float total_length = floorplan.estimate_output_network_length( connected_indice);
	return total_length*sw_cost;
}
float Datapath::generate_reg_output_network_energy(library *lib, Scm *scm,Storage_unit *reg ,
        unsigned long index)
{
        const long fu_num = functional_units.get_size();

        Register reg_temp(NULL);
        vector<int> fu_list;
        get_reg_output(reg_temp,fu_list, reg);
        float sw_cost = scm->get_total_wiresw(reg_temp.get_variables());
	assert(sw_cost==output_sw[index+fu_num]);

        //generate connected_indice
        RVector<unsigned long>  connected_indice;
        connected_indice.push_back(index+(unsigned long)fu_num);
        for(int i = 0; i<fu_list.size(); i++){
                connected_indice.push_back((unsigned long)(fu_list[i]));
        }
        float total_length = floorplan.estimate_output_network_length( connected_indice);
        return total_length*sw_cost;
}

///////////////////////////////////////////////////////////////////////
//Create the output network information for a fu
//For estimation of the interconnect energy in the case of shared buses:trunk_branch style Steiner Tree
pair<float, float> Datapath::generate_fu_shared_output_network_energy(library *lib, Scm *scm, FUPTR fu,
        unsigned long index)
{
        const long fu_num = functional_units.get_size();

        Register reg_temp(NULL);
        vector<int> reg_list;
	vector<float> reduced_link;
        get_fu_reduced_output(reg_temp,reg_list, reduced_link, fu, scm);
        float sw_cost = scm->get_total_wiresw(reg_temp.get_variables());
        assert(sw_cost==output_sw[index]);
        RVector<unsigned long>  connected_indice;
        connected_indice.push_back(index);
        for(int i=0; i<reg_list.size();i++){
                connected_indice.push_back((unsigned long)reg_list[i]+(unsigned long)fu_num);
        }
	float total_sc, reduced_sc, total_sc2, reduced_sc2;
	
	RVector<float> trunk_branches;
        float total_length = floorplan.estimate_output_network_length( connected_indice, trunk_branches, true);
	total_sc = total_length*sw_cost;	
	reduced_sc = trunk_branches[0]*sw_cost;
	for( int i = 0; i<reg_list.size(); i++) {
		reduced_sc +=(trunk_branches[i+1]*reduced_link[i]);
	}	
	
	trunk_branches.clear();
        total_length = floorplan.estimate_output_network_length( connected_indice, trunk_branches, false);
        total_sc2 = total_length*sw_cost;
        reduced_sc2 = trunk_branches[0]*sw_cost;
        for( int i = 0; i<reg_list.size(); i++) {
                reduced_sc2 +=(trunk_branches[i+1]*reduced_link[i]);
        }
	if(reduced_sc>reduced_sc2) reduced_sc = reduced_sc2;
	if(total_sc>total_sc2) total_sc = total_sc2;
	
        return pair<float, float>(total_sc, reduced_sc);
}
pair<float, float> Datapath::generate_reg_shared_output_network_energy(library *lib, Scm *scm,Storage_unit *reg ,
        unsigned long index)
{
        const long fu_num = functional_units.get_size();

        Register reg_temp(NULL);
        vector<int> fu_list;
	vector<float> reduced_link;
        get_reg_reduced_output(reg_temp,fu_list, reduced_link,reg, scm);
        float sw_cost = scm->get_total_wiresw(reg_temp.get_variables());
        assert(sw_cost==output_sw[index+fu_num]);

        //generate connected_indice
        RVector<unsigned long>  connected_indice;
        connected_indice.push_back(index+(unsigned long)fu_num);
        for(int i = 0; i<fu_list.size(); i++){
                connected_indice.push_back((unsigned long)(fu_list[i]));
        }
        float total_sc, reduced_sc, total_sc2, reduced_sc2;

        RVector<float> trunk_branches;
        float total_length = floorplan.estimate_output_network_length(connected_indice, trunk_branches, true);
        total_sc = total_length*sw_cost;
        reduced_sc = trunk_branches[0]*sw_cost;
        for( int i = 0; i<fu_list.size(); i++) {
                reduced_sc +=(trunk_branches[i+1]*reduced_link[i]);
        }

        trunk_branches.clear();
        total_length = floorplan.estimate_output_network_length( connected_indice, trunk_branches, false);
        total_sc2 = total_length*sw_cost;
        reduced_sc2 = trunk_branches[0]*sw_cost;
        for( int i = 0; i<fu_list.size(); i++) {
                reduced_sc2 +=(trunk_branches[i+1]*reduced_link[i]);
        }

        if(reduced_sc>reduced_sc2) reduced_sc = reduced_sc2;
        if(total_sc>total_sc2) total_sc = total_sc2;

        return pair<float, float>(total_sc, reduced_sc);
}

///////////////////////////////////////////////////////////////////////////
void Datapath::get_fu_output(Register & reg_temp,vector<int> &reg_list, FUPTR fu) const {

	EDGEPTR output_edge, edge;
        List_iterator<EDGEPTR> output_edgescan, edgescan;
        Storage_unit * reg;
        List_iterator<Storage_unit *> regscan;
        NODEPTR node;
        List_iterator<NODEPTR> nodescan;

	const List_ar<EDGEPTR> &output_list = reg_temp.get_variables();
	
	assert(output_list.get_size()==0);
	assert(reg_list.size()==0);

        FOR_EACH_LISTNODE( fu->get_operations(), nodescan )  {
            node = nodescan.get_item();
            assert(node);

            //for every output edge of the node
            FOR_EACH_LISTNODE( node->get_output_edges(), output_edgescan )  {
              output_edge =output_edgescan.get_item();
              assert(output_edge);
              assert( output_list.find(output_edge)==NULL );
		//Different operations do not have same output edge;
              reg_temp.add_variable( output_edge );
              int reg_index = 0;

              //check every register to see whether they take output from the node
              FOR_EACH_LISTNODE( storage_units,  regscan )  {
                reg = regscan.get_item();
                assert( reg );
                //if this register does take output from the node
                if( (((STORPTR)reg)->get_variables()).find( output_edge )!=NULL )  {
                  vector<int>::iterator result = find(reg_list.begin(), reg_list.end(),reg_index);
                  if(  result==reg_list.end() )  {//not found
                      reg_list.push_back(reg_index);
		      //07242002
		      break;//no need to go on;
                  }
                }//The output of fu feeds into reg
                reg_index++;
              }//For each register
            }//for each node output edge
        }//for each operation bound to fu
	return;
}
////////////////////////////////////////////////
void Datapath::get_reg_output(Register &reg_temp, vector<int> & fu_list, Storage_unit *reg) const {
	EDGEPTR output_edge, edge;
        List_iterator<EDGEPTR> output_edgescan, edgescan;
        FUPTR fu;
        List_iterator<FUPTR> fuscan;
        NODEPTR node;
        List_iterator<NODEPTR> nodescan;

        const List_ar<EDGEPTR> & output_list  = reg_temp.get_variables();


        FOR_EACH_LISTNODE(((STORPTR)reg)->get_variables(),output_edgescan){
            output_edge =output_edgescan.get_item();
            assert(output_edge);
            //Different registers do not have same variable;
            assert(output_list.find(output_edge)==NULL);
            //output_list.append(output_edge);
            reg_temp.add_variable(output_edge);
            int fu_index = 0;
            FOR_EACH_LISTNODE(functional_units,fuscan){
              fu = fuscan.get_item();
              assert(fu);
              FOR_EACH_LISTNODE(fu->get_operations(), nodescan){
                node = nodescan.get_item();
                assert(node);
                if((node->get_input_edges()).find(output_edge)!=NULL){
                  vector<int>::iterator result = find(fu_list.begin(), fu_list.end(),fu_index);
                  if(result==fu_list.end()){//not found
                      fu_list.push_back(fu_index);
		      //07242002
		      break;//no need to go on;
                  }
                }//The output of reg feeds into fu
              }//each node bound to this fu
              fu_index++;
            }//For each fu
        }  //for each variable bound to reg

	return;
}
//08022002
void Datapath::get_reg_input(vector<int> & fu_list, Storage_unit *reg) const {
        EDGEPTR input_edge, edge;
        List_iterator<EDGEPTR> input_edgescan, edgescan;
        FUPTR fu;
        List_iterator<FUPTR> fuscan;
        NODEPTR node;
        List_iterator<NODEPTR> nodescan;

        FOR_EACH_LISTNODE(((STORPTR)reg)->get_variables(),input_edgescan){
            input_edge =input_edgescan.get_item();
            assert(input_edge);
            //Different registers do not have same variable;
            int fu_index = 0;
            FOR_EACH_LISTNODE(functional_units,fuscan){
              fu = fuscan.get_item();
              assert(fu);
              FOR_EACH_LISTNODE(fu->get_operations(), nodescan){
                node = nodescan.get_item();
                assert(node);
                if((node->get_output_edges()).find(input_edge)!=NULL){
                  vector<int>::iterator result = find(fu_list.begin(), fu_list.end(),fu_index);
                  if(result==fu_list.end()){//not found
                      fu_list.push_back(fu_index);
                      //07242002
                      break;//no need to go on;
                  }
                }//The output of reg feeds into fu
              }//each node bound to this fu
              fu_index++;
            }//For each fu
        }  //for each variable bound to reg

        return;
}

void Datapath::get_fu_input(Register & reg_temp, FUPTR fu) const 
{
	NODEPTR node;
	List_iterator<NODEPTR> nodescan;
	EDGEPTR var;
	List_iterator<EDGEPTR> edgescan;
	
	reg_temp.clear_variables();
        FOR_EACH_LISTNODE(fu->get_operations(), nodescan){
	   node = nodescan.get_item();
           assert(node);
	   FOR_EACH_LISTNODE(node->get_input_edges(), edgescan) {
		var = edgescan.get_item();
		assert(var);
		if(reg_temp.get_variables().find(var)==NULL) {
			reg_temp.add_variable(var);
		}
	   }
	}
	return;		
}
void Datapath::get_fu_input(vector<int> & reg_list, FUPTR fu) const
{
        NODEPTR node;
        List_iterator<NODEPTR> nodescan;
        EDGEPTR var;
        List_iterator<EDGEPTR> edgescan;
	List_ar<STORPTR> inputs;
	reg_list.clear();	
        FOR_EACH_LISTNODE(fu->get_operations(), nodescan){
           node = nodescan.get_item();
           assert(node);
           FOR_EACH_LISTNODE(node->get_input_edges(), edgescan) {
                var = edgescan.get_item();
                assert(var);
		if(inputs.find(var->get_storage_unit())==NULL){
			inputs.append(var->get_storage_unit());
                }
           }
        }
	for(int index = 0; index <storage_units.get_size();index++) {
		if(inputs.find((STORPTR)storage_units[index])!=NULL)
			reg_list.push_back(index);
	}
        return;
}

//07242002
//The output edges are in reg_temp;
//the receiving registers are in reg_list;
//the switching activity for communication from fu to the reg is in reduced_link
void Datapath::get_fu_reduced_output(Register & reg_temp, vector<int> & reg_list, 
		vector<float> & reduced_link, FUPTR fu, Scm *scm) const 
{
        assert(fu);
	EDGEPTR var;
	List_iterator<EDGEPTR> varscan;
	Storage_unit *reg;
	reduced_link.clear();
	reg_list.clear();
        reg_temp.clear_variables();
	Register var_used(NULL);
        var_used.clear_variables();

        get_fu_output(reg_temp,reg_list, fu);

	for(int reg_index = 0; reg_index < reg_list.size(); reg_index ++) {
               reg = storage_units[reg_list[reg_index]];
               assert( reg );
               var_used.clear_variables();
               //if this register does take output from the node
                FOR_EACH_LISTNODE(reg_temp.get_variables(), varscan) {
                       var = varscan.get_item();
                       assert(var);
                       if(((Register *)reg)->get_variables().find( var )!=NULL )  {
                                var_used.add_variable(var);
                       }
                }
                assert(var_used.get_variables().get_size()>0); 
                reduced_link.push_back(scm->get_total_wiresw(var_used.get_variables()));
         }//For each register
	return;
}
void Datapath::get_reg_reduced_output(Register & reg_temp, vector<int> & fu_list,
                vector<float> & reduced_link, Storage_unit *reg, Scm *scm) const
{
        assert(reg);
	FUPTR fu;
	EDGEPTR var;
	List_iterator<EDGEPTR> varscan;
        reduced_link.clear();
        fu_list.clear();
        reg_temp.clear_variables();
        Register var_used(NULL), input_vars(NULL);
        var_used.clear_variables();

        get_reg_output(reg_temp,fu_list, reg);

        //check every register to see whether they take output from the node
        for(int fu_index = 0; fu_index < fu_list.size(); fu_index ++) {
               fu = functional_units[fu_list[fu_index]];
               assert(fu);
               var_used.clear_variables();
               input_vars.clear_variables();
               assert(input_vars.get_variables().get_size()==0);
               get_fu_input(input_vars, fu);

                FOR_EACH_LISTNODE(reg_temp.get_variables(), varscan) {
                       var = varscan.get_item();
                       assert(var);
                       if(((input_vars.get_variables()).find(var)!=NULL)) {
                             if((var_used.get_variables().find(var)==NULL)) {
                                    var_used.add_variable(var);
                             }
                        }
                }
                assert(var_used.get_variables().get_size()>0);
                reduced_link.push_back(scm->get_total_wiresw(var_used.get_variables()));
         }//For each register
	return;
}

void Datapath::generate_reduced_module_link(library *lib, Scm *scm)
{

        List_iterator<FUPTR> fuscan;
        FUPTR fu;
        EDGEPTR var ;
        List_iterator<EDGEPTR> varscan ;
        Storage_unit * reg;
        List_iterator<Storage_unit *> regscan;
        NODEPTR node;
        long fu_num = functional_units.get_size();
        long reg_num = storage_units.get_size();
        int reg_index, fu_index;
        assert(reg_num>=0 && fu_num>=0);
        Register var_used(NULL);
        Register reg_temp(NULL);


        //Note: there is no direct link between functional units.
        //Interconnection exist only betwen functional unit and register
        //clear the old values
        for(long i = 0; i<fu_num+reg_num; i++) {
                for(long j = i+1; j<fu_num + reg_num; j++) {
                        module_link(i,j) = 0.0;
                }
        }
        //First, check the output of a functional units
        fu_index = 0;
        FOR_EACH_LISTNODE(functional_units, fuscan){
                fu = fuscan.get_item();
                assert(fu);
                reg_temp.clear_variables();
                vector<int> reg_list;
		vector<float> reduced_link;

                get_fu_reduced_output(reg_temp,reg_list, reduced_link, fu,scm);
		for(int i = 0; i<reg_list.size(); i++) {
                        module_link(fu_index, reg_list[i] + fu_num) = reduced_link[i];
                }
                fu_index++;
        }//for each functional unit
         //Second, check the output of a register
        reg_index = 0;
        FOR_EACH_LISTNODE(storage_units, regscan )   {
                 reg = regscan.get_item();
                 assert( reg );
                 var_used.clear_variables();
                 assert(var_used.get_variables().get_size()==0);
                 reg_temp.clear_variables();
                 vector<int> fu_list;
		 vector<float> reduced_link;

                 get_reg_reduced_output(reg_temp,fu_list, reduced_link, reg, scm);
		 for(int i= 0; i<fu_list.size(); i++) {
			 module_link(fu_list[i], reg_index+fu_num) +=reduced_link[i];
		 }
                 reg_index++;
        }//for each reg;
        return;
}
////////////////////////////////////////////////////////////////////////////
void Datapath::create_graph(void){

	return;
}

