#ifndef WIRE_H
#define WIRE_H
#include <assert.h>
#define MAX_BIT_NUM 64

class wire_energy {
	
private:
	float wire_sw_energy[10];
	unsigned short bitwidth;

	enum pattern { SILENT = 0, UP = 1, DOWN = 2 };
	enum Triple { sss = 0, sts = 1, sst = 2, tst = 3, tsa = 4, \
		      ttt = 5, stt = 6, tta = 7, sta = 8, ata = 9};		

	pattern get_sw_pattern(bool A, bool B) const {
		if(A==B) return SILENT;
		else if(A==true)
			return DOWN;
		else	return UP;
	}
	float get_energy(bool A0, bool A1, bool B0, bool B1) const;
	float get_energy(bool A0, bool A1, bool A2,
				bool B0, bool B1, bool B2) const;

public:
	const double unit_;//pico Joule

	wire_energy(float *energy): unit_(0.000000000001) {
		for(int i = 0; i<10; i++){
			wire_sw_energy[i] = energy[i];
		}
		return;
	}		
	wire_energy(void): unit_(0.000000000001)  {
		//Unit: pJoule per mm 0.18 micron UMC 1P6M tech, minimum width.
		//UCSD data
		wire_sw_energy[0] = 0.0 ;    //sss
		wire_sw_energy[1] = 0.156 ;  //sts
		wire_sw_energy[2] = 0.033 ;  //sst
		wire_sw_energy[3] = 0.131 ;  //tst
		wire_sw_energy[4] = 0.0 ;    //tst'
		wire_sw_energy[5] = 0.002 ;  //ttt
		wire_sw_energy[6] = 0.045 ;  //stt
		wire_sw_energy[7] = 0.153 ;  //ttt'
		wire_sw_energy[8] = 0.327 ;  //stt'
		wire_sw_energy[9] = 0.560 ;  //t'tt'
		return;
	}
	~wire_energy(){ return;}
		
	float get_wire_energy(unsigned int previous, unsigned int current, int bitwidth) const;
	float get_pattern_energy(int i) const { return wire_sw_energy[i];}

};

typedef class wire_capacitance {

private:
	typedef class  unit_capacitance {
		public:
		const double area, fringing, coupling;
		double get_total(void ) const { return coupling; }
		unit_capacitance(double A, double F, double C):
	       		area(A),fringing(F), coupling(C){
			return;
		}
		~unit_capacitance() {return;}
	} unit_cap;
	//Currently, we have data for tech .25 .18 .15 .13 .1 .07
	unit_cap *cap_per_unit_length[6];

	float lamda[6];
	
public:
	const double length_unit, capacitance_unit;	

	float get_lamda(unsigned short index) const { return lamda[index]; }
	double get_cap(unsigned short tech, double length) const {
		return cap_per_unit_length[tech]->get_total() * length;
	}
	double get_unit_cap(unsigned short tech) const {
		return cap_per_unit_length[tech]->get_total();
	}

	wire_capacitance(void):length_unit(0.000001),//micro meter 
		capacitance_unit(0.000000000000001)//aF(10-15 F) 
	{
		//UCLA Jason Cong's data.
		//2 x min. width & spacing, using NTRS'97 and FastCap
		cap_per_unit_length[0] = new unit_cap(29.0, 41.8, 71.0);
		cap_per_unit_length[1] = new unit_cap(21.2, 30.2, 58.3);
		cap_per_unit_length[2] = new unit_cap(16.2, 24.8, 49.4);
		cap_per_unit_length[3] = new unit_cap(12.0, 18.3, 42.8);
		cap_per_unit_length[4] = new unit_cap(14.4, 14.1, 45.3);
		cap_per_unit_length[5] = new unit_cap(8.56, 14.8, 41.6);
		lamda[0] = 0.25; lamda[1] = 0.18; lamda[2] = 0.15;
		lamda[3] = 0.13; lamda[4] = 0.10; lamda[5] = 0.07;
	}	
	~wire_capacitance(void) {
		for( int i = 0; i<6; i++) {
			delete cap_per_unit_length[i];
		}
	}		
} wire_cap;

typedef class gate_info{

private:
	double cap[6];	
	double area[6];
public:
	const double unit_;
	//Minimum sized gate Pan & Cong data IEEE Trans. CAD, 2001, June
	double get_cap(unsigned short tech) const { return cap[tech]; }

	gate_info(void): unit_(0.000000000000001) {//aF(10-15 F)
		cap[0] = 282.0; area[0] = 0.25 * 0.25 ;
		cap[1] = 234.0; area[1] = 0.18 * 0.18 ;
		cap[2] = 220.0; area[2] = 0.15 * 0.15 ;
		cap[3] = 135.0; area[3] = 0.13 * 0.13 ;
		cap[4] = 72.0;  area[4] = 0.10 * 0.10 ;
		cap[5] = 66.0;  area[5] = 0.07 * 0.07 ;
		return;
	}
	~gate_info(void) { return; }
	//return the switching capacitance in pJoule
	float get_energy(float S, float Vdd, short tech) const {
		return S*S*Vdd*Vdd*0.5*cap[tech]/1000.0/1000.0;	
	}
	float get_area (float S, short tech) const {
		return S*S*area[tech];	
	}
} gate_cap;
#endif		
			
