/*  Copyright (c) 2000-2001 by Alternative System Concepts, Inc.  */
/*  All Rights Reserved                                           */
/*                                                                */
/*  THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF                */
/*       Alternative System Concepts, Inc.                        */
/*  The copyright notice above does not evidence any              */
/*  actual or intended publication of such source code.           */

///  <HEADER>
///     <PURPOSE>
///              
///     </PURPOSE>
///              
///  </HEADER>

/********************************************************************/
/*              FILE: libelem.h                                     */
/* Contains a description of a class Library_element and its subclasses */
/********************************************************************/
#ifndef KAMAL_LIBELEM_H
#define KAMAL_LIBELEM_H

#ifdef _IMPACT_
#include "componen.h"
#include "port.h"
#endif

#include "fritso.h"
#include "libport.h"
#include "array.h"
#include "symtab.h"
#include "matrix.h"

class Port_info : public FRITS_object {
friend class Library_element;
Library_element_port* portptr;	// pointer to the port
int portlist_index;		// port number in the list of ports
int delay_matrix_index;		// index of the entry in the delay matrix
Boolean is_data;		// DATA/CONTROL port

Port_info(Library_element_port* port)	{
	portptr = port;
	portlist_index = -1;
	delay_matrix_index = -1;
	is_data = TRUE;
}

Port_info(Library_element_port* port, int list_index, int matrix_index, Boolean data_attrib = TRUE)	{
	portptr = port;
	portlist_index = list_index;
	delay_matrix_index = matrix_index;
	is_data = data_attrib;
}

~Port_info() {};

public:
virtual const char* get_id() {return "Port_info";}

};


/********************************************************************/
#ifdef _IMPACT_
class Library_element : public Assurance, public FRITS_object {
#else
class Library_element {
#endif
friend class Datapath_element;
friend class Register;
private:

  char *name;
  int area;              //area attribute of this Library_element
  int bitwidth;		  
  float delay;             //delay attribute (ns units)		
  Matrix<float> *port_delay;	 
  Matrix<float> *bit_delay; 
  List_ar<class Library_element_port *> data_ports;  
  List_ar<class Library_element_port *> control_ports;  
  List_ar<class Library_element_port *> input_ports;  
  List_ar<class Library_element_port *> output_ports;  
  int number_input_ports;
  int number_output_ports;
  int number_ports;
  Port_info** port_infos;
#ifdef _IMPACT_
  Component *frits_component; //pointer to the corresponding FRITS VHDL component
  // to store port_infos by the port name
  Symbol_table* port_info_index;
#endif

protected:
#ifdef _IMPACT_
  Library_element(Component *comp); 
#else
  Library_element(const char* element_name, float element_area, float element_delay); 
#endif

  ~Library_element();

  //functions to set the fields of a Library_element
  Library_element* add_data_port(Library_element_port *new_port);
  Library_element* add_control_port(Library_element_port *new_port);

public:
  int get_bitwidth(void)  {return bitwidth;}
  //inline functions to get the fields of a Library_element
  inline const char *get_name() const {
    return (const char*)name;
  }

  inline const int get_area() {
    return area;
  }
  inline const float get_delay() {
    return delay;
  }
  const float get_delay(int input_port_index, int output_port_index);
  const float get_delay(const char* input_port_name, const char* output_port_name);

  inline const int number_of_ports(void) { return number_ports;  }
  
  virtual const char* get_id(void) {return "Library_element";}

 
  const char* get_port_name(int portindex)
	{ FRITS_SET_MESSAGE("get_port_name");
	  assert(portindex <= number_ports);
	  return (port_infos[portindex-1] ? 
		(port_infos[portindex-1])->portptr->get_name() : NULL); }

#ifdef _IMPACT_
  Component* get_frits_component(void) {return frits_component;}
#endif
};

class Functional_library_element : public Library_element {
private:
  Boolean* functions;    //array of functions that can be performed
  int pipeline_stages;	// self-explanatory, initialized in the
  float total_leakage;			// constructor
public:
#ifdef _IMPACT_
  Functional_library_element(Component *comp);
#else
  Functional_library_element(const char* element_name, float element_area, float element_delay);
#endif

  ~Functional_library_element() {if (functions) delete [] functions;}

  inline Functional_library_element* add_function(const Operator new_function) {
    functions[new_function]=TRUE;
    return this;
  }

  virtual const char* get_id(void) {return "Functional_library_element";}
  const float get_total_leakage(void) 	{return total_leakage;}
  const int get_pipeline_stages(void) 	{return pipeline_stages;}
  Boolean can_perform(Operator op)
	{
	return functions[(int)op];
	}

  Boolean can_perform(List_ar<Operator> &list)
	{
	List_iterator<Operator> list_iterator;
	FOR_EACH_LISTNODE(list,list_iterator)
	    {
	    if ( ! functions[(int)list_iterator.get_item()] )
		return FALSE;
	    }
	return TRUE;
	}

};

/********************************************************************/
class Storage_library_element : public Library_element {
private:
  Boolean load_enable;

  
public:  
#ifdef _IMPACT_
  Storage_library_element(Component *comp) : Library_element(comp) {
    FRITS_SET_CLASS("Storage_library_element");
    if ( comp->get_port()->search_interface_element("le") )
	load_enable = TRUE;
    else
	load_enable = FALSE;
#else
  Storage_library_element(const char* element_name, float element_area, float element_delay) : Library_element (element_name, element_area, element_delay) {
#endif
  }

  ~Storage_library_element() {
  }
  virtual const char* get_id(void) {return "Storage_library_element";};
  Boolean has_load_enable(void) const {return load_enable;}

};

/********************************************************************/
class Interconnect_library_element : public Library_element {
private:
  
  Boolean tri_state;
public:
#ifdef _IMPACT_
  Interconnect_library_element(Component *comp) : Library_element(comp) {
    FRITS_SET_CLASS("Interconnect_library_element");
    // A crude fix, should base the decision on an attribute
    if ( comp->get_port()->search_interface_element("select2") )
	tri_state = TRUE;
    else
	tri_state = FALSE;
#else
  Interconnect_library_element(const char* element_name, float element_area, float element_delay) : Library_element (element_name, element_area, element_delay) {
#endif
  }
  
  ~Interconnect_library_element() {
  }
  virtual const char* get_id(void) {return "Interconnect_library_element";}
  Boolean is_tristate(void) const	{return tri_state;}

};

/********************************************************************/
#endif 


