/*  Copyright (c) 2000-2001 by Alternative System Concepts, Inc.  */
/*  All Rights Reserved                                           */
/*                                                                */
/*  THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF                */
/*       Alternative System Concepts, Inc.                        */
/*  The copyright notice above does not evidence any              */
/*  actual or intended publication of such source code.           */

///  <HEADER>
///     <PURPOSE>
///              
///     </PURPOSE>
///              
///  </HEADER>

/********************************************************************/
/*        FILE: library_patch.h                                     */
/********************************************************************/
#ifndef LIBRARY_PATCH_H
#define LIBRARY_PATCH_H

#include "basic.h"
#include "parameter.h"
#include "matrix.h"
#include "list_ar.h"
#include "dfgnode.h"
#include "wire.h"
#ifdef _SCALP_
#include "libpack.h"
#include "libelem.h"
#endif

#define V_TH 0.6
#define REG_POWER_PER_BIT 0.00235 
//03262002 Lin
#define MUX_POWER_PER_BIT 0.00155//from Anand's power.vhdl averaged

#define REG_AREA_PER_BIT 370.0
//#define REG_AREA_PER_BIT 16.0
#define MUX_AREA_PER_BIT 148.0
//#define MUX_AREA_PER_BIT 16.0

#ifdef _OLD_LIBRARY_
typedef class libelement LIBEL;
typedef class libelement *LIBELPTR;
typedef class libelement FUNC_LIBEL;
typedef class libelement *FUNC_LIBELPTR;
typedef class libelement STOR_LIBEL;
typedef class libelement *STOR_LIBELPTR;
typedef class libelement INTCON_LIBEL;
typedef class libelement *INTCON_LIBELPTR;
#else
typedef class Library_element LIBEL;
typedef class Library_element *LIBELPTR;
typedef class Functional_library_element FUNC_LIBEL;
typedef class Functional_library_element *FUNC_LIBELPTR;
typedef class Storage_library_element STOR_LIBEL;
typedef class Storage_library_element *STOR_LIBELPTR;
typedef class Interconnect_library_element INTCON_LIBEL;
typedef class Interconnect_library_element *INTCON_LIBELPTR;
#endif
/********************************************************************/
//
// Class libelement stores info. about the types of modules
// available in the library, including their area, delay, and
// a model for their power dissipation.
//
#ifdef _SCALP_
class libelement : public Assurance {
#else
class libelement {
#endif
  friend class library;
  friend ostream &operator <<(ostream &, const libelement &);
  friend ostream &operator <<(ostream &, libelement *);

private:
  int address;
  char name[MAXSTRLEN];
  int bitwidth;                //bit width
  float area;
  float total_leakage;
  float delay;                 //delay at Vdd = 5 volts
  List_ar<float> *power_coeffs;   //power dissipation model
  Array<Boolean> operations;   //operations that can be performed by the module
  int pipestages;              //# of pipeline stages (> 1 if pipelined)
#ifdef _SCALP_
  Library_element *new_libelement;
#endif

public:
#ifdef _SCALP_
  libelement() : operations(ALLOPS) {
#else
  libelement() : operations(NUMNODETYPES) {
#endif
    name[0] = '\0';
    bitwidth = 0;
    area = 0.0;
    total_leakage =0.0;
    delay = 0.0;
    power_coeffs = NULL;
    pipestages = 0;
    operations.reset(F);
#ifdef _SCALP_
    new_libelement = NULL;
#endif
    return;
  }

  ~libelement() {
    if(power_coeffs) {
      delete power_coeffs;
    }
  }

#ifdef _SCALP_
  Library_element* get_frits_libelement(void)
	{return new_libelement;}
  void set_frits_libelement(Functional_library_element* element)
	{
	List_iterator<Operator> iterator;
	Operator op;
	new_libelement = element;
	strcpy(name, element->get_name());
	bitwidth = element->get_bitwidth();
	area = (float)element->get_area();
	delay = element->get_delay();
	pipestages = element->get_pipeline_stages();
	for (int i = 1; i <= ALLOPS; i++) {
	  if (element->can_perform((Operator)i)) operations[i-1]=T;
	  }
        }
#endif

  inline const int get_address() {return address;}
  inline const char *get_name() {return name;}
  inline const float get_area() {return area;}
  inline const int get_numstages() {return pipestages;}
  inline const float get_leakage() {return total_leakage;}
  //delay at an arbitrary voltage
  inline const float get_total_delay(const float voltage = 5.0) {
    return((delay*(5.0 - V_TH)*(5.0 - V_TH)*voltage)/(5.0*(voltage-V_TH)*(voltage-V_TH)));
  }

  //latency at an arbitrary voltage
  inline const float get_stage_delay(const float voltage = 5.0) {
    return((delay*(5.0 - V_TH)*(5.0 - V_TH)*voltage)/(5.0*(voltage-V_TH)*(voltage-V_TH)*((float)pipestages)));
  }

  //total number of control steps reqd. at given voltage and clock
  inline int get_total_csteps(const float voltage, const float clk) {
    assert(clk > 0.0);
    return(pipestages*CEIL((get_stage_delay(voltage))/clk));
  }

  //number of control steps reqd. per stage at given voltage and clock
  inline int get_stage_csteps(const float voltage, const float clk) {
    return(CEIL((get_stage_delay(voltage))/clk));
  }

  //power dissipation model
  float sw_cap(List_ar<unsigned int> &, List_ar<unsigned int> &);
  float sw_cap(unsigned int,unsigned int, short);

  inline const Boolean can_perform(Operator type, int nbits) {
#ifdef _SCALP_
    return (((Functional_library_element *)new_libelement)->can_perform(type));
#else
    return(((operations[(int)type-1] == T && bitwidth == nbits)?T:F));
#endif
  }

 
  inline List_ar<Operator> *get_functions() {
    List_ar<Operator> *retlist = new List_ar<Operator>;
    register int i;
    
    for(i = 0; i < (int)R_XNOR; i++) {
      if(operations[i]) {
	retlist->append((Operator) (i+1));
      }
    }
    return(retlist);
  }

 
  inline const int number_of_ports() {
#ifdef _SCALP_
    assert(new_libelement);
    return new_libelement->number_of_ports();
#else
    return 3;
#endif
  }
  //NOTE: Need to make this smarter, too !!
  inline const int number_of_input_ports() {
//    return(number_of_ports() - 1);
      return 2;
  }
};

/********************************************************************/

#ifdef _SCALP_
class library : public Assurance {
#else
class library {
#endif
private:

  //set of distinct module types available
  int elementcount;
  Array<libelement *> elements;

  //dummy library elements representing registers and muxes
  libelement *register_libelement;
  libelement *mux_libelement;
#ifdef _SCALP_
  // Having this pointer will enable access to storage and interconnect
  // elements of the package on a per-need basis.
  Library_package* libpack;
#endif

  //functions that are only called by other member functions
  int lib_parse();
  //used while parsing the library from a file
  libelement *get_libelement(char *);
  libelement *add_libelement(char *);
#ifdef PHYSICAL
  wire_energy Wire_En;
  double wire_scale_down;
  double gate_scale_down;
  double lamda_scale_down;
#endif  
public:
  library() : elements(NUMMODULETYPES)
#ifdef PHYSICAL
	 ,Wire_En()
#endif	 
  {
    elementcount = 0;
    elements.reset((libelement *)0);
    register_libelement = NULL;
    mux_libelement = NULL;
#ifdef _SCALP_
    libpack = NULL;
#endif
    return;
  }

  ~library() {
    register int i;
    //free the elements
    for(i = 0; i < elementcount; i++) {
      assert(elements[i]);
      delete elements[i];
    }
    if(register_libelement) delete register_libelement;
    if(mux_libelement) delete mux_libelement;
    return;
  }

#ifdef _SCALP_
  void extract_register(int bitwidth, Boolean load_enable = T);
  void extract_mux(int bitwidth, Boolean tri_state = F);
  void set_library_package(Library_package* libpackage);
#else
  inline void extract_register(int bitwidth, Boolean load_enable = T) {
    register_libelement = new STOR_LIBEL;
    return;
  }
  inline void extract_mux(int bitwidth, Boolean tri_state = F) {
    mux_libelement = new INTCON_LIBEL;
    return;
  }
#endif

  inline STOR_LIBELPTR get_register(int bitwidth, Boolean load_enable = T) {
    return(register_libelement);
  }
  inline INTCON_LIBELPTR get_mux(int bitwidth, Boolean tri_state = F) {
    return(mux_libelement);
  }
  inline const int numelements() {return elementcount;}

  //model for the register switched capacitance
  float reg_sw_cap(unsigned int, unsigned int, int);
#ifdef PHYSICAL
          float edge_sw(unsigned int, unsigned int, int);
// Scale down the capacitance for different technologies
	  // Note: the base is 0.18um tech
	  void set_wire_scale_down(double wire) { wire_scale_down = wire; return;}
	  double get_wire_scale_down(void) { return wire_scale_down; }
	  void set_gate_scale_down(double gate) { gate_scale_down = gate; return;}
	  double get_gate_scale_down(void) { return gate_scale_down; }
          void set_lamda_scale_down(double lamda_scale) { lamda_scale_down = lamda_scale; return;}
          double get_lamda_scale_down(void) { return lamda_scale_down; }
	  float get_reg_bit_power() const { return REG_POWER_PER_BIT; }
	  float get_mux_bit_power() const { return MUX_POWER_PER_BIT; }
	  
#endif

  //register area model
  	float get_reg_bit_area() const {return REG_AREA_PER_BIT;}
  
  //mux area model
  	float get_mux_bit_area() const {return MUX_AREA_PER_BIT;}

  inline libelement *get_nthelement(int id) {
    assert(id >= 0 && id < elementcount);
    return (elements[id]);
  }

  //return a List of all the library elements that can implement
  //an operation (dfgnode) of the given type
  List_ar<libelement *> *get_choices(Operator, int);

  //return a List of all the elements in the library
  List_ar<libelement *> *get_all_elements(int nbits);

// #ifndef _SCALP_
  //read the library from a file
  void read(char *);
// #endif

  //print out the library
  void display(ostream & = cout);

};

//Syntactic sugar for displaying a library
inline ostream & operator <<(ostream &output, library &lib)
{lib.display(output);return(output);}
/********************************************************************/
#endif /* LIBRARY_PATCH_H */

