/*  Copyright (c) 2000-2001 by Alternative System Concepts, Inc.  */
/*  All Rights Reserved                                           */
/*                                                                */
/*  THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF                */
/*       Alternative System Concepts, Inc.                        */
/*  The copyright notice above does not evidence any              */
/*  actual or intended publication of such source code.           */

///  <HEADER>
///     <PURPOSE>
///              
///     </PURPOSE>
///              
///  </HEADER>

/************************************************************************/
/*              FILE: rbtree.h                                    	*/
/* Contains a description of the RBtree template class  		*/
/* which implements red-black trees 					*/
/* Algorithms are taken from "Introduction to Algorithms",		*/
/* Cormen, Leiserson and Rivest, McGraw Hill, 1990, pp. 244-281		*/
/************************************************************************/

#ifndef RBTREE_H
#define RBTREE_H

#include "rbnode.h"
#include "list_ar.h"

template<class type, class key>
class RBtree 
#ifdef _IMPACT_
: public Assurance 
#endif
{
// This private message is used by the destructor to clear the tree
// The message deletes both subtrees of the node.
void delete_subtrees(RBnode<type,key>* node)
{
if ( !node ) return;
if ( node->get_left_child() ) 
	{
	delete_subtrees(node->get_left_child());
	delete node->get_left_child();
	node->set_left_child(NULL);
	}
if ( node->get_right_child() ) 
	{
	delete_subtrees(node->get_right_child());
	delete node->get_right_child();
	node->set_right_child(NULL);
	}
}

protected:

RBnode<type,key>* root;
int number_of_nodes;
RBnode<type,key>* current;	// Used by tree_next() message to track the current positon, NULL by default.

// Delivers a minimum of subtree stemming from a specified node.
// The default is the tree minimum.
RBnode<type,key>* subtree_min(RBnode<type,key>* node = NULL)
{
if ( !node ) node = root;
while ( node->get_left_child() ) node = node->get_left_child();
current = node;
return node;
}

void tree_insert(RBnode<type,key>* node)
{
// assert(node); // this is checked in the message that calls tree_insert()
number_of_nodes++;
if ( !root )
	{
	root = node;
	return;
	}
// x points to the curnent node, and y to x's parent
RBnode<type,key> *x, *y;
x = root;
y = NULL;
while (x) 
	{
	y = x;
	if ( node->get_value() < x->get_value() ) x = x->get_left_child();
	else  x = x->get_right_child();
	}
node->set_parent(y);
if ( !y ) root = node;  // this cannot reasonably happen, but is in the book
else if ( node->get_value() < y->get_value() ) y->set_left_child(node);
else y->set_right_child(node);
}

type search(key parameter, RBnode<type,key>* node)
{
if ( !root || !node ) return NULL;
if ( node->get_value() == parameter ) return node->get_actual_object();
else if ( node->get_value() > parameter ) // go left
	{
	if ( node->get_left_child() )
		return search(parameter, node->get_left_child());
	else	return NULL;
	}
else	// go right, node->get_value() < parameter
	{
	if ( node->get_right_child() ) 
		return search(parameter, node->get_right_child());
	else	return node->get_actual_object(); 
	}
}

// This search message return a list of all elements whose keys are <= than
// the specified parameter. By default, the search starts from the root.
List_ar<type> *search_all(key parameter, 
List_ar<type> *&list, RBnode<type,key>* node)
{
if ( !node || !root ) return list;
if ( !list ) mem_ok(list = new List_ar<type>);
if ( node->get_left_child() ) 
	search_all(parameter, list, node->get_left_child());
if ( node->get_value() <= parameter )
	{
	list->append(node->get_actual_object());
	if ( node->get_right_child() ) 
		search_all(parameter, list, node->get_right_child());
	}
return list;
}

// see p. 266 of the book: node => x, y => y.
void rotate_left(RBnode<type,key>* node)
{
FRITS_SET_MESSAGE("rotate_left");
assert(node);
RBnode<type,key>* y;
assert( (y=node->get_right_child()) != NULL);
node->set_right_child(y->get_left_child());
if (y->get_left_child()) y->get_left_child()->set_parent(node);
y->set_parent(node->get_parent());
if ( ! node->get_parent() ) root = y;
else if ( node == node->get_parent()->get_left_child() ) 
	node->get_parent()->set_left_child(y);
else 	node->get_parent()->set_right_child(y);
y->set_left_child(node);
node->set_parent(y);
}

// Reversed 'rotate_left': x => x, node => y.
void rotate_right(RBnode<type,key>* node)
{
FRITS_SET_MESSAGE("rotate_right");
assert(node);
RBnode<type,key>* x;
assert( (x=node->get_left_child()) != NULL);
node->set_left_child(x->get_right_child());
if (x->get_right_child()) x->get_right_child()->set_parent(node);
x->set_parent(node->get_parent());
if ( ! node->get_parent() ) root = x;
else if ( node == node->get_parent()->get_right_child() ) 
	node->get_parent()->set_right_child(x);
else 	node->get_parent()->set_left_child(x);
x->set_right_child(node);
node->set_parent(x);
}

public:
RBtree(RBnode<type,key>* node = NULL)
{
FRITS_SET_CLASS("RBtree");
root = node;
number_of_nodes = node ? 1 : 0;
current = NULL;
}

~RBtree() 
{
if ( root )
	{
	delete_subtrees(root);
	delete root;
	}
};

int get_number_of_nodes(void) {return number_of_nodes;}

// Straight from the book, p. 268
virtual void insert(RBnode<type,key>* node)
{
FRITS_SET_MESSAGE("insert");
assert(node);
RBnode<type,key> *x, *y;
tree_insert(node);
node->set_color(RED);
x = node;
while ( x != root && x->get_parent()->get_color() == RED )
	{
	if (x->get_parent() == x->get_parent()->get_parent()->get_left_child())
	    {
	    y = x->get_parent()->get_parent()->get_right_child();
	    if ( y && y->get_color() == RED )
		{
		x->get_parent()->set_color(BLACK);
		y->set_color(BLACK);
		x->get_parent()->get_parent()->set_color(RED);
		x = x->get_parent()->get_parent();
		}
	    else 
		{
		if ( x == x->get_parent()->get_right_child() )
		    {
		    x = x->get_parent();
		    rotate_left(x);
		    }
		x->get_parent()->set_color(BLACK);
		x->get_parent()->get_parent()->set_color(RED);
		rotate_right(x->get_parent()->get_parent());
		}
	    }
	else
	    {
	    y = x->get_parent()->get_parent()->get_left_child();
	    if ( y && y->get_color() == RED )
		{
		x->get_parent()->set_color(BLACK);
		y->set_color(BLACK);
		x->get_parent()->get_parent()->set_color(RED);
		x = x->get_parent()->get_parent();
		}
	    else 
		{
		if ( x == x->get_parent()->get_left_child() )
		    {
		    x = x->get_parent();
		    rotate_right(x);
		    }
		x->get_parent()->set_color(BLACK);
		x->get_parent()->get_parent()->set_color(RED);
		rotate_left(x->get_parent()->get_parent());
		}
	    }
	}
root->set_color(BLACK);
}

// This message returns the first node of the tree which matches
// the specified parameter. The message searches from root. 
virtual type search(key parameter)
	{return search(parameter, root);}

RBnode<type,key>* get_root(void) 	{return root;}

type search_min(void)
{
RBnode<type,key>* node;
if ( (node=tree_min()) ) return node->get_actual_object();
else return NULL;
}

List_ar<type>* search_all(key parameter, 
List_ar<type> *&list = NULL)
{
if ( !root ) return list;
if ( !list ) mem_ok(list = new List_ar<type>);
return search_all(parameter,list, root);
}

virtual const char* get_id(void) {return "RBtree";}

// Positions the current pointer at the minimum node and returns the node's
// pointer. The search starts from the root.
RBnode<type,key>* tree_min(void)		{return subtree_min(root);}

RBnode<type,key>* tree_successor(void)
{
FRITS_SET_MESSAGE("tree_successor");
assert(current);
RBnode<type,key>* y;
if ( current->get_right_child() ) 
	return subtree_min(current->get_right_child());

y = current->get_parent();
while ( y && current == y->get_right_child() )
	{
	current = y;
	// Gets NULL for y = root.
	y = y->get_parent();
	}
return (current = y);
}
};



#endif /* RBTREE_H */

