/********************************************************************/
/*              FILE: list_ar.h                                     */
/********************************************************************/
#ifndef ANAND_LIST_H
#define ANAND_LIST_H
#include <assert.h>
#include "basic.h"

template<class Type> class List_node;
template <class Type> class List_iterator;

/********************************************************************/
template<class type>
#ifdef _SCALP_
class List_ar : public Assurance {
#else
class List_ar {
#endif
friend class List_iterator<type>;
private:
  List_node<type> *head;
  List_node<type> *tail;
  int count;

  //used only by list_iterator, etc.
  //moved to public by wwang on 9/24/01
  //inline List_node<type> *get_head() const {return head;}
  inline List_node<type> *get_tail() const {return tail;}
  inline const int get_count() const {return count;}
public:
  List_ar() {
    FRITS_SET_CLASS("List_ar<type>");
    head = (List_node<type> *)NULL;
    tail = (List_node<type> *)NULL;
    count = 0;
  }

  ~List_ar() {
    //free all the List nodes.
    clear();
  }

  //Create a copy of this list and return it
  List_ar<type>& copy() const {
    FRITS_SET_MESSAGE("copy");
    List_ar<type> *target_list;
    register List_node<type> *lnode;
    target_list = new List_ar<type>;
    assert(target_list);
    //walk thro this list, and keep appending to target List
    for(lnode = head; lnode; lnode = lnode->next) 
      target_list->append(lnode->o);
    return *target_list;
  }

  //Copy another List into this List - note the difference to the previous copy() message
  void copy(const List_ar<type> &source_list) {
    register List_node<type> *lnode;

    FRITS_SET_MESSAGE("copy");
    if(this == &source_list) {
      return;
    }

    //clear existing contents
    clear();
    //walk thro source_list, and keep appending to this List
    for(lnode = source_list.head; lnode; lnode = lnode->next) {
      append(lnode->o);
    }

    return;
  }

  //moved here by wwang on 9/24/01, to handle the problem to scan a deleting list
  inline List_node<type> *get_head() const {return head;}

  void append(const type &obj_val) {
    register List_node<type> *lnode;
    
    FRITS_SET_MESSAGE("append");
    lnode = new List_node<type>(obj_val);
    assert(lnode != NULL);
    
    if(is_empty()) {
      lnode->next = NULL;
      lnode->prev = lnode;
      head = lnode;
      tail = lnode;
      count = 1;
    } else {
      lnode->prev = tail;
      lnode->next = NULL;
      tail->next = lnode;
      head->prev = lnode;
      tail = lnode;
      count++;
    }
    return;
  }

  void insert(const type &obj_val) {
    register List_node<type> *lnode;
    
    FRITS_SET_MESSAGE("insert");
    lnode = new List_node<type>(obj_val);
    assert(lnode);
    
    if(is_empty()) {
      lnode->next = NULL;
      head = lnode;
      tail = lnode;
      count = 1;
    } else {
      lnode->next = head;
      lnode->prev = tail;
      head->prev = lnode;
      head = lnode;
    }
    count++;
    return;
  }

  //add a node in the middle of a List, before a given node
  void add_before(List_iterator<type> &liter, const type &obj) {
    register List_node<type> *lnode;
    register List_node<type> *newlnode;

    FRITS_SET_MESSAGE("add_before");
    assert(!is_empty());
    lnode = liter.current;
    assert(lnode);
    if(lnode == head) {
      insert(obj);
    } else {
      newlnode = new List_node<type> (obj);
      newlnode->prev = lnode->prev;
      lnode->prev->next = newlnode;
      lnode->prev = newlnode;
      newlnode->next = lnode;
      count++;
    }
    return;
  }

  //add a node in the middle of a List, after a given node
  void add_after(List_iterator<type> &liter, const type &obj) {
    register List_node<type> *lnode;
    register List_node<type> *newlnode;

    FRITS_SET_MESSAGE("add_after");
    assert(!is_empty());
    lnode = liter.current;
    assert(lnode);
    if(lnode == tail) {
      append(obj);
    } else {
      newlnode = new List_node<type> (obj);
      newlnode->next = lnode->next;
      lnode->next->prev = newlnode;
      newlnode->prev = lnode;
      lnode->next = newlnode;
      count++;
    }
    return;
  }

  int remove(const type &obj) {
    register List_node<type> *lnode;
    register List_node<type> *next_lnode;
    int count = 0;

    FRITS_SET_MESSAGE("remove");
    for(lnode = head; lnode; lnode = next_lnode) {
      next_lnode = lnode->next;
      if(lnode->o == obj) {
	remove_node(lnode);
	count++;
      }
    }
    return(count);
  }

  //replace the nth occurence of obj with newobj
  //returns T if successful (obj occurs in the list n or more times),
  //F if unsuccessful
  Boolean replace_nth(const type &obj, const type &newobj, const int n) {
    register List_node<type> *lnode;
    register List_node<type> *next_lnode;
    register int count = 0;
    Boolean success = F;

    FRITS_SET_MESSAGE("replace_all");
    for(lnode = head; lnode; lnode = next_lnode) {
      next_lnode = lnode->next;
      if(lnode->o == obj) {
	count++;
	if(count == n) {
	  lnode->o = newobj;
	  success = T;
	  break;
	}
      }
    }
    return(success);
  }

  //replace all occurences of obj with newobj
  //returns the number of instances replaced
  int replace_all(const type &obj, const type &newobj) {
    register List_node<type> *lnode;
    register List_node<type> *next_lnode;
    int count = 0;

    FRITS_SET_MESSAGE("replace_all");
    for(lnode = head; lnode; lnode = next_lnode) {
      next_lnode = lnode->next;
      if(lnode->o == obj) {
	lnode->o = newobj;
	count++;
      }
    }
    return(count);
  }

#ifndef NDEBUG
  int is_empty() const {
    FRITS_SET_MESSAGE("is_empty");
    if(count == 0) {
      assert(!head && !tail);
      return T;
    } else {
      assert(head && tail);
      return F;
    }
  }
#else
  inline Boolean is_empty() const {
    FRITS_SET_MESSAGE("is_empty");
   return(count == 0? T : F);
  }
#endif

#ifndef NDEBUG
  void check() const {
    register int i;
    register List_node<type> *lnode = head;;
    
    FRITS_SET_MESSAGE("check");
    assert((!head && !tail && count == 0) || (head && tail && count > 0));
    assert(!head || (head->is_head() && head->prev == tail));
    assert(!tail || tail->is_tail());
    //assert that there is a path from head to tail that has count-1 edges
    for(i = 1; i < count; i++) {
      assert(lnode->next);
      assert(lnode->next->prev == lnode);
      lnode = lnode->next;
    }
    assert(lnode == tail);
    return;
  }
#else
  inline void check() const { 
    FRITS_SET_MESSAGE("check");
  }
#endif

  inline const int get_size() const {
    return count;
  }

// ATTENTION: Some gcc 2.7 problem - to be investigated later.
  List_node<type> *find(const type &obj) const {
    register List_node<type> *lnode;
    
    for(lnode = head; lnode; lnode = lnode->next) {
      if(lnode->o == obj) {
	return(lnode);
      }
    }
    return((List_node<type> *)NULL);
  }

  // This message returns a position of an object in the list.
  // First element is in the position 1. 0 indicates "NOT FOUND".
  // For most sequential searches, performance could be gained
  // by starting the search from the element found last, as opposed 
  // to the head. To realize this option, the list has to be made
  // circularly linked.
  // WARNING: DO NOT USE WITH CHAR* TYPES !!!!
  int get_position(const type &obj) const {
    register List_node<type> *lnode;
    int i;
    for(lnode = head, i = 1; lnode; lnode = lnode->next, i++) 
      if (lnode->o == obj) return(i);
    return 0;	// not found
  }

  void split() {
    FRITS_SET_MESSAGE("split");
    // not currently supported
    cerr << "ERROR: list.split() not implemented" << endl;
    exit(-1);
  }

  void cat() {
    FRITS_SET_MESSAGE("cat");
    // not currently supported
    cerr << "ERROR: List.cat() not implemented" << endl;
    exit(-1);
  }

  void clear() {
    register List_node<type> *lnode, *next_lnode;
    
    FRITS_SET_MESSAGE("clear");
    check();
    for(lnode = head; lnode; lnode = next_lnode) {
      next_lnode = lnode->next;
      if(lnode!=NULL)
	      delete lnode;
    }
    head = tail = (List_node<type> *)NULL;
    count = 0;
    
    return;
  }

  void remove_node(List_node<type> *lnode) {

    FRITS_SET_MESSAGE("remove_node");
    assert(lnode);
    assert(count >= 1);
    assert(lnode->prev);
    
    if(lnode == tail) {
      assert(!lnode->next);
      assert(head->prev == lnode);
      lnode->prev->next = lnode->next;
      if(lnode == head) {
	assert(count == 1);
	head = 0;
	tail = 0;
      } else {
	head->prev = lnode->prev;
	tail = lnode->prev;
      }
    } else if(lnode == head) {
      assert(count > 1); //count = 1 case handled above
      assert(lnode->prev == tail);
      assert(!lnode->prev->next);
      lnode->next->prev = lnode->prev;
      head = lnode->next;
    } else {
      //neither head nor tail
      assert(count > 2); //count = 1 & 2 cases handled above
      assert(lnode->next && lnode->prev->next);
      lnode->next->prev = lnode->prev;
      lnode->prev->next = lnode->next;
    }
    
    delete lnode;
    count--;
    return;
  }

  //accessing List elements using the [] operator
  //WARNING - VERY SLOW, USE WITH EXTREME CAUTION
  type &operator[](int index) const {
    register int i = 0;
    register List_node<type> *lnode;

    assert(index >= 0 && index < count);
    lnode = head;
    for(i = 0; i < index; i++) {
      lnode = lnode->next;
    }
    return(lnode->o);
  }

  void display(ostream & output = cout) const {
    List_node<type> *lnode;
    
    check();
    
    output << "List: <";
    for(lnode = head; lnode; lnode = lnode->next) {
      output << " " << ((type)lnode->o);
    }
    output << " >" << endl;
    
    return;
  }

};

//Syntactic sugar for displaying a List
template<class Type>
inline ostream &operator <<(ostream &output, const List_ar<Type> &l)
{l.display(output);return(output);}
/********************************************************************/
template<class type>
#ifdef _SCALP_
class List_node : public Assurance {
#else 
class List_node {
#endif
friend class List_ar<type>;
friend class List_iterator<type>;

private:
  List_node<type> *prev;
  List_node<type> *next;
  type o;
  //Since the constructor & destructor can be only called from List, they are protected.
  List_node(const type &obj_val) {
    FRITS_SET_CLASS("List_node<type>");
    o = obj_val;
    prev = (List_node<type> *)0;
    next = (List_node<type> *)0;
  }
  List_node() {
    FRITS_SET_CLASS("List_node<type>");
    o = (type) 0;
  }
  ~List_node() {}
public:
  inline Boolean is_head() {
    return ((this->prev->next == (List_node<type> *)0)? T : F);
  }
  //07252002
#ifdef PHYSICAL
  type get_current() {
	  return o;
  }
#endif  
//added by wwang on 9/24/01 to handle the deleting list problem
  inline List_node<type> *get_next() {
    return next;
  }

  inline Boolean is_tail() {
    FRITS_SET_MESSAGE("is_tail");
    return ((this->next == (List_node<type> *)0)? T : F);
  }
};

/********************************************************************/
template<class type>
#ifdef _SCALP_
class List_iterator : public Assurance {
#else
class List_iterator {
#endif
friend class List_ar<type>;
private:
  List_node<type> *current;  //current position in the list
  List_ar<type> *list;	     // so that you know what you are scanning

public:
  List_iterator() {
    FRITS_SET_CLASS("List_iterator");
    current = (List_node<type> *)NULL;
  }
  List_iterator(const List_ar<type> &list_toscan) {
    FRITS_SET_CLASS("List_iterator");
    current = list_toscan.get_head();
  }
  ~List_iterator() {}

inline void start(const List_ar<type> &list_toscan, Boolean reverse_order = F) {
    FRITS_SET_MESSAGE("start");
    current = (reverse_order == T) ? list_toscan.get_tail() : list_toscan.get_head();
    list = (List_ar<type>*)&list_toscan;
  }

  inline void increment() {
    FRITS_SET_MESSAGE("increment");
    assert(current);
    current = current->next;
  }

  inline void decrement() {
    FRITS_SET_MESSAGE("decrement");
    assert(current);
    if(current == list->get_head()) {
      current = NULL;
    } else {
      current = current->prev;
    }
  }

  inline List_node<type> *not_done() {
    FRITS_SET_MESSAGE("not_done");
    return current;
  }

  inline const type &get_item() {
    FRITS_SET_MESSAGE("get_item");
    return current->o;
  }

/* 
  This code is not reliable and should either be improved or gotten rid of.
  // Removes the current node from the list, updates the current pointer
  // to 'next' and returns it.
  inline List_node<type> *remove_item() {
    FRITS_SET_MESSAGE("remove_item");
    assert(current);
    if (current->next) {
      current = current->next;
      list->remove_node(current->prev);
    } else { //if the current node is the tail
      list->remove_node(current);
      current = NULL;
    }
    return current;
  }
*/

  inline void set_item(const type &val) {
    FRITS_SET_MESSAGE("set_item");
    current->o = val;
  }
};

//MACRO FOR LIST ITERATION
//l is a List<type> or List<type> &, l_iterptr is a List_iterator<type>
#define FOR_EACH_LISTNODE(l,l_iter) \
for((l_iter).start((l)); (l_iter).not_done(); (l_iter).increment())
//Walking through a list in reverse order
#define FOR_EACH_LISTNODE_REVERSE(l, l_iter) \
for((l_iter).start((l), T); (l_iter).not_done(); (l_iter).decrement())
/********************************************************************/
#endif
