/********************************************************************/
/*              FILE: matrix.h                                      */
/********************************************************************/
#ifndef ANAND_MATRIX_H
#define ANAND_MATRIX_H

#include "array.h"

/********************************************************************/
template<class type>
#ifdef _SCALP_
class Matrix : public Assurance {
#else
class Matrix {
#endif
private:
  int xsize;
  int ysize;
  Array<type> **elements;

public:
  Matrix(int xcount = 0, int ycount = 0) {
    register int i;

    FRITS_SET_CLASS("Matrix<type>");
    xsize = xcount;
    ysize = ycount;
    if(xcount != 0 && ycount != 0) {
      elements = new Array<type> *[xsize];
      assert(elements);
      for(i = 0; i < xsize; i++) {
	elements[i] = new Array<type>(ysize);
	assert(elements[i]);
      }
    } else {
      elements = (Array<type> **)NULL;
    }
    return;
  }

  ~Matrix () {
    register int i;
    if(xsize != 0 && ysize != 0) {
      for(i = 0; i < xsize; i++) {
	delete elements[i];
      }
      delete [] elements;
    } else {
      assert(!elements);
    }
    return;
  }

  /*resize the matrix. Copies elements of the old matrix into the new one.
   *If either the number of rows of the number of columns is lesser that
   *before, some elements are lost. If the number of rows or columns is
   *greater that before, the remaining entries are UNINITIALIZED
   */
  void resize(int newxcount, int newycount) {
    register int i, j;
    int xlimit, ylimit;
    Array<type> **newelements;

    FRITS_SET_MESSAGE("resize");

    assert_force(newxcount >= 0 && newycount >= 0);

    //special case when new dimensions are the same as the old ones
    if(newxcount == xsize && newycount == ysize) {
      return;
    }
    //special case when either newxcount or newycount is zero
    if(newxcount == 0 || newycount == 0) {
      if(xsize != 0 && ysize != 0) {
	assert(elements);
	for(i = 0; i < xsize; i++) {
	  delete elements[i];
	}
	delete [] elements;
      }
      xsize = newxcount;
      ysize = newycount;
      return;
    }

    /*allocate the new matrix*/
    newelements = new Array<type> *[newxcount];
    assert(newelements);
    for(i = 0; i < newxcount; i++) {
      newelements[i] = new Array<type>(newycount);
      assert(newelements[i]);
    }

    /*copy the elements of the old matrix into the new one*/
    xlimit = MIN(xsize, newxcount);
    ylimit = MIN(ysize, newycount);
    for(i = 0; i < xlimit; i++) {
      for(j = 0; j < ylimit; j++) {
	(*newelements[i])[j] = (*elements[i])[j];
      }
    }

    /*free the old matrix and replace it with the new one*/
    for(i = 0; i < xsize; i++) {
      delete elements[i];
    }
    delete [] elements;
    elements = newelements;
    xsize = newxcount;
    ysize = newycount;

    return;
  }

  //resize the matrix AND initialize the extra values
  void resize(int newxcount, int newycount, const type &initval) {
    register int i, j;
    int oldxcount, oldycount, xlimit, ylimit;

    FRITS_SET_MESSAGE("resize");

    oldxcount = get_xcount();
    oldycount = get_ycount();

    resize(newxcount, newycount);

    /*initialize the extra elements created, if any*/
    xlimit = MIN(oldxcount, newxcount);
    ylimit = MIN(oldycount, newycount);
    for(i = xlimit; i < newxcount; i++) {
      for(j = ylimit; j < newycount; j++) {
	(*elements[i])[j] = initval;
      }
    }

    return;
  }

  //accessing Matrix elements - the first [] returns an Array<type>
  //object reference, and the second [] is automatically taken care of
  //by the Array [] operator.
  inline Array<type> &operator[](int xindex) {
    assert(xindex >= 0 && xindex < xsize);
    assert(elements && elements[xindex]);
    return (*(elements[xindex]));
  }

  //initialize all elements of the Matrix
  void reset(const type &initval) {
    register int i;
    FRITS_SET_MESSAGE("reset");
    for(i = 0; i < xsize; i++) {
      elements[i]->reset(initval);
    }
  }

  inline int get_xcount() {
    FRITS_SET_MESSAGE("get_xcount");
    return xsize;
  }

  inline int get_ycount() {
    FRITS_SET_MESSAGE("get_ycount");
    return ysize;
  }

  //print out the Matrix
  void display(ostream &output = cout) {
    register int i;

    FRITS_SET_MESSAGE("display");
    assert(xsize >= 0 && ysize >= 0);
    output << "Matrix: <" << endl;
    for(i = 0; i < xsize; i++) {
      output << "  ";
      elements[i]->display(output);
    }
    output << ">" << endl;
  }
};

//Syntactic sugar for displaying a Matrix
template <class Type>
inline ostream &operator <<(ostream &output, Matrix<Type> &mat)
{mat.display(output);return(output);}
/********************************************************************/
#endif
