/********************************************************************/
/*           FILE: sched_ft.C                                       */
/********************************************************************/

#include "scheduler.h"
/********************************************************************/
/*Update the scalloc info data structure after adding comparisons
 */
void Scheduler::ft_update_schalloc_info(Dfg &flowgraph, library *lib, Schalloc_info &info)
{
  int i, oldcount;
  List_ar<LIBELPTR> *choices;
  LIBELPTR libel;
  NODEPTR node;

  assert(lib);

  if(info.moduletypes.get_size() != flowgraph.numnodes()) {
    oldcount = info.moduletypes.get_size();
    info.moduletypes.resize(flowgraph.numnodes());
    for(i = oldcount; i < info.moduletypes.get_size(); i++) {
      node = flowgraph.get_nthnode(i);
      assert(node);
      assert(node->get_func() == NOTEQ_);
      choices = lib->get_choices(NOTEQ_, flowgraph.bitwidth);
      assert_force(choices->get_size() == 1);
      libel = (*choices)[0];
      info.moduletypes[i] = libel->get_address();
      delete choices;
    }
  }
  if(info.nodebirths.get_size() != flowgraph.numnodes()) {
    info.nodebirths.resize(flowgraph.numnodes());
  }
  if(info.nodedeaths.get_size() != flowgraph.numnodes()) {
    info.nodedeaths.resize(flowgraph.numnodes());
  }
  if(info.functional_units.get_size() != flowgraph.numnodes()) {
    info.functional_units.resize(flowgraph.numnodes());
  }
  if(info.edgebirths.get_size() != flowgraph.numedges()) {
    info.edgebirths.resize(flowgraph.numedges());
  }
  if(info.edgedeaths.get_size() != flowgraph.numedges()) {
    info.edgedeaths.resize(flowgraph.numedges());
  }
  if(info.storage_units.get_size() != flowgraph.numedges()) {
    info.storage_units.resize(flowgraph.numedges());
  }

  return;
}
/********************************************************************/
void Scheduler::ft_restore_schalloc_info(Dfg &flowgraph, library *lib, Schalloc_info &info)
{
  assert(lib);

  if(info.moduletypes.get_size() != flowgraph.numnodes()) {
    assert(info.moduletypes.get_size() > flowgraph.numnodes());
    info.moduletypes.resize(flowgraph.numnodes());
  }
  if(info.nodebirths.get_size() != flowgraph.numnodes()) {
    assert(info.nodebirths.get_size() > flowgraph.numnodes());
    info.nodebirths.resize(flowgraph.numnodes());
  }
  if(info.nodedeaths.get_size() != flowgraph.numnodes()) {
    assert(info.nodedeaths.get_size() > flowgraph.numnodes());
    info.nodedeaths.resize(flowgraph.numnodes());
  }
  if(info.functional_units.get_size() != flowgraph.numnodes()) {
    assert(info.functional_units.get_size() > flowgraph.numnodes());
    info.functional_units.resize(flowgraph.numnodes());
  }
  if(info.edgebirths.get_size() != flowgraph.numedges()) {
    assert(info.edgebirths.get_size() > flowgraph.numnodes());
    info.edgebirths.resize(flowgraph.numedges());
  }
  if(info.edgedeaths.get_size() != flowgraph.numedges()) {
    assert(info.edgedeaths.get_size() > flowgraph.numnodes());
    info.edgedeaths.resize(flowgraph.numedges());
  }
  if(info.storage_units.get_size() != flowgraph.numedges()) {
    assert(info.storage_units.get_size() > flowgraph.numnodes());
    info.storage_units.resize(flowgraph.numedges());
  }
  return;
}
/********************************************************************/
void Scheduler::ft_implement_b_fu_move(Class_b_fu_move &move, Datapath *dp,
				       Dfg &flowgraph, library *lib,
				       Schalloc_info &cur_dfg_info,
				       Array<Boolean> &visited)
{
  int index;
  Boolean check;
  FUPTR new_fu;
  STORPTR new_su;
  NODEPTR node1, node2;
  EDGEPTR edge;
  List_iterator<int> intscan;
  List_ar<LIBELPTR> *libelemlistptr;
  LIBELPTR libel;
  List_ar<NODEPTR> orig_operations, copy_operations;
  Net *tmpnet;

  assert(dp && lib);
  assert(!move.splitting);

  move.secured_operations.clear();
  orig_operations.clear();
  copy_operations.clear();
  flowgraph.separate_orig_and_copy_operations(move.fu1->get_operations(),
						orig_operations, copy_operations);
  flowgraph.separate_orig_and_copy_operations(move.fu2->get_operations(),
					      orig_operations, copy_operations);
  switch(ft_sharetable->can_share_group(move.secured_operations, orig_operations,
					copy_operations)) {
  case 0:
    assert(!move.secured_operations.is_empty());
    break;
  case 1:
    assert(move.secured_operations.is_empty());
    break;
  default:
    assert_force(0);
    break;
  }

  //update cur_dfg_info to reflect the schedule after the move
  check = reschedule(flowgraph, lib, dp, cur_dfg_info, move, T);
  assert_force(check);

  //add dedicated functional units to the datapath that perform the comparisons
  //created as a result of the move, and dedicated registers to store the error
  //outputs, ie, the variables that are the outputs of the added comparison operations
  libelemlistptr = lib->get_choices(NOTEQ_, flowgraph.get_bitwidth());
  //simplification: assume only one comparison available in the library
  assert_force(libelemlistptr->get_size() == 1);
  libel = (*libelemlistptr)[0];
  assert(libel);
  delete libelemlistptr;
  libelemlistptr = NULL;
  FOR_EACH_LISTNODE(move.secured_operations, intscan) {
    index = intscan.get_item();
    node1 = flowgraph.get_nthnode(index);
    assert(node1);
    node2 = flowgraph.ft_get_checkpoint(node1);
    assert(node2);
    mem_ok( new_fu = new Functional_unit(libel) );
    new_fu->add_operation(node2);
    node2->set_functional_unit(new_fu);
    cur_dfg_info.functional_units[node2->get_address()] = new_fu;
    dp->add_functional_unit(new_fu);

    mem_ok( tmpnet = new Net );
    tmpnet->connect_to_port(new_fu, OUT1);
    new_fu->connect_to_net(tmpnet, OUT1);
    dp->add_net(tmpnet);

    //generate the interconnect network feeding new_fu
    dp->generate_interconnect_network(new_fu, lib->get_mux(flowgraph.get_bitwidth()));

    edge = node2->get_output_edge();
    assert(edge && edge->is_po());
    mem_ok( new_su = new STOR(NULL) );
    new_su->add_variable(edge);
    edge->set_storage_unit(new_su);
    cur_dfg_info.storage_units[edge->get_address()] = new_su;
    dp->add_storage_unit(new_su);

    mem_ok( tmpnet = new Net );
    tmpnet->connect_to_port(new_su, REGOUT);
    new_su->connect_to_net(tmpnet, REGOUT);
    dp->add_net(tmpnet);

    //The output of the register storing the error edge is a PO
    dp->mark_po(tmpnet);

    //generate the interconnect network feeding new_su
    dp->generate_interconnect_network(new_su, lib->get_mux(flowgraph.get_bitwidth()));
  }

  if(visited.get_size() != flowgraph.numnodes()) {
    assert(visited.get_size() < flowgraph.numnodes());
    visited.resize(flowgraph.numnodes(), F);
  }

  return;
}
/********************************************************************/
Boolean Scheduler::find_best_class_c_move(Dfg& flowgraph, Schalloc_info &info,
					  library& lib, Class_c_move &move, Boolean flag)
{
  static Array<Boolean> c_move_applied;
  static Array<float> weights;
  static Array<float> node_areas;
  List_iterator<NODEPTR> nodescan;
  List_iterator<EDGEPTR> edgescan;
  NODEPTR n, n1, orignode, copynode, outnode;
  EDGEPTR edge;
  float area;
  int i, max_weight_index;
  Boolean move_found = F, check;

  if(node_areas.get_size() != flowgraph.numnodes()) {
    node_areas.resize(flowgraph.numnodes());
  }

  if(weights.get_size() != flowgraph.numnodes()) {
    weights.resize(flowgraph.numnodes());
  }

  if(c_move_applied.get_size() != flowgraph.numnodes()) {
    c_move_applied.resize(flowgraph.numnodes(), F);
  }

  //If we are starting synthesis for a new Vdd, clock period pair, reset the
  //c_move_applied array
  if(flag) {
    c_move_applied.reset(F);
  }

  for(i = 0; i < flowgraph.maxlevel; i++) {
    FOR_EACH_LISTNODE(flowgraph.levellists[i], nodescan) {
      n = nodescan.get_item();
      assert(n->get_output_edge());
      area = 0;
      if(!n->get_output_edge()->is_po() && !c_move_applied[n->get_address()]) {
	FOR_EACH_FANIN_EDGE(n, edgescan) {
	  edge = edgescan.get_item();
	  assert(edge);
	  n1 = edge->input_node();
	  assert(n1 || edge->is_dfginput());
	  if(n1) {
	    area += node_areas[n1->get_address()];
	  }
	}
	area += lib.get_nthelement(info.moduletypes[n->get_address()])->get_area();
	weights[n->get_address()] = area/info.nodedeaths[n->get_address()];
	node_areas[n->get_address()] = area;
      }
    }
  }

  max_weight_index = 0;
  for(i = 0; i < 2*flowgraph.num_orig_nodes; i++) {
    assert(flowgraph.get_nthnode(i)->get_func() != NOTEQ_);
    n = flowgraph.get_nthnode(i);
    assert(n->get_output_edge());
    if(!n->get_output_edge()->is_po() && !c_move_applied[n->get_address()]) {
      assert(weights[i] > 0);

      //We transfer all fanouts of the selected operation in the copy to the
      //corresponding operation in the original. In order to guarantee that a
      //valid reschedule exists, we consider a class c move only if all fanouts
      //of the operation in the copy are scheduled AFTER the operation in the
      //original
      if(n->get_address() < flowgraph.num_orig_nodes) {
	orignode = n;
	copynode = flowgraph.get_nthnode(n->get_address() + flowgraph.num_orig_nodes);
      } else {
	copynode = n;
	orignode = flowgraph.get_nthnode(n->get_address() - flowgraph.num_orig_nodes);
      }
      check = F;
      edge = copynode->get_output_edge();
      FOR_EACH_FANOUT_NODE(edge, nodescan) {
	outnode = nodescan.get_item();
	assert(outnode);
	if(outnode->get_birth() < orignode->get_death()) {
	  check = T;
	  break; /*BREAK FROM THE FOR LOOP*/
	}
      }

      if(!check && weights[i] > weights[max_weight_index]) {
	max_weight_index = i;
      }
    }
  }

  if(max_weight_index > 0) {
    move.secured_operation = flowgraph.get_nthnode(max_weight_index);
    move_found = T;
    c_move_applied[max_weight_index] = T;
  } else {
    move_found = F;
  }

  return(move_found);
}
/********************************************************************/
void Scheduler::ft_implement_c_move(Class_c_move &move, Datapath *dp,
				       Dfg &flowgraph, library *lib,
				       Schalloc_info &cur_dfg_info, Array<Boolean> &visited)
{
  NODEPTR orignode, copynode, outnode, compare_node, innode;
  EDGEPTR origedge, copyedge, error_edge, inedge;
  FUPTR fu, new_fu;
  STORPTR new_su;
  LIBELPTR libel;
  List_ar<LIBELPTR> *libelemlistptr;
  List_iterator<NODEPTR> nodescan;
  List_iterator<EDGEPTR> edgescan;
  float cur_clock;
  List_ar<int> nodes_to_secure;
  int maxcycle;
  Net *tmpnet;
  List_ar<FUPTR> fulist;
  List_iterator<FUPTR> fuscan;

  if(move.secured_operation->get_address() < flowgraph.num_orig_nodes) {
    orignode = move.secured_operation;
    copynode = flowgraph.get_nthnode(orignode->get_address() + flowgraph.num_orig_nodes);
  } else {
    copynode = move.secured_operation;
    orignode = flowgraph.get_nthnode(copynode->get_address() - flowgraph.num_orig_nodes);
  }

  //connect each fanout node of copyedge EXCEPT that to a comparison to origedge
  origedge = orignode->get_output_edge();
  copyedge = copynode->get_output_edge();
  FOR_EACH_FANOUT_NODE(copyedge, nodescan) {
    outnode = nodescan.get_item();
    assert(outnode);
    if(outnode->get_address() < 2*flowgraph.num_orig_nodes) {
      //Replace each occurence of copyedge in the fanin list of outnode with origedge
      FOR_EACH_FANIN_EDGE(outnode, edgescan) {
	if(edgescan.get_item() == copyedge) {
	  edgescan.set_item(origedge);
	}
      }
      //Delete outnode from the sink list of copyedge
      copyedge->delete_sink_node(outnode);
      //Add outnode to the sink list of copyedge
      origedge->add_sink_node(outnode);
      //Need to remember the fu that outnode is mapped to, in order to re-generate its
      //interconnect networks
      fu = cur_dfg_info.functional_units[outnode->get_address()];
      assert(fu);
      if(!fulist.find(fu)) {
	fulist.append(fu);
      }
    }
  }

  //The DFG was updated by moving some fanout operations of copyedge to origedge
  //It is necessary to update the interconnect networks in front of the functional
  //units that perform these fanout operations
  FOR_EACH_LISTNODE(fulist, fuscan) {
    fu = fuscan.get_item();
    assert(fu);
    dp->generate_interconnect_network(fu,  lib->get_mux(flowgraph.get_bitwidth()));
  }

  if(flowgraph.ft_get_checkpoint(orignode) == NULL &&
     flowgraph.ft_get_checkpoint(copynode) == NULL) {
    nodes_to_secure.clear();
    nodes_to_secure.append(orignode->get_address());

    //update the flowgraph by adding the new comparisons
    //also updates the module selection information in the Schalloc_info object
    flowgraph.ft_add_comparisons(nodes_to_secure);
    //set the moduletypes for the added comparisons in the Schalloc_info object
    ft_update_schalloc_info(flowgraph, lib, cur_dfg_info);
    //re-levelize the DFG
    flowgraph.levelize();

    libelemlistptr = lib->get_choices(NOTEQ_, flowgraph.get_bitwidth());
    //simplification: assume only one comparison available in the library
    assert_force(libelemlistptr->get_size() == 1);
    libel = (*libelemlistptr)[0];
    assert(libel);
    delete libelemlistptr;
    libelemlistptr = NULL;

    compare_node = flowgraph.ft_get_checkpoint(orignode);
    assert(compare_node);
    error_edge = compare_node->get_output_edge();
    assert(error_edge && error_edge->is_po());

    //Assign lifetimes to compare_node and error_edge
    maxcycle = 0;
    FOR_EACH_FANIN_EDGE(compare_node, edgescan) {
      inedge = edgescan.get_item();
      assert(inedge);
      innode = inedge->input_node();
      assert(innode);
      maxcycle = MAX(maxcycle, cur_dfg_info.nodedeaths[innode->get_address()]);
    }
    assert(maxcycle > 0);
    cur_dfg_info.nodebirths[compare_node->get_address()] = maxcycle;
    cur_clock = (int) (dp->get_sample_period()/(float)dp->get_csteps());
    cur_dfg_info.nodedeaths[compare_node->get_address()] = maxcycle +
                                libel->get_total_csteps(dp->get_vdd(), cur_clock);

    cur_dfg_info.edgebirths[error_edge->get_address()] =
                                cur_dfg_info.nodedeaths[compare_node->get_address()];
    cur_dfg_info.edgedeaths[error_edge->get_address()] =
                   cur_dfg_info.edgebirths[error_edge->get_address()] + 1;

    //Create a dedicated functional unit and register, and assign the compare node and
    //error edge to them.
    mem_ok( new_fu = new Functional_unit(libel) );
    new_fu->add_operation(compare_node);
    cur_dfg_info.functional_units[compare_node->get_address()] = new_fu;
    compare_node->set_functional_unit(new_fu);
    dp->add_functional_unit(new_fu);
    
    mem_ok( tmpnet = new Net );
    tmpnet->connect_to_port(new_fu, OUT1);
    new_fu->connect_to_net(tmpnet, OUT1);
    dp->add_net(tmpnet);

    //generate the interconnect network feeding new_fu
    dp->generate_interconnect_network(new_fu, lib->get_mux(flowgraph.get_bitwidth()));

    mem_ok( new_su = new STOR(NULL) );
    new_su->add_variable(error_edge);
    cur_dfg_info.storage_units[error_edge->get_address()] = new_su;
    error_edge->set_storage_unit(new_su);
    dp->add_storage_unit(new_su);

    mem_ok( tmpnet = new Net );
    tmpnet->connect_to_port(new_su, REGOUT);
    new_su->connect_to_net(tmpnet, REGOUT);
    dp->add_net(tmpnet);

    //The output of the register storing the error edge is a PO
    dp->mark_po(tmpnet);

    //generate the interconnect network feeding new_su
    dp->generate_interconnect_network(new_su, lib->get_mux(flowgraph.get_bitwidth()));

  }

  //Since the fanout lists of origedge and copyedge have changed, we need to
  //update their lifetimes. origedge now feeds all the ex-fanouts of copyedge
  //too, as well as the newly added comparison operation. copyedge now feeds only
  //the comparison operation
  //FOR_EACH_FANOUT_NODE(origedge, nodescan) {
  //outnode = nodescan.get_item();
  //assert(outnode);
  //}   

  //resize the visited array to have an entry for the newly added comparison
  if(visited.get_size() != flowgraph.numnodes()) {
    visited.resize(flowgraph.numnodes(), F);
  }

  return;
}
/********************************************************************/
