/********************************************************************/
/*             FILE: sched_move.C                                   */
/********************************************************************/
#include "scheduler.h"

/********************************************************************/
/*This routine generates the next move for the iterative improvement
 *procedure. We attempt to generate a move such that the move does not
 *only affect operations that are already marked in the visited Array.
 *If such a move is not possible, the routine returns F. If one or more
 *such moves are possible, the routine returns T. If a move is found,
 *the visited Array is updated, and the Datapath cur_dp is changed
 *appropriately, and the scheduling and allocation information of the
 *operations & variables in the DFG is made in cur_dfg_info and dp.
 */
Boolean Scheduler::generate_move(Dfg &flowgraph, library *lib, Datapath *dp,
		      Schalloc_info &cur_dfg_info, Array<Boolean> &nodes_visited,
		      Array<Boolean> &edges_visited,
		      Scm *scms, const objective obj)
{
  float best_gain;
  Boolean a_move_found(F), better_b_fu_move_found(F), better_b_reg_move_found, check(F);
  Class_a_move best_a_move;
  Class_b_fu_move best_b_fu_move;
  Class_b_reg_move best_b_reg_move;

  assert(lib && dp);

  best_gain = -HUGEFLOAT;
#ifndef PHYSICAL  
  a_move_found = find_best_class_a_move(flowgraph, lib, dp, 
		  cur_dfg_info, nodes_visited, scms, obj, best_a_move, best_gain);
#else
  a_move_found = F;
#endif  
  better_b_fu_move_found = find_best_class_b_fu_move(flowgraph, lib, dp, cur_dfg_info,
	   nodes_visited, scms, obj, best_b_fu_move, best_gain);
  /*Next, search for a class B move that merges two registers into one.
   */
#ifndef PHYSICAL
  better_b_reg_move_found = find_best_class_b_reg_move(flowgraph, 
		  lib, dp, cur_dfg_info, edges_visited, scms, 
		  obj, best_b_reg_move, best_gain);
#else
  better_b_reg_move_found = F;
#endif  
 if (better_b_reg_move_found) {
    check = reschedule(flowgraph, lib, dp, cur_dfg_info, best_b_reg_move, T);
    assert_force(check);
    implement_move(best_b_reg_move, dp, flowgraph, lib, cur_dfg_info, edges_visited);
 //   cout << "---------generate_move(): FOUND A CLASS B REG MOVE-----------------" << endl;
 } else
 if(better_b_fu_move_found) 
 {
    	check = reschedule(flowgraph, lib, dp, cur_dfg_info, best_b_fu_move, T);
    	assert_force(check);
	cout<<"Fu Merged2"<<endl;
    	implement_move(best_b_fu_move, dp, flowgraph, lib, 
			cur_dfg_info, nodes_visited);
  } else if(a_move_found) 
  {
    //update cur_dfg_info to reflect the best move
    check = reschedule(flowgraph, lib, dp, cur_dfg_info, best_a_move, T);
    assert_force(check);
    implement_move(best_a_move, dp, flowgraph, cur_dfg_info, nodes_visited);
  }
  return((a_move_found || better_b_fu_move_found||better_b_reg_move_found)?T:F);
}
/***********************************************************************/
#ifdef PHYSICAL
Boolean Scheduler::generate_reg_move(Dfg &flowgraph, library *lib, Datapath *dp,
                      Schalloc_info &cur_dfg_info, Array<Boolean> &nodes_visited,
                      Array<Boolean> &edges_visited,
                      Scm *scms, const objective obj)
{
  float best_gain;
  Boolean better_b_reg_move_found, check(F);
  Class_b_reg_move best_b_reg_move;

  assert(lib && dp);

  best_gain = -HUGEFLOAT;

  better_b_reg_move_found = find_best_class_b_reg_move(flowgraph, lib, 
		  dp, cur_dfg_info, edges_visited, scms, obj, best_b_reg_move, best_gain);
 if (better_b_reg_move_found) {
    //check = reschedule(flowgraph, lib, dp, cur_dfg_info, best_b_reg_move, T);
    //assert_force(check);
    implement_move(best_b_reg_move, dp, flowgraph, lib, cur_dfg_info, edges_visited);
//    cout << "---------generate_move(): FOUND A CLASS B REG MOVE-----------------" << endl;
 } 
  return(better_b_reg_move_found);
}
#endif
/********************************************************************/
/*Given a DFG and the current datapath and scheduling allocation info,
 *this routine searches for the best move of class A possible at this point.
 *The "best" move of class A satisfies the following consitions:
 *(i) at least one of the operations mapped to the functional unit is
 *    not marked as visited.
 *(ii) a reschedule can be found to satisfy the move
 *NOTE: the DFG, datapath, and schalloc_info are NOT changed.
 *It a move is found, this routine returns T and sets best_gain and the
 *fields of best_a_move appropriately.
 */
Boolean Scheduler::find_best_class_a_move(Dfg &flowgraph, library *lib, Datapath *dp,
		       Schalloc_info &cur_dfg_info, Array<Boolean> &nodes_visited,
		       Scm *scms, const objective obj, Class_a_move &best_a_move,
		       float &best_gain)
{
  Boolean check, found;
  FUPTR fu;
  NODEPTR node;
  register int i;
  List_iterator<FUPTR> fuscan;
  List_iterator<NODEPTR> nodescan;
  Class_a_move cur_a_move;
  float cur_gain;

  assert(lib && dp);

  found = F;
  FOR_EACH_LISTNODE(dp->get_functional_units(), fuscan) 
  {
    fu = fuscan.get_item();

    //at least one operation mapped to the fu must be un-visited
    check = F;
    FOR_EACH_LISTNODE(fu->get_operations(), nodescan) {
      node = nodescan.get_item();
      assert(node);
      if(!nodes_visited[node->get_address()]) {
	check = T;
	break;
      }
    }
    if(check == F)
      continue;    /*WATCH OUT - A CONTINUE BEING USED*/

    for(i = 0; i < lib->numelements(); i++) 
    {
      if(i == fu->get_libelement()->get_address())
	continue;    /*WATCH OUT - A CONTINUE BEING USED*/
      //the new type should perform all the operations mapped to the fu
      check = T;
      FOR_EACH_LISTNODE(fu->get_operations(), nodescan) 
      {
	node = nodescan.get_item();
	if(!(lib->get_nthelement(i)->can_perform(node->get_func(),
						 flowgraph.get_bitwidth()))) 
	{  check = F;  break;	}
      }
      if(check == F)
	continue;    /*WATCH OUT - A CONTINUE BEING USED*/

      cur_a_move.set_fields(fu, fu->get_libelement(), lib->get_nthelement(i));
      switch(obj) 
      {
      	case AREA:
		cur_gain = compute_areagain(dp, lib, cur_a_move);
		break;
      	case POWER:
		cur_gain = compute_scgain(dp, cur_a_move, scms);
		break;
      	default:
		assert_force(0);
		break;
      }
      if(cur_gain > best_gain) 
      {
	if(reschedule(flowgraph, lib, dp, cur_dfg_info, cur_a_move, F)) 
	{  best_gain = cur_gain;  best_a_move.copy(cur_a_move);  found = T;}
      }
      
    } /*END FOR EACH LIBRARY ELEMENT*/
  } /*END FOR EACH FU*/

  return(found);
}
/********************************************************************/
/*Given a DFG and the current datapath and schalloc info, this routine
 *searches for the best class B move. The "best" possible class B move
 *should satisfy:
 *(i) it should have the best gain
 *(ii) it must be possible to reschedule the DFG in order to satisfy the
 *precedence constraints without violating the sample period.
 */
Boolean Scheduler::find_best_class_b_fu_move(Dfg &flowgraph, library *lib, Datapath *dp,
		       Schalloc_info &cur_dfg_info, Array<Boolean> &nodes_visited,
		       Scm *scms, const objective obj, Class_b_fu_move &best_move,
		       float &best_gain)
{
  List_ar<Operator> operlist;
  List_iterator<FUPTR> fuscan1, fuscan2;
  List_iterator<NODEPTR> nodescan;
  FUPTR fu1, fu2;
  NODEPTR node;
  Boolean check, found;
  Class_b_fu_move cur_b_move;
  float cur_gain;

  assert(lib && dp);
  found = F;
  FOR_EACH_LISTNODE(dp->get_functional_units(), fuscan1) {
    fu1 = fuscan1.get_item();
    assert(fu1);
    check = F;
    FOR_EACH_LISTNODE(fu1->get_operations(), nodescan) {
      node = nodescan.get_item();
      assert(node);
      if(!nodes_visited[node->get_address()]) {
	check = T;
      }
    }
    if(check == F) {
      continue; /*WATCH OUT - CONTINUE WITH THE INNER FOR LOOP*/
    }

    FOR_EACH_LISTNODE_REVERSE(dp->get_functional_units(), fuscan2) {
      fu2 = fuscan2.get_item();
      assert(fu2);
      
      if(fu1 == fu2) {
	continue; /*WATCH OUT - CONTINUE WITH THE INNER FOR LOOP*/
      }

      check = F;
      FOR_EACH_LISTNODE(fu2->get_operations(), nodescan) {
	node = nodescan.get_item();
	assert(node);
	if(!nodes_visited[node->get_address()]) {
	  check = T;
	}
      }
      if(check == F) {
	continue; /*WATCH OUT - CONTINUE WITH THE INNER FOR LOOP*/
      }

      //A class B move should not involve a change in the library template
      if(fu1->get_libelement() != fu2->get_libelement()) {
	continue;
      }

      cur_b_move.splitting = F;
      cur_b_move.fu1 = fu1;
      cur_b_move.fu2 = fu2;
      cur_b_move.split_operations.clear();
      switch(obj) {
      case AREA:
	cur_gain = compute_areagain(dp, lib, cur_b_move);
	break;
      case POWER:
	cur_gain = compute_scgain(dp, cur_b_move, scms);
	break;
      default:
	assert_force(0);
	break;
      }

      if(cur_gain > best_gain) {
	if(reschedule(flowgraph, lib, dp, cur_dfg_info, cur_b_move, F)) {
	  best_gain = cur_gain;
	  best_move.copy(cur_b_move);
	  found = T;
	}
      }
    } /*END FOR EACH FUNCTIONAL UNIT*/
  } /*END FOR EACH FUNCTIONAL UNIT*/

return(found);
}
/********************************************************************/
/*Given a DFG and the current datapath and schalloc info, this routine
 *searches for the best class B register move. The "best" possible class B move
 *should satisfy:
 *(i) it should have the best gain 
 *(ii) it must be possible to reschedule the DFG in order to satisfy the
 *precedence constraints without violating the sample period.
 */
Boolean Scheduler::find_best_class_b_reg_move(Dfg &flowgraph, library *lib,
			     Datapath *dp, Schalloc_info &cur_dfg_info,
			     Array<Boolean> & visited_edges, Scm *scms,
			     const objective obj, Class_b_reg_move &best_move,
			     float &best_gain)
{
  List_ar<Operator> operlist;
  List_iterator<Storage_unit *> storscan1, storscan2;
  List_iterator<NODEPTR> nodescan;
  List_iterator<EDGEPTR> edgescan;
  STORPTR su1, su2;
  NODEPTR node;
  EDGEPTR edge;
  Boolean check, found;
  Class_b_reg_move cur_b_move;
  float cur_gain;

  assert(lib && dp);

  found = F;
  FOR_EACH_LISTNODE(dp->get_storage_units(), storscan1) {
    su1 = (STORPTR) storscan1.get_item();
    assert(su1);
    check = F;
    FOR_EACH_LISTNODE(((STORPTR)su1)->get_variables(), edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      if(!visited_edges[edge->get_address()]) {
	check = T;
      }
    }
    if(check == F) {
      continue; /*WATCH OUT - CONTINUE WITH THE OUTER FOR LOOP*/
    }
    FOR_EACH_LISTNODE_REVERSE(dp->get_storage_units(), storscan2) {
      su2 = (STORPTR)storscan2.get_item();
      assert(su2);
      
      if(su1 == su2) {
	continue; /*WATCH OUT - CONTINUE WITH THE INNER FOR LOOP*/
      }
      check = F;
      FOR_EACH_LISTNODE(((STORPTR)su2)->get_variables(), edgescan) {
	edge = edgescan.get_item();
	assert(edge);
	if(!visited_edges[edge->get_address()]) {
	  check = T;
	}
      }
      if(check == F) {
	continue; /*WATCH OUT - CONTINUE WITH THE INNER FOR LOOP*/
      }
//     if(can_share_registers(su1->get_variables(), su2->get_variables())) {
      	   cur_b_move.splitting = F;
      	   cur_b_move.su1 = su1;
      	   cur_b_move.su2 = su2;
      	   switch(obj) {
      		case AREA:
			cur_gain = compute_areagain(flowgraph, dp, lib, cur_b_move);
			break;
      	        case POWER:
			cur_gain = compute_scgain(dp, cur_b_move, scms);
			break;
      		default:
			assert_force(0);
			break;
      	  }
          if(cur_gain > best_gain) {
	     if(reschedule(flowgraph, lib, dp, cur_dfg_info, cur_b_move, F)) {
	  	best_gain = cur_gain;
	 	best_move.copy(cur_b_move);
	  	found = T;
	     }
      	  }//end if cur_gain>best_gain
  //   }//end of can_share_registers
    } /*END FOR EACH STORAGE UNIT*/
  } /*END FOR EACH STORAGE UNIT*/
  //08022002
  /*
if(best_gain>=0)
	found = T;
else
	found = F;
	*/
  return(found);
}
/** to support perfect power management **/
/** Ganesh Lakshminarayana, 
  June 10th 1997 **/
/********************************************************************/
/**   extend variable lifetimes to enable perfect power management **/
/********************************************************************/

void extend_variable_lifetimes(Dfg& flowgraph) {
  int i = 0;
  FOR_EACH_EDGEINDEX(flowgraph, i) {
    EDGEPTR e = flowgraph.get_nthedge(i);
    /** for each operation fed by this variable **/
    NODEPTR n;
    List_iterator<NODEPTR> nodescan1;
    FOR_EACH_FANOUT_NODE(e, nodescan1) {
      n = nodescan1.get_item();

      Functional_unit* fu = n->get_functional_unit();
      List_iterator<NODEPTR> nodescan;
      int flag = 0;
      FOR_EACH_LISTNODE(fu->get_operations(), nodescan) {
        if (flag == 1) {
          e->set_death(MAX(e->get_death(), nodescan.get_item()->get_birth()));
        }
        if (nodescan.get_item() == n)
          flag = 1;
      }
    }
  }
}
/**/

/********************************************************************/
//Given a move of class A, modify the datapath to reflect the move
//This involves deleting the old functional unit, creating a new functional
//unit to replace the old one, and assigning the operations of the old functional
//unit to the new one. This routine also marks the operations affected by the move
//as visited
void Scheduler::implement_move(Class_a_move &move, Datapath *dp, Dfg &flowgraph,
			       Schalloc_info &cur_dfg_info, Array<Boolean> &nodes_visited)
{
  Boolean check;
  FUPTR old_fu, new_fu;
  List_iterator<NODEPTR> nodescan;
  NODEPTR node;
  int i;
  Net *net;

  old_fu = move.fu;
  assert(old_fu);

  assert_force(dp->get_functional_units().find(old_fu));
  //create a new functional unit of the new type
  new_fu = new Functional_unit(move.new_libelement);
  dp->add_functional_unit(new_fu);

  //assign each operation that was assigned to the old functional
  //unit to the new functional unit. This involves adding the operation
  //to the functional units list of operations, and updating schalloc_info
  //to reflect the operations functional unit
  FOR_EACH_LISTNODE(old_fu->get_operations(), nodescan) {
    node = nodescan.get_item();
    assert(node);
    cur_dfg_info.functional_units[node->get_address()] = new_fu;
    new_fu->add_operation(node);
//This was a bug
#ifdef PHYSICAL
   node->set_functional_unit(new_fu);	
#endif    
  }

  for(i = 1; i <= old_fu->get_number_of_nets(); i++) {
    net = old_fu->get_net(i);
    if (i <= 3) assert(net);
    
    if (net) {
      old_fu->disconnect_from_net(net, i);
      new_fu->connect_to_net(net, i);
    }
  }

  //delete the old functional unit from the list
  dp->delete_functional_unit(old_fu);
  
  //free the old functional unit
  delete old_fu;

  //mark the operations that are affected by the move as visited
  check = T;
  FOR_EACH_LISTNODE(new_fu->get_operations(), nodescan) {
    node = nodescan.get_item();
    assert(node);
    if(!nodes_visited[node->get_address()]) {
      check = F;
    }
    nodes_visited[node->get_address()] = T;
  }
  assert_force(check == F);

  return;
}
/********************************************************************/
//Given a move of class B, modify the datapath and the dfg schalloc info
//to reflect the move. This involves deleting the two old functional units,
//creating a new functional unit to replace the old ones, and assigning the
//operations of both the old functional units to the new one. This routine
//also marks the operations affected as a result of the current move as visited
void Scheduler::implement_move(Class_b_fu_move &move, Datapath *dp, Dfg &flowgraph,
		   library *lib, Schalloc_info &cur_dfg_info, Array<Boolean> &nodes_visited)
{
  FUPTR new_fu;
  List_iterator<NODEPTR> nodescan;
  List_ar<STORPTR> fanout_registers;
  List_iterator<STORPTR> suscan;
  STORPTR su;
  NODEPTR node;
  EDGEPTR edge;
  Boolean check;
  Net *tmpnet;
  int j, num_fu_input_ports;
  Mux *intcon_unit;
  
  assert(dp && lib);
  //Mark all operations associated with the move as visited
  assert(move.fu1 && dp->get_functional_units().find(move.fu1));
  check = T;
  FOR_EACH_LISTNODE(move.fu1->get_operations(), nodescan) {
    node = nodescan.get_item();
    assert(node);
    if(!nodes_visited[node->get_address()]) {
      check = F;
    }
    nodes_visited[node->get_address()] = T;
  }
  assert_force(check == F);

  if(!move.splitting) {
    assert(move.fu2 && dp->get_functional_units().find(move.fu2));
    check = T;
    FOR_EACH_LISTNODE(move.fu2->get_operations(), nodescan) {
      node = nodescan.get_item();
      assert(node);
      if(!nodes_visited[node->get_address()]) {
	check = F;
      }
      nodes_visited[node->get_address()] = T;
    }
    assert_force(check == F);
  }

  //create a new functional unit, add it to the data path
  // create an output net for the unit
  new_fu = new Functional_unit(move.fu1->get_libelement());
  mem_ok( tmpnet = new Net );
  new_fu->connect_to_net(tmpnet, OUT1);
  dp->add_net(tmpnet);

  if(move.splitting) { //SPLITTING MOVE
    assert(move.fu1 && !move.fu2 && !move.split_operations.is_empty());

    //transfer the split operations to the new functional unit
    FOR_EACH_LISTNODE(move.split_operations, nodescan) {
      node = nodescan.get_item();
      assert(node);
      move.fu1->delete_operation(node);
      new_fu->add_operation(node);
      cur_dfg_info.functional_units[node->get_address()] = new_fu;
      node->set_functional_unit(new_fu);
    }
  } else { //SHARING MOVE
    assert(move.fu1 && move.fu2 && move.split_operations.is_empty());
    //transfer the operations of fu1 and fu2 to new_fu
    FOR_EACH_LISTNODE(move.fu1->get_operations(), nodescan) {
      node = nodescan.get_item();
      assert(node);
      node->set_birth(cur_dfg_info.nodebirths[node->get_address()]);
      node->set_death(cur_dfg_info.nodedeaths[node->get_address()]);
      new_fu->add_operation(node);
      cur_dfg_info.functional_units[node->get_address()] = new_fu;
      node->set_functional_unit(new_fu);
    }
    FOR_EACH_LISTNODE(move.fu2->get_operations(), nodescan) {
      node = nodescan.get_item();
      assert(node);
      node->set_birth(cur_dfg_info.nodebirths[node->get_address()]);
      node->set_death(cur_dfg_info.nodedeaths[node->get_address()]);
      new_fu->add_operation(node);
      cur_dfg_info.functional_units[node->get_address()] = new_fu;
      node->set_functional_unit(new_fu);
    }

  }

  dp->add_functional_unit(new_fu);
  dp->generate_interconnect_network(new_fu, lib->get_mux(flowgraph.get_bitwidth()));
  //Interconnect network to be re-generated for all the datapath elements that are
  //fed by this functional unit
  //(i) Get a list of edges sourced bu this fu
  
  //(ii) associate each element of this list with a register
  //(iii) generate interconnect network for the register
  FOR_EACH_LISTNODE(new_fu->get_operations(), nodescan) {
    edge = nodescan.get_item()->get_output_edge();
    assert(edge);
    su = edge->get_storage_unit();
    assert(su);
    if (fanout_registers.find(su) == NULL) {
      fanout_registers.append(su);
    }
  }
  FOR_EACH_LISTNODE(fanout_registers, suscan) {
    su = suscan.get_item();
    assert(su);
    dp->generate_interconnect_network(su, lib->get_mux(flowgraph.get_bitwidth()));
  }

  if(!move.splitting) {
    //NOTE: assumes that for a fu, input ports are numbered 0..n-1 and 
    //the output port is n
    {
      tmpnet = move.fu1->get_net(OUT1);
      assert(tmpnet);
      dp->delete_net(tmpnet);
      delete tmpnet;

      num_fu_input_ports = move.fu1->get_libelement()->number_of_input_ports();
      for(j = 1; j <= num_fu_input_ports; j++) {
	//delete the interconnect unit connected to each input port of move.fu1,
	//and the net that implements the above connection
	tmpnet = move.fu1->get_net(j);
	//This net should have exactly 2 connections - the output port
	//of an intcon_unit and the jth input port of this functional unit
	assert(tmpnet && tmpnet->number_of_connections() == 2);
	
	intcon_unit = (Mux *)tmpnet->get_driver();
	assert(intcon_unit && !strcmp(intcon_unit->get_id(), "Mux"));
	
	dp->delete_interconnect_unit(intcon_unit);
	dp->delete_net(tmpnet);
	delete intcon_unit;
	delete tmpnet;
      }/*END FOR j = 1 TO num_fu_input_ports*/
      dp->delete_functional_unit(move.fu1);
      delete move.fu1;
    }

    //NOTE: assumes only 1 output port, and NO control ports
    {
      tmpnet = move.fu2->get_net(OUT1);
      assert(tmpnet);
      dp->delete_net(tmpnet);
      delete tmpnet;

      num_fu_input_ports = move.fu2->get_libelement()->number_of_input_ports();
      for(j = 1; j <= num_fu_input_ports; j++) {
	//delete the interconnect unit connected to each input port of move.fu2,
	//and the net that implements the above connection
	tmpnet = move.fu2->get_net(j);
	//This net should have exactly 2 connections - the output port
	//of an intcon_unit and the jth input port of this functional unit
	assert(tmpnet && tmpnet->number_of_connections() == 2);

	intcon_unit = (Mux *)tmpnet->get_driver();
	assert(intcon_unit && !strcmp(intcon_unit->get_id(), "Mux"));

	dp->delete_interconnect_unit(intcon_unit);
	dp->delete_net(tmpnet);
	delete intcon_unit;
	delete tmpnet;
      }/*END FOR j = 1 TO num_fu_input_ports*/
      dp->delete_functional_unit(move.fu2);
      delete move.fu2;
    }
  }
  return;
}
/********************************************************************/
//Given a reg move of class B, modify the datapath and the dfg schalloc info
//to reflect the move. This involves deleting the two old storage units,
//creating a new storage unit to replace the old ones, and assigning the
//variables of both the old storage units to the new one. This routine
//also marks the variables affected as a result of the current move as visited
void Scheduler::implement_move(const Class_b_reg_move &move, Datapath *dp, Dfg &flowgraph,
		   library *lib, Schalloc_info &cur_dfg_info, Array<Boolean> &edges_visited)
{
  STORPTR new_su, su;
  FUPTR fu;
  List_iterator<EDGEPTR> edgescan;
  List_iterator<STORPTR> suscan;
  List_iterator<FUPTR> fuscan;
  List_iterator<NODEPTR> nodescan;
  List_ar<FUPTR> fanout_fus;
  List_ar<STORPTR> fanout_regs;
  EDGEPTR edge;
  Boolean check;
  Net *tmpnet;
  Mux *intcon_unit;
  
  assert(dp && lib);

  //mark all variables involved in the move as visited
  assert(move.su1);
  assert(dp->get_storage_units().find(move.su1));
  check = T;
  FOR_EACH_LISTNODE(move.su1->get_variables(), edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    if(!edges_visited[edge->get_address()]) {
      check = F;
    }
    edges_visited[edge->get_address()] = T;
  }
  assert_force(check == F);
  if(!move.splitting) {
    assert(move.su2);
    assert(dp->get_storage_units().find(move.su2));
    check = T;
    FOR_EACH_LISTNODE(move.su2->get_variables(), edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      if(!edges_visited[edge->get_address()]) {
	check = F;
      }
      edges_visited[edge->get_address()] = T;
    }
    assert_force(check == F);
  }
  
  //Create a new storage unit, add it to the data path
  //Create an output net for the unit
  mem_ok( new_su = new STOR(lib->get_register(flowgraph.get_bitwidth())) );
  mem_ok( tmpnet = new Net );
  new_su->connect_to_net(tmpnet, REGOUT);
  dp->add_net(tmpnet);

  //In the case of a splitting move, move the split variables to new_su
  //Otherwise, move all variables assigned to move.su1 and move.su2 to new_su
  if(move.splitting) { //SPLITTING MOVE
    assert(move.su1 && !move.su2 && !move.split_variables.is_empty());
    FOR_EACH_LISTNODE(move.split_variables, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      move.su1->delete_variable(edge);
      new_su->add_variable(edge);
      cur_dfg_info.storage_units[edge->get_address()] = new_su;
      edge->set_storage_unit(new_su);
    }
  } else { //SHARING MOVE
    assert(move.su1 && move.su2 && move.split_variables.is_empty());
    //transfer the variables of su1 and su2 to new_su
    FOR_EACH_LISTNODE(move.su1->get_variables(), edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      edge->set_birth(cur_dfg_info.edgebirths[edge->get_address()]);
      edge->set_death(cur_dfg_info.edgedeaths[edge->get_address()]);
      new_su->add_variable(edge);
      cur_dfg_info.storage_units[edge->get_address()] = new_su;
      edge->set_storage_unit(new_su);
    }
    FOR_EACH_LISTNODE(move.su2->get_variables(), edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      edge->set_birth(cur_dfg_info.edgebirths[edge->get_address()]);
      edge->set_death(cur_dfg_info.edgedeaths[edge->get_address()]);
      new_su->add_variable(edge);
      cur_dfg_info.storage_units[edge->get_address()] = new_su;
      edge->set_storage_unit(new_su);
    }
  }
  dp->add_storage_unit(new_su);
  dp->generate_interconnect_network(new_su, lib->get_mux(flowgraph.get_bitwidth()));

  //If new_su stores a PO variable, mark the net connected to the REGOUT port of new_su as a
  //PO net
  FOR_EACH_LISTNODE(flowgraph.outputs, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    if(cur_dfg_info.storage_units[edge->get_address()] == new_su) {
      tmpnet = new_su->get_net(REGOUT);
      assert(tmpnet);
      dp->mark_po(tmpnet);
      break; /*BREAK FROM THE FOR LOOP*/
    }
  }

  //Now, we gennerate the interconnect networks for all the fus and the loopin registers fed 
  //by this register
  FOR_EACH_LISTNODE(new_su->get_variables(), edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    FOR_EACH_FANOUT_NODE(edge, nodescan) {
      fu = nodescan.get_item()->get_functional_unit();
      if (!fanout_fus.find(fu)) {
	fanout_fus.append(fu);
      }
    }
  }
  FOR_EACH_LISTNODE(fanout_fus, fuscan) {
    fu = fuscan.get_item();
    assert(fu);
    dp->generate_interconnect_network(fu, lib->get_mux(flowgraph.get_bitwidth()));
  }

  FOR_EACH_LISTNODE(new_su->get_variables(), edgescan) {
    edge = edgescan.get_item();
    if (edge->is_loopout()) {
      su = edge->get_loopin_link()->get_storage_unit();
      assert(su);
      if (!fanout_regs.find(su)) {
	fanout_regs.append(su);
      }
    }
  }
  FOR_EACH_LISTNODE(fanout_regs, suscan) {
    su = suscan.get_item();
    assert(su);
    dp->generate_interconnect_network(su, lib->get_mux(flowgraph.get_bitwidth()));
  }
  
  if(!move.splitting) {
    //Delete the net connected to the REGOUT port of move.su1,
    //the net connected to the REGIN port of move.su1, and the
    //interconnect unit driving this net. Then, delete move.su1
    //from the datapath and free it
    {
      tmpnet = move.su1->get_net(REGOUT);
      if(dp->is_output(move.su1)) {
	dp->unmark_po(tmpnet);
      }
      dp->delete_net(tmpnet);
      delete tmpnet;

      tmpnet = move.su1->get_net(REGIN);
      //This net should have exactly 2 connections - the output port
      //of an intcon_unit and the input port of this storage unit
      assert(tmpnet && tmpnet->number_of_connections() == 2);
      intcon_unit = (Mux *)tmpnet->get_driver();
      assert(intcon_unit && !strcmp(intcon_unit->get_id(), "Mux"));

      dp->delete_interconnect_unit(intcon_unit);
      dp->delete_net(tmpnet);
      delete intcon_unit;
      delete tmpnet;

      dp->delete_storage_unit(move.su1);
      delete move.su1;
    }

    //Do the same for move.su2
    {
      tmpnet = move.su2->get_net(REGOUT);
      if(dp->is_output(move.su2)) {
	dp->unmark_po(tmpnet);
      }
      dp->delete_net(tmpnet);
      delete tmpnet;

      tmpnet = move.su2->get_net(REGIN);
      //This net should have exactly 2 connections - the output port
      //of an intcon_unit and the input port of this storage unit
      assert(tmpnet && tmpnet->number_of_connections() == 2);
      intcon_unit = (Mux *)tmpnet->get_driver();
      assert(intcon_unit && !strcmp(intcon_unit->get_id(), "Mux"));

      dp->delete_interconnect_unit(intcon_unit);
      dp->delete_net(tmpnet);
      delete intcon_unit;
      delete tmpnet;

      dp->delete_storage_unit(move.su2);
      delete move.su2;
    }
  }

  return;
}
/********************************************************************/
