/********************************************************************/
/*              FILE: sched_util.C                                  */
/* This file contains functions that perform various consistency    */
/* checks on the synthesis results, including basic schedule and    */
/* resource sharing validation, etc.                                */
/********************************************************************/
#include "scheduler.h"

/********************************************************************/
/*This routine creates a copy of a given flowgraph and datapath.
 */
void Scheduler::copy_flowgraph_and_dp(Dfg &source_dfg, Datapath &source_dp,
				      Dfg &target_dfg, Datapath &target_dp,
				      library *lib)
{
  int i;
  List_iterator<NODEPTR> nodescan;
  List_iterator<EDGEPTR> edgescan;
  List_iterator<FUPTR> fuscan;
  List_iterator<Storage_unit *> regscan;
  NODEPTR node, copynode;
  EDGEPTR edge, copyedge;
  FUPTR fu, new_fu;
  STORPTR reg, new_reg;

  Mux *intcon_unit;
  Net *tmpnet;
  int j, num_fu_input_ports;;
  char portname[10];
  
  assert(lib);

  //check_dfg_and_datapath(source_dfg, source_dp);

  //clear this DFG first;
  target_dfg.clear();

  strcpy(target_dfg.name, source_dfg.name);
  target_dfg.nodecount = source_dfg.numnodes();
  target_dfg.edgecount = source_dfg.numedges();
  target_dfg.bitwidth = source_dfg.get_bitwidth();

  //create nodes in target_dfg corresponding to source_dfg
  for(i = 0; i < source_dfg.numnodes(); i++) {
    node = source_dfg.get_nthnode(i);
    assert(node);
    copynode = new NODE;;
    assert(copynode);
    copynode->copy(*node);
    target_dfg.nodes[i] = copynode;
  }

  //create edges in target_dfg corresponding to source_dfg
  for(i = 0; i < source_dfg.numedges(); i++) {
    edge = source_dfg.get_nthedge(i);
    assert(edge);
    copyedge = new EDGE;
    assert(copyedge);
    copyedge->copy(*edge);
    target_dfg.edges[i] = copyedge;
  }
  
  //build the fanin and fanout lists for the nodes in target_dfg
  for(i = 0; i < source_dfg.numnodes(); i++) {
    node = source_dfg.get_nthnode(i);
    assert(node);
    copynode = target_dfg.get_nthnode(i);
    assert(copynode);

    copynode->get_input_edges().clear();
    copynode->get_output_edges().clear();
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      copyedge = target_dfg.get_nthedge(edge->get_address());
      copynode->add_input_edge(copyedge);
    }
    edge = node->get_output_edge();
    copyedge = target_dfg.get_nthedge(edge->get_address());
    copynode->set_output_edge(copyedge);
  }

  //build the fanin and fanout lists for the edges in target_dfg
  for(i = 0; i < source_dfg.numedges(); i++) {
    copyedge = target_dfg.get_nthedge(i);
    edge = source_dfg.get_nthedge(i);
    assert(copyedge);
    assert(edge);

    copyedge->get_sink_nodes().clear();
    copyedge->get_source_nodes().clear();
    FOR_EACH_FANOUT_NODE(edge, nodescan) {
      node = nodescan.get_item();
      copynode = target_dfg.get_nthnode(node->get_address());
      copyedge->add_sink_node(copynode);
    }

    node = edge->input_node();
    assert(node || edge->is_pi() || edge->is_loopin() || edge->is_constant());
    if(node) {
      copynode = target_dfg.get_nthnode(node->get_address());
      copyedge->set_source_node(copynode);
    }
  }

  //create the inputs, outputs, constants, loopins, and loopouts of target_dfg
  FOR_EACH_LISTNODE(source_dfg.inputs, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    copyedge = target_dfg.get_nthedge(edge->get_address());
    assert(copyedge);
    target_dfg.inputs.append(copyedge);
  }
  FOR_EACH_LISTNODE(source_dfg.outputs, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    copyedge = target_dfg.get_nthedge(edge->get_address());
    assert(copyedge);
    target_dfg.outputs.append(copyedge);
  }
  FOR_EACH_LISTNODE(source_dfg.loopins, edgescan) {
    edge = edgescan.get_item();
    assert(edge && edge->get_loopout_link());
    copyedge = target_dfg.get_nthedge(edge->get_address());
    assert(copyedge);
    copyedge->set_loopout_link(target_dfg.get_nthedge(edge->get_loopout_link()->get_address()));
    target_dfg.loopins.append(copyedge);
  }
  FOR_EACH_LISTNODE(source_dfg.constants, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    copyedge = target_dfg.get_nthedge(edge->get_address());
    assert(copyedge);
    target_dfg.constants.append(copyedge);
  }
  FOR_EACH_LISTNODE(source_dfg.loopouts, edgescan) {
    edge = edgescan.get_item();
    assert(edge && edge->get_loopin_link());
    copyedge = target_dfg.get_nthedge(edge->get_address());
    assert(copyedge);
    copyedge->set_loopin_link(target_dfg.get_nthedge(edge->get_loopin_link()->get_address()));
    target_dfg.loopouts.append(copyedge);
  }
  
  target_dfg.levelize();

  target_dp.clear();
  target_dp.set_vdd(source_dp.get_vdd());
  target_dp.set_sample_period(source_dp.get_sample_period());
  target_dp.set_csteps(source_dp.get_csteps());
#ifdef PHYSICAL
  	//copy the module_size and module_link in datapath
	target_dp.set_link(source_dp.get_link());
	target_dp.set_size(source_dp.get_size());
	target_dp.set_area_after_floorplan(source_dp.get_area_after_floorplan());
	target_dp.set_module_sc(source_dp.get_module_sc());
	target_dp.set_wire_sc(source_dp.get_wire_sc());
	target_dp.set_mux_sc(source_dp.get_mux_sc());
	target_dp.set_clock_tree_sc(source_dp.get_clock_tree_sc());
	target_dp.set_bitwidth(source_dp.get_bitwidth());
#endif  
  //construct the functional units of target_dp
  FOR_EACH_LISTNODE(source_dp.get_functional_units(), fuscan) {
    fu = fuscan.get_item();
    assert(fu);

    new_fu = new FU(fu->get_libelement());
    target_dp.add_functional_unit(new_fu);
    //create a net for the fu output port
    mem_ok( tmpnet = new Net );
    new_fu->connect_to_net(tmpnet, OUT1);
    target_dp.add_net(tmpnet);

    FOR_EACH_LISTNODE(fu->get_operations(), nodescan) {
      node = nodescan.get_item();
      assert(node);
      //add the corresponding node in target_dfg to new_fu
      copynode = target_dfg.get_nthnode(node->get_address());
      assert(copynode && copynode->get_address() == node->get_address());
      new_fu->add_operation(copynode);
      copynode->set_functional_unit(new_fu);
    }
  }

  //construct the storage units of target_dp
  FOR_EACH_LISTNODE(source_dp.get_storage_units(), regscan) {
    reg = (STORPTR) regscan.get_item();
    assert(reg); 

    new_reg = new STOR(reg->get_libelement());
    target_dp.add_storage_unit(new_reg);
    //create a net for the storage unit output port
    mem_ok( tmpnet = new Net );
    new_reg->connect_to_net(tmpnet, REGOUT);
    target_dp.add_net(tmpnet);

    FOR_EACH_LISTNODE(reg->get_variables(), edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      //add the corresponding edge in target_dfg to new_reg
      copyedge = target_dfg.get_nthedge(edge->get_address());
      assert(copyedge && copyedge->get_address() == edge->get_address());
      new_reg->add_variable(copyedge);
      copyedge->set_storage_unit(new_reg);
    }
  }

  //add interconnect units to connect the functional units and registers
  //all interconnect units are initially 1-input since no sharing has been done
  //NOTE: The code below AUTOMATICALLY marks input nets to the datapath
  FOR_EACH_LISTNODE(target_dp.get_functional_units(), fuscan) {
    new_fu = fuscan.get_item();
    assert(new_fu);
    target_dp.generate_interconnect_network(new_fu, lib->get_mux(target_dfg.get_bitwidth()));
  }

  FOR_EACH_LISTNODE(target_dp.get_storage_units(), regscan) {
    new_reg = (STORPTR) regscan.get_item();
    assert(new_reg);
    target_dp.generate_interconnect_network(new_reg, lib->get_mux(target_dfg.get_bitwidth()));
  }

  //mark output nets of the datapath
  FOR_EACH_LISTNODE(target_dfg.outputs, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    new_reg = (STORPTR) edge->get_storage_unit();
    assert(new_reg);
    tmpnet = new_reg->get_net(REGOUT);
    assert(tmpnet);
    if(!target_dp.get_po_nets().find(tmpnet)) {
      target_dp.mark_po(tmpnet);
    }
  }

  //check_dfg_and_datapath(target_dfg, target_dp);
  return;
}
/********************************************************************/
// This routine performs various consistency checks on a Dfg and Datapath
// to ensure that they satisfy basic invariant properties, such as:
// -> operations (variables) that are simultaneously alive cannot share a
// resource,
// -> data dependencies are indeed satisfied during scheduling, etc.
// Checks are also performed to ensure the consistency of the Dfg and Datapath
// data structures, and the operations, variables, and datapath elements

void Scheduler::check_dfg_and_datapath(Dfg &flowgraph, Datapath &dp)
{
  int i, j, total, stage_csteps;
  float cur_clock;
  NODEPTR node, node1;
  EDGEPTR edge, edge1;
  FUPTR fu;
  STORPTR su;
  LIBELPTR libel;
  List_iterator<FUPTR> fuscan;
  List_iterator<Storage_unit *> suscan;
  List_iterator<NODEPTR> nodescan, nodescan1;
  List_iterator<EDGEPTR> edgescan, edgescan1;

  assert(flowgraph.numnodes() > 0 && flowgraph.numedges() > 0 &&
	 flowgraph.numedges() >= flowgraph.numnodes());

  for(i = 0; i < flowgraph.numnodes(); i++) {
    node = flowgraph.get_nthnode(i);
    assert(node && node->get_address() == i);
    assert(node->get_birth() >= 0 && node->get_death() > node->get_birth());
    fu = node->get_functional_unit();
    assert(fu);
    assert(dp.get_functional_units().find(fu));
    assert(fu->get_operations().find(node));
  }

  for(i = 0; i < flowgraph.numedges(); i++) {
    edge = flowgraph.get_nthedge(i);
    assert(edge && edge->get_address() == i);
    assert(edge->get_birth() >= 0 && edge->get_death() > edge->get_birth());
    su = edge->get_storage_unit();
    assert(su);
    assert(dp.get_storage_units().find(su));
    assert(su->get_variables().find(edge));
  }

  cur_clock = (int)(dp.get_sample_period()/(float)dp.get_csteps());
  total = 0;
  FOR_EACH_LISTNODE(dp.get_functional_units(), fuscan) {
    fu = fuscan.get_item();
    assert(fu);
    libel = fu->get_libelement();
    assert(libel);
    stage_csteps = libel->get_stage_csteps(dp.get_vdd(), cur_clock);

    total += fu->get_operations().get_size();
    i = 0;
    FOR_EACH_LISTNODE(fu->get_operations(), nodescan) {
      node = nodescan.get_item();
      assert(node);
      j = 0;
      FOR_EACH_LISTNODE(fu->get_operations(), nodescan1) {
	node1 = nodescan1.get_item();
	assert(node1);
	if(i == j) {
	  continue;
	} else {
	  assert( node->get_birth() >= (node1->get_birth() + stage_csteps) ||
		  node1->get_birth() >= (node->get_death() + stage_csteps) );
	}
	j++;
      }
      i++;
    }
  }
  assert(total == flowgraph.numnodes());

  total = 0;
  FOR_EACH_LISTNODE(dp.get_storage_units(), suscan) {
    su = (STORPTR) suscan.get_item();
    assert(su);
    total += su->get_variables().get_size();
    i = 0;
    FOR_EACH_LISTNODE(su->get_variables(), edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      j = 0;
      FOR_EACH_LISTNODE(su->get_variables(), edgescan1) {
	edge1 = edgescan1.get_item();
	assert(edge1);
	if(i == j) {
	  continue;
	} else {
	  assert(edge->get_birth() >= edge1->get_death() || edge1->get_birth() >= edge->get_death());
	}
	j++;
      }
      i++;
    }
  }
  assert(total == flowgraph.numedges());

  return;
}
/********************************************************************/
void Scheduler::check_dfg_and_datapath(Dfg &flowgraph, Schalloc_info &info, Datapath &dp)
{
  int i, j, total, stage_csteps;
  float cur_clock;
  NODEPTR node, node1;
  EDGEPTR edge, edge1;
  FUPTR fu;
  STORPTR su;
  LIBELPTR libel;
  List_iterator<FUPTR> fuscan;
  List_iterator<Storage_unit *> suscan;
  List_iterator<NODEPTR> nodescan, nodescan1;
  List_iterator<EDGEPTR> edgescan, edgescan1;

  assert(flowgraph.numnodes() > 0 && flowgraph.numedges() > 0);

  for(i = 0; i < flowgraph.numnodes(); i++) {
    node = flowgraph.get_nthnode(i);
    assert(node && node->get_address() == i);
    assert(info.nodebirths[node->get_address()] >= 0 &&
	   info.nodedeaths[node->get_address()] > info.nodebirths[node->get_address()]);
    fu = info.functional_units[node->get_address()];
    assert(fu);
    assert(dp.get_functional_units().find(fu));
    assert(fu->get_operations().find(node));
    assert(info.moduletypes[node->get_address()] == fu->get_libelement()->get_address());
  }

  for(i = 0; i < flowgraph.numedges(); i++) {
    edge = flowgraph.get_nthedge(i);
    assert(edge && edge->get_address() == i);
    assert(info.edgebirths[edge->get_address()] >= 0 &&
	   info.edgedeaths[edge->get_address()] > info.edgebirths[edge->get_address()]);
    su = info.storage_units[edge->get_address()];
    assert(su);
    assert(dp.get_storage_units().find(su));
    assert(su->get_variables().find(edge));
  }

  cur_clock = (int)(dp.get_sample_period()/(float)dp.get_csteps());
  total = 0;
  FOR_EACH_LISTNODE(dp.get_functional_units(), fuscan) {
    fu = fuscan.get_item();
    assert(fu);
    libel = fu->get_libelement();
    assert(libel);
    stage_csteps = libel->get_stage_csteps(dp.get_vdd(), cur_clock);

    total += fu->get_operations().get_size();
    i = 0;
    FOR_EACH_LISTNODE(fu->get_operations(), nodescan) {
      node = nodescan.get_item();
      assert(node);
      j = 0;
      FOR_EACH_LISTNODE(fu->get_operations(), nodescan1) {
	node1 = nodescan1.get_item();
	assert(node1);
	if(i == j) {
	  continue;
	} else {
	  assert( (info.nodebirths[node->get_address()] >=
		   info.nodebirths[node1->get_address()] + stage_csteps) ||
		  (info.nodebirths[node1->get_address()] >=
		   info.nodebirths[node->get_address()] + stage_csteps) );
	}
	j++;
      }
      i++;
    }
  }
  assert(total == flowgraph.numnodes());

  total = 0;
  FOR_EACH_LISTNODE(dp.get_storage_units(), suscan) {
    su = (STORPTR) suscan.get_item();
    assert(su);
    total += su->get_variables().get_size();
    i = 0;
    FOR_EACH_LISTNODE(su->get_variables(), edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      j = 0;
      FOR_EACH_LISTNODE(su->get_variables(), edgescan1) {
	edge1 = edgescan1.get_item();
	assert(edge1);
	if(i == j) {
	  continue;
	} else {
	  assert(info.edgebirths[edge->get_address()] >= info.edgedeaths[edge1->get_address()] ||
		 info.edgebirths[edge1->get_address()] >= info.edgedeaths[edge->get_address()]);
	}
	j++;
      }
      i++;
    }
  }
  assert(total == flowgraph.numedges());

  return;
}
/********************************************************************/
void Scheduler::write_genesis_dfg(Dfg &flowgraph, Datapath &dp, char *outfilename)
{
  EDGEPTR edge, loopin_edge;
  FUPTR fu;
  STORPTR reg;
  List_iterator<FUPTR> fuscan;
  List_iterator<Storage_unit *> regscan;
  NODEPTR node;
  List_iterator<NODEPTR> nodescan;
  List_iterator<EDGEPTR> edgescan; 
  List_ar<Operator> *funclist;
  int i;
  assert(outfilename);
  ofstream output(outfilename, ios::out);

  if(!output) {
    cerr << "ERROR: Could not open File " << outfilename << " for output" << endl;
    exit(-1);
  }
  output << "#DFG input file for Genesis generated by SCALP" << endl << endl;
  output << ".n " << flowgraph.get_bitwidth() << endl;
  output << endl;

  output << ".cycle " << flowgraph.get_numcsteps() << endl;
  output << endl;

  output << "#--------primary input declarations---------#" << endl;
  FOR_EACH_LISTNODE(flowgraph.inputs, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    output << ".i " << edge->get_name() << " ; " << edge->get_birth() << endl;
  }

  output << "#--------primary output declarations---------#" << endl;
  FOR_EACH_LISTNODE(flowgraph.outputs, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    output << ".o " << edge->get_name() << " ; " << edge->get_death() << endl;
  }

  output << "#----------------------delays----------------#" << endl;
  FOR_EACH_LISTNODE(flowgraph.loopouts, edgescan) {
    edge = edgescan.get_item();
    loopin_edge = edge->get_loopin_link();
    assert(loopin_edge);
    assert(loopin_edge->is_loopin());
    output << ".delay " << edge->get_name() << " " << loopin_edge->get_name() << " 1 ; "
      << loopin_edge->get_value() << endl;
  }

  output << "#------------------constants-----------------#" << endl;
  FOR_EACH_LISTNODE(flowgraph.constants, edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    assert(edge->is_constant());
    output << ".constant " << edge->get_name() << " ; " << edge->get_value() << endl;
  }

  output << "#----------------operations------------------#" << endl;
  for(i = 0; i < flowgraph.numnodes(); i++) {
    node = flowgraph.get_nthnode(i);
    assert(node);
    output << ".m";
    display_genesis_operator(output, node->get_func());
    output << " " << node->get_name() << " ;";
    FOR_EACH_FANIN_EDGE(node, edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      output << " " << edge->get_name();
    }
    edge = node->get_output_edge();
    assert(edge);
    output << " " << edge->get_name();
    output << " ; " << node->get_birth() << " " ;
    //Check for multicycled or pipelined operations
    if(node->get_death() > node->get_birth() + 1) {
      assert(node->get_functional_unit() && node->get_functional_unit()->get_libelement());
      if(node->get_functional_unit()->get_libelement()->get_numstages() > 1) {
	//structural pipelining
	output << "ERROR: STRUCTURAL PIPELINING IS NOT SUPPORTED IN GENESIS DFG OUTPUT" << endl;
      } else {
	//multicycled operation
	output << "MULTICYCLE " << (node->get_death() - node->get_birth());
      }
    }
    output << endl;
  }

  output << "#-----------------functional unit sharing info--------------" << endl;
  FOR_EACH_LISTNODE(dp.get_functional_units(), fuscan) {
    fu = fuscan.get_item();
    assert(fu);
    //print out a comment with the libelement name
    output << " # " << fu->get_libelement()->get_name() << endl;

    output << ".= ";
    //ALUS NOT ALLOWED SINCE GENESIS CANNOT HANDLE THEM IN A GENERAL MANNER
    assert(fu->get_libelement());
    funclist = fu->get_libelement()->get_functions();
    assert(funclist);
    assert_force(funclist->get_size() == 1);
    display_genesis_operator(output, (*funclist)[0]);
    delete funclist;
    funclist = NULL;
    output  << " ;";
    FOR_EACH_LISTNODE(fu->get_operations(), nodescan) {
      node = nodescan.get_item();
      assert(node);
      output << " " << node->get_name();
    }
    output << endl;
  }
//lzhong 012902 Originally commented out. But I need it.:-)
  output << "-----------------register sharing info-------------------" << endl;
  FOR_EACH_LISTNODE(dp.get_storage_units(), regscan) {
    reg = (STORPTR) regscan.get_item();
    assert(!strcmp(reg->get_id(), "Register"));
    assert(reg);
    output << ".= r ;";
    FOR_EACH_LISTNODE(reg->get_variables(), edgescan) {
      edge = edgescan.get_item();
      assert(edge);
      output << " " << edge->get_name();
    }
    output << endl;
  }

  return;
}
/********************************************************************/
void Scheduler::display_genesis_operator(ostream &output, Operator op)
{
  switch(op) {
  case PLUS_:
    output << "+";
    break;
  case MINUS_:
    output << "-";
    break;
  case MULT_:
    output << "*";
    break;
  case NOTEQ_:
    output << "-";
    break;
  case OR_:
    output << "|";
    break;
  case AND_:
    output << "&";
    break;
  case DEV_:
    output << "/";
  default:
    output << "?";
    cerr << "ERROR: Unexpected operator " << op << endl;
    //assert_force(0);
    break;
  }

  return;
}
/********************************************************************/

