/********************************************************************/
/*          FILE: scheduler.h                                       */
/********************************************************************/
#ifndef ANAND_SCHEDULER_H
#define ANAND_SCHEDULER_H

#include "parameter.h"
#include "basic.h"
#include "dfg.h"
#include "library_patch.h"
#include "datapath.h"
#include "scm.h"
#include "fsm.h"
#include "retpipe.h"
#ifdef PHYSICAL
#include "i_macro.h"
#endif

#define MAX_CHAINING_FACTOR 1
#define VDD_GRANULARITY 0.1
#define MAX_NUM_PASSES 100
//temp 07182002
#define MAX_NUM_MOVES_PER_PASS 10 
//#define MAX_NUM_MOVES_PER_PASS 10

/*forward declarations*/
struct Class_a_move;
struct Class_b_fu_move;
struct Class_b_reg_move;
struct Precedence_constraint;
typedef struct Precedence_constraint PRECON;
typedef struct Precedence_constraint *PRECONPTR;
typedef enum objective_type {
  AREA = 0,
  POWER = 1,
} objective;
/*************************************************************************/
struct interval
  {
    int left;
    int right;
    interval(int l, int r)
    {
      left = l;
      right = r;
    };
    interval()
    {
      left = right = 0;
    };
    void set_right(int r)
    {
      right = r;
    };
    void set_left(int l)
    {
      left = l;
    };
    Boolean intersect(interval I)
    {
      if (I.left == left)
	return T;
      if (I.left < left)
	{
	  if (I.right > left)
	    return T;
	  else
	    return F;
	}
      else
	{
	  if (right > I.left)
	    return T;
	  else
	    return F;
	}
    };
};
/********************************************************************/
struct Precedence_constraint {
  List_ar<NODEPTR> predecessors;
  List_ar<NODEPTR> successors;

  Precedence_constraint(const NODEPTR newpred = NULL, const NODEPTR newsucc = NULL) : predecessors(), successors() {
    if(newpred) predecessors.append(newpred);
    if(newsucc) successors.append(newsucc);
  }
  ~Precedence_constraint() {}

  //Add a NODEPTR to the predecessors list
  void add_predecessor(const NODEPTR pred) {
    assert(pred && !predecessors.find(pred));
    predecessors.append(pred);
  }

  //Add a NODEPTR to the successors list
  void add_successor(const NODEPTR succ) {
    assert(succ && !successors.find(succ));
    successors.append(succ);
  }

  void operator = (const Precedence_constraint &source) {
    predecessors.copy(source.predecessors);
    successors.copy(source.successors);
  }

  void clear() {
    predecessors.clear();
    successors.clear();
  }
};
inline ostream &operator <<(ostream &output, PRECON &precedence_constraint) {
  output << "Pred.: " << precedence_constraint.predecessors << ", Succ.: "
    << precedence_constraint.successors;
  return output;
}
/********************************************************************/
#ifdef _SCALP_
class Scheduler : public Assurance {
#else
class Scheduler {
#endif
#ifdef PHYSICAL
	public:
		virtual void set_scm( Scm *scms)
		{ scm = scms; return; }
	private:
		Scm *scm;	
	protected:
#else		
private:
#endif		

  /*************************************************************/
  /*Routines used solely by other Scheduler:: member functions */
  /*************************************************************/

  //Compute the lifetime of an edge during initial ASAP scheduling
  void compute_edge_lifetime(EDGEPTR, Dfg &, library *, float, float);

  //Compute the lifetime of an edge during precedence constrained scheduling
  void compute_edge_lifetime(EDGEPTR , Dfg &, Schalloc_info &, Array<int> &, Array<int> &, Array<int> &, Array<int> &, library *, float, float);

  //perform an As-Soon-As-Possible schedule of the DFG
  int asap_schedule(Dfg &, library *, float, float);

  //Compute the lifetime of a node during precedence constrained scheduling
  void precedence_constrained_schedule_process_node(Schalloc_info &, Array<int> &, Array<int> &, library *, float, float, NODEPTR, Array<PRECON> &, Array<PRECON> &);

  //precedence-constrained scheduling for re-scheduling
  int precedence_constrained_schedule(Dfg &, library *, float, float, Schalloc_info &, Boolean, Array<PRECON> &, Array<PRECON> &);

  //perform an As-Late-As-Possible schedule of the DFG
  //NOT IMPLEMENTED YET
  void alap_schedule(register Dfg &, library *, float, float);

  //perform an initial module type selection for each dfgnode
  void fastest_map(Dfg &, library *);

  //perform a one-to-one allocation to create the Datapath
  void initial_allocation(Dfg &, Datapath *, library *);

  //Generate an upper bound on the clock period for a given Vdd
  //Dfg and library
  const int compute_min_csteps(Dfg &,library *,const float,const float,const int);

  //Generate a lower bound on the clock period for a given Vdd
  //Dfg and library
  const int compute_max_csteps(Dfg &, library *, const float, const float);

  //compute the fastest possible sampling period for an implementation
  //of the given DFG using the given library
  float compute_min_sample_period(Dfg &, library *);

  //compute the vdd scaling possible for a given delay laxity
  float compute_vdd(const float, const float, const float);

  //check if a candidate clock period can be pruned
  Boolean clk_can_be_pruned(float, float, float, Dfg &, library *);

  //check if a candidate supply voltage can be pruned

  //perform iterative improvement from a given starting point
  virtual void iterative_improvement(Dfg &, library *, Datapath * &, const float, const float, const objective &, Scm * = (Scm *)NULL);
#ifdef PHYSICAL  
  virtual void iterative_reg_improvement(Dfg &, library *, Datapath * &, const float, const float, const objective &, Scm * = (Scm *)NULL);
#endif
  //evaluate the switched capacitance cost function
  virtual float compute_sccost(Datapath *, library *, Scm *);

  //evaluate the area cost function
  virtual float compute_areacost(Dfg &, Datapath *, library *);

  //generate the next move
  virtual Boolean generate_move(Dfg &, library *, Datapath *, Schalloc_info &, 
			Array<Boolean> &, Array<Boolean> &, Scm *, const objective);
#ifdef PHYSICAL  
  virtual Boolean generate_reg_move(Dfg &, library *, Datapath *, Schalloc_info &,
                        Array<Boolean> &, Array<Boolean> &, Scm *, const objective);
#endif

  //find the best class A move
  Boolean find_best_class_a_move(Dfg &, library *, Datapath *, Schalloc_info &,
				 Array<Boolean> &, Scm *, const objective,
				 Class_a_move &, float &);

  //find the best class B fu move
  virtual Boolean find_best_class_b_fu_move(Dfg &, library *, Datapath *, Schalloc_info &,
				 Array<Boolean> &, Scm *, const objective,
				 Class_b_fu_move &, float &);

  //find the best class B reg move
  virtual Boolean find_best_class_b_reg_move(Dfg &, library *, Datapath *, Schalloc_info &,
				 Array<Boolean> &, Scm *, const objective,
				 Class_b_reg_move &, float &);

  //compute the gain in switched capacitance due to a class A move
  float compute_scgain(Datapath *, Class_a_move &, Scm *);

  //compute the gain in switched capacitance due to a class B fu move
  virtual float compute_scgain(Datapath *, Class_b_fu_move &, Scm *);
#ifdef _OLD_BUG_  
  virtual float compute_scgain(Datapath *, Class_b_reg_move &, Scm *);
#endif  

  //compute the area gain due to a class A move
  virtual float compute_areagain(Datapath *, library *, Class_a_move &);

  //compute the gain for a class B fu move
  virtual float compute_areagain(Datapath *, library *, Class_b_fu_move &);

  //compute the gain for a class B reg move
  virtual float compute_areagain(Dfg &, Datapath *, library *, Class_b_reg_move &);

  //generate precedence constraints for a list of nodes that must share a resource
  void generate_node_precedence_constraints(const List_ar<NODEPTR> &, Schalloc_info &, Array<PRECON> &);

  //generate precedence constraints for a list of edges that must share a resource
  void generate_edge_precedence_constraints(const List_ar<EDGEPTR> &, Schalloc_info &, Array<PRECON> &);

  //generate precedence constraints for rescheduling for a Class_a_move
  void generate_precedence_constraints(Dfg &, Datapath *, Schalloc_info &, Class_a_move &, Array<PRECON> &, Array<PRECON> &);

  //generate precedence constraints for rescheduling for a Class_b_fu_move
  void generate_precedence_constraints(Dfg &, Datapath *, Schalloc_info &, Class_b_fu_move &, Array<PRECON> &, Array<PRECON> &);

  //generate precedence constraints for rescheduling for a Class_b_reg_move
  void generate_precedence_constraints(Dfg &, Datapath *, Schalloc_info &, Class_b_reg_move &, Array<PRECON> &, Array<PRECON> &);

  //reschedule to enable a class A move
  Boolean reschedule(Dfg &, library *, Datapath *, Schalloc_info &, Class_a_move &, Boolean);

  //reschedule to enable a class B fu move
  Boolean reschedule(Dfg &, library *, Datapath *, Schalloc_info &, Class_b_fu_move &, Boolean);

  //reschedule to enable a class B reg move 
  Boolean reschedule(Dfg &, library *, Datapath *, Schalloc_info &, Class_b_reg_move &, Boolean);

  //Checks the condition that when two registers are merged, the primary 
  //inputs occur together at the beginnning of the list and the POs at the
  //end, where the order is determined by the birth-time.
  Boolean valid_register_merge(Class_b_reg_move &, Schalloc_info &);

  //change the datapath to reflect a Class_a_move
  void implement_move(Class_a_move &, Datapath *, Dfg &, Schalloc_info &, Array<Boolean> &);

  //change the datapath to reflect a Class_b_fu_move
  void implement_move(Class_b_fu_move &, Datapath *, Dfg &, library *, Schalloc_info &, Array<Boolean> &);

  //change the datapath to reflect a Class_b_reg_move
  void implement_move(const Class_b_reg_move &, Datapath *, Dfg &, library *, Schalloc_info &, Array<Boolean> &);

  //perform some basic validity checks on the DFG and Datapath
  void check_dfg_and_datapath(Dfg &, Datapath &);
  void check_dfg_and_datapath(Dfg &, Schalloc_info &, Datapath &);

  //write out a DFG file with scheduling and allocation information in genesis format
  void write_genesis_dfg(Dfg &, Datapath &, char *);

  //display an operator
  void display_genesis_operator(ostream &, Operator);

  //create a copy of the dfg and datapath
  void copy_flowgraph_and_dp(Dfg &, Datapath &, Dfg &, Datapath &, library *);

  //returns true if there is a directed path between two edges e1, e2
  Boolean DFS_edge(EDGEPTR e1, EDGEPTR e2);

  //returns true if the variable mapped to two registers do not have overlapping
  //life times
  Boolean can_share_registers(const List_ar<EDGEPTR> &edgelist1, const List_ar<EDGEPTR> &edgelist2);

  //Does register sharing on the scheduled DFG(Does not change the schedule)
  void reg_share(Dfg& flowgraph, Datapath *cur_dp, Schalloc_info& cur_dfg_info, library *lib);
  
public:
  Scheduler() {
    return;
  }

  ~Scheduler() {
    return;
  }

  void display(ostream & = cout);

  //function that performs low power scheduling
  virtual Datapath *lpschedule(Dfg &, library *, const float, const float, Scm *, const Boolean, char *, const int);

  //function that performs area targetted scheduling
  virtual Datapath *areaschedule(Dfg &, library *, const float, const float, const int);

  FSM* create_fsm(Datapath *);

};
/********************************************************************/
struct Class_a_move
#ifdef _SCALP_
: public Assurance
#endif
{
  FUPTR fu;
  FUNC_LIBELPTR old_libelement;
  FUNC_LIBELPTR new_libelement;
#ifdef PHYSICAL
  float gain;
#endif  
  Class_a_move() {
    FRITS_SET_CLASS("Class_a_move");
    fu = NULL;
    old_libelement = NULL;
    new_libelement = NULL;
    return;
  }
  ~Class_a_move() {
    return;
  }

  inline void copy(Class_a_move &source) {
    fu = source.fu;
#ifdef PHYSICAL
    gain = source.gain;
#endif    
    old_libelement = source.old_libelement;
    new_libelement = source.new_libelement;
    return;
  }
  inline void set_fields(FUPTR new_fu, const FUNC_LIBELPTR old_libel, const FUNC_LIBELPTR new_libel) {
    FRITS_SET_MESSAGE("set_fields");
    assert(new_fu && old_libel && new_libel && old_libel != new_libel);
    fu = new_fu;
    old_libelement = old_libel;
    new_libelement = new_libel;
#ifdef PHYSICAL
	gain =0;
#endif    	
    return;
  }

  inline void display(ostream &output = cout) {
    output << "Class_a_move: " << (void *)this << endl;
    output << "  fu: " << fu << endl;
    output << "  old_libelement: " << old_libelement << endl;
    output << "  new_libelement: " << new_libelement << endl;
#ifdef PHYSICAL
	output << " gain: "<<gain<<endl;
#endif    	
    output << endl;
    
    return;
  }
};
/********************************************************************/
struct Class_b_fu_move {
#ifdef _SCALP_
: public Assurance
#endif
  Boolean splitting;             /*Whether this is a splitting or sharing move*/
  FUPTR fu1;
  FUPTR fu2;
#ifdef PHYSICAL
	float gain;
#endif      	
  List_ar<NODEPTR> split_operations;  /*List of operations split off for a splitting move*/

  Class_b_fu_move() : split_operations() {
    splitting = F;
    fu1 = NULL;
    fu2 = NULL;
    return;
  }
  ~Class_b_fu_move() {
    return;
  }

  inline void copy(Class_b_fu_move &source) {
    fu1 = source.fu1;
    fu2 = source.fu2;
    split_operations.copy(source.split_operations);
#ifdef PHYSICAL
  	gain = source.gain;
#endif      	
    return;
  }

  inline void display(ostream &output = cout) {
    output << "Class_b_fu_move: " << ((void *)this) << endl;
    output << "  fu1: " << fu1 << endl;
    output << "  fu1: " << fu2 << endl;
    output << "  split_operations: " << split_operations << endl;
#ifdef PHYSICAL
    output << " gain : "<<gain<<endl;
#endif    
    output << endl;
    return;
  }
};
/********************************************************************/
struct Class_b_reg_move {
  /*Whether this is a splitting or sharing move*/
  /*0 => sharing, 1=> splitting*/
  Boolean splitting;
  STORPTR su1;
  STORPTR su2;
  List_ar<EDGEPTR> split_variables;  /*List of variables split off for a splitting move*/
#ifdef PHYSICAL
  float gain;
#endif  
  Class_b_reg_move() : split_variables() {
    splitting = F;
    su1 = NULL;
    su2 = NULL;
    return;
  }
  ~Class_b_reg_move() {
    return;
  }

  inline void copy(Class_b_reg_move &source) {
    splitting = source.splitting;
    su1 = source.su1;
    su2 = source.su2;
    split_variables.copy(source.split_variables);
#ifdef PHYSICAL
	gain = source.gain;
#endif    	
  }

  inline void display(ostream &output = cout) {
    output << "Class_b_reg_move: " << ((void *)this) << endl;
    output << "  su1: " << su1 << endl;
    output << "  su1: " << su2 << endl;
    output << "  split_variables: " << split_variables << endl;
#ifdef PHYSICAL
	output << "gain: "<<gain<<endl;
#endif    	
    output << endl;
    return;
  }
};
/********************************************************************/
/********************************************************************/
#endif
