/*
 * FILE: graph.c
 */

#include "graphsr.h"

/* declare up front all functions in this file which could be static */
extern void searchc();
extern int give_name_to_scc();

GRAPHPTR gr_open(first, last)
int first, last;
{
  GRAPHPTR graph;

  assert(last - first >= 0);

  graph = (GRAPHPTR)SR_MALLOC(sizeof(GRAPH));
  graph->no_vertices = last - first + 1;
  graph->first = first;
  graph->last = last;

  graph->vlist = (LISTPTR *)SR_MALLOC(gr_nv(graph)*sizeof(LISTPTR));
  graph->vlist = &graph->vlist[-first];
  l_init_listarray(graph->vlist, first, last);

  graph->active = (BOOLEAN *)SR_MALLOC(gr_nv(graph)*sizeof(BOOLEAN));
  graph->active = &graph->active[-first];
  sr_init_boolarray(graph->active, first, last);

  graph->scc = (LISTPTR *)NULL;
  graph->v2scc = (int *)NULL;
  graph->scc_valid = 0;

  return(graph);
}

GRAPHPTR gr_close(graph)
GRAPHPTR graph;
{
  int i;

  if (graph) {
    /* free graph->vlist */
    for (i = gr_first(graph); i <= gr_last(graph); i++)
      if (graph->vlist[i])
        graph->vlist[i] = l_free(graph->vlist[i]);
    graph->vlist = (LISTPTR *)sr_free((char *)&graph->vlist[gr_first(graph)]);
    graph->active = (BOOLEAN *)sr_free((char *)&graph->active[gr_first(graph)]);

    /* free graph->scc */
    if (graph->scc) {
      for (i = gr_first(graph); i <= gr_last(graph); i++)
        if (graph->scc[i]) graph->scc[i] = l_free(graph->scc[i]);
      graph->scc = (LISTPTR *)sr_free((char *)&graph->scc[gr_first(graph)]);
    }

    /* free graph->v2scc */
    if (graph->v2scc)
      graph->v2scc = (int *)sr_free((char *)&graph->v2scc[gr_first(graph)]);

    /* miscellaneous */
    graph->no_vertices = graph->first = graph->last = graph->scc_valid = 0;

    /* free the graph itself */
    graph = (GRAPHPTR)sr_free((char *)graph);
  }

  return(GRAPHNULL);
}

int gr_add_arc(graph, v1, v2) /* Return 1 if an arc was really added else 0 */
GRAPHPTR graph;
int v1, v2;
{
  int retval = 0;

  assert(graph);
  assert(v1 >= gr_first(graph) && v1 <= gr_last(graph));
  assert(v2 >= gr_first(graph) && v2 <= gr_last(graph));

  if (!l_find(graph->vlist[v1], v2)) {
    graph->scc_valid = 0;
#ifndef NDEBUG
    /* REMOVE THIS DEBUG STATEMENT */
    if (v1 == 1134 && v2 == -1153)
      fflush(stdout);
#endif
    if (v1 == -(v2)) l_insert(graph->vlist[v1], v2);
    else l_append(graph->vlist[v1], v2);
    retval = 1;
  }

  return(retval);
}

int gr_delete_arc(graph, v1, v2) /* Return 1 if the arc was really deleted. */
GRAPHPTR graph;
int v1, v2;
{
  int retval = 0;
  LISTPTR found;

  assert(graph);
  assert(v1 >= gr_first(graph) && v1 <= gr_last(graph));
  assert(v2 >= gr_first(graph) && v2 <= gr_last(graph));

  found = l_find(graph->vlist[v1], v2);
  if (found) {
    l_delete(&graph->vlist[v1], found);
    graph->scc_valid = 0;
    retval = 1;
  }

  return(retval);
}

void gr_print(graph, filename)
GRAPHPTR graph;
char *filename;
{
  int i;
  FILE *fp = strcmp(filename, "stdout") ? sr_fopen(filename, "w") : stdout;

  if (!graph) {
    fprintf(fp, "Adjacency List: Graph is NULL!\n");
  } else {
    fprintf(fp, "Adjacency List:\n");
    for (i = gr_first(graph); i <= gr_last(graph); i++) {
      fprintf(fp, "%3d: ", i);
      if (graph->vlist[i]) l_print(graph->vlist[i], fp);
      else fprintf(fp, "\n");
    }
    fprintf(fp, "\n");

    if (strcmp(filename, "stdout")) { fclose(fp); fp = (FILE *)NULL; }
  }

  return;
}

void gr_print_scc(graph, filename)
GRAPHPTR graph;
char *filename;
{
  int i;
  FILE *fp = strcmp(filename, "stdout") ? sr_fopen(filename, "w") : stdout;
  LISTPTR sortedlist;

  if (!graph) {
    fprintf(fp, "Strongly Connected Components: Graph is NULL!\n");
  } else {
    if (!graph->scc) {
      fprintf(fp, "Strongly Connected Components: Graph->scc is NULL!\n");
    } else {
      fprintf(fp, "Strongly Connected Components:\n");
      for (i = gr_first(graph); i <= gr_last(graph); i++) {
        if (graph->scc[i]) {
          fprintf(fp, "Component %d: ", i);
          sortedlist = l_copy(graph->scc[i]);
          l_sort(sortedlist, (int (*)())NULL);
          l_print(sortedlist, fp);
          sortedlist = l_free(sortedlist);
        }
      }
    }
  }

  if (strcmp(filename, "stdout")) { fclose(fp); fp = (FILE *)NULL; }

  return;
}

/* Find the strongly connected components of the implication graph. */
void gr_compute_scc(graph)
GRAPHPTR graph;
{
  int v;
  LISTPTR l;
  int *visited, *dfs_no, *lowlink;
  STACKAPTR dfs_stack;
  int dfs_number = 0;

  assert(gr_scc_not_valid(graph));

  gr_init_scc(graph);

  /* allocate temporary memory */
  visited = (int *)SR_MALLOC(gr_nv(graph)*sizeof(int));
  visited = &visited[-gr_first(graph)];

  dfs_no = (int *)SR_MALLOC(gr_nv(graph)*sizeof(int));
  dfs_no = &dfs_no[-gr_first(graph)];

  lowlink = (int *)SR_MALLOC(gr_nv(graph)*sizeof(int));
  lowlink = &lowlink[-gr_first(graph)];

  /* initialize */
  sr_init_intarray(visited, gr_first(graph), gr_last(graph));
  sr_init_intarray(lowlink, gr_first(graph), gr_last(graph));
  sr_init_intarray(dfs_no, gr_first(graph), gr_last(graph));

  dfs_stack = sta_open(gr_nv(graph));

  /* do it */
  for (v = gr_first(graph); v <= gr_last(graph); v++)
    if (!visited[v])
      searchc(graph, v, visited, dfs_no, lowlink, dfs_stack, &dfs_number);

  /* free temporary memory */
  dfs_stack = sta_close(dfs_stack);
  visited = (int *)sr_free((char *)&visited[gr_first(graph)]);
  dfs_no = (int *)sr_free((char *)&dfs_no[gr_first(graph)]);
  lowlink = (int *)sr_free((char *)&lowlink[gr_first(graph)]);

  /* find each vertex's connected component and put it in v2scc */
  for (v = gr_first(graph); v <= gr_last(graph); v++) {
    if (graph->scc[v]) {
      /* v is a scc in graph */
      for (l = graph->scc[v]; l != LISTNULL; l = l->next) {
        assert(!graph->v2scc[l->o]);
        graph->v2scc[l->o] = v;
      }
    }
  }

  graph->scc_valid = 1;

  return;
}

void searchc(graph, v, visited, dfs_no, lowlink, dfs_stack, dfs_number)
GRAPHPTR graph;
register int v;
int *visited, *dfs_no, *lowlink;
STACKAPTR dfs_stack;
int *dfs_number;
{
  register int w;
  register int x;
  register LISTPTR l;

  /* mark v as visited */
  visited[v] = 1;
  dfs_no[v] = ++*dfs_number;
  lowlink[v] = dfs_no[v];
  sta_push(dfs_stack, v);

  for (l = graph->vlist[v]; l != LISTNULL; l = l->next) {
    w = l->o;
    if (!visited[w]) {
      searchc(graph, w, visited, dfs_no, lowlink, dfs_stack, dfs_number);
      lowlink[v] = lowlink[v] < lowlink[w] ? lowlink[v] : lowlink[w];
    } else if (dfs_no[w] < dfs_no[v] && sta_in(dfs_stack, w)) {
      lowlink[v] = dfs_no[w] < lowlink[v] ? dfs_no[w] : lowlink[v];
    }
  }

  if (lowlink[v] == dfs_no[v]) {
    register LISTPTR scc_list = LISTNULL;

    do {
      x = sta_pop(dfs_stack);
      l_insert(scc_list, x);
    } while (x != v);

    /* end of strongly connected component */
    graph->scc[give_name_to_scc(scc_list)] = scc_list;
  }

  return;
}

int give_name_to_scc(scclist)
LISTPTR scclist;
{
  register int smallest;
  register int sign;
  register LISTPTR l;

  assert(scclist);

  smallest = ABS(scclist->o);
  sign = scclist->o;

  for (l = scclist->next; l != LISTNULL; l = l->next) {
    if (ABS(l->o) < smallest) {
      smallest = ABS(l->o);
      sign = l->o;
    }
  }

  return(sign);
}

/*
 * build_dag() returns the dag of the strongly connected components
 * of the input argument 'graph'.  It assumes the the strongly connected
 * components of 'graph' have been built already.
 */
GRAPHPTR gr_build_dag(graph)
GRAPHPTR graph;
{
  register int i, j;
  register LISTPTR l0, l1;
  QUEUEPTR c_scc;
  GRAPHPTR dag;

  /* make sure that the graph and its scc are in sync. */
   assert(graph->scc_valid);

  /* construct the dag */
  dag = gr_open(gr_first(graph), gr_last(graph));
  c_scc = q_open(gr_first(graph), gr_last(graph));

  for (i = gr_first(graph); i <= gr_last(graph); i++) {
    if (gr_scc(graph, i)) {
      assert(q_empty(c_scc));
      assert(q_really_empty(c_scc));

      /* 'i' is the name of the current scc. Find the children scc's */
      for (l0 = gr_scc(graph, i); l0 != LISTNULL; l0 = l0->next) {
        for (l1 = gr_adj(graph, l0->o); l1 != LISTNULL; l1 = l1->next) {
          if (gr_active(graph,l1->o) && gr_v2scc(graph, l1->o) != i) {
            q_enq(c_scc, graph->v2scc[l1->o]);
          }
        }
      }

      /* c_scc consists of children scc's of scc i */
      while (q_not_empty(c_scc)) {
        j = q_deq(c_scc);
        if (!l_find(dag->vlist[i], j)) {
          dag->scc_valid = 0;
          if (i == -j) l_insert(dag->vlist[i], j);
          else l_append(dag->vlist[i], j);
        }
      }
    }
  }

  /* close children scc queue */
  c_scc = q_close(c_scc);

#ifndef NDEBUG
  if (gr_scc_not_valid(dag)) gr_compute_scc(dag);
  for (i = gr_first(dag); i <= gr_last(dag); i++) {
    if (l_len(gr_scc(dag, i)) > 1) {
      fprintf(stdout,
        "ERROR: in build_dag(), dag has cycle! scc = %d\n", i);
      fflush(stdout);
    }
    assert(l_len(gr_scc(dag, i)) <= 1);
  }
  gr_free_scc(dag);
#endif

  return(dag);
}

void gr_init_scc(graph)
GRAPHPTR graph;
{
  register int i;

  assert(graph);
  graph->scc_valid = 0;

  if (graph->scc) {
    /* free graph->scc */
    assert(graph->v2scc);
    for (i = gr_first(graph); i <= gr_last(graph); i++) {
      graph->v2scc[i] = 0;
      if (graph->scc[i]) graph->scc[i] = l_free(graph->scc[i]);
    }
  } else { /* very first time we compute the sccs for the graph. */
    /* allocate space and initialize */
    assert(!graph->v2scc);
    graph->scc = (LISTPTR *)SR_MALLOC(gr_nv(graph)*sizeof(LISTPTR));
    graph->scc = &graph->scc[-gr_first(graph)];
    l_init_listarray(graph->scc, gr_first(graph), gr_last(graph));

    graph->v2scc = (int *)SR_MALLOC(gr_nv(graph)*sizeof(int));
    graph->v2scc = &graph->v2scc[-gr_first(graph)];
    sr_init_intarray(graph->v2scc, gr_first(graph), gr_last(graph));
  }

  return;
}

void gr_free_scc(graph)
GRAPHPTR graph;
{
  graph->scc_valid = 0;

  if (graph->scc) {
    int i;
    assert(graph->v2scc);

    /* free graph->scc */
    for (i = gr_first(graph); i <= gr_last(graph); i++)
      if (graph->scc[i]) graph->scc[i] = l_free(graph->scc[i]);
    graph->scc = (LISTPTR *)sr_free((char *)&graph->scc[gr_first(graph)]);

    /* free graph->v2scc */
    graph->v2scc = (int *)sr_free((char *)&graph->v2scc[gr_first(graph)]);
  } else {
    assert(!graph->v2scc);
  }

  return;
}

/*
 * gr_topsort() returns a topologically sorted list
 * of the vertices in the input dag.
 */
LISTPTR gr_topsort(dag, dagroots)
GRAPHPTR dag;
LISTPTR dagroots;
{
  int *visited;
  LISTPTR  l, topsortlist = LISTNULL;
  void topsortvisit();

  assert(dag);

  /* allocate temporary memory */
  visited = (int *)SR_MALLOC(gr_nv(dag)*sizeof(int));
  visited = &visited[-gr_first(dag)];

  sr_init_intarray(visited, gr_first(dag), gr_last(dag));

  for (l = dagroots; l != LISTNULL; l = l->next)
    topsortvisit(dag, l->o, &topsortlist, visited);

  /* free temporary memory */
  visited = (int *)sr_free((char *)&visited[gr_first(dag)]);

  return(topsortlist);
}

/*
 * topsortvisit()
 *
 * Accepts a graph, a vertex v, a descendant list, and a visited array.  All
 * vertices reachable from v are placed in the list. The elements in the list
 * are in topological order.  Visited is an integer array with bounds
 * graph->first to graph->last initialized to all zeros.
 */
void topsortvisit(graph, v, deslist, visited)
GRAPHPTR graph;
int v;
LISTPTR *deslist;
int visited[];
{
  LISTPTR l;

  assert(!visited[v]);
  visited[v] = 1;

  for (l = graph->vlist[v]; l != LISTNULL; l = l->next)
    if (!visited[l->o]) topsortvisit(graph, l->o, deslist, visited);

  if (deslist) l_insert(*deslist, v);

  return;
}

/*
 * visit()
 *
 * Accepts a graph, a vertex v, a descendant list, and a visited array.  All
 * vertices reachable from v are placed in the list. The elements in the list
 * are in REVERSE topological order.  Visited is an integer array with bounds
 * graph->first to graph->last initialized to all zeros.
 */
void visit(graph, v, deslist, visited)
GRAPHPTR graph;
int v;
LISTPTR *deslist;
int visited[];
{
  LISTPTR l;

  assert(!visited[v]);
  visited[v] = 1;

  for (l = graph->vlist[v]; l != LISTNULL; l = l->next)
    if (!visited[l->o]) visit(graph, l->o, deslist, visited);

  if (deslist) l_append(*deslist, v);

  return;
}

GRAPHPTR gr_copy(graph)
GRAPHPTR graph;
{
  GRAPHPTR newgraph;
  int i;

  assert(graph);

  newgraph = gr_open(gr_first(graph), gr_last(graph));

  for (i = gr_first(newgraph); i <= gr_last(newgraph); i++)
    newgraph->vlist[i] = l_copy(graph->vlist[i]);

  newgraph->scc_valid = graph->scc_valid;

  if (graph->scc) {
    assert(graph->v2scc);
    newgraph->scc = (LISTPTR *)SR_MALLOC(gr_nv(newgraph)*sizeof(LISTPTR));
    newgraph->scc = &newgraph->scc[-gr_first(newgraph)];
    newgraph->v2scc = (int *)SR_MALLOC(gr_nv(newgraph)*sizeof(int));
    newgraph->v2scc = &newgraph->v2scc[-gr_first(newgraph)];
    for (i = gr_first(newgraph); i <= gr_last(newgraph); i++) {
      newgraph->scc[i] = l_copy(graph->scc[i]);
      newgraph->v2scc[i] = graph->v2scc[i];
    }
  } else {
    assert(!graph->v2scc);
  }

  return(newgraph);
}

void gr_expand(graph, first, last)
GRAPHPTR graph;
int first, last;
{
  int oldfirst, oldlast, origsize, newsize, i, *newv2scc;
  LISTPTR *newvlist, *newscc;

  assert(graph);
  assert(gr_nv(graph) > 0);
  assert(graph->vlist);

  oldfirst = gr_first(graph);
  oldlast = gr_last(graph);

  assert(first < last);
  assert(first < oldfirst || (!first && !oldfirst));
  assert(last > oldlast);

  origsize = oldlast - oldfirst + 1;
  newsize = last - first + 1;
  assert(newsize > origsize);

  graph->no_vertices = newsize;
  graph->first = first;
  graph->last = last;

  /* allocate new 'vlist', 'scc', and 'v2scc' arrays */
  newvlist = (LISTPTR *)SR_MALLOC(newsize*sizeof(LISTPTR));
  newvlist = &newvlist[-first];

  newscc = (LISTPTR *)NULL;
  if (graph->scc) {
    newscc = (LISTPTR *)SR_MALLOC(newsize*sizeof(LISTPTR));
    newscc = &newscc[-first];
  }

  newv2scc = (int *)NULL;
  if (graph->v2scc) {
    newv2scc = (int *)SR_MALLOC(newsize*sizeof(int));
    newv2scc = &newv2scc[-first];
  }

  /* initialize new 'vlist', 'scc', and 'v2scc' arrays */
  for (i = first; i < oldfirst; i++) {
    newvlist[i] = LISTNULL;
    if (graph->scc) newscc[i] = LISTNULL;
    if (graph->v2scc) newv2scc[i] = 0;
  }

  for (i = oldfirst; i <= oldlast; i++) {
    newvlist[i] = graph->vlist[i];
    if (graph->scc) newscc[i] = graph->scc[i];
    if (graph->v2scc) newv2scc[i] = graph->v2scc[i];
  }

  for (i = oldlast + 1; i <= last; i++) {
    newvlist[i] = LISTNULL;
    if (graph->scc) newscc[i] = LISTNULL;
    if (graph->v2scc) newv2scc[i] = 0;
  }

  /* transfer new 'vlist', 'scc', and 'v2scc' arrays to graph */
  sr_free((char *)&graph->vlist[oldfirst]);
  graph->vlist = newvlist; newvlist = (LISTPTR *)NULL;

  if (graph->scc) {
    sr_free((char *)&graph->scc[oldfirst]);
    graph->scc = newscc; newscc = (LISTPTR *)NULL;
  }

  if (graph->v2scc) {
    sr_free((char *)&graph->v2scc[oldfirst]);
    graph->v2scc = newv2scc; newv2scc = (int *)NULL;
  }

  return;
}

void gr_unexpand(graph, first, last)
GRAPHPTR graph;
int first, last;
{
  int oldfirst, oldlast, origsize, newsize, i, *newv2scc;
  LISTPTR *newvlist, *newscc;

  assert(graph);
  assert(gr_nv(graph) > 0);
  assert(graph->vlist);

  oldfirst = gr_first(graph);
  oldlast = gr_last(graph);

  assert(first < last);
  assert(oldfirst < first || (!first && !oldfirst));
  assert(oldlast > last);

  origsize = oldlast - oldfirst + 1;
  newsize = last - first + 1;
  assert(origsize > newsize);

  graph->no_vertices = newsize;
  graph->first = first;
  graph->last = last;

  /* allocate new 'vlist', 'scc', and 'v2scc' arrays */
  newvlist = (LISTPTR *)SR_MALLOC(newsize*sizeof(LISTPTR));
  newvlist = &newvlist[-first];

  newscc = (LISTPTR *)NULL;
  if (graph->scc) {
    newscc = (LISTPTR *)SR_MALLOC(newsize*sizeof(LISTPTR));
    newscc = &newscc[-first];
  }

  newv2scc = (int *)NULL;
  if (graph->v2scc) {
    newv2scc = (int *)SR_MALLOC(newsize*sizeof(int));
    newv2scc = &newv2scc[-first];
  }

  /* initialize new 'vlist', 'scc', and 'v2scc' arrays */
  for (i = oldfirst; i < first; i++) {
    assert(!graph->vlist[i]);
    if (graph->scc) {
      graph->scc[i] = l_free(graph->scc[i]);
    }
  }

  for (i = first; i <= last; i++) {
    newvlist[i] = graph->vlist[i];
    if (graph->scc) newscc[i] = graph->scc[i];
    if (graph->v2scc) newv2scc[i] = graph->v2scc[i];
  }

  for (i = last + 1; i <= oldlast; i++) {
    assert(!graph->vlist[i]);
    if (graph->scc) {
      graph->scc[i] = l_free(graph->scc[i]);
    }
  }

  /* transfer new 'vlist', 'scc', and 'v2scc' arrays to graph */
  sr_free((char *)&graph->vlist[oldfirst]);
  graph->vlist = newvlist; newvlist = (LISTPTR *)NULL;

  if (graph->scc) {
    sr_free((char *)&graph->scc[oldfirst]);
    graph->scc = newscc; newscc = (LISTPTR *)NULL;
  }

  if (graph->v2scc) {
    sr_free((char *)&graph->v2scc[oldfirst]);
    graph->v2scc = newv2scc; newv2scc = (int *)NULL;
  }

  return;
}

void gr_size(graph)
GRAPHPTR graph;
{
  int i;
  int ecount = 0, vcount = 0;
  LISTPTR l;

  assert(graph);

  for (i = gr_first(graph); i <= gr_last(graph); i++) {
    if (gr_adj(graph, i)) {
      vcount++;
      for (l = gr_adj(graph, i); l != LISTNULL; l = l->next)
        if (gr_active(graph,l->o)) ecount++;
    }
  }

  fprintf(stdout, "* Graph size: vertices in use = %d; edges = %d;\n",
          vcount, ecount);

  return;
}

int gr_identical(g1, g2)
GRAPHPTR g1, g2;
{
  register int i;

  assert(gr_last(g1) == -gr_first(g1));
  assert(gr_nv(g1) == gr_last(g1) - gr_first(g1) + 1);
  assert(gr_last(g2) == -gr_first(g2));
  assert(gr_nv(g2) == gr_last(g2) - gr_first(g2) + 1);

  if (gr_nv(g2) > gr_nv(g1)) {
    for (i = gr_first(g2); i < gr_first(g1); i++) {
      assert(i);
      assert(!gr_adj(g2, i));
    }
    for (i = gr_first(g1); i <= gr_last(g1); i++) {
      if (!i) continue;
      assert(l_cmp(gr_adj(g1, i), gr_adj(g2, i)));
    }
    for (i = gr_last(g1) + 1; i <= gr_last(g2); i++) {
      assert(i);
      assert(!gr_adj(g2, i));
    }
  } else if (gr_nv(g1) > gr_nv(g2)) {
    for (i = gr_first(g1); i < gr_first(g2); i++) {
      assert(i);
      assert(!gr_adj(g1, i));
    }
    for (i = gr_first(g2); i <= gr_last(g2); i++) {
      if (!i) continue;
      assert(l_cmp(gr_adj(g1, i), gr_adj(g2, i)));
    }
    for (i = gr_last(g2) + 1; i <= gr_last(g1); i++) {
      assert(i);
      assert(!gr_adj(g1, i));
    }
  } else {
    for (i = gr_first(g1); i <= gr_last(g1); i++) {
      if (!i) continue;
      assert(l_cmp(gr_adj(g1, i), gr_adj(g2, i)));
    }
  }

  return(1);
}

BOOLEAN gr_empty(graph)
GRAPHPTR graph;
{
  int i;

  for (i = gr_first(graph); i <= gr_last(graph); i++)
    if (graph->vlist[i])
      return(F);

  return(T);
}

void gr_make_empty(graph, changelist)
GRAPHPTR graph;
register LISTPTR changelist;
{
  register int v;
  register LISTPTR *vlist;

  assert(graph);

  vlist = graph->vlist;
  assert(vlist);

  while (changelist) {
    v = changelist->o;
    assert(v >= gr_first(graph) && v <= gr_last(graph));
    if (vlist[v]) vlist[v] = l_free(vlist[v]);
    changelist = changelist->next;
  }

  return;
}

