/*
 * FILE: graph.h
 */

#ifndef GRAPHDEF
#define GRAPHDEF

#include "sr.h"
#include "listsr.h"
#include "stackasr.h"
#include "queuesr.h"

#define INTSIZE (sizeof(int)*8)

typedef struct graph GRAPH;
typedef struct graph *GRAPHPTR;

#define GRAPHNULL (GRAPHPTR)NULL

struct graph {
  int first, last;
  int no_vertices;
  BOOLEAN *active;              /* v is active when active[v] is T */
  LISTPTR *vlist;               /* adj list representation */
  LISTPTR *scc;                 /* records the strongly cc */
  int *v2scc;                   /* records the scc corres to vertex */
  int scc_valid;                /* 1 when scc corresponds to vlist, else 0 */
};

/* graph.c */

extern GRAPHPTR gr_open( /* int first, last; */ );
/*
 * gr_open() takes as input the first and last node numbers.  It returns a
 * pointer to a new graph structure appropriately initialized.
 */

extern GRAPHPTR gr_close( /* GRAPHPTR old; */ );
/*
 * gr_close() takes as input the old graph you would like to free.  It frees
 * the graph and returns GRAPHNULL.
 */

extern int gr_add_arc( /* GRAPHPTR graph; int v1, v2; */ );
/*
 * gr_add_arc() returns 1 if an arc was really added else it returns 0.
 */

extern int gr_delete_arc( /* GRAPHPTR graph; int v1, v2; */ );
/*
 * gr_delete_arc() returns 1 if the arc was really deleted else it returns 0.
 */

extern void gr_print( /* GRAPHPTR graph; char *filename; */ );
/*
 * gr_print() takes as input the graph you would like to print and the name of
 * the file in which you would like it to be printed.  If filename == "stdout",
 * the standard output is used.  It prints the graph to the file and returns
 * nothing.
 */

extern void gr_print_scc( /* GRAPHPTR graph; char *filename; */ );
/*
 * gr_print_scc() takes as input the graph of which you would like to print the
 * strongly connected components and the name of the file in which you would
 * like it to be printed.  If filename == "stdout", the standard output is used.
 * It prints the strongly connected components of the graph to the file and
 * returns nothing.
 */

extern void gr_compute_scc( /* GRAPHPTR graph; */ );
/*
 * gr_compute_scc() takes as input a graph.  It computes all of the strongly
 * connected components of the graph and returns nothing.
 */

extern GRAPHPTR gr_build_dag( /* GRAPHPTR graph; */ );
/*
 * gr_build_dag() takes as input a graph.  It creates and returns a Directed
 * Acyclic Graph of the strongly connected components of the input argument.
 * It assumes that the strongly connected components of 'graph' have been
 * built already.
 */

extern void gr_init_scc( /* GRAPHPTR graph; */ );
/*
 * gr_init_scc() takes as input a graph.  It inits the scc and v2scc arrays.
 * if necessary, it mallocs the arrays.
 */

extern void gr_free_scc( /* GRAPHPTR graph; */ );
/*
 * gr_free_scc() takes as input a graph.  It frees all of the strongly
 * connected components of the graph and returns nothing.
 */

extern LISTPTR gr_topsort( /* GRAPHPTR dag; */ );
/*
 * gr_topsort() takes as input a dag.  It returns a topologically sorted list
 * of the vertices in the input dag.
 */

extern void topsortvisit( /* GRAPHPTR graph; int v; LISTPTR *deslist; int visited[]; */ );
/*
 * topsortvisit() accepts a graph, a vertex v, a pointer to a descendant list,
 * and an integer array.  All vertices reachable from v are placed in the list.
 * The elements in the list are in topological order.  Visited has bounds
 * graph->first to graph->last and should be initialized to all zeros.
 */

extern void visit( /* GRAPHPTR graph; int v; LISTPTR *deslist; int visited[]; */ );
/*
 * visit() accepts a graph, a vertex v, a pointer to a descendant list, and an
 * integer array.  All vertices reachable from v are placed in the list. The
 * elements in the list are in REVERSE topological order.  Visited has bounds
 * graph->first to graph->last and should be initialized to all zeros.
 */

extern GRAPHPTR gr_copy( /* GRAPHPTR graph; */ );
/*
 * gr_copy() accepts a graph as input.  It copies the graph and returns the new
 * graph.
 */

extern void gr_expand( /* GRAPHPTR graph; int newfirst, newlast; */ );
/*
 * gr_expand() accepts a graph and new first and last values as input.  It
 * expands the graph to the new size (maintaining its contents) and returns
 * nothing.
 */

extern void gr_unexpand( /* GRAPHPTR graph; int newfirst, newlast; */ );
/*
 * gr_unexpand() accepts a graph and new first and last values as input.  It
 * contracts the graph to the new size (maintaining its contents) and returns
 * nothing.
 */

extern void gr_size( /* GRAPHPTR graph; */ );
/*
 * gr_size() accepts a graph as input.  It prints the size of the graph to the
 * standard output and returns nothing.
 */

extern int gr_identical( /* GRAPHPTR g1, g2; */ );
/*
 * gr_identical() accepts two graphs as input.  It asserts that the two graphs
 * are identical and returns 1.  Usually used: assert(gr_identical(g1, g2));
 */

extern BOOLEAN gr_empty( /* GRAPHPTR graph; */ );
/*
 * gr_empty() accepts a graph as input.  It returns T if and only if the
 * adjacency lists (vlist) are LISTNULL from gr_first(graph) to gr_last(graph).
 */

extern void gr_make_empty( /* GRAPHPTR graph; LISTPTR changelist; */ );
/*
 * gr_make_empty() accepts a graph and a list of integers as input.  It frees
 * the adjacency list for every integer in changelist.  It returns nothing.
 */

#define GR_ADD_ARC(graph, v1, v2) \
{ \
  assert(graph != GRAPHNULL); \
  assert(v1 >= gr_first(graph) && v1 <= gr_last(graph)); \
  assert(v2 >= gr_first(graph) && v2 <= gr_last(graph)); \
  if (l_find(graph->vlist[v1], v2) == LISTNULL) { \
    graph->scc_valid = 0; \
    if (v1 == -(v2)) { \
      l_insert(graph->vlist[v1], v2); \
    } else { \
      l_append(graph->vlist[v1], v2); \
    } \
  } \
}

#define GR_DELETE_ARC(graph, v1, v2) \
{ \
  LISTPTR gr_found; \
  assert(graph != GRAPHNULL); \
  assert(v1 >= gr_first(graph) && v1 <= gr_last(graph)); \
  assert(v2 >= gr_first(graph) && v2 <= gr_last(graph)); \
  graph->scc_valid = 0 ;  \
  gr_found = l_find(graph->vlist[v1], v2); \
  if (gr_found != LISTNULL) (void)l_delete(&graph->vlist[v1], gr_found); \
}

#define gr_nv(g) ((g)->no_vertices)
/*
 * gr_nv() takes as input a graph.  It returns the number of vertices allowed in
 * the graph.
 */

#define gr_first(g) ((g)->first)
/*
 * gr_first() takes as input a graph.  It returns the lowest vertex number
 * allowed in the graph.
 */

#define gr_last(g) ((g)->last)
/*
 * gr_last() takes as input a graph.  It returns the highest vertex number
 * allowed in the graph.
 */

#define gr_adj(g,v) ((g)->vlist[v])
/*
 * gr_adj() takes as input a graph and a vertex (integer).  It returns a list of
 * all vertices which are adjacent to v (v2 is adjacent to v1 if and only if
 * there is an edge from v1 to v2).
 */

#define gr_scc(g,v) ((g)->scc[v])
/*
 * gr_scc() takes as input a graph and a vertex (integer).  It returns a list
 * of vertices belonging to the strongly connected component named v.
 */

#define gr_v2scc(g,v) ((g)->v2scc[v])
/*
 * gr_v2scc() takes as input a graph and a vertex (integer).  It returns the
 * name (integer) of the strongly connected component to which v belongs.
 */

#define gr_scc_valid(g) ((g)->scc_valid)
/*
 * gr_scc_valid() takes as input a graph.  It returns 1 when the strongly
 * connected components correspond to (or are up to date with) the graph
 * itself, else it returns 0.
 */

#define gr_scc_not_valid(g) (!(g)->scc_valid)
/*
 * gr_scc_not_valid() takes as input a graph.  It returns 1 when the strongly
 * connected components do not correspond to (or are not up to date with) the
 * graph itself, else it returns 0.
 */

#define gr_in(g,v1,v2) (l_find(gr_adj(g, v1), v2) != LISTNULL)
/*
 * gr_in() takes as input a graph and two vertices (integers).  It returns TRUE
 * if edge (v1, v2) is in the graph, and FALSE if edge (v1, v2) is not in the
 * graph.
 */

#define gr_in_scc(g,v1,v2) (l_find(gr_scc(g, v1), v2) != LISTNULL)
/*
 * gr_in_scc() takes as input a graph and two vertices (integers).  It returns
 * TRUE if v2 is in the strongly connected component named v1, and FALSE if v2
 * is not in the strongly connected component named v1.
 */

#define gr_active(graph, v) ((graph)->active[v])

#define gr_activate(graph, v) ((graph)->active[v] = T)

#define gr_deactivate(graph, v) ((graph)->active[v] = F)

#endif

