/*
 * FILE: graph-util.c
 */

#include "pretg.h"

/* edge structure allocation chunk maxima */
#define EDGECHUNK 4096
#define EDGEMAXALLOC 4096

/* globals for use in allocating data structures */
static LISTPTR Edgeheaplist = LISTNULL;
static EDGEPTR Edgeheap[EDGEMAXALLOC];
static int Edgeheapcount = 0;

EDGEPTR edgecreate(weight, vertex)
int weight;
int vertex;
{
  int i;
  EDGEPTR new;

  if (!Edgeheaplist) {
    EDGEPTR edges;

    assert(Edgeheapcount < EDGEMAXALLOC - 1);
    Edgeheap[Edgeheapcount++]=edges=(EDGEPTR)SR_MALLOC(EDGECHUNK*sizeof(EDGE));
    for (i = 0; i < EDGECHUNK; i++) l_append(Edgeheaplist, (int)&edges[i]);
  }

  /* pop a edge off of Edgeheaplist */
  new = (EDGEPTR)l_delete(&Edgeheaplist, Edgeheaplist);
  new->wt = weight;
  new->vertex = vertex;

  return(new);
}

EDGEPTR edgefree(old)
EDGEPTR old;
{
  if (old) {
    old->wt = -1;
    old->vertex = -1;

    /* push old onto Edgeheaplist */
    l_insert(Edgeheaplist, (int)old);
  }

  return(EDGENULL);
}

void edge_end()
{
  while (Edgeheapcount > 0) {
    Edgeheapcount--;
    Edgeheap[Edgeheapcount]=(EDGEPTR)sr_free((char *)Edgeheap[Edgeheapcount]);
  }

  Edgeheaplist = l_free(Edgeheaplist);

  return;
}

/* wt_graph stuff - vertices correspond to nodes of the 
   Nodetable (except for FF nodes & POaddr node). Wts
   correspond to the number of FF's between nodes */

WT_GRAPHPTR create_wt_graph(nv)
int nv;
{
  WT_GRAPHPTR wt_graph;
  
  wt_graph = (WT_GRAPHPTR)SR_MALLOC(nv*sizeof(WT_GRAPH));
  l_init_listarray(wt_graph, 0, nv-1);

  return(wt_graph);
}

/* recursively add edges to wt_graph */

void r_add_edges(wt_graph, nv, vo, v, w)
WT_GRAPHPTR wt_graph;
int nv, vo, v, w;

{
  LISTPTR l;
  NODEPTR n1, n2;

  /* vo is the original v (the vertex from which an edge will be placed) */
  /* v is the vertex we are currently traversing (may be a PPI) */
  /* w is the current weight (# of PPIs traversed so far) */

  assert(wt_graph);
  assert(vo >= 0 && vo < nv);
  assert(w >= 0);

  n1 = Nodetable[V2N(v)];
  assert(V2N(v) > 0 && V2N(v) < Nodecount);
  assert(n1);
  assert(n1->address == V2N(v));
  assert(N2V(n1->address) == v);
  assert(v == vo || n1->type == PPI);

  for (l = n1->out; l != LISTNULL; l = l->next) {
    assert(l->o > 0 && l->o < Nodecount);
    n2 = Nodetable[l->o];
    assert(n2);
    assert(n2->type != PI);
    assert(n2->address == l->o);
    assert(V2N(N2V(n2->address)) == n2->address);
    if (l->o != POaddr) {
      if (n2->type == PPI) r_add_edges(wt_graph, nv, vo, N2V(l->o), w+1);
      else l_append(wt_graph[vo], (int)edgecreate(w, N2V(l->o)));
    }
  }

  return;
}

WT_GRAPHPTR make_wt_graph(nv)
int nv;

{
  int v;
  WT_GRAPHPTR wt_graph = WTNULL;

  /* allocate memory for wt_graph */
  wt_graph = create_wt_graph(2*nv);

  /* make edges for wt_graph */
  for (v = 0; v < nv; v++) {
    assert(V2N(v) != POaddr);
    assert(Nodetable[V2N(v)]->type != PPI);
    r_add_edges(wt_graph, nv, v, v, 0);
  }

  return(wt_graph);
}

void print_wt_graph(wt_graph)
WT_GRAPHPTR wt_graph;
{
  int i, cp;

  extern int edge_printstr();

  assert(wt_graph && Nvertices);

  fprintf(stdout, "Printing Edge Weight Graph:\n");

  for (i = 0; i < Nvertices; i++) {
    assert(V2N(i) > 0 && V2N(i) < Nodecount);
    assert(Nodetable[V2N(i)]);
    assert(Nodetable[V2N(i)]->name && Nodetable[V2N(i)]->name[0]);
    cp = fprintf(stdout, "%s: ", Nodetable[V2N(i)]->name);
    l_oprint(wt_graph[i], stdout, edge_printstr, " ", cp, cp);
    fprintf(stdout, "\n");
  }
  fprintf(stdout, "#\n");
  fflush(stdout);

  return;
}

WT_GRAPHPTR reverse_wt_graph(wt_graph,nv)
WT_GRAPHPTR wt_graph;
int nv;
{
  int i;
  LISTPTR l;
  EDGEPTR arc;
  WT_GRAPHPTR rcpy_wt_graph;

  rcpy_wt_graph = create_wt_graph(nv);

  for (i = 0; i < nv; i++)
    for (l = wt_graph[i]; l != LISTNULL; l = l->next) {
      arc = (EDGEPTR) l->o;
      l_append(rcpy_wt_graph[arc->vertex], (int)edgecreate(arc->wt,i));
    }
  return(rcpy_wt_graph);
}

WT_GRAPHPTR free_wt_graph(wt_graph,nv)
WT_GRAPHPTR wt_graph;
int nv;
{
  int i;
  LISTPTR l;

  assert(wt_graph);

  for (i = 0; i < nv;i++) {
    for (l = wt_graph[i]; l != LISTNULL; l = l->next)
      l->o = (int)edgefree((EDGEPTR)l->o);
    wt_graph[i] = l_free(wt_graph[i]);
  }

  wt_graph = (WT_GRAPHPTR)sr_free((char *)wt_graph);

  return(WTNULL);
}

/* depth first search to compute the number of sink 
   nodes reachable from a source node and vice-versa */

static int *visited, *count;

void vsit(wt_graph, root, v)
WT_GRAPHPTR wt_graph;   
int root, v;
{
  LISTPTR l;
  EDGEPTR arc;

  assert(!visited[v]);
  visited[v] = 1;

  for (l = wt_graph[v]; l!=LISTNULL; l = l->next) {
    arc = (EDGEPTR)l->o;
    if (!visited[arc->vertex]) vsit(wt_graph,root,arc->vertex);
  }

  if (!wt_graph[v]){
    count[root]++;
    count[v]--;
  }

  return;
}

int *dfs_wt_graph(wt_graph,indeg,nv)
WT_GRAPHPTR wt_graph;
int nv, *indeg;
{
  int i,j;

  count = (int *) SR_MALLOC(2*nv * sizeof(int));
  visited = (int *) SR_MALLOC(nv * sizeof(int));

  for (i=0;i<2*nv;i++) count[i] = 0;
  for(i=0;i<nv;i++)
    if (!indeg[i]) {
      assert(l_find(Inputs,V2N(i)));
      for (j=0;j<nv;j++) visited[j] = 0;
      vsit(wt_graph,i,i);
    }

  /* free arrays */
  visited = (int *) sr_free((char *) visited);

  return(count);
}

/*
void remove_arcs_vertices(wt_graph)
WT_GRAPHPTR wt_graph;

{
  int i;
  LISTPTR l, next_l;
  EDGEPTR arc;

  for (i = 0; i < Nvertices; i++)
    if (!visited[i])
      for (l = wt_graph[i]; l != LISTNULL; l = next_l) {
	next_l = l->next;
	arc = (EDGEPTR) l->o;
	l_delete(&(wt_graph[i]), l);
	arc = edgefree(arc);
      }
  visited = (int *) sr_free((char *) visited);
  return;
} */
