/*
 * FILE: main.c
 */

#include "pretg.h"
#include "pretg.e"

static char *Opts = "d:o:RVMm:T ";

int main(argc, argv)
int argc;
char *argv[];
{
  double time;

  extern void parse_options(), parse_args();
  extern void insert_buffers_at_POs();
  extern void check_inputs_outputs();

  sr_print_header(argc, argv);

  /* get start time */
  time = get_time(RUSAGE_SELF);

  /* catch signals which cause core dumps */
  catch_core_dump_signals();

  fprintf(stdout, "#\n"); 
  fflush(stdout);

  /* initialize Program name */
  Program = argv[0];

  /* get options from the command line and read in the circuit */
  parse_options(argc, argv);
  parse_args(argc, argv);

  /* keep the inputs, outputs clean. No Constants allowed */
  insert_buffers_at_POs();
  check_inputs_outputs();

  /* do it */
  pretg_main();

  /* free memory if we are keeping track */
  if (sr_tmfile) free_all();

  /* Report time used. */
  time = get_time(RUSAGE_SELF) - time;
  fprintf(stdout, "#\n# Time Used: %.2lf sec\n#\n", time);
  fflush(stdout);

  return(NORMAL);
}

void parse_options(argc, argv)
int argc;
char *argv[];
{
  register int c;
  extern char *optarg;
  extern FILE *sr_tmfile;
  extern unsigned long sr_trackmem;
  extern void usage_error();

  while ((c = getopt(argc, argv, Opts)) != EOF) {
    switch (c) {
      /* legal options */
      case 'd': Debug = atoi(optarg); break;
      case 'o': if (!strcmp(optarg, "bench")) OutputLang = BENCH;
                else if (!strcmp(optarg, "blif")) OutputLang = BLIF;
                else if (!strcmp(optarg, "verilog")) OutputLang = VERILOG;
                else if (!strcmp(optarg, "pwc")) OutputLang = PWC;
                else usage_error();
                break;
      case 'R': Use_Rvalues = T; break;
      case 'V': Verbose = T; break;
      case 'M': sr_tmfile = sr_fopen("allocfree", "w"); break;
      case 'm': sr_trackmem = (unsigned long)atol(optarg); break;
      case 'T': Trace = T; break;

      /* illegal options */
      case '-': case '?': usage_error(); break;
      default: assert_false(); break;
    }
  }

  return;
}

void parse_args(argc, argv)
int argc;
char *argv[];
{
  extern int optind;
  extern void parse();

  if (optind != argc - 1) {
    fprintf(stderr, "\nERROR: missing circuit filename\n");
    usage_error();
  } else {
    /* parse argv[optind] */
    Circuit_Filename = sr_strsav(argv[optind]);
    Circuit_Basename = rmext(sr_strsav(Circuit_Filename));
    Circuit_Modelname = basename(Circuit_Basename);
    parse(&Circuit_Filename);
  }

  return;
}

void parse(infile)
char **infile;
{
  extern FILE *bench_in, *blif_in;
  extern int bench_lineno, blif_lineno;
  extern void construct_primary_lists();

  assert(infile && *infile && **infile);
  assert(Nodetable == (NODEPTR *)NULL);
  assert(Hashtable == H_TABLENULL);

  /* create new hash table */
  Hashtable = h_createtable(HASHSIZE);

  /* determine file type */
  if (InputLang == NOLANG) {
    int len = strlen(*infile);
    if ((*infile)[len-6] == '.' && (*infile)[len-5] == 'b' &&
        (*infile)[len-4] == 'e' && (*infile)[len-3] == 'n' &&
        (*infile)[len-2] == 'c' && (*infile)[len-1] == 'h') {
      InputLang = BENCH;
    } else if ((*infile)[len-5] == '.' &&
               (*infile)[len-4] == 'b' && (*infile)[len-3] == 'l' &&
               (*infile)[len-2] == 'i' && (*infile)[len-1] == 'f') {
      InputLang = BLIF;
    } else {
      InputLang = BENCH;
    }
  }

  /* open *infile and set line number */
  Lineno = InputLang == BENCH ? &bench_lineno : &blif_lineno;
  if (InputLang == BENCH) bench_lineno = 1; else blif_lineno = 1;

  if (access(*infile, R_OK)) {
    char buf[MAXSTRLEN];
    sprintf(buf, "%s.%s", *infile, InputLang == BENCH ? "bench" : "blif");
    *infile = sr_free(*infile);
    *infile = sr_strsav(buf);
  }

  if (InputLang == BENCH) bench_in = sr_fopen(*infile, "r");
  else blif_in = sr_fopen(*infile, "r");

  /* parse the input file */
  fprintf(stdout, "# currently reading \"%s\"\n", *infile);
  fflush(stdout);
  if (InputLang == BENCH) bench_parse(); else blif_parse();

  /* unset line number */
  Lineno = (int *)NULL;

  /* close *infile */
  if (InputLang == BENCH) bench_in = sr_fclose(bench_in);
  else blif_in = sr_fclose(blif_in);

  return;
}

void insert_buffers_at_POs()
{
  LISTPTR l;
  NODEPTR po_node, buff_node;
  char name[PLEN];
  
  assert(!Nodelist);
  assert(POaddr);
  assert(Nodetable[POaddr]->in);

  l = l_copy(Nodetable[POaddr]->in);
  for (l; l!=LISTNULL; l = l->next){
    assert(l->o > 0 && l->o < Nodecount);
    po_node = get_node(l->o, Nodelist);
    sprintf(name, "pre_%s", po_node->name);
    buff_node = mkbuff(po_node, name, F);
    insert_as_PO(po_node, buff_node);
  }
  l = l_free(l);
  
  return;
}

void check_inputs_outputs()
{
  int i;
  
  assert(POaddr);

  for (i=1; i<Nodecount; i++) {
    if ((Nodetable[i]->in && Nodetable[i]->type == PI) ||
	(!Nodetable[i]->in && Nodetable[i]->type != PI))
      fprintf(stderr, "#\n# ERROR : check signal %s ", Nodetable[i]->name);
    if ((Nodetable[i]->out && Nodetable[i]->func == GPO) ||
	(!Nodetable[i]->out && Nodetable[i]->func != GPO))
      fprintf(stderr, "#\n# ERROR : check signal %s ", Nodetable[i]->name);
    if (l_find(Nodetable[POaddr]->in, i)) {
      assert(l_len(Nodetable[i]->out) == 1);
      assert(l_find(Nodetable[i]->out,POaddr));
      assert(Nodetable[i]->func == GBUFF);
    }
  }

  /* Construct list of primary inputs, outputs and constants */
  construct_primary_lists();
  assert(!Constants)

  return;
}

void construct_primary_lists()
{
  int i, count;
  NODEPTR *table;

  table = Nodetable;
  count = Nodecount;

  assert(table);
  assert(count > 0);
  assert(!Inputs);
  assert(!PIcount);
  assert(!Outputs);
  assert(!POcount);
  assert(!Constants);
  assert(!CONcount);
  assert(POaddr > 0 && POaddr < Nodecount);
  assert(Nodetable[POaddr] == POnode);

  for (i = 0; i < count; i++) {
    if (table[i]) {
      if (table[i]->func == GPI) {
        assert(table[i]->type == PI);
        l_append(Inputs, i);
        PIcount++;
      }
      if (table[i]->func == GZERO || table[i]->func == GONE) {
        l_append(Constants, i);
        CONcount++;
      }
      if (l_find(Nodetable[POaddr]->in, i)) {
	l_append(Outputs, i);
	POcount++;
      }
    }
  }

  assert(POcount == l_len(POnode->in));

  return;
}

void usage_error()
{
  /* static char *Opts = "d:i:o:Rs:VMm:T "; */

  assert(Opts);

  fprintf(stderr, "\nUsage:\n\n");
  fprintf(stderr, "        %s [-%s] filename\n\n", Program, Opts);
  fprintf(stderr, "options:\n\n");
  fprintf(stderr, "  -d <int>   set debug level to <int>\n");
  fprintf(stderr, "  -o <lang>  set output language to <lang> (bench or blif or verilog or pwc)\n");
  fprintf(stderr, "  -R         use retime values to retime circuit\n");
  fprintf(stderr, "  -V         verbose mode\n");
  fprintf(stderr, "  -M         check memory a/f in \"allocfree\" for use by afc\n");
  fprintf(stderr, "  -m <int>   track memory location <int>\n");
  fprintf(stderr, "  -T         trace parsing\n");

  fprintf(stderr, "\n");
  exit(USAGE__ERROR);

  return;
}


