/*
 * FILE: print.c
 */

#include "pretg.h"

/* Printing Utilities */

void print_circuit(filename)
char *filename;
{
  int i;
  FILE *fp = strcmp(filename, "stdout") ? sr_fopen(filename, "w") : stdout;

  extern void print_bench_header(), print_blif_header();
  extern void print_verilog_header(), print_pwc_header();
  extern void print_bench_trailer(), print_blif_trailer();
  extern void print_verilog_trailer(), print_pwc_trailer();

  assert(check_connectivity());

  switch (OutputLang) {
    case BENCH: print_bench_header(fp); break;
    case BLIF: print_blif_header(fp, filename); break;
    case VERILOG: print_verilog_header(fp); break;
    case PWC: print_pwc_header(fp); break;
    default: assert_false(); break;
  }

  for (i = 0; i < Nodecount; i++) if (Nodetable[i]) {
    switch (OutputLang) {
      case BENCH: print_bench_gate(i, fp); break;
      case BLIF: print_blif_gate(i, fp); break;
      case VERILOG: print_verilog_gate(i, fp); break;
      case PWC: print_pwc_gate(i, fp); break;
      default: assert_false(); break;
    }
  }

  switch (OutputLang) {
    case BENCH: print_bench_trailer(fp); break;
    case BLIF: print_blif_trailer(fp); break;
    case VERILOG: print_verilog_trailer(fp); break;
    case PWC: print_pwc_trailer(fp); break;
    default: assert_false(); break;
  }

  fp = fp != stdout ? sr_fclose(fp) : (FILE *)NULL;

  return;
}

int print_bench_gate(v, fp)
int v;
FILE *fp;
{
  int cp;
  NODEPTR n = Nodetable[v];

  assert(v > 0 && v < Nodecount);
  assert(n);
  assert(fp);

  if (n->func != GPO && n->func != GPI && n->func != GDFF) {
    assert(n->type != PI && n->type != PPI);
    cp = fprintf(fp, "%s = %s%s", n->name, func2str(n), n->in ? "(" : "");
    if (n->in) l_oprint(n->in, fp, printstr, ", ", cp, cp);
    fprintf(fp, "%s\n", n->in ? ")" : "");
    fflush(fp);
  } else {
    assert(n == POnode || n->type == PI || n->type == PPI);
  }

  return(0);
}

int print_blif_gate(v, fp)
int v;
FILE *fp;
{
  char *bf;
  int cp;
  NODEPTR n = Nodetable[v];

  assert(v > 0 && v < Nodecount);
  assert(n);
  assert(fp);

  if (n->func != GPO && n->func != GPI && n->func != GDFF) {
    assert(n->type != PI && n->type != PPI);
    cp = fprintf(fp, ".names ");
    l_oprint(n->in, fp, printstr, " ", cp, cp);
    fprintf(fp, " %s\n%s", n->name, n->blifunc ? n->blifunc :
            (bf=blifunc(n)) ? bf : strcat(func2str(n), "\n"));
  } else {
    assert(n == POnode || n->type == PI || n->type == PPI);
  }
  fflush(fp);

  return(0);
}

int print_verilog_gate(v, fp)
int v;
FILE *fp;
{
  assert(v > 0 && v < Nodecount);
  fflush(fp);
  return(0);
}

int print_pwc_gate(v, fp)
int v;
FILE *fp;
{
  assert(v > 0 && v < Nodecount);
  fflush(fp);
  return(0);
}

void print_bench_header(fp)
FILE *fp;
{
  NODEPTR n, inp;
  LISTPTR l, inputs = LISTNULL, outputs = LISTNULL, ffs = LISTNULL;

  /* separate Inputs list into PIs and PPIs */
  for (l = Inputs; l != LISTNULL; l = l->next) {
    assert(l->o > 0 && l->o < Nodecount);
    assert(Nodetable[l->o] && Nodetable[l->o]->address == l->o);
    assert(Nodetable[l->o]->type == PI || Nodetable[l->o]->type == PPI);
    if (Nodetable[l->o]->type == PI) l_append(inputs, (int)Nodetable[l->o]);
    else l_append(ffs, (int)Nodetable[l->o]);
  }

  while (inputs) {
    n = (NODEPTR)l_delete(&inputs, inputs);
    assert(n);
    assert(n->type == PI);
    assert(!n->in);
    fprintf(fp, "INPUT(%s)\n", n->name);
  }
  fprintf(fp, "\n");

  assert(POaddr > 0 && POaddr < Nodecount);
  assert(POnode && Nodetable[POaddr] == POnode);

  /* put POnode->in (the Outputs list) in outputs */
  for (l = POnode->in; l != LISTNULL; l = l->next) {
    assert(l->o > 0 && l->o < Nodecount);
    assert(Nodetable[l->o] && Nodetable[l->o]->address == l->o);
    assert(ispo(Nodetable[l->o]));
    l_append(outputs, (int)Nodetable[l->o]);
  }

  while (outputs) {
    n = (NODEPTR)l_delete(&outputs, outputs);
    assert(n);
    assert(ispo(n));
    fprintf(fp, "OUTPUT(%s)\n", n->name);
  }
  fprintf(fp, "\n\n");

  while (ffs) {
    n = (NODEPTR)l_delete(&ffs, ffs);
    assert(n);
    assert(n->type == PPI);
    assert(n->in && !n->in->next);
    assert(n->in->o > 0 && n->in->o < Nodecount);
    inp = Nodetable[n->in->o];
    assert(inp && inp->name && inp->name[0]);
    assert(isppo(inp));
    fprintf(fp, "%s = %s(%s)\n", n->name, func2str(n), inp->name);
  }

  fflush(fp);

  return;
}

void print_blif_header(fp, filename)
FILE *fp;
char *filename;
{
  int cp;
  NODEPTR n, inp;
  LISTPTR l, inputs = LISTNULL, outputs = LISTNULL, regs = LISTNULL;
  char *tempname, *modelname;

  if (fp != stdout)  {
    /*char *modelname = basename(rmext(sr_strsav(filename)));*/
    tempname = sr_strsav(filename);
    modelname = basename(rmext(tempname));
    fprintf(fp, ".model %s\n", modelname);
    modelname = tempname = sr_free(tempname);
  } else {
    fprintf(fp, ".model %s\n", Circuit_Modelname);
  }

  /* separate Inputs list into PIs and PPIs */
  for (l = Inputs; l != LISTNULL; l = l->next) {
    assert(l->o > 0 && l->o < Nodecount);
    assert(Nodetable[l->o] && Nodetable[l->o]->address == l->o);
    assert(Nodetable[l->o]->type == PI || Nodetable[l->o]->type == PPI);
    if (Nodetable[l->o]->type == PI) l_append(inputs, (int)Nodetable[l->o]);
    else l_append(regs, (int)Nodetable[l->o]);
  }

  cp = fprintf(fp, ".inputs");
  while (inputs) {
    n = (NODEPTR)l_delete(&inputs, inputs);
    assert(n);
    assert(n->type == PI);
    assert(!n->in);
    cp += fprintf(fp, " %s", n->name);
    if (cp > 72 && inputs) cp = fprintf(fp, " \\\n       ") - 2;
  }
  fprintf(fp, "\n");

  assert(POaddr > 0 && POaddr < Nodecount);
  assert(POnode && Nodetable[POaddr] == POnode);

  /* put POnode->in (the Outputs list) in outputs */
  for (l = POnode->in; l != LISTNULL; l = l->next) {
    assert(l->o > 0 && l->o < Nodecount);
    assert(Nodetable[l->o] && Nodetable[l->o]->address == l->o);
    assert(ispo(Nodetable[l->o]));
    l_append(outputs, (int)Nodetable[l->o]);
  }

  cp = fprintf(fp, ".outputs");
  while (outputs) {
    n = (NODEPTR)l_delete(&outputs, outputs);
    assert(n);
    assert(ispo(n));
    cp += fprintf(fp, " %s", n->name);
    if (cp > 72 && outputs) cp = fprintf(fp, " \\\n        ") - 2;
  }
  fprintf(fp, "\n");

  cp = 0;
  while (regs) {
    n = (NODEPTR)l_delete(&regs, regs);
    assert(n);
    assert(n->type == PPI);
    assert(n->in && !n->in->next);
    assert(n->in->o > 0 && n->in->o < Nodecount);
    inp = Nodetable[n->in->o];
    assert(inp && inp->name && inp->name[0]);
    assert(isppo(inp));
    cp += fprintf(fp,".latch    %s %s   %d\n", inp->name,n->name,n->initstate);
  }

  fflush(fp);

  return;
}

void print_verilog_header(fp)
FILE *fp;
{
  fflush(fp);
  return;
}

void print_pwc_header(fp)
FILE *fp;
{
  fflush(fp);
  return;
}

void print_bench_trailer(fp)
FILE *fp;
{
  fflush(fp);
  return;
}

void print_blif_trailer(fp)
FILE *fp;
{
  fprintf(fp, ".end\n");
  fflush(fp);

  return;
}

void print_verilog_trailer(fp)
FILE *fp;
{
  fflush(fp);
  return;
}

void print_pwc_trailer(fp)
FILE *fp;
{
  fflush(fp);
  return;
}

void print_wt_graph_blif(wt_graph, nv, filename)
WT_GRAPHPTR wt_graph;
int nv;
char *filename;
{
  int i, n, ncnt = 1, lcount = 0, *indeg, save_poaddr;
  int max_wt, min_wt, prev_min;
  int nodecount = 1, save_nodecount, picount = 0, save_picount;
  int pocount = 0, save_pocount, concount = 0, save_concount;
  EDGEPTR e;
  LISTPTR l, f, newregs;
  LISTPTR inputs = LISTNULL, save_inputs, outputs = LISTNULL, save_outputs;
  LISTPTR constants = LISTNULL, save_constants, nodelist = LISTNULL;
  NODEPTR node, n1, n2, ponode, save_ponode;
  NODEPTR *nodetable = (NODEPTR *)NULL, *save_nodetable;
  H_TABLEPTR hashtable = H_TABLENULL, save_hshtbl;

  extern LISTPTR make_pregs();
  extern void expand_pnodetable();

  assert(wt_graph);
  assert(!Nodelist);

  /* allocate hashtable */
  hashtable = h_createtable(HASHSIZE);

  /* allocate and initialize indeg array */
  indeg = (int *)SR_MALLOC(nv*sizeof(int));
  sr_init_intarray(indeg, 0, nv-1);
  for (i = 0; i < nv; i++) {
    for (l = wt_graph[i]; l != LISTNULL; l = l->next) {
      e = (EDGEPTR)l->o;
      assert(e);
      assert(e->vertex >= 0 && e->vertex < nv);
      indeg[e->vertex]++;
    }
  }

  /* make ponode */
  ponode = nodecreate(1, (char *)NULL, NONT);
  strcpy(ponode->name, "PRIMARY_OUTPUTS");
  assert(!h_lookup(hashtable, ponode->name));
  h_add(hashtable, ponode->name, (char *)ponode);
  ponode->func = GPO;

  /* make all combinational nodes */
  for (i = 0; i < nv; i++) {
    assert(N2V(V2N(i)) == i);
    assert(Nodetable[V2N(i)]);
    assert(Nodetable[V2N(i)]->address == V2N(i));
    node = nodecopy(Nodetable[V2N(i)], F);
    assert(!h_lookup(hashtable, node->name));
    h_add(hashtable, node->name, (char *)node);
    assert(node->type != PPI);
    assert(indeg[i] || wt_graph[i]);
    if (!indeg[i]) {
      l_append(inputs, ncnt);
      picount++;
      node->func = GPI;
      node->type = PI;
      if (node->blifunc) node->blifunc = sr_free(node->blifunc);
    }
    if (!wt_graph[i]){
      l_append(outputs, ncnt);
      pocount++;
      l_append(ponode->in, ncnt);
      ponode->n_fanin++;
      assert(node->type == NONT);
      if (node->blifunc) node->blifunc = sr_free(node->blifunc);
    }
    if (Nodetable[V2N(i)]->type == VDD || Nodetable[V2N(i)]->type == VSS) {
      node->type = Nodetable[V2N(i)]->type;
      node->func = Nodetable[V2N(i)]->func;
      assert(node->type != VDD || node->func == GONE);
      assert(node->type != VSS || node->func == GZERO);
      l_append(constants, ncnt);
      concount++;
    }
    node->address = ncnt++;
    l_append(nodelist, (int)node);
  }

  /* connect primary outputs to ponode */
  assert(l_len(nodelist) == nv);
  ponode->address = ncnt++;
  l_append(nodelist, (int)ponode);
  for (l = ponode->in; l != LISTNULL; l = l->next) {
    f = l_lookup(nodelist, l->o-1);
    node = (NODEPTR)f->o;
    assert(node);
    assert(node->address == l->o);
    assert(node->type != PPI);
    l_append(node->out, ponode->address);
    node->n_fanout++;
  }

  /* make all registers and connect up all the nodes */
  for (i = 0; i < nv; i++) {
    f = l_lookup(nodelist, i);
    n1 = (NODEPTR)f->o;
    assert(n1);
    assert(n1->address == i+1);
    assert(n1->type != PPI);

    max_wt = 0; min_wt = -1;
    for (l = wt_graph[i]; l != LISTNULL; l = l->next) {
      e = (EDGEPTR)l->o;
      assert(e);
      assert(e->vertex >= 0 && e->vertex < nv);

      max_wt = MAX(e->wt,max_wt);
    }
    
    do {
      prev_min = min_wt;
      min_wt = max_wt;
      for (l = wt_graph[i]; l != LISTNULL; l = l->next) {
	e =(EDGEPTR)l->o;
	if (e->wt > prev_min)
	  min_wt = MIN(e->wt,min_wt);
      }

      assert(min_wt > prev_min);

      if (min_wt) {
	node = nodecreate(ncnt++,(char *)NULL,NONT);
	sprintf(node->name, "FB%d", ncnt-1);
	assert(!h_lookup(hashtable, node->name));
	h_add(hashtable, node->name, (char *)node);
	node->func = GBUFF;
	l_append(nodelist, (int)node);
	if (prev_min != -1) n = min_wt - prev_min;
	else n = min_wt;
	newregs = make_pregs(n,&ncnt,&lcount,&nodelist,&inputs,hashtable);
	attach(n1, (NODEPTR)newregs->o);
	attach((NODEPTR)l_tail(newregs)->o, node);
	n1 = node;
	newregs = l_free(newregs);
      }
	  
      for (l = wt_graph[i]; l != LISTNULL; l = l->next) {
	e = (EDGEPTR)l->o;
	if (e->wt == min_wt) {
	  f = l_lookup(nodelist, e->vertex);
	  n2 = (NODEPTR)f->o;
	  assert(n2);
	  assert(n2->address == e->vertex+1);
	  assert(n2->type != PPI);

	  assert(n1);
	  assert(n1->type != PPI);
	  attach(n1, n2);
	}
      }
    } while (min_wt != max_wt);
  }

  /* fix primary output buffers with more than one input 
  assert(!posav);
  for (l = ponode->in; l != LISTNULL; l = l->next) {
    f = l_lookup(nodelist, l->o-1);
    assert(f);
    node = (NODEPTR)f->o;
    assert(node);
    assert(node->address == l->o);
    assert(node->type != PPI);
    assert(node->func == GBUFF);
    assert(l_len(node->in) == node->n_fanin);
    while (node->n_fanin > 1) {
      n = nodecopy(node, F);
      do strcat(n->name, "_0");
      while (h_lookup(hashtable, n->name) && strlen(n->name) < PLEN-2);
      assert(!h_lookup(hashtable, n->name));
      h_add(hashtable, n->name, (char *)n);
      n->address = ncnt++;
      l_append(nodelist, (int)n);
      f = l_lookup(nodelist, node->in->o-1);
      assert(f);
      assert((NODEPTR)f->o);
      assert(((NODEPTR)f->o)->address == node->in->o);
      detach((NODEPTR)f->o, node);
      attach((NODEPTR)f->o, n);
      l_append(posav, (int)n);
    }
  }
  while (posav) attach((NODEPTR)l_delete(&posav, posav), ponode); */

  /* make nodetable and nodecount */
  assert(nodelist);
  expand_pnodetable(nodelist, l_len(nodelist), &nodecount, &nodetable);
  nodelist = l_free(nodelist);
  assert(ncnt == nodecount);

  /* print out the circuit now */
  save_nodetable = Nodetable; Nodetable = nodetable;
  save_nodecount = Nodecount; Nodecount = nodecount;
  save_inputs = Inputs; Inputs = inputs;
  save_picount = PIcount; PIcount = picount;
  save_outputs = Outputs; Outputs = outputs;
  save_pocount = POcount; POcount = pocount;
  save_constants = Constants; Constants = constants;
  save_concount = CONcount; CONcount = concount;
  save_ponode = POnode; POnode = ponode;
  save_poaddr = POaddr; POaddr = ponode->address;
  save_hshtbl = Hashtable; Hashtable = hashtable;

  assert(!Nodelist);
  /*remove_buffers((int *)NULL);CHAK Sept 16*/
  print_circuit(filename);

  Nodetable = save_nodetable; save_nodetable = (NODEPTR *)NULL;
  Nodecount = save_nodecount; save_nodecount = 0;
  Inputs = save_inputs; save_inputs = LISTNULL;
  PIcount = save_picount; save_picount = 0;
  Outputs = save_outputs; save_outputs = LISTNULL;
  POcount = save_pocount; save_pocount = 0;
  Constants = save_constants; save_constants = LISTNULL;
  CONcount = save_concount; save_concount = 0;
  POnode = save_ponode; save_ponode = NODENULL;
  POaddr = save_poaddr; save_poaddr = 0;
  Hashtable = save_hshtbl; save_hshtbl = H_TABLENULL;

  /* free nodetable, inputs, and indeg */
  for (i = 0; i < nodecount; i++) {
    if (nodetable[i]) {
      nodetable[i]->n_fanin = nodetable[i]->n_fanout = 0;
      nodetable[i]->in = l_free(nodetable[i]->in);
      nodetable[i]->out = l_free(nodetable[i]->out);
      nodetable[i]->type = NONT;
      nodetable[i] = nodefree(nodetable[i]);
    }
  }
  nodetable = (NODEPTR *)sr_free((char *)nodetable);
  nodecount = 0;
  inputs = l_free(inputs);
  outputs = l_free(outputs);
  constants = l_free(constants);
  indeg = (int *)sr_free((char *)indeg);

  /* free hashtable */
  hashtable = h_freetable(hashtable);

  return;
}

void expand_pnodetable(newlist, n, count, table)
LISTPTR newlist;
int n, *count;
NODEPTR **table;
{
  int i;
  LISTPTR l;
  NODEPTR *newtable = (NODEPTR *)NULL;

  assert(newlist);
  assert(n == l_len(newlist));

  assert(table && count);
  assert(*count > 0);
  assert(*count != 1 || !*table);
  assert(*count == 1 || *table);

  newtable = (NODEPTR *)SR_MALLOC((*count+n)*sizeof(NODEPTR));
  newtable[0] = NODENULL;

#ifndef NDEBUG
  for (l = newlist; l != LISTNULL; l = l->next)
    assert(!l_find(l->next, l->o));
#endif

  /* transfer *table to newtable */
  assert(!*table || !(*table)[0]);
  for (i = 1; i < *count; i++) newtable[i] = (*table)[i];
  for (i = *count; i < *count + n; i++) newtable[i] = NODENULL;
  if (*table) sr_free((char *)*table);
  *table = newtable;
  newtable = (NODEPTR *)NULL;

  for (l = newlist; l != LISTNULL; l = l->next) {
    NODEPTR node = (NODEPTR)l->o;
    assert(node);
    assert(node->address >= *count && node->address < *count + n);
    assert(!(*table)[node->address]);
    (*table)[node->address] = node;
  }

  *count += n;

  return;
}

LISTPTR make_pregs(n, newcount, lcount, nodelist, inputs, hashtable)
int n, *newcount, *lcount;
LISTPTR *nodelist, *inputs;
H_TABLEPTR hashtable;
{
  int lc;
  LISTPTR retptr = LISTNULL;
  NODEPTR n1, n2 = NODENULL;

  assert(n);
  assert(newcount && lcount && nodelist && inputs && hashtable);

  *lcount += n;
  lc = *lcount;

  for (; n > 0; n--) {
    n1 = nodecreate((*newcount)++, (char *)NULL, NONT);
    sprintf(n1->name, "reg%d", --lc);
    assert(!h_lookup(hashtable, n1->name));
    h_add(hashtable, n1->name, (char *)n1);
    l_append(*inputs, *newcount-1);
    n1->func = GDFF;
    n1->type = PPI;
    l_append(*nodelist, (int)n1);
    if (n2) attach(n1, n2);
    l_insert(retptr, (int)(n2=n1));
  }

  return(retptr);
}

int printstr(v, fp)
int v;
FILE *fp;
{
  NODEPTR node = get_node(v, Nodelist);
  assert(v > 0 && v < Nodecount || Nodelist);
  assert(node && node->address == v);
  assert(node->name && node->name[0]);
  return(fprintf(fp, "%s", node->name));
}

int edge_printstr(e, fp)
int e;
FILE *fp;
{
  int v;
  NODEPTR node;
  EDGEPTR edge = (EDGEPTR)e;

  assert(edge);
  assert(edge->vertex >= 0 && edge->vertex < Nvertices);

  v = V2N(edge->vertex);
  node = Nodetable[v];
  assert(v > 0 && v < Nodecount);
  assert(node && node->name && node->name[0]);
  return(fprintf(fp, "%s(%d)", node->name, edge->wt));
}

char *blifunc(node)
NODEPTR node;
{
  static char blifuncbuf[MAXSTRLEN];
  int i=0, j, k;

  assert(node);
  blifuncbuf[0] = '\0';

  if (node->func == GZERO || node->func == GONE) {
    sprintf(blifuncbuf, "%c\n", node->func == GZERO ? '0' : '1');
  } else if (node->func == GBUFF || node->func == GNOT) {
    sprintf(blifuncbuf, "%c 1\n", node->func == GNOT ? '0' : '1');
  } else if (node->func == GAND) {
    for (j = 0; j < node->n_fanin; j++)
      blifuncbuf[i++] = '1';
    blifuncbuf[i++] = ' ';
    blifuncbuf[i++] = '1';
    blifuncbuf[i++] = '\n';
    blifuncbuf[i++] = '\0';
  } else if (node->func == GNOR) {
    for (j = 0; j < node->n_fanin; j++)
      blifuncbuf[i++] = '0';
    blifuncbuf[i++] = ' ';
    blifuncbuf[i++] = '1';
    blifuncbuf[i++] = '\n';
    blifuncbuf[i++] = '\0';
  } else if (node->func == GOR) {
    for (k = 0; k < node->n_fanin; k++) {
      for (j = 0; j < node->n_fanin; j++)
        blifuncbuf[i++] = j == k ? '1' : '-';
      blifuncbuf[i++] = ' ';
      blifuncbuf[i++] = '1';
      blifuncbuf[i++] = '\n';
    }
    blifuncbuf[i++] = '\0';
  } else if (node->func == GNAND) {
    for (k = 0; k < node->n_fanin; k++) {
      for (j = 0; j < node->n_fanin; j++)
        blifuncbuf[i++] = j == k ? '0' : '-';
      blifuncbuf[i++] = ' ';
      blifuncbuf[i++] = '1';
      blifuncbuf[i++] = '\n';
    }
    blifuncbuf[i++] = '\0';
  } else if (node->func == GXOR) {
    if (node->n_fanin == 2) {
      strcpy(blifuncbuf, "01 1\n10 1\n");
    } else if (node->n_fanin == 3) {
      strcpy(blifuncbuf, "001 1\n010 1\n100 1\n111 1\n");
    } else assert_false();
  } else if (node->func == GXNOR) {
    if (node->n_fanin == 2) {
      strcpy(blifuncbuf, "00 1\n11 1\n");
    } else if (node->n_fanin == 3) {
      strcpy(blifuncbuf, "000 1\n110 1\n101 1\n011 1\n");
    } else assert_false();
  }

  return(blifuncbuf[0] ? blifuncbuf : (char *)NULL);
}



