/*
 * FILE: util.c
 */

#include "pretg.h"

/* Data Structure Utilities */

/* string allocation chunk maxima */
#define STRCHUNK 4096
#define STRMAXALLOC 4096

/* globals for use in allocating data structures */
static LISTPTR Strheaplist = LISTNULL;
static char *Strheap[STRMAXALLOC];
static int Strheapcount = 0;

static int strcreate_count = 0;
static int strfree_count = 0;

char *strcreate(s, line, file)
char *s;
int line;
char *file;
{
  int i;
  char *new;

  assert(s && s[0]);

  if (!Strheaplist) {
    char *strs;

    assert(Strheapcount < STRMAXALLOC - 1);
    Strheap[Strheapcount++] = strs = SR_MALLOC(STRCHUNK*PLEN);
    for (i = 0; i < STRCHUNK*PLEN; i += PLEN)
      l_append(Strheaplist, (int)&strs[i]);
  }

  /* pop a str off of Strheaplist */
  new = (char *)l_delete(&Strheaplist, Strheaplist);

  /* initialize it with s */
  strcpy(new, strchk(s));

  if (sr_trackmem == (unsigned long)new) {
    printf("%lua; LINE: %d; FILE: %s;\n", (unsigned long)new,
            line, file);
    fflush(stdout);
  }

  strcreate_count++;

  return(new);
}

char *strchk(s)
char *s;
{
  if (strlen(s) >= PLEN) {
    fprintf(stdout, "# Warning: \"%s\" exceeds PLEN (%d)\n", s, PLEN-1);
    s[PLEN-1] = '\0';
    fprintf(stdout, "#          truncating to \"%s\"\n\n", s);
    fflush(stdout);
  }

  return(s);
}

char *strfree(old, line, file)
char *old;
int line;
char *file;
{
  if (old) {
    old[0] = '\0';

    if (sr_trackmem == (unsigned long)old) {
      printf("%luf; LINE: %d; FILE: %s;\n", (unsigned long)old,
              line, file);
      fflush(stdout);
    }

    /* push old onto Strheaplist */
    assert(!l_find(Strheaplist, (int)old));
    l_insert(Strheaplist, (int)old);
  }

  strfree_count++;

  return((char *)NULL);
}

void str_end()
{
  assert(l_len(Strheaplist) <= Strheapcount*STRCHUNK);

  if (l_len(Strheaplist) < Strheapcount*STRCHUNK) {
    register int i, j, count = 0;
    register FILE *fp = sr_fopen("str_allocfree", "w");

    for (i = 0; i < Strheapcount; i++) {
      for (j = 0; j < STRCHUNK*PLEN; j += PLEN) {
        if (Strheap[i][j]) {
          count++;
          fprintf(fp, "%lua\n", (unsigned long)&Strheap[i][j]);
        }
      }
    }

    if (count < Strheapcount*STRCHUNK - l_len(Strheaplist)) {
      printf("str_end: %d strings freed with str != \"\"\n", count);
    }

    if (count > Strheapcount*STRCHUNK - l_len(Strheaplist)) {
      printf("str_end: %d strings not freed with str == \"\"\n",
             count - (Strheapcount*STRCHUNK - l_len(Strheaplist)));
    }

    printf("str_end: %d strings; %d not freed; created %d; freed %d;\n",
           Strheapcount*STRCHUNK, Strheapcount*STRCHUNK - l_len(Strheaplist),
           strcreate_count, strfree_count);
    fflush(stdout);
    fp = sr_fclose(fp);
  }

  while (Strheapcount > 0) {
    Strheapcount--;
    Strheap[Strheapcount] = sr_free(Strheap[Strheapcount]);
  }

  Strheaplist = l_free(Strheaplist);

  return;
}

/* node structure allocation chunk maxima */
#define NODECHUNK 4096
#define NODEMAXALLOC 4096

/* globals for use in allocating data structures */
static LISTPTR Nodeheaplist = LISTNULL;
static NODEPTR Nodeheap[NODEMAXALLOC];
static int Nodeheapcount = 0;

NODEPTR nodecreate(addr, prefix, type)
int addr;
char *prefix;
NODETYPE type;
{
  int i;
  NODEPTR new;

  extern char *new_name();

  if (!Nodeheaplist) {
    NODEPTR nodes;

    assert(Nodeheapcount < NODEMAXALLOC - 1);
    Nodeheap[Nodeheapcount++]=nodes=(NODEPTR)SR_MALLOC(NODECHUNK*sizeof(NODE));
    for (i = 0; i < NODECHUNK; i++) l_append(Nodeheaplist, (int)&nodes[i]);
  }

  /* pop a node off of Nodeheaplist */
  new = (NODEPTR)l_delete(&Nodeheaplist, Nodeheaplist);
  new->address = addr;
  if (prefix) {
    strcpy(new->name, strchk(new_name(prefix, addr)));
    h_add(Hashtable, new->name, (char *)new);
  } else for (i = 0; i < PLEN; i++) new->name[i] = '\0';
  new->func = NOFUNC;
  new->blifunc = (char *)NULL;
  new->sisnode = (char *)NULL;
  if (type == PI || type == PPI) {
    assert(!l_find(Inputs, addr));
    l_append(Inputs, addr);
    PIcount++;
    new->func = type == PI ? GPI : GDFF;
  } else if (type == VDD || type == VSS) {
    assert(!l_find(Constants, addr));
    l_append(Constants, addr);
    CONcount++;
    new->func = type == VDD ? GONE : GZERO;
  }
  new->type = type;
  new->n_fanout = new->n_fanin = 0;
  new->out = new->in = LISTNULL;
  new->inout = 0;
  new->initstate = 3;

  return(new);
}

NODEPTR nodefree(old)
NODEPTR old;
{
  if (old) {
    /* assert(!l_find(ISinputs, (int)node)); */
    if (old->type == PI || old->type == PPI) {
      assert(l_find(Inputs, old->address));
      l_delete(&Inputs, l_find(Inputs, old->address));
      PIcount--;
      assert(!l_find(Inputs, old->address));
      old->type = NONT;
    }
    if (old->type == VDD || old->type == VSS) {
      assert(l_find(Constants, old->address));
      l_delete(&Constants, l_find(Constants, old->address));
      CONcount--;
      assert(!l_find(Constants, old->address));
      old->type = NONT;
    }
    old->address = 0;
    old->name[0] = '\0';
    old->func = NOFUNC;
    if (old->blifunc) old->blifunc = sr_free(old->blifunc);
    if (old->sisnode) old->sisnode = sr_free(old->sisnode);
    assert(old->type == NONT);
    /* old->n_fanin = old->n_fanout = 0; */
    assert(!old->n_fanin && !old->n_fanout);
    /* old->in = l_free(old->in); old->out = l_free(old->out); */
    assert(!old->in && !old->out);
    old->inout = 0;
    old->initstate = 3;

    /* push old onto Nodeheaplist */
    l_insert(Nodeheaplist, (int)old);
  }

  return(NODENULL);
}

NODEPTR nodecopy(node, connections)
NODEPTR node;
BOOLEAN connections;
{
  NODEPTR new = NODENULL;

  if (node) {
    new = nodecreate(node->address, (char *)NULL, NONT);
    strcpy(new->name, node->name);
    new->type = node->type;
    new->func = node->func;
    if (node->blifunc) new->blifunc = sr_strsav(node->blifunc);
    new->sisnode = (char *)NULL;
    if (connections) {
      new->n_fanout = node->n_fanout;
      new->n_fanin = node->n_fanin;
      new->out = l_copy(node->out);
      new->in = l_copy(node->in);
    }
    new->inout = node->inout;
    new->initstate = node->initstate;
  }

  return(new);
}

void node_end()
{
  while (Nodeheapcount > 0) {
    Nodeheapcount--;
    Nodeheap[Nodeheapcount]=(NODEPTR)sr_free((char *)Nodeheap[Nodeheapcount]);
  }

  Nodeheaplist = l_free(Nodeheaplist);

  return;
}

char *new_name(prefix, num)
char *prefix;
int num;
{
  int count = 0;
  static char new_name_buf[MAXSTRLEN];

  assert(Hashtable);

  sprintf(new_name_buf, "%s%d", prefix, num);
  while (h_lookup(Hashtable, new_name_buf))
    sprintf(new_name_buf, "%s%d_%d", prefix, num, count++);

  return(new_name_buf);
}

void catch_core_dump_signals()
{
  extern void interrupt_handler(), c_error();

  if (signal(SIGQUIT, interrupt_handler) == (void (*)())(-1)) c_error(SIGQUIT);
  if (signal(SIGILL,  interrupt_handler) == (void (*)())(-1)) c_error(SIGILL);
  if (signal(SIGTRAP, interrupt_handler) == (void (*)())(-1)) c_error(SIGTRAP);
  if (signal(SIGABRT, interrupt_handler) == (void (*)())(-1)) c_error(SIGABRT);
  if (signal(SIGEMT,  interrupt_handler) == (void (*)())(-1)) c_error(SIGEMT);
  if (signal(SIGFPE,  interrupt_handler) == (void (*)())(-1)) c_error(SIGFPE);
  if (signal(SIGBUS,  interrupt_handler) == (void (*)())(-1)) c_error(SIGBUS);
  if (signal(SIGSEGV, interrupt_handler) == (void (*)())(-1)) c_error(SIGSEGV);
  if (signal(SIGSYS,  interrupt_handler) == (void (*)())(-1)) c_error(SIGSYS);

  return;
}

/* The signals which cause core dumps: */

#define SIGSTR(sig) \
  ((sig == SIGQUIT) ? "SIGQUIT (quit)"                                  : \
   (sig == SIGILL)  ? "SIGILL  (illegal instruction)"                   : \
   (sig == SIGTRAP) ? "SIGTRAP (trace trap)"                            : \
   (sig == SIGABRT) ? "SIGABRT (abort (generated by abort(3) routine))" : \
   (sig == SIGEMT)  ? "SIGEMT  (emulator trap)"                         : \
   (sig == SIGFPE)  ? "SIGFPE  (arithmetic exception)"                  : \
   (sig == SIGBUS)  ? "SIGBUS  (bus error)"                             : \
   (sig == SIGSEGV) ? "SIGSEGV (segmentation violation)"                : \
   (sig == SIGSYS)  ? "SIGSYS  (bad argument to system call)"           : \
   (sig == SIGKILL) ? "SIGKILL (kill signal)"                           : \
   "unknown signal")

/*
 * The handler routine can be declared:
 *
 *      void handler(sig, code, scp, addr)
 *      int sig, code;
 *      struct sigcontext *scp;
 *      char *addr;
 *
 * Here sig is the signal number; code is a parameter  of  cer-
 * tain  signals  that  provides  additional  detail;  scp is a
 * pointer to the sigcontext structure (defined in <signal.h>),
 * used to restore the context from before the signal; and addr
 * is additional address information.  See sigvec(2)  for  more
 * details.
 */

void interrupt_handler(sig, code, scp, addr)
int sig, code;
struct sigcontext *scp;
char *addr;
{
  (void)fprintf(stderr, "ERROR: caught %s\n", SIGSTR(sig), code, scp, addr);
  (void)fflush(stderr);
  exit(-1);
}

void c_error(sig)
int sig;
{
  (void)fprintf(stderr, "ERROR: cannot catch %s\n", SIGSTR(sig));
  (void)fflush(stderr);
  exit(-1);
}

#define EXTCHAR '.'

char *rmext(filename)
char *filename;
{
  char *s = &filename[strlen(filename)-1];

  while (s > filename && *s != EXTCHAR) s--;
  if (*s == EXTCHAR) *s = '\0';

  return(filename);
}

#define BASECHAR '/'

char *basename(filename)
char *filename;
{
  char *s = &filename[strlen(filename)-1];

  while (s > filename && *s != BASECHAR) s--;
  if (*s == BASECHAR) s++;

  return(s);
}

void allocate_nodes(nodelist)
LISTPTR *nodelist;
{
  register int i;
  register LISTPTR l;

  assert(nodelist);
  assert(Nodecount == l_len(*nodelist) + 1);

  /* allocate Nodetable */
  Nodetable = (NODEPTR *)SR_MALLOC(Nodecount*sizeof(NODEPTR));

  /* initialize Nodetable up to Nodecount */
  Nodetable[0] = NODENULL;
  for (l = *nodelist, i = 1; l != LISTNULL; l = l->next) {
    assert(i > 0 && i < Nodecount);
    assert((NODEPTR)l->o);
    assert(((NODEPTR)l->o)->address == i);
    Nodetable[i++] = (NODEPTR)l->o;
  }
  assert(i == Nodecount);

  /* free *nodelist */
  if (*nodelist) *nodelist = l_free(*nodelist);

  return;
}

/* Circuit Utilities */

/*
 * This function changes each nodes' list of inputs from a list of node name
 * strings to a list of node numbers, sets up the output lists for all nodes,
 * and sets up the "PRIMARY_OUTPUT" node.
 */

void fix_connectivity()
{
  int i, addr, newcount = Nodecount;
  LISTPTR l, next_l;
  NODEPTR node, inputnode, n /*, buffernode */;

  assert(!Nodetable[0]);
  assert(!Nodelist);

  /*
   * change each nodes' list of inputs from a list of node name strings to a
   * list of node numbers and set up the output lists for all nodes
   */
  for (i = 1; i < Nodecount; i++) {
    node = Nodetable[i];
    assert(node);
    assert(node->address == i);
    for (l = node->in; l != LISTNULL; l = l->next) {
      inputnode = (NODEPTR)h_lookup(Hashtable, (char *)l->o);
      if (!inputnode) {
        fprintf(stdout, "# WARNING: turning node %s into a ", (char *)l->o);
        fprintf(stdout, "primary input because it has no source\n");
        fflush(stdout);
        inputnode = nodecreate(newcount++, (char *)NULL, NONT);
        strcpy(inputnode->name, (char *)l->o);
        h_add(Hashtable, inputnode->name, (char *)inputnode);
        inputnode->func = GPI;
        inputnode->type = PI;
        l_append(Nodelist, (int)inputnode);
      }
      strfree((char *)l->o, __LINE__, __FILE__);
      l->o = inputnode->address;
      l_append(inputnode->out, node->address);
      inputnode->n_fanout++;
    }
  }

  /* put new nodes into Nodetable */
  if (Nodelist) {
    expand_nodetable(Nodelist, l_len(Nodelist));
    Nodelist = l_free(Nodelist);
    assert(newcount == Nodecount);
  }

  /* make the "PRIMARY_OUTPUT" node and put it in Nodetable */
  for (l = Outputs; l != LISTNULL; l = next_l) {
    next_l = l->next;
    node = (NODEPTR)h_lookup(Hashtable, (char *)l->o);
    if (!node) {
      fprintf(stdout, "# WARNING: ignoring output node %s", (char *)l->o);
      fprintf(stdout, " because it has no source\n");
      fflush(stdout);
      strfree((char *)l_delete(&Outputs, l), __LINE__, __FILE__);
    } else {
      assert(Nodetable[node->address] == node);
      strfree((char *)l->o, __LINE__, __FILE__); l->o = node->address;
    }
  }

  POnode = nodecreate(newcount++, (char *)NULL, NONT);
  strcpy(POnode->name, "PRIMARY_OUTPUT");
  assert(!h_lookup(Hashtable, POnode->name));
  h_add(Hashtable, POnode->name, (char *)POnode);
  POnode->func = GPO;
  POnode->n_fanin = l_len(Outputs);
  POnode->in = Outputs;
  Outputs = LISTNULL;
  for (l = POnode->in; l != LISTNULL; l = l->next) {
    Nodetable[l->o]->n_fanout++;
    l_append(Nodetable[l->o]->out, POaddr=POnode->address);
  }
  assert(!Nodelist);
  l_append(Nodelist, (int)POnode);
  expand_nodetable(Nodelist, 1);
  Nodelist = l_free(Nodelist);
  assert(newcount == Nodecount);

  if (InputLang == BENCH) {
    /* set GIOBF's corresponding PIs' inout fields to the GIOBF */
    for (i = 1; i < Nodecount; i++) {
      node = Nodetable[i];
      if (node) {
        assert(node->address == i);
        if (node->func == GIOBF) {
          assert(node->inout > 0 && node->inout < Nodecount);
          inputnode = Nodetable[node->inout];
          assert(inputnode);
          assert(inputnode->func == GPI);
          assert(inputnode->type == PI);
          assert(!inputnode->inout);
          inputnode->inout = i;
        }
      }
    }
  }

  /* make sure all nodes have fanout */
  for (i = 1; i < Nodecount; i++) {
    node = Nodetable[i];
    if (node) {
      assert(node->address == i);
      if (node != POnode && !node->n_fanout) {
        if (node->type == NONT || node->type == PPI) {
          fprintf(stdout, "# WARNING: turning node %s into a", node->name);
          fprintf(stdout, " primary output because it has no fanout\n");
          fflush(stdout);
          assert(Nodetable[POaddr] == POnode);
          node->n_fanout++;
          l_append(node->out, POaddr);
          POnode->n_fanin++;
          l_append(POnode->in, i);
        } else {
          assert(node->type == PI || node->type == VDD || node->type == VSS);
          fprintf(stdout, "# WARNING: removing node %s ", node->name);
          fprintf(stdout, "from the circuit because it has no fanout\n");
          fflush(stdout);

          /* delete node from Hashtable and Nodetable */
          addr = node->address;
          n = (NODEPTR)h_delete(Hashtable, node->name);
          assert(n == node);
          assert(!node->n_fanin && !node->n_fanout && !node->in && !node->out);
          node->type = NONT;
          Nodetable[addr] = node = n = nodefree(node);
        }
      }
    }
  }

  return;
}

BOOLEAN check_connectivity()
{
  int i;

  extern void single_conn_check();

  for (i = 0; i < Nodecount; i++) {
    NODEPTR node = get_node(i, Nodelist);
    if (node) single_conn_check(node);
  }

  return(T);
}

void single_conn_check(node)
NODEPTR node;
{
  LISTPTR l1, l2, loop;
  BOOLEAN found;

  extern LISTPTR find_acyclic_loop();

  assert(node);
  assert(node->address > 0);
  assert(node->address < Nodecount);
  assert(Nodetable[node->address] == node);
  assert(node->n_fanin == l_len(node->in));
  assert(node->n_fanout == l_len(node->out));

  switch (node->func) {
    case GPI:
      assert(!node->blifunc);
      assert(node->type == PI || node->type == PPI);
      assert(!node->n_fanin && !node->in);
      if (!node->n_fanout) {
        assert(!node->out);
        if (Verbose) {
          fprintf(stdout, "# WARNING: %sinput node %s has no fanout.\n",
            node->type == PPI ? "pseudo " : "", node->name);
          fflush(stdout);
        }
      }
      if (node->inout > 0) {
        assert(Nodetable[node->inout]);
        assert(Nodetable[node->inout]->func == GIOBF);
        assert(Nodetable[node->inout]->inout == node->address);
      }
      break;
    case GPO:
      assert(!node->blifunc);
      assert(node == POnode);
      assert(!node->n_fanout && !node->out);
      if (!node->n_fanin) {
        assert(!node->in);
        if (Verbose) {
          fprintf(stdout, "# WARNING: No Primary Outputs in the circuit.\n");
          fflush(stdout);
        }
      }
      break;
    case GAND: case GNAND: case GOR: case GNOR: case GXOR: case GXNOR:
      assert(node->n_fanin > 1);
      assert(!node->inout);
      break;
    case GBUFF: case GNOT:
      assert(node->n_fanin == 1);
      assert(!node->inout);
      break;
    case GDFF:
      assert(!node->blifunc);
      assert(node->n_fanin == 1);
      assert(!node->inout);
      break;
    case GIOBF:
      assert(!node->blifunc);
      assert(node->n_fanin == 2);
      assert(node->inout > 0);
      assert(node->inout < Nodecount);
      assert(Nodetable[node->inout]);
      assert(Nodetable[node->inout]->func == GPI);
      assert(!Nodetable[node->inout]->n_fanin);
      assert(!Nodetable[node->inout]->in);
      assert(Nodetable[node->inout]->inout == node->address);

      loop = find_acyclic_loop(node);
      if (loop) {
        fprintf(stdout, "ERROR: found acyclic loop:\n");
        l_oprint_wocr(loop, stdout, print_blif_gate, "");
        fflush(stdout);
      }
      break;
    case GST3:
      assert(!node->blifunc);
      assert(node->n_fanin == 2);
      assert(!node->inout);
      break;
    case GZERO: case GONE:
      assert(!node->n_fanin && !node->in);
      assert(!node->inout);
      break;
    case GCOMPLEX:
      assert(node->blifunc);
      break;
    default: assert_false(); break;
  }

  for (l1 = node->in; l1 != LISTNULL; l1 = l1->next) {
    NODEPTR n1 = get_node(l1->o, Nodelist);
    assert(n1);
    for (l2 = n1->out, found = F; l2 != LISTNULL; l2 = l2->next) {
      NODEPTR n2 = get_node(l2->o, Nodelist);
      assert(n2);
      assert(n2->address == l2->o);
      if (n2 == node) {
        if (found ) {
          char buf[MAXSTRLEN];
          sprintf(buf, "duplicate input from gate %s to gate %s",
                  n1->name, node->name);
          sr_error(buf);
        }
        found = T;
      }
    }
    assert(found);
  }

  for (l1 = node->out; l1 != LISTNULL; l1 = l1->next) {
    NODEPTR n1 = get_node(l1->o, Nodelist);
    assert(n1);
    for (l2 = n1->in, found = F; l2 != LISTNULL; l2 = l2->next) {
      NODEPTR n2 = get_node(l2->o, Nodelist);
      assert(n2);
      assert(n2->address == l2->o);
      if (n2 == node) {
        if (found ) {
          char buf[MAXSTRLEN];
          sprintf(buf, "duplicate input from gate %s to gate %s",
                  node->name, n1->name);
          sr_error(buf);
        }
        found = T;
      }
    }
    assert(found);
  }

  return;
}

LISTPTR find_acyclic_loop(node)
NODEPTR node;
{
  LISTPTR pathlist = LISTNULL;
  register BOOLEAN *visited;

  assert(Nodetable);
  assert(node);
  assert(node->func == GIOBF);
  assert(node->inout > 0 && node->inout < Nodecount);
  assert(node->n_fanin == 2);
  assert(node->in && node->in->next && !node->in->next->next);

  visited = (BOOLEAN *)SR_MALLOC(Nodecount*sizeof(BOOLEAN));
  sr_init_boolarray(visited, 0, Nodecount-1);

  /* put the first input to node on the pathlist */
  assert(node->in->o > 0 && node->in->o < Nodecount);
  visited[node->in->o] = T;
  l_insert(pathlist, node->in->o);

  while (pathlist) {
    register int sig = pathlist->o;

    if (sig == node->inout) {
      /* found a loop: append and insert node->address to pathlist */
      l_insert(pathlist, node->address);
      l_append(pathlist, node->address);
      break;
    } else {
      register LISTPTR l;

      /* find an unvisited input to sig */
      for (l = Nodetable[sig]->in; l != LISTNULL; l = l->next) {
        sig = l->o;
        assert(sig > 0 && sig < Nodecount);
        if (!visited[sig]) break;
      }

      if (l) {
        visited[sig] = T;
        l_insert(pathlist, sig);
      } else {
        l_delete(&pathlist, pathlist);
      }
    }
  }

  visited = (BOOLEAN *)sr_free((char *)visited);

  return(pathlist);
}

NODEPTR get_node(num, nodelist)
int num;
LISTPTR nodelist;
{
  NODEPTR retptr = NODENULL;

  if (num < Nodecount) {
    assert(!Nodetable[num] || Nodetable[num]->address == num);
    retptr = Nodetable[num];
  } else {
    LISTPTR l;
    for (l = nodelist; l != LISTNULL && !retptr; l = l->next) {
      NODEPTR n = (NODEPTR)l->o;
      assert(n);
      if (n->address == num) retptr = n;
    }
  }

  return(retptr);
}

void attach(n1, n2)
NODEPTR n1, n2;
{
  assert(n1 && n2);
  assert(n1->n_fanout == l_len(n1->out));
  assert(n1->n_fanin == l_len(n1->in));
  assert(n2->n_fanout == l_len(n2->out));
  assert(n2->n_fanin == l_len(n2->in));

  n1->n_fanout++;
  assert(!l_find(n1->out, n2->address));
  l_append(n1->out, n2->address);
  assert(l_find(n1->out, n2->address));

  n2->n_fanin++;
  assert(!l_find(n2->in, n1->address));
  l_append(n2->in, n1->address);
  assert(l_find(n2->in, n1->address));

  return;
}

/* attach n1 to n2 with the position of the input to n2 being inputnum */
void attach_pos(n1, n2, inputnum)
NODEPTR n1, n2;
int inputnum;
{
  LISTPTR half2;

  assert(n1 && n2);
  assert(n1->n_fanout == l_len(n1->out));
  assert(n1->n_fanin == l_len(n1->in));
  assert(n2->n_fanout == l_len(n2->out));
  assert(n2->n_fanin == l_len(n2->in));
  assert(inputnum >= 0);

  n1->n_fanout++;
  assert(!l_find(n1->out, n2->address));
  l_append(n1->out, n2->address);
  assert(l_find(n1->out, n2->address));

  n2->n_fanin++;
  assert(!l_find(n2->in, n1->address));
  half2 = l_cut(&n2->in, inputnum);
  l_append(n2->in, n1->address);
  if (half2) n2->in = l_cat(n2->in, half2);
  assert(l_find(n2->in, n1->address));

  return;
}

void detach(n1, n2)
NODEPTR n1, n2;
{
  LISTPTR found;

  assert(n1);
  assert(n1->n_fanout > 0);
  assert(n2);
  assert(n2->n_fanin > 0);

  found = l_find(n1->out, n2->address);
  assert(found);
  l_delete(&n1->out, found);
  assert(!l_find(n1->out, n2->address));
  n1->n_fanout--;

  found = l_find(n2->in, n1->address);
  assert(found);
  l_delete(&n2->in, found);
  assert(!l_find(n2->in, n1->address));
  n2->n_fanin--;

  return;
}

BOOLEAN attached(n1, n2)
NODEPTR n1, n2;
{
  assert(n1);
  assert(n2);
  if (l_find(n1->out, n2->address) && l_find(n2->in, n1->address))
    return(T);
  else return(F);
}

void expand_nodetable(newlist, n)
LISTPTR newlist;
int n;
{
  int i;
  LISTPTR l;
  NODEPTR *newtable = (NODEPTR *)NULL;

  assert(newlist);
  assert(n == l_len(newlist));

  assert(Nodecount > 0);
  assert(Nodecount != 1 || !Nodetable);
  assert(Nodecount == 1 || Nodetable);

  newtable = (NODEPTR *)SR_MALLOC((Nodecount+n)*sizeof(NODEPTR));
  newtable[0] = NODENULL;

#ifndef NDEBUG
  for (l = newlist; l != LISTNULL; l = l->next)
    assert(!l_find(l->next, l->o));
#endif

  /* transfer Nodetable to newtable */
  assert(!Nodetable || !Nodetable[0]);
  for (i = 1; i < Nodecount; i++) newtable[i] = Nodetable[i];
  for (i = Nodecount; i < Nodecount + n; i++) newtable[i] = NODENULL;
  if (Nodetable) sr_free((char *)Nodetable);
  Nodetable = newtable;
  newtable = (NODEPTR *)NULL;

  for (l = newlist; l != LISTNULL; l = l->next) {
    NODEPTR node = (NODEPTR)l->o;
    assert(node);
    assert(node->address >= Nodecount && node->address < Nodecount + n);
    assert(!Nodetable[node->address]);
    Nodetable[node->address] = node;
  }

  Nodecount += n;

  return;
}

void shrink_nodetable(count)
int *count;
{
  int i = Nodecount-1;

  while (i > 0 && !Nodetable[i]) i--;
  assert(i < Nodecount-1);

  Nodecount = i+1;

  if (Nodelist) fix_nodelist(count);

  return;
}

void fix_nodelist(count)
int *count;
{
  LISTPTR l, lf, f;
  NODEPTR n, nf;

  assert(count && Nodelist);
  *count = Nodecount;

  for (l = Nodelist; l != LISTNULL; l = l->next) {
    n = (NODEPTR)l->o;
    assert(n && n->address >= *count);

    for (lf = n->in; lf != LISTNULL; lf = lf->next) {
      assert(lf->o > 0);
      nf = get_node(lf->o, Nodelist);
      assert(nf);
      assert(nf->address == lf->o);
      f = l_find(nf->out, n->address);
      assert(f);
      f->o = *count;
    }

    for (lf = n->out; lf != LISTNULL; lf = lf->next) {
      assert(lf->o > 0);
      nf = get_node(lf->o, Nodelist);
      assert(nf);
      assert(nf->address == lf->o);
      f = l_find(nf->in, n->address);
      assert(f);
      f->o = *count;
    }

    if (n->type == PI || n->type == PPI) {
      f = l_find(Inputs, n->address);
      assert(f);
      f->o = *count;
      assert(n->address == *count || !l_find(Inputs, n->address));
    }

    if (n->type == VDD || n->type == VSS) {
      f = l_find(Constants, n->address);
      assert(f);
      f->o = *count;
      assert(n->address == *count || !l_find(Constants, n->address));
    }

    n->address = (*count)++;
  }

  assert(check_connectivity());

  return;
}

NODEPTR mkbuff(node, name, transfer_fanout)
NODEPTR node;
char *name;
BOOLEAN transfer_fanout;
{
  NODEPTR n;

  assert(node);

  n = nodecreate(Nodecount, name, NONT);
  n->func = GBUFF;
  n->blifunc = sr_strsav("1 1\n");

  assert(name || !n->name[0]);
  assert(n->name[0] || !name);

  /* put n into Nodetable */
  assert(!Nodelist);
  l_append(Nodelist, (int)n);
  expand_nodetable(Nodelist, 1);
  Nodelist = l_free(Nodelist);

  if (transfer_fanout) move_fanouts(node, n);

  attach(node, n);

  return(n);
}

void move_fanouts(n1, n2)
NODEPTR n1, n2;
{
  int len = l_len(Nodelist);
  LISTPTR l, f, find_next_dup();
  NODEPTR n;

  assert(n1);
  assert(n2);

  if (ispo(n1)) swap_names(n1, n2);

  for (l = n1->out; l != LISTNULL; l = l->next) {
    assert(l->o > 0 && l->o < Nodecount + len);
    n = get_node(l->o, Nodelist);
    assert(n);
    assert(n->address == l->o);
    f = l_find(n->in, n1->address);
    assert(f);
    f->o = n2->address;
  }

  assert(n1->n_fanout == l_len(n1->out));
  assert(n2->n_fanout == l_len(n2->out));

  n2->out = l_cat(n2->out, n1->out); n1->out = LISTNULL;
  n2->n_fanout += n1->n_fanout; n1->n_fanout = 0;

  assert(n1->n_fanout == l_len(n1->out));
  assert(n2->n_fanout == l_len(n2->out));

  l = find_next_dup(n2->out);
  if (l) {
    char buf[MAXSTRLEN];
    n = get_node(l->o, Nodelist);
    assert(n);
    assert(n->address == l->o);
    sprintf(buf, "duplicate input from gate %s to gate %s",n2->name,n->name);
    sr_error(buf);
  }

  return;
}

void insert_as_PO(n1, n2)
NODEPTR n1, n2;
{
  int len = l_len(Nodelist);
  LISTPTR l, f, find_next_dup();
  NODEPTR n;
  BOOLEAN connect = F;

  assert(n1);
  assert(n2);

  if (ispo(n1)) swap_names(n1, n2);

  for (l = n1->out; l != LISTNULL; l = l->next) {
    assert(l->o > 0 && l->o < Nodecount + len);
    if (l->o == POaddr) {
      connect = T;
      n = get_node(l->o, Nodelist);
      assert(n);
      assert(n->address == l->o);
      f = l_find(n->in, n1->address);
      assert(f);
      f->o = n2->address;
    }
  }
  assert(connect);

  assert(n1->n_fanout == l_len(n1->out));
  assert(n2->n_fanout == l_len(n2->out));
  l_append(n2->out, POaddr);
  l = l_find(n1->out, POaddr);
  assert(l);
  l_delete(&(n1->out), l);
  n2->n_fanout++; n1->n_fanout--;
  assert(n1->n_fanout == l_len(n1->out));
  assert(n2->n_fanout == l_len(n2->out));

  l = find_next_dup(n2->out);
  if (l) {
    char buf[MAXSTRLEN];
    n = get_node(l->o, Nodelist);
    assert(n);
    assert(n->address == l->o);
    sprintf(buf, "duplicate input from gate %s to gate %s",n2->name,n->name);
    sr_error(buf);
  }

  return;
}

int remove_buffers(count)
int *count;
{
  int i, retval = 0;

  extern NODEPTR rmbuff();

  for (i = 0; i < Nodecount; i++) {
    NODEPTR node = Nodetable[i];
    if (node && node->func == GBUFF) {
      assert(node->address == i);
      if (!(node = rmbuff(node))) retval++;
    }
  }

  if (!Nodetable[Nodecount-1]) shrink_nodetable(count);

  return(retval);
}

/* return node if the buffer was NOT removed, NODENULL if it was */
NODEPTR rmbuff(node)
NODEPTR node;
{
  int addr;
  NODEPTR inp = NODENULL, n;

  assert(node);

  addr = node->address;
  assert(addr > 0 && addr < Nodecount);
  assert(Nodetable[addr] == node);
  assert(node->type == NONT);
  assert(node->func == GBUFF);
  assert(node->in || !node->out);
  assert(!node->in || !node->in->next);
  assert(node->n_fanin == 0 || node->n_fanin == 1);
  assert(!node->in || node->in->o > 0);

  if (node->in) {
    inp = get_node(node->in->o, Nodelist);
    assert(inp);
    assert(inp->address == node->in->o);
  }

  if (!ispo(node) || !inp || !ispo(inp)) {
    if (inp) {
      detach(inp, node);
      assert(!node->n_fanin && !node->in);
      move_fanouts(node, inp);
    }

    /* delete node from Hashtable and Nodetable */
    assert(h_lookup(Hashtable, node->name));
    n = (NODEPTR)h_delete(Hashtable, node->name);
    assert(n == node);
    assert(node == Nodetable[addr]);
    Nodetable[addr] = node = n = nodefree(node);
  }

  return(node);
}

void fix_registers()
{
  extern int fix_regs();
  assert(check_connectivity());
  while (fix_regs()) assert(check_connectivity());
  if (!Nodetable[Nodecount-1]) shrink_nodetable((int *)NULL);
  return;
}

int fix_regs()
{
  int i, lowreg, count = 0;
  BOOLEAN haspo = F;
  NODEPTR node, n, n1, nlow;
  LISTPTR l, regs = LISTNULL, nonregs = LISTNULL;

  for (i = 0; i < Nodecount; i++) {
    node = Nodetable[i];
    assert(!node || l_len(node->out) == node->n_fanout);

    if (node && node != POnode && isppo(node) && node->n_fanout > 1) {
      assert(node->out);
      assert(!regs && !nonregs && !haspo);

      /* find the register with the lowest address (or PO if there is one) */
      for (l = node->out, lowreg = Nodecount; l != LISTNULL; l = l->next) {
        n = Nodetable[l->o];
        assert(l->o > 0 && l->o < Nodecount);
        assert(n);
        if (n->type == PPI) {
          if (!haspo) {
            if (!ispo(n)) {
              if (l->o < lowreg) lowreg = l->o;
            } else {
              haspo = T; lowreg = l->o;
            }
          } else {
            /* should never have more than one register PO */
            assert(!ispo(n));
            /* if (ispo(n) && l->o < lowreg) lowreg = l->o; */
          }
        }
      }

      nlow = Nodetable[lowreg];

      /* separate fanout into regs and nonregs */
      for (l = node->out; l != LISTNULL; l = l->next) {
        n = Nodetable[l->o];
        if (n->type == PPI) l_append(regs, l->o);
        else l_append(nonregs, l->o);
      }

      assert(l_find(regs, lowreg));

      if (regs && regs->next) {
        for (l = regs; l != LISTNULL; l = l->next) {
          n = Nodetable[l->o];
          assert(l->o > 0 && l->o < Nodecount);
          assert(n);
          if (l->o != lowreg) {
            assert(n != nlow);
            /* if (ISfile) {
              // fix initial state circuit //
              assert(ISfile && nlow->sisnode && n->sisnode);
              fprintf(ISfile, "%s = BUFF(%s)\n", n->sisnode, nlow->sisnode);
              assert(l_find(ISinputs, (int)n));
              l_delete(&ISinputs, l_find(ISinputs, (int)n));
              assert(!l_find(ISinputs, (int)n));
            } */
            move_fanouts(n, nlow);
            detach(node, n);
            n1 = (NODEPTR)h_delete(Hashtable, n->name);
            assert(n1 == n);
            assert(!n->n_fanin && !n->n_fanout && !n->in && !n->out);
            Nodetable[l->o] = n1 = n = nodefree(n);
            count++;
          }
        }
      }

      /* reset variables for next iteration */
      regs = l_free(regs);
      nonregs = l_free(nonregs);
      haspo = F;
    }
  }

  return(count);
}

void display_lineno(fpin, line)
int line;
FILE *fpin;
{
  static int lineno_charcount = 0;

  if (!lineno_charcount) {
    fprintf(stdout, "# line ");
    lineno_charcount = fprintf(stdout, "%d", line);
  } else {
    int i;
    for (i = 0; i < lineno_charcount; i++) fprintf(stdout, KL);
    fprintf(stdout, CE);
    lineno_charcount = fprintf(stdout, "%d", line);
  }

  if (!fpin) {
    fprintf(stdout, "\n");
    lineno_charcount = 0;
  }

  fflush(stdout);

  return;
}

double get_time(who)
int who;
{
  double time;
  struct rusage ts[1];

  assert(who == RUSAGE_SELF || who == RUSAGE_CHILDREN);

  getrusage(who, ts);
  time = (double)ts->ru_utime.tv_sec + (double)ts->ru_utime.tv_usec/1000000.0;

  return(time);
}

void free_all()
{
  int i;

  /* free Circuit_Filename and Circuit_Basename */
  assert(Circuit_Filename);
  Circuit_Filename = sr_free(Circuit_Filename);
  assert(Circuit_Basename);
  Circuit_Basename = sr_free(Circuit_Basename);

  /* free Hashtable */
  Hashtable = h_freetable(Hashtable);

  /* free Nodetable */
  for (i = 0; i < Nodecount; i++) {
    if (Nodetable[i]) {
      Nodetable[i]->n_fanin = Nodetable[i]->n_fanout = 0;
      Nodetable[i]->in = l_free(Nodetable[i]->in);
      Nodetable[i]->out = l_free(Nodetable[i]->out);
      Nodetable[i] = nodefree(Nodetable[i]);
    }
  }
  Nodetable = (NODEPTR *)sr_free((char *)Nodetable);
  Nodecount = 0;

  if (Node2vertex) Node2vertex = (int *)sr_free((char *)Node2vertex);
  if (Vertex2node) Vertex2node = (int *)sr_free((char *)Vertex2node);
  Nvertices = 0;

  /* l_end() MUST be called LAST */
  node_end();
  edge_end();
  str_end();
  l_end();

  if (sr_tmfile) sr_tmfile = sr_fclose(sr_tmfile);

  return;
}

int isppo(n)
NODEPTR n;
{
  int retval = 0;
  LISTPTR l;

  assert(n);

  for (l = n->out; l != LISTNULL && !retval; l = l->next) {
    assert(l->o > 0 && l->o < Nodecount);
    assert(Nodetable[l->o]);
    if (Nodetable[l->o]->type == PPI) retval = 1;
  }

  return(retval);
}


LISTPTR find_next_dup(list)
LISTPTR list;
{
  LISTPTR f = LISTNULL;
  for (; list && !f; list = list->next) f = l_find(list->next, list->o);
  return(f);
}

void swap_names(n1, n2)
NODEPTR n1, n2;
{
  char tmp[PLEN];
  NODEPTR n;

  /* swap names of n1 and n2 */
  n = (NODEPTR)h_delete(Hashtable, n1->name);
  assert(n == n1);

  n = (NODEPTR)h_delete(Hashtable, n2->name);
  assert(n == n2);

  strcpy(tmp, n1->name);
  strcpy(n1->name, n2->name);
  strcpy(n2->name, tmp);

  h_add(Hashtable, n1->name, (char *)n1);
  h_add(Hashtable, n2->name, (char *)n2);

  assert((NODEPTR)h_lookup(Hashtable, n1->name) == n1);
  assert((NODEPTR)h_lookup(Hashtable, n2->name) == n2);

  return;
}

void give_names_to_registers()
{
  int i, is;
  NODEPTR n;

  assert(!Nodelist);
  assert(ISfile);

  for (i = 0; i < Nodecount; i++) {
    n = Nodetable[i];
    if (n && n->type == PPI) {
      fprintf(ISfile, "OUTPUT(%s)\n", n->name);
      /* l_append(ISinputs, (int)n); */
    }
  }

  fprintf(ISfile, "\n");

  for (i = 0; i < Nodecount; i++) {
    n = Nodetable[i];
    if (n && n->type == PPI) {
      is = n->initstate;
      fprintf(ISfile,"REQD%c(%s)\n", is==0 ? '0' : is==1 ? '1' : 'X', n->name);
    }
  }

  fprintf(ISfile, "\n");

  for (i = 0; i < Nodecount; i++) {
    n = Nodetable[i];
    if (n && n->type == PPI) {
      name_register(n);
      fprintf(ISfile, "%s = BUFF(%s)\n", n->name, n->sisnode);
    }
  }

  fflush(ISfile);

  return;
}

void name_register(n)
NODEPTR n;
{
  char buf[MAXSTRLEN];

  assert(n && n->type == PPI && n->func == GDFF && !n->sisnode);

  sprintf(buf, "IS%d", IScount++);
  n->sisnode = sr_strsav(buf);

  return;
}

char *func2str(node)
NODEPTR node;
{
  static char funcbuf[MAXSTRLEN];
  char cmplx[MAXSTRLEN];
  int i;

  assert(node);

  switch (node->func) {
    case GPI:      strcpy(funcbuf, "INPT");    break;
    case GAND:     strcpy(funcbuf, "AND");     break;
    case GNAND:    strcpy(funcbuf, "NAND");    break;
    case GOR:      strcpy(funcbuf, "OR");      break;
    case GNOR:     strcpy(funcbuf, "NOR");     break;
    case GXOR:     strcpy(funcbuf, "XOR");     break;
    case GXNOR:    strcpy(funcbuf, "XNOR");    break;
    case GBUFF:    strcpy(funcbuf, "BUFF");    break;
    case GNOT:     strcpy(funcbuf, "NOT");     break;
    case GDFF:     strcpy(funcbuf, "DFF");     break;
    case GIOBF:    strcpy(funcbuf, "IOBF");    break;
    case GST3:     strcpy(funcbuf, "ST3");     break;
    case GZERO:    strcpy(funcbuf, "ZERO");    break;
    case GONE:     strcpy(funcbuf, "ONE");     break;
    case GCOMPLEX:
      strcpy(cmplx, node->blifunc);
      for (i = 0; cmplx[i]; i++) if (cmplx[i] == '\n') cmplx[i] = '\\';
      sprintf(funcbuf, "[%s]", cmplx);
      break;
    case NOFUNC:   strcpy(funcbuf, "UNKNOWN"); break;
    default:       assert_false();             break;
  }

  return(funcbuf);
}

int printl(list) 
LISTPTR list; 
{
 l_print(list, stdout); 
 return(1); 
}

int printl_names(list)
LISTPTR list;
{
  l_oprint(list, stdout, printstr, " ", 0, 0);
  fprintf(stdout, "\n"); fflush(stdout);
  return(1);
}

