/********************************************************************/
/*This routine estimates the minimum supply voltage at which the DFG
 *be implemented to still meet the specified sample period constraint.
 *First, we perform a module selection that maps each operation to the
 *fastest functional unit template that can perform it. Then, the longest
 *path in the DFG is found (by adding up delays of operations along the
 *path). The voltage at which the longest path delay just meets the
 *given sample period constraint gives the minimum supply voltage.
 */
const float scheduler::compute_min_vdd(dfg *flowgraph, library *lib,
			     const float sample_period)
{
  float min_voltage, tmp_clock;
  float execution_period;

  assert_force(flowgraph && lib);
  assert_force(sample_period > 0.0);

  //map to fastest modules
  fastest_map(flowgraph, lib);
  FOR_EACH_NODEINDEX(*flowgraph, i) {
    node =flowgraph-> getnthnode(i);
    assert(node);
    libel = lib->get_element(node->moduletype);
    if(libel->get_latency(5.0) < tmp_clock) {
      tmp_clock = libel->get_delay(5.0);
    }
  }
  asap_schedule(flowgraph, lib, 5.0, tmp_clock);

  min_voltage = compute_vdd(reference_vdd, delay_ratio);

  assert_force(min_voltage > 0.0 && min_voltage <= 5.0);
  assert_force(min_voltage/VDD_GRANULARITY ==  (float)((int)(min_voltage/VDD_GRANULARITY)));

  flowgraph->reset_scheduling_info();
  flowgraph->reset_allocation_info();

  return(min_voltage);
}
/********************************************************************/
const float scheduler::find_longest_path(dfg *flowgraph,library *lib)
{
  register EDGEPTR edge;
  register NODEPTR node;
  array<float> atimes;
  list_iterator<EDGEPTR> edgescan;
  list_iterator<NODEPTR> nodescan;
  float max_atime;
  register int i;

  assert_force(flowgraph && lib);
  //assert that the DFG has been levelized
  assert(flowgraph->levellists.getsize() == flowgraph->numnodes() &&
	 flowgraph->maxlevel < flowgraph->levellists.getsize());

  atimes.resize(flowgraph->numedges());
  atimes.reset(-1.0);

  //set arrival times of all primary inputs to 0.0
  FOR_EACH_LISTNODE(flowgraph->inputs,edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    assert(edge->address >= 0 && edge->address < flowgraph->numedges());
    assert(edge->inputtype == PI);
    atimes[edge->address] = 0.0;
  }

  //set arrival times of all primary inputs to 0.0
  FOR_EACH_LISTNODE(flowgraph->constants,edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    assert(edge->address >= 0 && edge->address < flowgraph->numedges());
    assert(edge->inputtype == CONSTANT_);
    atimes[edge->address] = 0.0;
  }

  //set arrival times of all loopins to 0.0
  FOR_EACH_LISTNODE(flowgraph->loopins,edgescan) {
    edge = edgescan.get_item();
    assert(edge);
    assert(edge->address >= 0 && edge->address < flowgraph->numedges());
    assert(edge->inputtype == LOOPIN);
    atimes[edge->address] = 0.0;
  }

  for(i = 0; i <= flowgraph->maxlevel; i++) {
    FOR_EACH_LISTNODE(flowgraph->levellists[i], nodescan) {
      node = nodescan.get_item();

      max_atime = -1.0;
      FOR_EACH_FANIN_EDGE(node, edgescan) {
	edge = edgescan.get_item();
	assert(edge);
	assert(edge->address >= 0 && edge->address < flowgraph->numedges());
	assert(atimes[edge->address] >= 0.0);
	max_atime = MAX(max_atime, atimes[edge->address]);
      }	

      edge = node->output_edge();
      assert(edge);
      assert(edge->address >= 0 && edge->address < flowgraph->numedges());
      assert(atimes[edge->address] == -1.0);
      assert(max_atime >= 0.0);
      atimes[edge->address] = max_atime + ((lib->get_element(node->moduletype))->get_delay());
    }
  }

  max_atime = -1.0;
  FOR_EACH_EDGEINDEX(*flowgraph, i) {
    edge = flowgraph->getnthedge(i);
    assert(edge && edge->address == i);
    max_atime = MAX(max_atime, atimes[edge->address]);
  }

  assert(max_atime >= 0.0);
  return(max_atime);
}
/********************************************************************/
/*This routine searches for the class A move that results in the maximum
 *gain, i.e., the maximum decrease in the estimated switched capacitance.
 *The gain is evaluated using the switched capacitance matrices for
 *functional units.
 */
void scheduler::compute_best_class_a_move(dfg *flowgraph, library *lib, datapath *dp, 
					  class_a_move &best_move,array<Boolean> &visited)
{
  float best_gain, cur_gain;

  best_gain = -HUGEFLOAT;
  FOR_EACH_FUNCTIONAL_UNIT(datapath, fu) {
    if(fu has been visited) {
      continue;
    }
    for_each_new_type_for_fu {
      //move refers to changing the type of fu from its old type to the new type
      compute_gain(flowgraph, lib, dp, move);
      compute cur_gain for the move as the sw cap for fu - that for the new fu;
      if(cur_gain > best_gain) {
	if(reschedule(flowgraph, lib, dp, move, F)) {
	  best_gain = cur_gain;
	  set best_move to the current move;
	}
      }
    }
  }

  return;
}
/********************************************************************/
/*BEGIN UNUSED*/
  for(level = 0; level <= flowgraph->maxlevel; level++) {
    FOR_EACH_LISTNODE(flowgraph->levellists[level], node_liter) {
      node = node_liter.get_item();
      assert(node);

      /*node's birth is computed as the max. of the deaths of its fanins*/
      maxcycle = 0;
      FOR_EACH_FANIN_EDGE(node, edgescan) {
	edge = edgescan.get_item();
	assert(edge);
	innode = edge->input_node();
	assert((edge->is_dfginput() == T && !innode) ||
	       (edge->is_dfginput() == F && innode));
	if(innode) {
	  index = innode->get_address();
	  assert(nodebirths[index] >= 0 &&
		 nodedeaths[index] > nodebirths[index]);
	  maxcycle = MAX(maxcycle, nodedeaths[index]);
	}
      }
      nodebirths[node->get_address()] = maxcycle;

      /*node's death is computed as its birth plus the number of cycles
       *taken for it.
       */
      libel = lib->get_element(info.moduletypes[index]);
      assert(libel);
      /*ANAND 4/5/94*/
      //ASSUMES THAT THE REGISTER, MUX, WIRING DELAY ESTIMATES ARE INCLUDED IN THE
      //FU ESTIMATES - NEED TO REFINE
      nodedeaths[index] = nodebirths[index] + libel->get_total_csteps(vdd,clk);
    }
  }
/*END UNUSED*/
/********************************************************************/
    //Add new interconnect units to feed new_fu, and delete the interconnect
    //units that feed move.fu1 and move.fu2

    //NOTE: assumes only 1 output port, and NO control ports
    num_fu_input_ports = new_fu->get_libelement()->number_of_input_ports();

    for(j = 1; j <= num_fu_input_ports; j++) {
      //create the interconnect unit that feeds the jth input port of new_fu
      mem_ok( intcon_unit = new Mux(lib->get_mux(target_dfg.get_bitwidth())) );
      target_dp.add_interconnect_unit(intcon_unit);
      mem_ok( tmpnet = new Net );
      sprintf(portname, "in%d", j);
      tmpnet->connect_to_port(new_fu, portname);
      new_fu->connect_to_net(tmpnet, portname);
      tmpnet->connect_to_port(intcon_unit, "out");
      intcon_unit->connect_to_net(tmpnet, "out");
      target_dp.add_net(tmpnet);

      FOR_EACH_LISTNODE(new_fu->get_operations(), nodescan) {
	node = nodescan.get_item();
	assert(node);
	assert(node->number_input_edges() == num_fu_input_ports);
	assert(node->get_functional_unit() == new_fu);

	edge = (node->get_input_edges())[j-1];
	assert(edge);
	new_reg = edge->get_storage_unit();
	assert(new_reg);

	//If no input port of intcon_unit is already connected to the output of su,
	//create a new input port and do so
	tmpnet = new_reg->get_net("out");
	assert(tmpnet);

	if(!tmpnet->is_connected_to_input_port_by_name(intcon_unit)) {
	  sprintf(portname, "in%d", intcon_unit->get_number_of_nets());
	  tmpnet->connect_to_port(intcon_unit, portname);
	  intcon_unit->connect_to_net(tmpnet, portname);
	}

      } /*END FOR EACH OPERATION MAPPED TO new_fu*/
    }/*END FOR j = 1 TO num_fu_input_ports*/
