/********************************************************************/
/*        FILE: dfgval.C                                            */
/********************************************************************/
#include "dfgval.h"

int Dfg_value::bits_per_digit = 8*sizeof(unsigned int);

/********************************************************************/
/*If the bitwidth is NOT a multiple of bits_per_digit, the
 *MS Digit will have some unused bits. This function resets
 *these unused bits to 0.
 */
void Dfg_value::truncate() {
  register int msd_index, i;

  FRITS_SET_MESSAGE("truncate");
  msd_index = digits.get_size();
  i = bitwidth % bits_per_digit;
  digits[msd_index] = digits[msd_index] % (1 << i);
  return;
}
/********************************************************************/
/*Given two digits and a carryin, this routine computes the carry out
 *of this stage.
 */
unsigned int compute_carry(unsigned int x, unsigned int y,
			   unsigned int carryin)
{
  assert(carryin == 0 || carryin == 1);
  unsigned int result = 0;

  if(x == ~0) {
    if(y > 0 || carryin > 0) {
      result = 1;
    }
  } else {
    if((x + carryin) > ~y) {
      result = 1;
    }
  }
  return(result);
}
/********************************************************************/
/*  val1 + val2
 *The result has bitwidth equal to MAX(bw1, bw2) + 1, where bw1 and
 *bw2 are the bitwidths of val1 and val2. Addition is performed
 *digit-wise for efficiency. It is assumed that the machine add
 *instruction truncates the result in the case of overflow (e.g.,
 *if bits_per_digit is 32, and we add (2^32)-1 and 1, we get 0.
 *This assumption is required to avoid performing bit-wise addition.
 */
Dfg_value operator +(Dfg_value &val1, Dfg_value &val2)
{
  register int i, j, k;
  unsigned int result_bw;
  unsigned int carry;

  assert(val1.bitwidth > 0 && val2.bitwidth > 0);

  result_bw = MAX(val1.bitwidth, val2.bitwidth) + 1;
  Dfg_value result(result_bw);

  //Perform addition, digit-by-digit from the LS Digit to the MS Digit.
  carry = 0;
  i = (val1.digits.get_size())-1;
  j = (val2.digits.get_size())-1;
  k = (result.digits.get_size())-1;
  for(; i >= 0 && j >= 0; i--,j--,k--) {
    carry = compute_carry(val1.digits[i], val2.digits[j], carry);
    result.digits[k] = val1.digits[i] + val2.digits[j] + carry;
  }

  //If one of the values has a larger number of digits than the other,
  //fill in the gap
  assert(i < 0 || j < 0);
  for( ; i >= 0 || j >= 0; i--,j--,k--) {
    if(i >= 0) {
      carry = compute_carry(val1.digits[i], 0, carry);
    } else if(j >= 0) {
      carry = compute_carry(0, val2.digits[j], carry);
    }
  }

  //handle the case when the result can have 1 digit more than either operand
  assert(k <= 0);
  if(k == 0) {
    result.digits[k] = carry;
  }

  return(result);
}
/********************************************************************/
/*Given two digits a and b, and a borrowin, this routine computes
 *whether a borrow 
 */
unsigned int compute_borrow(unsigned int a, unsigned int b,
			    unsigned int borrowin)
{
  unsigned int borrowout = 0;

  if(b == ~0) {
    if(borrowin == 1) {
      borrowout = 1;
    }
  } else {
    if(a < (b + borrowin)) {
      borrowout = 1;
    }
  }
  return(borrowout);
}
/********************************************************************/
/*  val1 - val2
 *This routine assumes that
 *(i) val1 has a bitwidth that is lesser than that of val2
 *(ii) val1 CANNOT be less than val2, i.e, the result cannot be negative.
 *This is necessary due to the assumption that negative numbers are not handled
 *in the number representation. Subtraction is also performed digit-wise,
 *for efficiency. Again, it is assumed that if the result of using the machine
 *subtract instruction is negative, say -n where n is positive,
 *the result is stored as (2^bits_per_digit)-n.
 */
Dfg_value operator -(Dfg_value &val1, Dfg_value &val2)
{
  unsigned int i, j;
  unsigned int borrow;

  assert(val2.bitwidth > 0 && val1.bitwidth >= val2.bitwidth);
  Dfg_value result(val1.bitwidth);

  //Perform subtraction, digit-by-digit from the LS Digit to the MS Digit.
  //At any digit, if the first operand's digit is less than the second
  //operand's digit, a borrow is generated.
  borrow = 0;
  i = val1.digits.get_size()-1;
  j = val2.digits.get_size()-1;
  for(; i >= 0 && j >= 0; i--,j--) {
    result.digits[i] = val1.digits[i] - val2.digits[j] - borrow;
    borrow = compute_borrow(val1.digits[i], val2.digits[j], borrow);
  }

  //if val1 has a larger number of digits than val2, fill in the remaining
  for( ; i >= 0; i--) {
    result.digits[i] = val1.digits[i] - borrow;
    borrow = compute_borrow(val1.digits[i], 0, borrow);
  }

  //ensure that no borrow out occured at the MS Digit, ie, that
  //the result is not negative
  if(borrow) {
    cerr << "ERROR: Underflow during simulation, negative values not allowed";
    exit(-1);
  }
  return(result);
}
/********************************************************************/
/*  val1 * val2
 *The result has bitwidth bw1 + bw2, where bw1 and bw2 are the
 *bitwidths of val1 and val2, respectively.
 */
Dfg_value operator *(Dfg_value &val1, Dfg_value &val2)
{
  register int i;
  assert(val1.bitwidth > 0 && val2.bitwidth > 0);

  Dfg_value result(val1.bitwidth + val2.bitwidth);
  Dfg_value partial_product(val1.bitwidth + val2.bitwidth);
  
  for(i = val2.bitwidth-1; i >= 0; i--) {
    if(val2.get_bit(i)) {
      partial_product = val1 << i;
      result = result + partial_product;
    }
  }
}
/********************************************************************/
/*Left shifts a Dfg_value by "shift" bit-positions.
 *The bitwidth of the result is bw + shift, where bw is
 *the bitwidth of the original value being shifted
 */
Dfg_value operator <<(Dfg_value &val, const unsigned int shift)
{
  register int i;

  assert(shift >= 0);
  Dfg_value result(val.bitwidth + shift);

  //reset the LS Bits to 0's
  for(i = result.bitwidth-1; i >= (result.bitwidth-(1+shift)); i--) {
    result.set_bit(i, 0);
  }

  //fill in the remaining bits of result with corresponding bits from val
  for( ; i >= 0; i--) {
    result.set_bit(i, val.get_bit(i));
  }

  return(result);
}
/********************************************************************/
/*  val1 < val2
 *The result has a bitwidth of 1 (since it can be either 0 or 1.
 */
Dfg_value operator <(Dfg_value &val1, Dfg_value &val2)
{
  register int i, j;
  Dfg_value result(1);
  BOOLEAN done;

  i = 0; j = 0;
  //First, handle the extra bits in val1 or val2
  if(val1.digits.get_size() >= val2.digits.get_size()) {
    for( ; i <= (val1.digits.get_size() - val2.digits.get_size())-1; i++) {
      if(val1.digits[i] > 0) {
	result.set_bit(0, 0);
	done = T;
	break;
      }
    }
  } else {
    for( ; j <= (val2.digits.get_size() - val1.digits.get_size())-1; j++) {
      if(val2.digits[j] > 0) {
	result.set_bit(0, 1);
	done = T;
	break;
      }
    }
  }

  //If the result has not been decided by now, go on to compare
  //the common bits of val1 and val2
  if(!done) {
    for( ; i < val1.digits.get_size() && j < val2.digits.get_size();
	i++, j++) {
      if(val1.digits[i] < val2.digits[j]) {
	result.set_bit(0, 1);
	done = T;
	break;
      } else if(val1.digits[i] > val2.digits[j]) {
	result.set_bit(0, 0);
	done = T;
	break;
      }
    }
  }

  //if the result has not been decided by now, it means that
  //val1 and val2 are equal
  if(!done) {  //val1 and val2 are equal
    assert(i == val1.digits.get_size() && j == val2.digits.get_size());
    result.set_bit(0, 0);
    done = T;
  }

  return(result);
}
/********************************************************************/
/*  val1 > val2
 *The result has a bitwidth of 1 (since it can be either 0 or 1).
 */
Dfg_value operator >(Dfg_value &val1, Dfg_value &val2)
{
  register int i, j;
  Dfg_value result(1);
  BOOLEAN done;

  i = 0; j = 0;
  //First, handle the extra bits in val1 or val2
  if(val1.digits.get_size() >= val2.digits.get_size()) {
    for( ; i <= (val1.digits.get_size() - val2.digits.get_size())-1; i++) {
      if(val1.digits[i] > 0) {
	result.set_bit(0, 1);
	done = T;
	break;
      }
    }
  } else {
    for( ; j <= (val2.digits.get_size() - val1.digits.get_size())-1; j++) {
      if(val2.digits[j] > 0) {
	result.set_bit(0, 0);
	done = T;
	break;
      }
    }
  }

  //If the result has not been decided by now, go on to compare
  //the common bits of val1 and val2
  if(!done) {
    for( ; i < val1.digits.get_size() && j < val2.digits.get_size();
	i++, j++) {
      if(val1.digits[i] < val2.digits[j]) {
	result.set_bit(0, 0);
	done = T;
	break;
      } else if(val1.digits[i] > val2.digits[j]) {
	result.set_bit(0, 1);
	done = T;
	break;
      }
    }
  }

  //if the result has not been decided by now, it means that
  //val1 and val2 are equal
  if(!done) {  //val1 and val2 are equal
    assert(i == val1.digits.get_size() && j == val2.digits.get_size());
    result.set_bit(0, 0);
    done = T;
  }

  return(result);
}
/********************************************************************/
/*  val1 == val2
 *The result has a bitwidth of 1 (since it can be either 0 or 1).
 */
Dfg_value operator ==(Dfg_value &val1, Dfg_value &val2)
{
  register int i, j;
  Dfg_value result(1);
  BOOLEAN done;

  i = 0; j = 0;
  //First, handle the extra bits in val1 or val2
  if(val1.digits.get_size() >= val2.digits.get_size()) {
    for( ; i <= (val1.digits.get_size() - val2.digits.get_size())-1; i++) {
      if(val1.digits[i] > 0) {
	result.set_bit(0, 0);
	done = T;
	break;
      }
    }
  } else {
    for( ; j <= (val2.digits.get_size() - val1.digits.get_size())-1; j++) {
      if(val2.digits[j] > 0) {
	result.set_bit(0, 0);
	done = T;
	break;
      }
    }
  }

  //If the result has not been decided by now, go on to compare
  //the common bits of val1 and val2
  if(!done) {
    for( ; i < val1.digits.get_size() && j < val2.digits.get_size();
	i++, j++) {
      if(val1.digits[i] != val2.digits[j]) {
	result.set_bit(0, 0);
	done = T;
	break;
      }
    }
  }

  //if the result has not been decided by now, it means that
  //val1 and val2 are equal
  if(!done) {  //val1 and val2 are equal
    assert(i == val1.digits.get_size() && j == val2.digits.get_size());
    result.set_bit(0, 1);
    done = T;
  }

  return(result);
}
/********************************************************************/
/*  val1 != val2
 *The result has a bitwidth of 1 (since it can be either 0 or 1).
 *This operator uses the already defined == operator.
 */
Dfg_value operator !=(Dfg_value &val1, Dfg_value &val2)
{
  Dfg_value result(1);

  result = (val1 == val2);

  if(result.get_bit(0) == 0) {
    result.set_bit(0, 1);
  } else {
    result.set_bit(0, 0);
  }

  return(result);
}
/********************************************************************/
/* val1 & val2
 *The bitwidth of the result is equal to MAX(bw1, bw2) where
 *bw1 and bw2 are the bitwidths of val1 and val2, respectively.
 */
Dfg_value operator &(Dfg_value &val1, Dfg_value &val2)
{
  register int i, j;

  assert(val1.bitwidth > 0 && val2.bitwidth > 0);
  Dfg_value result(MAX(val1.bitwidth, val2.bitwidth));

  i = 0; j = 0;
  if(val1.digits.get_size() >= val2.digits.get_size()) {
    //First, handle the extra digits in val1
    for( ; i <= (val1.digits.get_size() - val2.digits.get_size())-1; i++) {
      result.digits[i] = 0;
    }
    //Next, handle the rest of the digits
    for( ; i < val1.digits.get_size(); i++, j++) {
      assert(j < val2.digits.get_size());
      result.digits[i] = val1.digits[i] & val2.digits[j];
    }
  } else {
    //First, handle the extra digits in val2
    for( ; j <= (val2.digits.get_size() - val1.digits.get_size())-1; j++) {
      result.digits[j] = 0;
    }
    //Next, handle the rest of the digits
    for( ; j < val2.digits.get_size(); j++, i++) {
      assert(i < val1.digits.get_size());
      result.digits[j] = val1.digits[i] & val2.digits[j];
    }
  }

  return(result);
}
/********************************************************************/
/* val1 | val2
 *The bitwidth of the result is equal to MAX(bw1, bw2) where
 *bw1 and bw2 are the bitwidths of val1 and val2, respectively.
 */
Dfg_value operator |(Dfg_value &val1, Dfg_value &val2)
{
  register int i, j;

  assert(val1.bitwidth > 0 && val2.bitwidth > 0);
  Dfg_value result(MAX(val1.bitwidth, val2.bitwidth));

  i = 0; j = 0;
  if(val1.digits.get_size() >= val2.digits.get_size()) {
    //First, handle the extra digits in val1
    for( ; i <= (val1.digits.get_size() - val2.digits.get_size())-1; i++) {
      result.digits[i] = val1.digits[i];
    }
    //Next, handle the rest of the digits
    for( ; i < val1.digits.get_size(); i++, j++) {
      assert(j < val2.digits.get_size());
      result.digits[i] = val1.digits[i] & val2.digits[j];
    }
  } else {
    //First, handle the extra digits in val2
    for( ; j <= (val2.digits.get_size() - val1.digits.get_size())-1; j++) {
      result.digits[j] = val2.digits[j];
    }
    //Next, handle the rest of the digits
    for( ; j < val2.digits.get_size(); j++, i++) {
      assert(i < val1.digits.get_size());
      result.digits[j] = val1.digits[i] | val2.digits[j];
    }
  }

  return(result);
}
/********************************************************************/
/* ~val1
 *This operator implements the bitwise complement operator.
 *The bitwidth of the result is the same as that of the operand.
 */
Dfg_value operator ~(Dfg_value &val)
{
  register int i;

  assert(val.bitwidth > 0);
  Dfg_value result(val.bitwidth);

  //perform the ones complement digit-wise
  for(i = 0; i < val.digits.get_size(); i++) {
    result.digits[i] = ~(val.digits[i]);
  }

  //reset any extra 1s introduced in the unused MSBs
  result.truncate();

  return(result);
}
/********************************************************************/
void Dfg_value::display(ostream &output)
{
  register int i;
  for(i = 0; i < digits.get_size(); i++) {
    output << digits[i];
  }
}
/********************************************************************/
