/********************************************************************/
/*        FILE: dfgval.h                                            */
/********************************************************************/

#ifndef ANAND_DFGVAL_H
#define ANAND_DFGVAL_H

#include "array.h"

/********************************************************************/
#ifdef _SCALP_
class Dfg_value : public Assurance {
#else
class Dfg_value {
#endif
  friend Dfg_value operator +(Dfg_value &, Dfg_value &);
  friend Dfg_value operator -(Dfg_value &, Dfg_value &);
  friend Dfg_value operator *(Dfg_value &, Dfg_value &);
  friend Dfg_value operator <<(Dfg_value &, const unsigned int);
  friend Dfg_value operator <(Dfg_value &, Dfg_value &);
  friend Dfg_value operator <=(Dfg_value &, Dfg_value &);
  friend Dfg_value operator >(Dfg_value &, Dfg_value &);
  friend Dfg_value operator >=(Dfg_value &, Dfg_value &);
  friend Dfg_value operator ==(Dfg_value &, Dfg_value &);
  friend Dfg_value operator !=(Dfg_value &, Dfg_value &);
  friend Dfg_value operator &(Dfg_value &, Dfg_value &);
  friend Dfg_value operator |(Dfg_value &, Dfg_value &);
  friend Dfg_value operator ~(Dfg_value &);
private:
  static int bits_per_digit;
  unsigned int bitwidth;
  Array<unsigned int> digits;

  //set a given bit position
  inline void set_bit(unsigned int position, unsigned int bitval) {
    register unsigned int digit_index, digit_position;

    assert(position < bitwidth);
    assert(bitval == 0 || bitval == 1);
    digit_index = position/bits_per_digit;
    digit_position = position % bits_per_digit;
    if(bitval) {
      digits[digit_index] = digits[digit_index] | (1 << digit_position);
    } else {
      digits[digit_index] = digits[digit_index] & (~(1 << digit_position));
    }
    return;
  }

  //retrieve the value at a given bit powition
  inline unsigned int get_bit(unsigned int position) {
    register unsigned int digit_index, digit_position;

    assert(position < bitwidth);
    digit_index = position/bits_per_digit;
    digit_position = position % bits_per_digit;
    if(digits[digit_index] & (1 << digit_position)) {
      return 1;
    } else {
      return 0;
    }
  }

  //In cases where the bitwidth is NOT a multiple of bits_per_digit,
  //this function fills in the unused bits in the MS Digit to zero
  void truncate();
  
public:
  Dfg_value(const unsigned int bitwidth) : digits() {
    int size;

    FRITS_SET_CLASS("Dfg_value");
    assert(bitwidth > 0);
    size = bitwidth/bits_per_digit;
    if(bitwidth % bits_per_digit) size++;
    digits.resize(size);
  }
  ~Dfg_value() {
    //the destructor for the digits array gets automatically called
    //from here.
  }

  //set the values of the digits from a given Array
  inline void set_digits(Array<unsigned int> &newdigits) {
    assert(newdigits.get_size() == digits.get_size());
    digits.copy(newdigits);
  }

  //assign another Dfg_value (source_val) to this Dfg_value
  //If source_val has a smaller bitwidth than this Dfg_value,
  //the extra MSB's are filled with 0's
  //If source_val has a larger bitwidth than this Dfg_value,
  //the extra MSB's are truncated
  inline Dfg_value &operator = (Dfg_value &source_val) {
    register int i, j;

    //special case of assigning a value to itself
    if(this == &source_val) {
      return *this;
    } else {
      //copy beginning from the LS Digit till
      //one of the two arrays is exhausted
      for(i = digits.get_size()-1, j = source_val.digits.get_size()-1;
	  i >= 0 && j >= 0; i--,j--) {
	digits[i] = source_val.digits[i];
      }
      //if unassigned digits remain in this Dfgedge_value,
      //force them to 0
      for( ; i >= 0; i--) {
	digits[i] = 0;
      }
      //truncate this Dfg_value by forcing the
      //appropriate bits in the MS Digit to 0
      truncate();
    }
    return(*this);
  }

  //print out the contents of this Dfg_value in Hex
  void display(ostream & = cout);
};
/********************************************************************/
//Syntactic sugar for displaying a Dfg_value
inline ostream &operator <<(ostream &output, Dfg_value &val)
{val.display(output);return(output);}
#endif
