/********************************************************************/
/*              FILE: scm.c                                         */
/********************************************************************/
#include "scm.h"

/********************************************************************/
/*This routine performs a simulation of the given DFG, to extract
 *the switched capacitance matrices. The vectors are read from the
 *file that has the given name. If no file with the given name
 *exists, RANDOM vectors are used.
 */
void Scm::extract_scmatrices(Dfg &flowgraph, library *lib, char *vecfile)
{
  register int i,j,k;
  register NODEPTR node1, node2;
  register EDGEPTR edge, edge1, edge2;
  extern FILE *vec_in;
  Boolean use_random_vectors;
  Boolean first_iteration;
  libelement *libel;
  Array<unsigned int> *vals1, *vals2;
  Array<unsigned int> *previous_values, *current_values;
  static List_ar<unsigned int> loopout_values;
  static List_ar<unsigned int> vallist1, vallist2, vallist3, vallist4;
  List_iterator<EDGEPTR> edgescan;

#ifdef SCM_DEBUG
  char scminfo_filename[MAXSTRLEN];
  ofstream scminfo_file;
#endif

#ifdef _SCALP_
    set_message("extract_scmatrices");
#endif
  assert_force(lib)

  intra_sc.resize(lib->numelements(), (Matrix<float> *)NULL);
  inter_sc.resize(lib->numelements(), (Matrix<float> *)NULL);
  reg_intra_sc.resize(flowgraph.numedges(), flowgraph.numedges(), 0.0);
  reg_inter_sc.resize(flowgraph.numedges(), flowgraph.numedges(), 0.0);
  //allocate the sc matrices, each of dimensions nodeount X nodecount
  for(i = 0; i < lib->numelements(); i++) {
    if(!intra_sc[i]) {
      intra_sc[i] =
	new Matrix<float>(flowgraph.numnodes(),flowgraph.numnodes());
    } else {
      intra_sc[i]->resize(flowgraph.numnodes(),flowgraph.numnodes());
    }
    intra_sc[i]->reset(0.0);
    if(!inter_sc[i]) {
      inter_sc[i] =
	new Matrix<float>(flowgraph.numnodes(),flowgraph.numnodes());
    } else {
      inter_sc[i]->resize(flowgraph.numnodes(),flowgraph.numnodes());
    }
    inter_sc[i]->reset(0.0);
  }
  //allocate two Arrays for storing the current and previous iteration
  //values, and initialize all values to ZERO
  vals1 = new Array<unsigned int> (flowgraph.numedges());
  vals2 = new Array<unsigned int> (flowgraph.numedges());
  vals1->reset(0);
  vals2->reset(0);
  previous_values = vals1;
  current_values = vals2;
  //allocate the loopout_values list to the appropriate size
  loopout_values.clear();
  for(i = 0; i < flowgraph.loopins.get_size(); i++) {
    loopout_values.append(0);
  }

  if(!vecfile) {
    use_random_vectors = T;
    vec_in = NULL;
    cout << "*" << endl;
    cerr << "*  No vector file specified for SC simulation - "
      << "using random vectors instead" << endl;
  } else {
    use_random_vectors = F;
    vec_in = anand_fopen(vecfile, "r");
    cout << "*" << endl;
    cout << "*  Reading vectors from file \"" << vecfile << "\"" << endl;
  }
#ifdef SCM_DEBUG
  strcpy(scminfo_filename, flowgraph.get_name());
  strcat(scminfo_filename, ".scminfo");
  scminfo_file.open(scminfo_filename,ios::out);
  if(!scminfo_file) {
    cerr << "Could not open file " << "scminfo" << " for output" << endl;
    exit(-1);
  }
#endif

  first_iteration = T;
  vector_number = 0;
  while(get_vector(use_random_vectors, flowgraph.inputs.get_size())) {
    //simulate the Dfg for the current iteration, and capture the
    //edge values in the current_values array
    flowgraph.simulate(cur_vector, loopout_values, *current_values, first_iteration);
    if(first_iteration) {
      first_iteration = F;
    }

#ifdef SCM_DEBUG
    flowgraph.display_values(*current_values, scminfo_file);
#endif

    //update the scmatrices entries;
    for(i = 0; i < lib->numelements(); i++) {
      libel = lib->get_nthelement(i);
      assert(libel);

      for(j = 0; j < flowgraph.numnodes(); j++) {
	node1 = flowgraph.get_nthnode(j);
	assert(node1 && node1->get_address() == j);
	if(!libel->can_perform(node1->get_func(), flowgraph.bitwidth)) {
	  continue;
	}

	for(k = j; k < flowgraph.numnodes(); k++) {
	  node2 = flowgraph.get_nthnode(k);
	  assert(node2 && node2->get_address() == k);
	  if(!libel->can_perform(node2->get_func(), flowgraph.bitwidth)) {
	    continue;
	  }

	  vallist1.clear(); vallist2.clear();
	  vallist3.clear(); vallist4.clear();
	  FOR_EACH_FANIN_EDGE(node1, edgescan) {
	    edge = edgescan.get_item();
	    assert(edge);
	    vallist1.append((*current_values)[edge->get_address()]);
	    vallist3.append((*previous_values)[edge->get_address()]);
	  }
	  FOR_EACH_FANIN_EDGE(node2, edgescan) {
	    edge = edgescan.get_item();
	    assert(edge);
	    vallist2.append((*current_values)[edge->get_address()]);
	    vallist4.append((*previous_values)[edge->get_address()]);
	  }
	  //update intra_sc[i][j][k]
	  (*intra_sc[i])[j][k] += libel->sw_cap(vallist1, vallist2);
	  
	  //update intra_sc[i][k][j]
	  if(j != k) {
	    (*intra_sc[i])[k][j] += libel->sw_cap(vallist2, vallist1);
	  }
	  
	  //update inter_sc[i][j][k]
	  (*inter_sc[i])[j][k] += libel->sw_cap(vallist3, vallist2);

	  //update inter_sc[i][k][j]
	  if(j != k) {
	    (*inter_sc[i])[k][j] += libel->sw_cap(vallist4, vallist1);
	  }

	} //END FOR EACH Dfg NODE
      } //END FOR EACH Dfg NODE
    } //END FOR EACH LIBELEMENT

    //SWITCHED CAPACITANCES FOR REGISTERS
    for(i = 0; i < flowgraph.numedges(); i++) {
      for(j = i; j < flowgraph.numedges(); j++) {
	edge1 = flowgraph.get_nthedge(i);
	assert(edge1 && edge1->get_address() == i);
	edge2 = flowgraph.get_nthedge(j);
	assert(edge2 && edge2->get_address() == j);

	reg_intra_sc[i][j] += lib->reg_sw_cap((*current_values)[i],
			       (*current_values)[j], flowgraph.bitwidth);
	if( i != j) {
	  reg_intra_sc[j][i] += lib->reg_sw_cap((*current_values)[j],
				 (*current_values)[i], flowgraph.bitwidth);
	}

	reg_inter_sc[i][j] += lib->reg_sw_cap((*previous_values)[i],
			       (*current_values)[j], flowgraph.bitwidth);
	if( i != j) {
	  reg_inter_sc[j][i] += lib->reg_sw_cap((*previous_values)[j],
		  	         (*current_values)[i], flowgraph.bitwidth);
	}

      }
    }

    //transfer current values to previous values, and reset current values
    if(current_values == vals1) {
      assert(previous_values == vals2);
      current_values = vals2;
      previous_values = vals1;
    } else {
      assert(current_values == vals2 && previous_values == vals1);
      current_values = vals1;
      previous_values = vals2;
    }
    current_values->reset(0);
  }

  //Normalize the SCM entries by the number of CDFG iterations simulated
  for(i = 0; i < lib->numelements(); i++) {
    libel = lib->get_nthelement(i);
    assert(libel);

    for(j = 0; j < flowgraph.numnodes(); j++) {
      node1 = flowgraph.get_nthnode(j);
      assert(node1 && node1->get_address() == j);
      if(!libel->can_perform(node1->get_func(), flowgraph.bitwidth)) {
	continue;
      }

      for(k = j; k < flowgraph.numnodes(); k++) {
	node2 = flowgraph.get_nthnode(k);
	assert(node2 && node2->get_address() == k);
	if(!libel->can_perform(node2->get_func(), flowgraph.bitwidth)) {
	  continue;
	}
	(*intra_sc[i])[j][k] /= (float) vector_number;
	if(j != k) {
	  (*intra_sc[i])[k][j] /= (float) vector_number;
	}
	(*inter_sc[i])[j][k] /= (float) vector_number;
	if(j != k) {
	  (*inter_sc[i])[k][j] /= (float) vector_number;
	}
      }
    }
  }
  for(i = 0; i < flowgraph.numedges(); i++) {
    for(j = i; j < flowgraph.numedges(); j++) {
      edge1 = flowgraph.get_nthedge(i);
      assert(edge1 && edge1->get_address() == i);
      edge2 = flowgraph.get_nthedge(j);
      assert(edge2 && edge2->get_address() == j);

      reg_intra_sc[i][j] /=  (float) vector_number;
      if(i != j) {
	reg_intra_sc[j][i] /=  (float) vector_number;
      }
      reg_inter_sc[i][j] /=  (float) vector_number;
      if(i != j) {
	reg_inter_sc[j][i] /=  (float) vector_number;
      }
    }
  }

  cout << endl << "*" << endl;
  delete vals1;
  delete vals2;
  if(vecfile) {
    vec_in = anand_fclose(vec_in);
  }

#ifdef SCM_DEBUG
  display(scminfo_file, lib);
#endif

  return;
}
/********************************************************************/
/*This routine is used to generate the vectors for the switched
 *capacitance simulation. If the parameter use_random_vectors is
 *T, random vectors are used. Otherwise, the vectors are read from
 *the given input file. The vector is assumed to consist of integer
 *values, separated by (non-newline) white space. Each vector
 *is assumed to span a single line unless a "\" is the last character
 *seen on the line, in which case the current vector is assumed to be
 *continued on the next line. If EOF was reached, the function returns
 *F, otherwise, it returns T.
 */
Boolean Scm::get_vector(const Boolean use_random_vectors, const int num_inputs)
{
  extern FILE *vec_in;
  unsigned int val;
  register int i;
  int vec_errors;

#ifdef _SCALP_
    set_message("get_vector");
#endif
  assert(num_inputs > 0);

  cur_vector.clear();
  if(use_random_vectors) {
    if(vector_number == MAX_RANDOM_VECTORS) {
      return F;
    }
    for(i = 0; i < num_inputs; i++) {
      //generate a random number, and truncate to the bit-width
      //of the DFG, and store it in val
      val = 0;
      cur_vector.append(val);
    }
    vector_number++;
  } else {
    assert_force(vec_in);
    vec_errors = vec_parse();
    if(vec_errors) {
      cerr << "Errors occured while parsing the vector file" << endl;
      exit(-1);
    }
    if(cur_vector.get_size() == 0) {
      return F;
    }
    vector_number++;
    if (cur_vector.get_size() != num_inputs) {
      cerr << "ERROR: Vector number " << vector_number
	<< " has too many/too few values" << endl;
      exit(-1);
    }
  }

  cout << "
" << "*  Vector: " << vector_number;

  return T;
}
/********************************************************************/
const float Scm::get_total_sc(const int type, const List_ar<NODEPTR> &operations)
{
  List_iterator<NODEPTR> nodescan;
  NODEPTR node, node1, first_node;
  Boolean first = T;
  float retval = 0.0;

  FOR_EACH_LISTNODE(operations, nodescan) {
    node1 = nodescan.get_item();
    assert(node1);
    if(!first) {
      assert(node != node1);
      assert(node->get_birth() <= node1->get_birth());
      retval += get_fu_intra_entry(type, node->get_address(),node1->get_address());
    } else {
      first_node = node1;
      first = F;
    }
    node = node1;
  }
  retval += get_fu_inter_entry(type, node1->get_address(), first_node->get_address());

  return(retval);
}
/********************************************************************/
const float Scm::get_total_sc(const List_ar<EDGEPTR> &variables)
{
  List_iterator<EDGEPTR> edgescan;
  EDGEPTR edge, edge1, first_edge;
  Boolean first = T;
  float retval = 0.0;

  FOR_EACH_LISTNODE(variables, edgescan) {
    edge1 = edgescan.get_item();
    assert(edge1);
    if(!first) {
      assert(edge != edge1);
      assert(edge->get_birth() <= edge1->get_birth());
      retval += get_reg_intra_entry(edge->get_address(), edge1->get_address());
      } else {
	first_edge = edge1;
	first = F;
      }
      edge = edge1;
    }
    retval += get_reg_inter_entry(edge1->get_address(), first_edge->get_address());

  return(retval);
}
/********************************************************************/
void Scm::display(ofstream &outfile, library *lib)
{
  register int i;

  assert(lib);
  assert(outfile);

  for(i = 0; i < lib->numelements(); i++) {
    //print out library element name
    outfile << "SC Matrices for library element: " <<
      lib->get_nthelement(i)->get_name() << " :" << endl;
    outfile << "Intra-Interation Switched Capacitance Estimates" << endl;
    intra_sc[i]->display(outfile);
    outfile << "Inter-Interation Switched Capacitance Estimates" << endl;
    inter_sc[i]->display(outfile);
  }

  outfile << "SC Matrices for registers: " << endl;
  outfile << "Intra-Interation Switched Capacitance Estimates" << endl;
  reg_intra_sc.display(outfile);
  outfile << "Inter-Interation Switched Capacitance Estimates" << endl;
  reg_inter_sc.display(outfile);

  return;
}
/********************************************************************/
