/******************************************************************************/
/*                                                                            */
/*  AUTHOR:  Cameron Brien                                                    */
/*           Department of Electrical Engineering, Princeton University       */
/*  FILE:    tigerdisp.pde                                                    */
/*  RUN:     Run by executing within the Processing environment. Tested with  */
/*           Processing 99.                                                   */
/*  PURPOSE: Display tool for visual analysis of the dynamic behaviour of     */
/*           modern DPLL SAT solvers                                          */
/*                                                                            */
/******************************************************************************/



/****** Notes ******/
/* This tool plots opposite each other the run-time metrics of two solvers. In
   this particular instance of the tool, the tool appears as follows:
   /-----------------------------\      /-----------------------------\
   | decision depth, solver 0    |      | decision depth, solver 1    |
   \-----------------------------/      \-----------------------------/
   
   /-----------------------------\      /-----------------------------\
   | average depth, solver 0     |      | average depth, solver 1     |
   \-----------------------------/      \-----------------------------/
   
   /-----------------------------\      /-----------------------------\
   | total implications, solver 0|      | total implications, solver 1|
   \-----------------------------/      \-----------------------------/
   
   learnt clause histogram, solver 0    learnt clause histogram, solver 1
   ---------------------------------    ---------------------------------
   
   By default, the metrics in the first plot (in this case, decision depth)
   are plotted in the background of the other two plots. The histogram is 
   displayed with nothing in the background. One can zoom in on the second two
   plots (in this case, average depth and total implications) by clicking the 
   mouse in the zoom marker. Moving the mouse along the zoom marker also scans
   along the histogram.
   
   The tool may be adjusted to display different plots. Additionally, other
   metrics may be fed in a put in place of those displayed in this instance.
*/



/****** Constants and globals ******/
final int SOLVER0 = 0;
final int SOLVER1 = 1;

/* Metric files to read */
final String inSolver0 = "metrics.sss-sat-1.shuffled.cnf.haifa.dump";
final String inSolver1 = "metrics.sss-sat-1.shuffled.cnf.minisat.dump";

/* The number of metrics to display */
int sz = 0;
final int overFlow = 500;

/* The size of the averaging 'window' for the average depth */
final int avgWindow = 100;

/* Which plots should be drawn */
boolean drawPlot0 = true;
boolean drawPlot1 = true;
boolean drawPlot2 = true;

/* Number of buckest to display in the histogram */
final int numHistBuckets = 14;

/* Depths of assignment */
int depthSolver0[];
int depthSolver1[];

/* Running average of depths of assignment */
int depthAvgSolver0[];
int depthAvgSolver1[];

/* New implications from this decision */
int deltaImpSolver0[];
int deltaImpSolver1[];

/* Total implications at this decision */
int impSolver0[];
int impSolver1[];

/* Learnt clauses */
int clauseSolver0[][];
int clauseSolver1[][];

/* Winner */
int winner;
int winnerLen = 0;

/* Depth statistics */
int maxDepth = 0;
int maxDepthAvg = 0;

/* Implication statistics */
int maxDeltaImp = 0;
int maxImp = 0;

/* Clause statistics */
int maxClause = 0;

/* Run-time counts */
boolean firstRun = true;
boolean zoomChange = true;

/* Zoomed waveform range */
int range0 = 0;
int range1 = 0;

/* Display settings */
/* Resolution */
final int dispX = 1200;
final int dispY = 1000;

/* Colors */
final int bgColor = 60;
final int waveColor = 100;
final int impColor = #FFCC00;
final int avgColor = #CCFF00;
final int rectColor = 50;

/* Font */
PFont font0;

/* Horizontal and veritcal cut-offs for displaying solver 0 and solver 1 */
final float leftSplit0 = 0.05;
final float leftSplit1 = 0.45;
final float rightSplit0 = 0.55;
final float rightSplit1 = 0.95;

final float overviewY0 = 0.02;
final float overviewY1 = 0.22;

final float zoomY0 = 0.28;
final float zoomY1 = 0.48;

final float zoomMarkY0 = 0.53;
final float zoomMarkY1 = 0.55;

final float zoom2Y0 = 0.61;
final float zoom2Y1 = 0.81;

final float histogramY0 = 0.87;
final float histogramY1 = 0.92;



/****** Methods ******/
/* Initializes the environment. Loads the SAT solver metric dumps for solver 0
   and solver 1 solutions into memory. Takes no arguments. Starts at run-time.
   Returns void. */
void setup() {
    /* Initialize the display and text */
    size(dispX, dispY);
    background(bgColor);
    font0 = loadFont("Courier-12.vlw");
    int readAmount = 0;
    
    /* Load solver metrics into memory */
    String linesSolver0[] = loadStrings(inSolver0);
    String linesSolver1[] = loadStrings(inSolver1);
    
    /* Determine winner */
    if (linesSolver0.length < linesSolver1.length) {
        winner = SOLVER0;
        readAmount = linesSolver0.length + overFlow - 2;
        winnerLen = linesSolver0.length / 2;
    }
    else {
        winner = SOLVER1;
        readAmount = linesSolver1.length + overFlow - 2;
        winnerLen = linesSolver1.length / 2;
    }
    
    println(readAmount);
    
    /* Initialize metric arrays */
    sz = winnerLen + overFlow;
    int countSolver0 = 0;
    int countSolver1 = 0;
    depthSolver0 = new int[sz];
    depthSolver1 = new int[sz];
    depthAvgSolver0 = new int[sz];
    depthAvgSolver1 = new int[sz];
    deltaImpSolver0 = new int[sz];
    deltaImpSolver1 = new int[sz];
    impSolver0 = new int[sz];
    impSolver1 = new int[sz];
    clauseSolver0 = new int[sz][numHistBuckets];
    clauseSolver1 = new int[sz][numHistBuckets];
    
    /* Initialize ranges */
    range0 = 0;
    range1 = sz;
    
    int temp1 = 0;
    
    /* Parse metrics into various arrays */
    for (int i = 0; i < readAmount; i++) {
        String cleaned;
        String pieces[];
        
        /* Load Solver 0 */
        if (i < linesSolver0.length) {
            /* Parse input file */
            cleaned = trim(linesSolver0[i]);
            pieces = split(cleaned);

            /* Depth, implications */
            if ((i % 2) == 0) {            
                /* Error checking */
                if (pieces.length != 3) {
                    println("DBG:\nIgnoring line " + (i + 1));
                    println(pieces.length);
                    continue;
                }
                
                temp1++;
                
                /* Record the metrics */
                depthSolver0[countSolver0] = toInt(pieces[0]);
                impSolver0[countSolver0] = toInt(pieces[1]);
                deltaImpSolver0[countSolver0] = toInt(pieces[2]);
                
                if (countSolver0 == 0) {
                    depthAvgSolver0[0] = 1;
                }
                else {
                    for (int j = 0; (j < avgWindow) && (j < countSolver0); j++) {
                        depthAvgSolver0[countSolver0] += depthSolver0[countSolver0 - j];
                    }
                    
                    if (avgWindow > countSolver0) {
                        depthAvgSolver0[countSolver0] = depthAvgSolver0[countSolver0]
                                                     / countSolver0;
                    }
                    else {
                        depthAvgSolver0[countSolver0] = depthAvgSolver0[countSolver0]
                                                     / avgWindow;
                    }
                }
                
                if (depthSolver0[countSolver0] > maxDepth) { 
                    maxDepth = depthSolver0[countSolver0];
                }
                
                if (depthAvgSolver0[countSolver0] > maxDepthAvg) {
                    maxDepthAvg = depthAvgSolver0[countSolver0];
                }
                
                if (impSolver0[countSolver0] > maxImp) {
                    maxImp = impSolver0[countSolver0];
                }
            }
            /* Clauses */
            else {
                /* Error checking */
                if (pieces.length != numHistBuckets) {
               //     println("DBG:\nIgnoring line " + (i + 1));
                    continue;
                }
                
                /* Record the metrics */
                for (int j = 0; j < numHistBuckets; j++) {
                    clauseSolver0[countSolver0][j] = toInt(pieces[j]);
                    
                    if (clauseSolver0[countSolver0][j] > maxClause) {
                        maxClause = clauseSolver0[countSolver0][j];
                    }
                }
                
                countSolver0++;
            }
        }
                
        /* Load Solver 1 */
        if (i < linesSolver1.length) {
            /* Parse input file */
            cleaned = trim(linesSolver1[i]);
            pieces = split(cleaned);
            
            /* Depth, implications */
            if ((i % 2) == 0) {
                /* Error checking */
                if (pieces.length != 3) {
                //    println("Ignoring line " + (i + 1));
                    continue;
                }
                
                /* Record the metrics */
                depthSolver1[countSolver1] = toInt(pieces[0]);
                impSolver1[countSolver1] = toInt(pieces[1]);
                deltaImpSolver1[countSolver1] = toInt(pieces[2]);
                
                if (countSolver1 == 0) {
                    depthAvgSolver1[0] = 1;
                }
                else {
                    for (int j = 0; (j < avgWindow) && (j < countSolver1); j++) {
                        depthAvgSolver1[countSolver1] += depthSolver1[countSolver1 - j];
                    }
                    
                    if (avgWindow > countSolver1) {
                        depthAvgSolver1[countSolver1] = depthAvgSolver1[countSolver1]
                                                     / countSolver1;
                    }
                    else {
                        depthAvgSolver1[countSolver1] = depthAvgSolver1[countSolver1]
                                                     / avgWindow;
                    }
                }
                
                if (depthSolver1[countSolver1] > maxDepth) {
                    maxDepth = depthSolver1[countSolver1];
                }
                
                if (depthAvgSolver1[countSolver1] > maxDepthAvg) {
                    maxDepthAvg = depthAvgSolver1[countSolver1];
                }
                
                if (impSolver1[countSolver1] > maxImp) {
                    maxImp = impSolver1[countSolver1];
                }
            }
            /* Clauses */
            else {
                /* Error checking */
                if (pieces.length != numHistBuckets) {
                //    println("DBG:\nIgnoring line " + (i + 1));
                    continue;
                }
                
                /* Record the metrics */
                for (int j = 0; j < numHistBuckets; j++) {
                    clauseSolver1[countSolver1][j] = toInt(pieces[j]);
                    
                    if (clauseSolver1[countSolver1][j] > maxClause) {
                        maxClause = clauseSolver1[countSolver1][j];
                    }
                } 
                
                countSolver1++;   
            }
        }   
    }
}


/* Draws the metric displays on the screen. Takes no arguments. Called at
   run-time. Returns void. */
void draw() {
    /* Draw overview waveform */
    if (firstRun) {
        drawOverviewWaveform();
        drawHistogram(0);
        
         /* Label plots and show winner */
        textAlign(LEFT);
        textFont(font0, 12);
        
        text("Depth", leftSplit0 * width, overviewY1 * height + 20);
        text("Average depth", leftSplit0 * width, zoomY1 * height + 20);
        text("Total Implications", leftSplit0 * width, zoom2Y1 * height + 20);
            
        if (winner == SOLVER0)  {
            text("WINNER!", leftSplit0 * width, histogramY1 * height + 20);
        }
        else { 
            text("WINNER!", rightSplit0 * width, histogramY1 * height + 20);
        }
    }
    
    /* Draw overview reference marker */
    drawZoomMarker();
    drawZoomWaveform();
    drawZoom2Waveform();

    zoomChange = false;
    firstRun = false;
}


/* Draws the overview waveform on the screen. Takes no arguments. Returns void. */
void drawOverviewWaveform() {
    /* Spans of each waveform */
    float xSpan = leftSplit1 - leftSplit0;
    float ySpan = overviewY1 - overviewY0;
    
    /* Drawing points */
    float x0 = leftSplit0;
    float y0 = overviewY0;
    
    /* Draw Solver 0 waveform */
    stroke(rectColor);
    fill(rectColor);
    rect(x0 * width, y0 * height, xSpan * width, ySpan * height + 5);
    stroke(waveColor);
    
    for (int i = 0; i < sz; i++) {
        float x1 = (xSpan * ((float) i / (float) sz)) + leftSplit0;
        float y1 = (ySpan * ((float) depthSolver0[i] / (float) maxDepth))
                    + overviewY0;
    
        line(x1 * width, y1 * height, x0 * width, y0 * height);
        
        x0 = x1;
        y0 = y1;
    }
    
    /* Drawing points */
    x0 = rightSplit0;
    y0 = overviewY0;
    
    /* Draw Solver 1 waveform */
    stroke(rectColor);
    rect(x0 * width, y0 * height, xSpan * width, ySpan * height + 5);
    stroke(waveColor);
    
    for (int i = 0; i < sz; i++) {
        float x1 = (xSpan * ((float) i / (float) sz)) + rightSplit0;
        float y1 = (ySpan * ((float) depthSolver1[i] / (float) maxDepth))
                    + overviewY0;
    
        line(x1 * width, y1 * height, x0 * width, y0 * height);
        
        x0 = x1;
        y0 = y1;
    }
}


/* Draws the zoomed view waveform on the screen. Takes no arguments. Returns void. */
void drawZoomWaveform() {
    if (!zoomChange) return;

    /* Spans of each waveform */
    float xSpan = leftSplit1 - leftSplit0;
    float ySpan = zoomY1 - zoomY0;
    
    /* Drawing points */
    float x0 = leftSplit0;
    float y0 = zoomY0;
    
    /* Draw Solver 0 depth waveform */
    stroke(rectColor);
    fill(rectColor);
    rect(x0 * width, y0 * height, xSpan * width, ySpan * height + 5);
    stroke(waveColor);
    
    int range = range1 - range0;
    
    for (int i = range0; i < range1; i++) {
        float x1 = (xSpan * ((float) (i - range0) / (float) range)) 
                   + leftSplit0;
        float y1 = (ySpan * ((float) depthSolver0[i] / (float) maxDepth))
                    + zoomY0;
    
        line(x1 * width, y1 * height, x0 * width, y0 * height);
        
        x0 = x1;
        y0 = y1;
    }
    
    /* Drawing points */
    x0 = leftSplit0;
    y0 = zoomY0;
    
    /* Draw Solver 0 average depth waveform */
    if (drawPlot1) {
        stroke(avgColor);
        
        for (int i = range0; i < range1; i++) {
            float x1 = (xSpan * ((float) (i - range0) / (float) range)) 
                       + leftSplit0;
            float y1 = (ySpan * ((float) depthAvgSolver0[i] /
                        (float) maxDepthAvg)) + zoomY0;
        
            line(x1 * width, y1 * height, x0 * width, y0 * height);
            
            x0 = x1;
            y0 = y1;
        }
    }
        
    /* Drawing points */
    x0 = rightSplit0;
    y0 = zoomY0;
    
    /* Draw Solver 1 depth waveform */
    stroke(rectColor);
    rect(x0 * width, y0 * height, xSpan * width, ySpan * height + 5);
    stroke(waveColor);
    
    for (int i = range0; i < range1; i++) {
        float x1 = (xSpan * ((float) (i - range0) / (float) range)) 
                   + rightSplit0;
        float y1 = (ySpan * ((float) depthSolver1[i] / (float) maxDepth))
                    + zoomY0;
    
        line(x1 * width, y1 * height, x0 * width, y0 * height);
        
        x0 = x1;
        y0 = y1;
    }
    
    /* Drawing points */
    x0 = rightSplit0;
    y0 = zoomY0;
    
    /* Draw Solver 1 average depth waveform */
    if (drawPlot1) {
        stroke(avgColor);
        
        for (int i = range0; i < range1; i++) {
            float x1 = (xSpan * ((float) (i - range0) / (float) range)) 
                       + rightSplit0;
            float y1 = (ySpan * ((float) depthAvgSolver1[i] /
                        (float) maxDepthAvg)) + zoomY0;
        
            line(x1 * width, y1 * height, x0 * width, y0 * height);
            
            x0 = x1;
            y0 = y1;
        }
    }
}



/* Draws the zoomed view waveform on the screen. Takes no arguments. Returns void. */
void drawZoom2Waveform() {
    if (!zoomChange) return;

    /* Spans of each waveform */
    float xSpan = leftSplit1 - leftSplit0;
    float ySpan = zoom2Y1 - zoom2Y0;
    
    /* Drawing points */
    float x0 = leftSplit0;
    float y0 = zoom2Y0;
    
    /* Draw Solver 0 depth waveform */
    stroke(rectColor);
    fill(rectColor);
    rect(x0 * width, y0 * height, xSpan * width, ySpan * height + 5);
    stroke(waveColor);
    
    int range = range1 - range0;
    
    for (int i = range0; i < range1; i++) {
        float x1 = (xSpan * ((float) (i - range0) / (float) range)) 
                   + leftSplit0;
        float y1 = (ySpan * ((float) depthSolver0[i] / (float) maxDepth))
                    + zoom2Y0;
    
        line(x1 * width, y1 * height, x0 * width, y0 * height);
        
        x0 = x1;
        y0 = y1;
    }
    
    /* Drawing points */
    x0 = leftSplit0;
    y0 = zoom2Y0;
    
    /* Draw Solver 0 delta implication waveform */
    if (drawPlot2) {
        stroke(impColor);
         
        for (int i = range0; i < range1; i++) {
            float x1 = (xSpan * ((float) (i - range0) / (float) range)) 
                       + leftSplit0;
            float y1 = (ySpan * ((float) impSolver0[i] / (float) maxImp))
                        + zoom2Y0;
        
            line(x1 * width, y1 * height, x0 * width, y0 * height);
            
            x0 = x1;
            y0 = y1;
        }
    }
        
    /* Drawing points */
    x0 = rightSplit0;
    y0 = zoom2Y0;
    
    /* Draw Solver 1 depth waveform */
    stroke(rectColor);
    rect(x0 * width, y0 * height, xSpan * width, ySpan * height + 5);
    stroke(waveColor);
    
    for (int i = range0; i < range1; i++) {
        float x1 = (xSpan * ((float) (i - range0) / (float) range)) 
                   + rightSplit0;
        float y1 = (ySpan * ((float) depthSolver1[i] / (float) maxDepth))
                    + zoom2Y0;
    
        line(x1 * width, y1 * height, x0 * width, y0 * height);
        
        x0 = x1;
        y0 = y1;
    }
    
    /* Drawing points */
    x0 = rightSplit0;
    y0 = zoom2Y0;
    
    /* Draw Solver 1 delta implication waveform */
    if (drawPlot2) {
        stroke(impColor);
        
        for (int i = range0; i < range1; i++) {
            float x1 = (xSpan * ((float) (i - range0) / (float) range)) 
                       + rightSplit0;
            float y1 = (ySpan * ((float) impSolver1[i] / (float) maxImp))
                        + zoom2Y0;
        
            line(x1 * width, y1 * height, x0 * width, y0 * height);
            
            x0 = x1;
            y0 = y1;
        }
    }
}



/* Draws a reference marker under the overview waveform on the screen. Takes no
   arguments. Returns void. */
void drawZoomMarker() {
    /* Spans of each waveform */
    float xSpan = leftSplit1 - leftSplit0;
    float ySpan = zoomMarkY1 - zoomMarkY0;
    
    int range = range1 - range0;
    
    /* Draw rectangles if first time */
    if (firstRun) {
        fill(rectColor);
        stroke(rectColor);
        rect(leftSplit0 * width, zoomMarkY0 * height, xSpan * width,
             ySpan * height + 1);
        rect(rightSplit0 * width, zoomMarkY0 * height, xSpan * width,
             ySpan * height + 1);
    }
        
    /* Check for marker in valid vertical range */
    if ((mouseY > zoomMarkY0 * height) && 
       (mouseY < zoomMarkY1 * height)) {        
        int i = -1;
        int mouseOffset = -1;

        /* Check on Solver 0 side */
        if ((mouseX > leftSplit0 * width) && (mouseX < leftSplit1 * width)) {
            /* Draw rectangles */
            stroke(rectColor);
            fill(rectColor);
            rect(leftSplit0 * width, zoomMarkY0 * height, xSpan * width,
                 ySpan * height + 1);
            rect(rightSplit0 * width, zoomMarkY0 * height, xSpan * width,
                 ySpan * height + 1);
                 
            stroke(waveColor);

            /* Draw lines */
            mouseOffset = (int) (mouseX - leftSplit0 * width);          
            line(mouseX, zoomMarkY0 * height, mouseX, 
                 zoomMarkY1 * height);
            int xDraw = (int) (mouseOffset + rightSplit0 * width);
            line(xDraw, zoomMarkY0 * height, xDraw,
                 zoomMarkY1 * height);
                 
            /* Calculation for depth information */
            i = (int) (range * mouseOffset / (xSpan * width)) + range0;
        }
        else if ((mouseX > rightSplit0 * width) && 
                 (mouseX < rightSplit1 * width)) {
            /* Draw rectangles */
            fill(rectColor);
            stroke(rectColor);
            rect(leftSplit0 * width, zoomMarkY0 * height, xSpan * width,
                 ySpan * height + 1);
            rect(rightSplit0 * width, zoomMarkY0 * height, xSpan * width,
                 ySpan * height + 1 );
            
            stroke(waveColor);

            /* Draw lines */    
            mouseOffset = (int) (mouseX - rightSplit0 * width); 
            line(mouseX, zoomMarkY0 * height, mouseX, 
                 zoomMarkY1 * height);
            int xDraw = (int) (mouseOffset + leftSplit0 * width);
            line(xDraw, zoomMarkY0 * height, xDraw,
                 zoomMarkY1 * height);
                 
            /* Calculation for depth information */
            i = (int) (range * mouseOffset / (xSpan * width)) + range0;
        }
        
        if (i >= 0) {
            /* Clear existing value */
            fill(bgColor);
            stroke(bgColor);
            rect(leftSplit0 * width, zoomMarkY1 * height + 2, 
                 width - leftSplit0 * width, 30);
            rect(rightSplit0 * width, zoomMarkY1 * height + 2, 
                 xSpan * width, 30);
            
            textFont(font0, 12);
            fill(waveColor); 
            
            /* Draw Solver 0 */
            textAlign(LEFT);
            text("(step, depth, impl, delta-impl) =",
                 mouseOffset + leftSplit0 * width, zoomMarkY1 * height + 12);
            text("(" + i + ", " + depthSolver0[i] + ", " + 
                 impSolver0[i] + ", " + 
                 deltaImpSolver0[i] + ")",
                 mouseOffset + leftSplit0 * width, zoomMarkY1 * height + 24);
            
            /* Draw Solver 1 */
            textAlign(RIGHT);
            text("(step, depth, impl, delta-impl) =",
                 mouseOffset + rightSplit0 * width, zoomMarkY1 * height + 12);
            text("(" + i + ", " + depthSolver1[i] + ", " + 
                 impSolver1[i] + ", " + 
                 deltaImpSolver1[i] + ")",
                 mouseOffset + rightSplit0 * width, zoomMarkY1 * height + 24);
                 
            drawHistogram(i);
        
            fill(rectColor);
        }
    }
}


/* Draws the learnt clause histogram on the screen. Takes index into histogram
   array as argument. Returns void. */
void drawHistogram(int hist_i) {
    /* Spans of each waveform */
    float xSpan = leftSplit1 - leftSplit0;
    float ySpan = histogramY1 - histogramY0;
    
    /* Drawing points */
    float x0 = leftSplit0;
    float y0 = histogramY1;
    
    /* Draw Solver 0 histogram */
    stroke(rectColor);
    fill(rectColor);
    rect(x0 * width, histogramY0 * height, xSpan * width, ySpan * height + 5);
    stroke(waveColor);
    fill(waveColor);
    
    for (int i = 0; i < numHistBuckets; i++) {
        float x1 = (xSpan * ((float) (i) / (float) numHistBuckets)) + leftSplit0;
        float y1 = histogramY1 - (ySpan * ((float) clauseSolver0[hist_i][i] / (float) maxClause));
    
        rect(x0 * width, y1 * height, (x1 - x0) * width, (y0 - y1) * height);
        stroke(rectColor);
        line(x0 * width, histogramY0 * height, x0 * width, histogramY1 * height);
        stroke(waveColor);
        
        x0 = x1;
    }
    
    /* Drawing points */
    x0 = rightSplit0;
    y0 = histogramY1;
    
    /* Draw Solver 1 histogram */
    stroke(rectColor);
    fill(rectColor);
    rect(x0 * width, histogramY0 * height, xSpan * width, ySpan * height + 5);
    stroke(waveColor);
    fill(waveColor);
    
    for (int i = 0; i < numHistBuckets; i++) {
        float x1 = (xSpan * ((float) (i) / (float) numHistBuckets)) + rightSplit0;
        float y1 = histogramY1 - (ySpan * ((float) clauseSolver1[hist_i][i] / (float) maxClause));
    
        rect(x0 * width, y1 * height, (x1 - x0) * width, (y0 - y1) * height);
        stroke(rectColor);
        line(x0 * width, histogramY0 * height, x0 * width, histogramY1 * height);
        stroke(waveColor);
        
        x0 = x1;
    }
    
}


void mousePressed() {
    /* Mouse is in marker range */
    if ((mouseY > zoomMarkY0 * height) && (mouseY < zoomMarkY1 * height)) {
        /* Spans of each waveform */
        float xSpan = leftSplit1 - leftSplit0;
        int range = range1 - range0;        
        
        int i = -1;
        
        /* Solver 0 */
        if ((mouseX > leftSplit0 * width) && (mouseX < leftSplit1 * width)) {
            /* Get mouse information */
            int mouseOffset = (int) (mouseX - leftSplit0 * width);
            
            i = (int) (range * mouseOffset / (xSpan * width)) + range0;            
        }
        /* Solver 1 */
        else if ((mouseX > rightSplit0 * width) &&
                (mouseX < rightSplit1 * width)) {
            /* Get mouse information */
            int mouseOffset = (int) (mouseX - rightSplit0 * width);
            
            i = (int) (range * mouseOffset / (xSpan * width)) + range0;
        }
        
        /* Range correction */
        range0 = i - (range / 4);
        range1 = i + (range / 4);

        if (range0 < 0) range0 = 0;
        if (range1 > sz) range1 = sz;
        
        /* Note change in zoom */
        zoomChange = true;   
        
        println("DBG:\nrange0 = " + range0);
        println("DBG:\nrange1 = " + range1);
    }
}


void keyPressed() {
    int range = range1 - range0;
    
    if (key == '+') {
        range0 = range0 + (range / 4);
        range1 = range1 - (range / 4);
        
        if (range0 < 0) range0 = 0;
        if (range1 > sz) range1 = sz;
        
        /* Note change in zoom */
        zoomChange = true;   
        
        println("DBG:\nrange0 = " + range0);
        println("DBG:\nrange1 = " + range1);
    }
    else if (key == '-') {
        range0 = range0 - (range / 2);
        range1 = range1 + (range / 2);
        
        if (range0 < 0) range0 = 0;
        if (range1 > sz) range1 = sz;
        
        /* Note change in zoom */
        zoomChange = true;   
        
        println("DBG:\nrange0 = " + range0);
        println("DBG:\nrange1 = " + range1);
    }
}

