// Copyright (c) 2015 Princeton University
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * Neither the name of Princeton University nor the
//       names of its contributors may be used to endorse or promote products
//       derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY PRINCETON UNIVERSITY "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL PRINCETON UNIVERSITY BE LIABLE FOR ANY
// DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
// ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

`include "define.vh"

// Filename: mem_test_seq_driver.v
// Author: mmckeown
// Description: This module implements a driver for
//              sequential memory tests,
//              i.e. read followed by a write, no
//              concurrent operations.  The actual
//              address and data that is being tested
//              is generated by submodules and can vary

module mem_test_seq_driver(
    input                               clk,
    input                               rst_n,
    input                               en,

    input      [2:0]                    test_mode_en,

    input      [2:0]                    addr_mode,
    input      [`PHY_ADDR_WIDTH-1:0]    addr_incr_val,
    input      [7:0]                    data_payload_flits,
    input      [`PHY_ADDR_WIDTH-1:0]    mem_top_addr,

    output reg                          test_mem_val,
    output reg [`NOC_DATA_WIDTH-1:0]    test_mem_data,
    input                               test_mem_rdy,

    input                               mem_test_val,
    input      [`NOC_DATA_WIDTH-1:0]    mem_test_data,
    output reg                          mem_test_rdy,

    output reg                          test_running,
    output reg                          test_done,
    output reg                          test_timeout,
    output reg                          test_passed
);

///////////////////////
// Type declarations //
///////////////////////

// Memory load header flit
// CHIPID: 14'd0
// XPOS: 8'd0
// YPOS: 8'd0
// FBITS: 4'd0
// PAYLOAD LENGTH: 8'd2
// MESSAGE TYPE: MSG_TYPE_LOAD_MEM
// MSHR/TAG: 8'haa
// RESERVED: 6'd0
localparam                          MEM_LOAD_HEADER_FLIT = {14'd0, 8'd0, 8'd0, 4'd0, 8'd2, 
                                                            `MSG_TYPE_LOAD_MEM, 8'haa, 6'd0};

// Memory store header flit
// CHIPID: 14'd0
// XPOS: 8'd0
// YPOS: 8'd0
// FBITS: 4'd0
// PAYLOAD LENGTH: dynamic
// MESSAGE TYPE: MSG_TYPE_STORE_MEM
// MSHR/TAG: 8'haa
// RESERVED: 6'd0
localparam                          MEM_STORE_HEADER_FLIT = {14'd0, 8'd0, 8'd0, 4'd0, 8'd0,
                                                             `MSG_TYPE_STORE_MEM, 8'haa, 6'd0};

// Memory load ack header flit
// CHIPID: 14'h2aaa
// XPOS: 8'haa
// YPOS: 8'haa
// FBITS: 4'ha
// PAYLOAD LENGTH: dynamic
// MESSAGE TYPE: MSG_TYPE_LOAD_MEM_ACK
// MSHR/TAG: 8'haa
// RESERVED: 6'd0
localparam                          MEM_LOAD_ACK_HEADER = {14'h2aaa, 8'haa, 8'haa, 4'ha, 8'd0,
                                                           `MSG_TYPE_LOAD_MEM_ACK, 8'haa, 6'd0};

// Memory store ack header flit
// CHIPID: 14'h2aaa
// XPOS: 8'haa
// YPOS: 8'haa
// FBITS: 4'ha
// PAYLOAD LENGTH: 8'd0
// MESSAGE TYPE: MSG_TYPE_STORE_MEM_ACK
// MSHR/TAG: 8'haa
// RESERVED: 6'd0
localparam                          MEM_STORE_ACK_HEADER = {14'h2aaa, 8'haa, 8'haa, 4'ha, 8'd0,
                                                            `MSG_TYPE_STORE_MEM_ACK, 8'haa, 6'd0};

// Flit3 for memory requests
// SRC CHIPID: 14'h2aaa
// SRC X: 8'haa
// SRC Y: 8'haa
// SRC FBITS: 4'ha
// RESERVED: 30'd0
localparam                          MEM_REQ_FLIT3 = {14'h2aaa, 8'haa, 8'haa, 4'ha, 30'd0};

// High level test states
localparam                          TEST_STATE_RESET = 2'd0;
localparam                          TEST_STATE_RUN = 2'd1;
localparam                          TEST_STATE_PASS = 2'd2;
localparam                          TEST_STATE_FAIL = 2'd3;

// Low level memory request states
localparam                          MEM_STATE_STORE_REQ = 2'd0;
localparam                          MEM_STATE_STORE_RESP = 2'd1;
localparam                          MEM_STATE_LOAD_REQ = 2'd2;
localparam                          MEM_STATE_LOAD_RESP = 2'd3;

// Timeout counter
localparam                          TIMEOUT_COUNTER_WIDTH = 31;
localparam                          TIMEOUT_COUNTER_THRESHOLD = 31'd1073741823;

// Test modes from one hot test_mode_en
localparam                          TEST_MODE_WALKING_ONES = 3'b001;
localparam                          TEST_MODE_ADDRESS_OWN = 3'b010;
localparam                          TEST_MODE_RANDOM = 3'b100;

// States
reg  [1:0]                          test_state_f;
reg  [1:0]                          test_state_next;
reg  [1:0]                          mem_req_state_f;
reg  [1:0]                          mem_req_state_next;

// Timeout signals
reg                                 timeout_rst_n;
wire                                timeout_trigger;

// Address and data registers
reg  [`PHY_ADDR_WIDTH-1:0]          test_address_f;
reg  [`PHY_ADDR_WIDTH-1:0]          test_address_next;
reg  [511:0]                        test_data_f;
reg  [511:0]                        test_data_next;
reg  [`NOC_DATA_WIDTH-1:0]          test_data_f_2d [7:0];

// Shifted response data for sdctrl_test, it expects the L2
// to pick correct data from 8B response, so we need to do the
// same
reg  [`NOC_DATA_WIDTH-1:0]          mem_test_data_shifted;

// Test data matches incoming data
reg                                 test_data_matches;

// Flit counters
reg  [7:0]                          head_flit_counter_f;
reg  [7:0]                          head_flit_counter_next;
reg  [7:0]                          data_flit_counter_f;
reg  [7:0]                          data_flit_counter_next;

// Test address/data generator signals
reg                                 test_state_reset;

wire [`PHY_ADDR_WIDTH-1:0]          walk_ones_address_next;
wire [511:0]                        walk_ones_data_next;
wire                                walk_ones_done;

wire [`PHY_ADDR_WIDTH-1:0]          address_own_address_next;
wire [511:0]                        address_own_data_next;
wire                                address_own_done;

wire                                random_addrdata_next_val;
wire [`PHY_ADDR_WIDTH-1:0]          random_address_next;
wire [511:0]                        random_data_next;
wire                                random_done;

// Multiplexed next address and data from generators
reg                                 gen_addrdata_next_rdy;
reg                                 gen_addrdata_next_val;
reg  [`PHY_ADDR_WIDTH-1:0]          gen_address_next;
reg  [511:0]                        gen_data_next;
reg                                 gen_done;

//////////////////////
// Sequential Logic //
//////////////////////

always @ (posedge clk)
begin
    if(~rst_n)
    begin
        test_state_f <= TEST_STATE_RESET;
        mem_req_state_f <= MEM_STATE_STORE_REQ;

        head_flit_counter_f <= 8'd0;
        data_flit_counter_f <= 8'd0;
    end
    else
    begin
        test_state_f <= test_state_next;
        mem_req_state_f <= mem_req_state_next;

        head_flit_counter_f <= head_flit_counter_next;
        data_flit_counter_f <= data_flit_counter_next;
    end

    test_address_f <= test_address_next;
    test_data_f <= test_data_next;
end

/////////////////////////
// Combinational Logic //
/////////////////////////

always @ *
begin
    test_running = (test_state_f == TEST_STATE_RUN);
    test_passed = (test_state_f == TEST_STATE_PASS);
    test_timeout = timeout_trigger;
    test_done = test_state_f[1];
end

always @ *
    mem_test_rdy = mem_req_state_f[0];

always @ *
    test_state_reset = (test_state_f == TEST_STATE_RESET);

always @ *
begin
    test_data_f_2d[0] = test_data_f[`NOC_DATA_WIDTH-1:0];
    test_data_f_2d[1] = test_data_f[(`NOC_DATA_WIDTH*2)-1:`NOC_DATA_WIDTH];
    test_data_f_2d[2] = test_data_f[(`NOC_DATA_WIDTH*3)-1:(`NOC_DATA_WIDTH*2)];
    test_data_f_2d[3] = test_data_f[(`NOC_DATA_WIDTH*4)-1:(`NOC_DATA_WIDTH*3)];
    test_data_f_2d[4] = test_data_f[(`NOC_DATA_WIDTH*5)-1:(`NOC_DATA_WIDTH*4)];
    test_data_f_2d[5] = test_data_f[(`NOC_DATA_WIDTH*6)-1:(`NOC_DATA_WIDTH*5)];
    test_data_f_2d[6] = test_data_f[(`NOC_DATA_WIDTH*7)-1:(`NOC_DATA_WIDTH*6)];
    test_data_f_2d[7] = test_data_f[(`NOC_DATA_WIDTH*8)-1:(`NOC_DATA_WIDTH*7)];
end

// Multiplex test mode outputs
always @ *
begin
    case (test_mode_en)
        TEST_MODE_WALKING_ONES:
        begin
            gen_addrdata_next_val = 1'b1;
            gen_address_next = walk_ones_address_next;
            gen_data_next = walk_ones_data_next;
            gen_done = walk_ones_done;
        end
        TEST_MODE_ADDRESS_OWN:
        begin
            gen_addrdata_next_val = 1'b1;
            gen_address_next = address_own_address_next;
            gen_data_next = address_own_data_next;
            gen_done = address_own_done;
        end
        TEST_MODE_RANDOM:
        begin
            gen_addrdata_next_val = random_addrdata_next_val;
            gen_address_next = random_address_next;
            gen_data_next = random_data_next;
            gen_done = random_done;
        end
        default:
        begin
            gen_addrdata_next_val = 1'bx;
            gen_address_next = {`PHY_ADDR_WIDTH{1'bx}};
            gen_data_next = 512'bx;
            gen_done = 1'bx;
        end
    endcase
end

// Need to shift the response data for sdctrl_test
// as SD expects the L2 will pick out the correct
// data out of the 8B, it ignores the bottom 3
// bits of the address
always @ *
begin
    mem_test_data_shifted = mem_test_data >> {test_address_f[2:0], 3'b000};
end

// Logic to check if incoming data matches expected
always @ *
begin
    test_data_matches = 1'b0;

    case (addr_mode)
        3'd0:
        begin
            // No data never matches I guess?
            test_data_matches = 1'b0;
        end
        3'd1:
        begin
`ifdef SDCTRL_TEST
            if (mem_test_data_shifted[7:0] == test_data_f_2d[0][7:0])
`else // ifndef SDCTRL_TEST
            if (mem_test_data[7:0] == test_data_f_2d[0][7:0])
`endif // endif SDCTRL_TEST
                test_data_matches = 1'b1;
        end
        3'd2:
        begin
`ifdef SDCTRL_TEST
            if (mem_test_data_shifted[15:0] == test_data_f_2d[0][15:0])
`else // ifndef SDCTRL_TEST
            if (mem_test_data[15:0] == test_data_f_2d[0][15:0])
`endif // endif SDCTRL_TEST
                test_data_matches = 1'b1;
        end
        3'd3:
        begin
`ifdef SDCTRL_TEST
            if (mem_test_data_shifted[31:0] == test_data_f_2d[0][31:0])
`else // ifndef SDCTRL_TEST
            if (mem_test_data[31:0] == test_data_f_2d[0][31:0])
`endif // endif SDCTRL_TEST
                test_data_matches = 1'b1;
        end
`ifdef SDCTRL_TEST
        3'd4:
        begin
            // SD only supports up to 8 bytes and always returns 8 flits.
            // It replicates the data to fill the remaining splace in the 8
            // flits.  So we should check all flits against the first 8bytes
            // of the test data.  If we check other flits according to test
            // data, as in the default case, it would fail.
            if (mem_test_data == test_data_f_2d[0])
                test_data_matches = 1'b1;
        end
`endif // endif SDCTRL_TEST
        default:
        begin
            if (mem_test_data == test_data_f_2d[data_flit_counter_f])
                test_data_matches = 1'b1;
        end
    endcase
end

// State machine
always @ *
begin
    // Defaults
    test_state_next = test_state_f;
    mem_req_state_next = mem_req_state_f;
    timeout_rst_n = 1'b1;
    gen_addrdata_next_rdy = 1'b0;
    test_address_next = test_address_f;
    test_data_next = test_data_f;
    head_flit_counter_next = head_flit_counter_f;
    data_flit_counter_next = data_flit_counter_f;
    test_mem_val = 1'b0;
    test_mem_data = {`NOC_DATA_WIDTH{1'bx}};

    case (test_state_f)
        TEST_STATE_RESET:
        begin
            // If we are enabled and have valid
            // next address and data
            if (en && gen_addrdata_next_val)
                test_state_next = TEST_STATE_RUN;
        
            mem_req_state_next = MEM_STATE_STORE_REQ;
            timeout_rst_n = 1'b0;

            gen_addrdata_next_rdy = 1'b1;

            test_address_next = gen_address_next;
            test_data_next = gen_data_next;

            head_flit_counter_next = 8'd0;
            data_flit_counter_next = 8'd0;
        end
        TEST_STATE_RUN:
        begin
            // Check for timeout
            if (timeout_trigger)
                test_state_next = TEST_STATE_FAIL;
            else
            begin
                case (mem_req_state_f)
                    MEM_STATE_STORE_REQ:
                    begin
                        // Send header flit
                        if (head_flit_counter_f == 8'd0)
                        begin
                            test_mem_val = 1'b1;
                            test_mem_data = MEM_STORE_HEADER_FLIT;
                            test_mem_data[`MSG_LENGTH] = 8'd2 + data_payload_flits;
                        end
                        // Send flit 2 (address flit)
                        else if (head_flit_counter_f == 8'd1)
                        begin
                            test_mem_val = 1'b1;
                            test_mem_data = {`NOC_DATA_WIDTH{1'b0}};
                            test_mem_data[`MSG_ADDR_] = test_address_f;
                            test_mem_data[`MSG_DATA_SIZE_] = addr_mode; 
                        end
                        // Send flit 3 (source flit)
                        else if (head_flit_counter_f == 8'd2)
                        begin
                            test_mem_val = 1'b1;
                            test_mem_data = MEM_REQ_FLIT3; 
                        end
                        // Send data flits
                        else
                        begin
                            test_mem_val = 1'b1;
                            test_mem_data = test_data_f_2d[data_flit_counter_f];
                        end

                        // Header flit counter
                        if (test_mem_rdy && head_flit_counter_f < 8'd3)
                        begin
                            head_flit_counter_next = head_flit_counter_f + 8'd1;
                        end
                        // Data flit counter and flits left to send
                        else if (test_mem_rdy && data_flit_counter_f < (data_payload_flits - 8'd1))
                        begin
                            data_flit_counter_next = data_flit_counter_f + 8'd1;
                        end
                        // Next state
                        else if (test_mem_rdy && data_flit_counter_f == (data_payload_flits - 8'd1))
                        begin
                            mem_req_state_next = MEM_STATE_STORE_RESP;
                            timeout_rst_n = 1'b0;
                            head_flit_counter_next = 8'd0;
                            data_flit_counter_next = 8'd0;
                        end
                    end
                    MEM_STATE_STORE_RESP:
                    begin
                        // Look for store ack
                        if (mem_test_val && mem_test_data == MEM_STORE_ACK_HEADER)
                        begin
                            mem_req_state_next = MEM_STATE_LOAD_REQ;
                            timeout_rst_n = 1'b0;
                            head_flit_counter_next = 8'd0;
                            data_flit_counter_next = 8'd0;
                        end
                        else if (mem_test_val)
                        begin
                            test_state_next = TEST_STATE_FAIL;
                        end
                    end
                    MEM_STATE_LOAD_REQ:
                    begin
                        // Send header flit
                        if (head_flit_counter_f == 8'd0)
                        begin
                            test_mem_val = 1'b1;
                            test_mem_data = MEM_LOAD_HEADER_FLIT;
                        end
                        // Send flit 2 (address flit)
                        else if (head_flit_counter_f == 8'd1)
                        begin
                            test_mem_val = 1'b1;
                            test_mem_data = {`NOC_DATA_WIDTH{1'b0}};
                            test_mem_data[`MSG_ADDR_] = test_address_f;
                            test_mem_data[`MSG_DATA_SIZE_] = addr_mode;
                        end
                        // Send flit 3 (source flit)
                        else if (head_flit_counter_f == 8'd2)
                        begin
                            test_mem_val = 1'b1;
                            test_mem_data = MEM_REQ_FLIT3;
                        end
                        
                        // Header flit counter
                        if (test_mem_rdy && head_flit_counter_f < 8'd2)
                        begin
                             head_flit_counter_next = head_flit_counter_f + 8'd1;
                        end
                        // Next state
                        else if (test_mem_rdy && head_flit_counter_f == 8'd2)
                        begin
                            mem_req_state_next = MEM_STATE_LOAD_RESP;
                            timeout_rst_n = 1'b0;
                            head_flit_counter_next = 8'd0;
                            data_flit_counter_next = 8'd0;
                        end
                    end
                    MEM_STATE_LOAD_RESP:
                    begin
                        // Look for load response header
                        if (head_flit_counter_f == 8'd0 && mem_test_val && 
`ifdef SDCTRL_TEST  
                            // SD always returns 8 flits, just replicates data
                            mem_test_data == 
                            (MEM_LOAD_ACK_HEADER | ({{`NOC_DATA_WIDTH-`MSG_LENGTH_WIDTH{1'b0}},`MSG_LENGTH_WIDTH'd8}<<`MSG_LENGTH_LO)))
`else // ifndef SDCTRL_TEST
                            mem_test_data == 
                            (MEM_LOAD_ACK_HEADER | ({{`NOC_DATA_WIDTH-`MSG_LENGTH_WIDTH{1'b0}},data_payload_flits}<<`MSG_LENGTH_LO)))
`endif // endif SDCTRL_TEST
                        begin
                            head_flit_counter_next = 8'd1;
                            data_flit_counter_next = 8'd0;
                        end
                        // Look for correct data or if we have already received all
                        // correct data
                        else if (head_flit_counter_f == 8'd1 && ((mem_test_val && test_data_matches) ||
`ifdef SDCTRL_TEST
                                  // SD always returns 8 flits, just
                                  // replicates data
                                  data_flit_counter_f == 8'd8))
`else // ifndef SDCTRL_TEST
                                  data_flit_counter_f == data_payload_flits))
`endif // endif SDCTRL_TEST
                        begin
                            // Check if this is the last flit or if we have received
                            // all flits already
`ifdef SDCTRL_TEST
                            // SD always returns 8 flits
                            if (data_flit_counter_f >= 8'd7)
`else // ifndef SDCTRL_TEST
                            if (data_flit_counter_f >= (data_payload_flits - 8'b1))
`endif // endif SDCTRL_TEST
                            begin
                                // Check if test case is done
                                if (gen_done)
                                begin
                                    test_state_next = TEST_STATE_PASS;
                                end
                                // Otherwise, get next address and data
                                else
                                begin
                                    // If we have valid next data, go back
                                    // to store request state
                                    if (gen_addrdata_next_val)
                                        mem_req_state_next = MEM_STATE_STORE_REQ;
                                    timeout_rst_n = 1'b0;
                                    gen_addrdata_next_rdy = 1'b1;
                                    test_address_next = gen_address_next;
                                    test_data_next = gen_data_next;

                                    // If we have valid data, reset counters for
                                    // next state
                                    if (gen_addrdata_next_val)
                                    begin
                                        head_flit_counter_next = 8'd0;
                                        data_flit_counter_next = 8'd0;
                                    end
                                    // Otherwise, increment counter if we have not
                                    // done so already to indicate we have
                                    // received all flits
`ifdef SDCTRL_TEST
                                    else if (data_flit_counter_f == 8'd7)
`else // ifndef SDCTRL_TEST
                                    else if (data_flit_counter_f == (data_payload_flits - 8'b1))
`endif // endif SDCTRL_TEST
                                    begin
                                        data_flit_counter_next = data_flit_counter_f + 8'd1;
                                    end
                                end
                            end
                            // If not increment data flit counter
                            else
                            begin
                                data_flit_counter_next = data_flit_counter_f + 8'd1;
                            end
                        end
                        else if (mem_test_val)
                        begin
                            test_state_next = TEST_STATE_FAIL;
                        end
                    end
                    default:
                    begin
                        test_state_next = TEST_STATE_RESET;
                        mem_req_state_next = MEM_STATE_STORE_REQ;
                        timeout_rst_n = 1'b1;
                        test_address_next = {`PHY_ADDR_WIDTH{1'bx}};
                        test_data_next = 512'bx;
                        head_flit_counter_next = 8'd0;
                        data_flit_counter_next = 8'd0;
                        test_mem_val = 1'b0;
                        test_mem_data = {`NOC_DATA_WIDTH{1'bx}};
                    end
                endcase
            end
        end
        TEST_STATE_PASS: ;
        TEST_STATE_FAIL: ;
        default:
        begin
            test_state_next = TEST_STATE_RESET;
            mem_req_state_next = MEM_STATE_STORE_REQ;
            timeout_rst_n = 1'b1;
            test_address_next = {`PHY_ADDR_WIDTH{1'bx}};
            test_data_next = 512'bx;
            head_flit_counter_next = 8'd0;
            data_flit_counter_next = 8'd0;
            test_mem_val = 1'b0;
            test_mem_data = {`NOC_DATA_WIDTH{1'bx}};
        end
    endcase
end

//////////////////////////
// Sub-module Instances //
//////////////////////////

// Timeout counter
alarm_counter #(
    .COUNTER_WIDTH(TIMEOUT_COUNTER_WIDTH),
    .ALARM_THRESHOLD(TIMEOUT_COUNTER_THRESHOLD)
)timeout_counter(
    .clk(clk),
    .rst_n(timeout_rst_n),
    .en(test_running),
    .trigger(timeout_trigger)
);

walk_ones_addrdata walk_ones_addrdata(
    .en(test_mode_en[0]),

    .addr_incr_val(addr_incr_val),
    .mem_top_addr(mem_top_addr),
    
    .test_state_reset(test_state_reset),
    
    .curr_addr(test_address_f),
    .curr_data(test_data_f),

    .next_addr(walk_ones_address_next),
    .next_data(walk_ones_data_next),

    .done(walk_ones_done)
);

address_own_addrdata address_own_addrdata(
    .en(test_mode_en[1]),

    .addr_incr_val(addr_incr_val),
    .mem_top_addr(mem_top_addr),

    .test_state_reset(test_state_reset),

    .curr_addr(test_address_f),
    .curr_data(test_data_f),

    .next_addr(address_own_address_next),
    .next_data(address_own_data_next),

    .done(address_own_done)
);

random_addrdata random_addrdata(
    .clk(clk),
    .rst_n(rst_n),
    .en(test_mode_en[2]),

    .addr_incr_val(addr_incr_val),
    .mem_top_addr(mem_top_addr),

    .test_state_reset(test_state_reset),

    .next_rdy(gen_addrdata_next_rdy),
    .next_val(random_addrdata_next_val),
    .next_addr(random_address_next),
    .next_data(random_data_next),

    .done(random_done)
);

endmodule
