function [thresh] = mask_threshold(type, j, P)
% [masking_threshold] = mask_threshold(type, location_of_masker_bin,
%                                     power_spectral_density_at_j)
%
% mask_threshold returns the auditory mask (in dB SPL) that
% results due to the input masker.
% The mask is padded with zeros to get the appropriate length.
% location_of_masker_bin here is the index used to access the vector
% of frequencies.
%
% The user should also supply the power spectral density of the masker
% so that all calculations can be made. Note also that there are two
% different threshold types:
%
%  type = 0      threshold = NOISE threshold
%  type = 1      threshold = TONE  threshold


FFTlength=512;
fs=44100;
f=[1:FFTlength/2]*(fs/FFTlength);
ATH=ath(f);
b=hz2bark(f);
% determine where masker is in barks
maskerloc=b(j);

% set up range of the resulting function
% in barks
low=maskerloc-3;
high=maskerloc+8;
% in discrete bins
lowbin=max(find(b<low));
if (isempty(lowbin))
   lowbin = 1;
end
highbin=max(find(b<high));

% calculate spreading function
SF = spreading_function(j, P, lowbin, highbin, b);

% which type are we dealing with?  The equations are slightly different.
if (type==0)
   % calculate noise threshold
   threshold=P-.175*b(j)+SF-2.025;
else
   % calculate tone threshold
   threshold=P-.275*b(j)+SF-6.025;
end

% finally, note that the lowest value in threshold corresponds
% to the frequency bin at lowbin.
start=lowbin;
thresh=-Inf*ones(1,256);
thresh(start:start+length(threshold)-1)=threshold;

% -----------------------------------------------------------------
function spread = spreading_function(masker_bin, power, low, high, b)
% spreading_around_masker = spreading_function(masker_bin, masker_power,
%                                              low_bin, high_bin, bark_spectrum)
%
% spreading_function takes a masker_bin and related power to determine
% the spreading function that surrounds it.  This function indicates
% the minimum level that nearby frequencies must attain before they
% are detectable by the human ear.  Along with the power in the 
% masker and other data, this can be used to determine whether a 
% signal is audible over a masker.

masker_bark=b(masker_bin);

for i=low:high,
   maskee_bark=b(i);
   deltaz=maskee_bark-masker_bark;  % difference in Barks
   
   if ((deltaz>=-3.5) && (deltaz<-1))
      spread(i-low+1)=17*deltaz-0.4*power+11;
   elseif ((deltaz>=-1) && (deltaz<0))
      spread(i-low+1)=(0.4*power+6)*deltaz;
   elseif ((deltaz>=0) && (deltaz<1))
      spread(i-low+1)=-17*deltaz;
   elseif ((deltaz>=1) && (deltaz<8.5))
      spread(i-low+1)=(0.15*power-17)*deltaz-0.15*power;
   end
   
   % Alternatively can use this more agressive mask from Camastra et al.
%    spread(i-low+1)=15.81+7.5*(deltaz+0.474)-17.5*sqrt(1+(deltaz+0.474).^2);
   
%    spread(i-low+1)=15.8111389+7.5*(1.05*deltaz+0.474)-17.5*sqrt(1+1.05*(deltaz+0.474).^2)+8*min(0,(1.05*deltaz-0.5).^2-2*(1.05*deltaz-0.5));
end
