% example_lrcov.m
% This is an example of the "long-run covariability" 
% This version: 12/29/2017, mww
% 
clear all;
small = 1.0e-10;

% -- File Directories   
parmdir = 'lrcov_parms_dir/';        % Location of various pre-computed parameters and covariance matrices (these are supplied with the replication files)
progdir = 'matlab_lrcov_programs/';  % Location of various utility programs needed for LRCOV  (these are supplied with the replication files)

% -- Load Example data (the data are supplied with replication files)
% ---------- THIS IS THE GDP-CONSUMPTION DATASET ------------------
load yc_example_data;    % This is dataset supplied with replication file
                         % First column is growth of GDP per capita
                         % Second colum is growth of Consumption per capita
                         % Both deflated by PCE price index
                         % Sample period is 1948:Q2 through 2015:Q4.
                         % Note there are no missing values
x = yc_example_data(:,1);
y = yc_example_data(:,2);
% ---------- THIS IS THE INTEREST RATE DATASET ------------------
% load irate_example_data;    % This is dataset supplied with replication file
%                             % First column is the 3-month TBill rate
%                             % Second colum is 10Y TBond Rate
%                             % Sample period is 1953:Q2 through 2015:Q4.
%                             % Note there are no missing values
% x = irate_example_data(:,1);
% y = irate_example_data(:,2);

% Parameters for LRCOV function
% ---------------------------------
lowest_period_years = 11;  % This is low-period cut-off .. here in years
nper = 4;                  % Number of periods per year .. these are quarterly data

T_obs = size(y,1);
lowest_period = nper*lowest_period_years;
q = floor(2*T_obs/lowest_period);
lowest_period_years_q = 2*T_obs/(nper*q);
highest_period_years_q = 2*T_obs/(nper);

% Other Parameters
i_corr = 1;   % 1: compute correlation
i_bet = 1;    % 1: compute regression coefficient of y on x (beta)
i_sig = 1;    % 1: compute sigma_y|x (standard deivation of regression error)

% ------------------ Set Path, etc. ------------------------
p = path; % Change path so that Matlab accesses utility programs
path(progdir,p);

% --------------------------- Print Introductory Results -----------------------------
fprintf('\n\n');
fprintf('Results from LR covariability Analysis \n');
fprintf('   Data Set: example data \n');
fprintf('   Number of observations: %4i \n',T_obs);
fprintf('   Lowest Period in years: %3.1f \n',lowest_period_years_q);
fprintf('   Highest Period in years: %3.1f \n',highest_period_years_q);
fprintf('   q value: %2i \n\n',q);
fprintf('\n\n');
if q > 12;
 fprintf('You will need to files for q = %2i \n',q);
 fprintf('  Make sure you have downloaded these files and they are in parmdir folder listed in this program near line 9 \n');
 fprintf('\n\n');
end;
% ------------------ Carry out the analysis ------------------------
[corr,bet,sig,misc] = lrcov(x,y,q,i_corr,i_bet,i_sig,parmdir);

path(p);  % Restore original path


% --------------------------- Print Results -----------------------------
if i_corr == 1;
 fprintf('   Long-run correlation \n');
 fprintf('     Posterior Median: %5.3f \n',corr.posterior_median);
 for j = 1:length(corr.level);
  fprintf('     Coverage Probability: %5.2f \n',corr.level(j));
  fprintf('        Confidence Interval %5.3f to %5.3f \n',[corr.conf_interval(j,:)]);
 end;
 fprintf('  \n');
 fprintf('     Note: Additional features of posterior \n');
 fprintf('      Posterior Mean: %5.3f \n',corr.posterior_mean);
 for j = 1:length(corr.level);
  fprintf('      Coverage Probability: %5.2f \n',corr.level(j));
  fprintf('        Equal-tail credible interval %5.3f to %5.3f \n',[corr.posterior_cs(j,:)]);
 end;
 fprintf('\n\n');
end;

if i_bet == 1;
 fprintf('   Long-run regression coefficient \n');
 fprintf('     Posterior Median: %5.3f \n',bet.posterior_median);
 for j = 1:length(bet.level);
  fprintf('     Coverage Probability: %5.2f \n',bet.level(j));
  fprintf('        Confidence Interval %5.3f to %5.3f \n',[bet.conf_interval(j,:)]);
 end;
 fprintf('  \n');
 fprintf('     Note: Additional features of posterior \n');
 fprintf('      Posterior Mean: %5.3f \n',bet.posterior_mean);
 for j = 1:length(bet.level);
  fprintf('      Coverage Probability: %5.2f \n',bet.level(j));
  fprintf('        Equal-tail credible interval %5.3f to %5.3f \n',[bet.posterior_cs(j,:)]);
 end;
 fprintf('\n\n');
end;

if i_sig == 1;
 fprintf('   Long-run regression standard error \n');
 fprintf('     Posterior Median: %5.3f \n',sig.posterior_median);
 for j = 1:length(sig.level);
  fprintf('     Coverage Probability: %5.2f \n',sig.level(j));
  fprintf('        Confidence Interval %5.3f to %5.3f \n',[sig.conf_interval(j,:)]);
 end;
 fprintf('  \n');
 fprintf('     Note: Additional features of posterior \n');
 fprintf('      Posterior Mean: %5.3f \n',sig.posterior_mean);
 for j = 1:length(sig.level);
  fprintf('      Coverage Probability: %5.2f \n',sig.level(j));
  fprintf('        Equal-tail credible interval %5.3f to %5.3f \n',[sig.posterior_cs(j,:)]);
 end;
 fprintf('\n\n');
end;
