/* proc (6)=resolkw(systm,driver,track,comp); */

/* Rational expectations solution program implementing 
the various steps described in R.G.King and M.W.Watson
"System Reduction and Model Solution Algorithms for
Linear Rational Expectations Models." The inputs and 
outputs are summarized by the function statement:

	 [M,PI,ylist,xlist,lpd]=resolkw(systm,driver,track,comp)

There are two required inputs to the program:

  (1) The file 'systm.gss' must describe the 
		linear rational expectations model to be solved.
		(the input file need not be called 'systm.gss', but
		 can take any name that the user supplies in the call
		 to the resolkw.prc program).  The RE model takes the form:

		  A Ey(t+1)|I(t) = B y(t) + C0 x(t) + ... Cn Ex(t+n)|I(t)

		systm.gss must contain 
		  (a) the matrices A,B,CF=[C0 ... Cn]
		  (b) a vector of predetermined variable locations (lpd)
		  (c) the number of exogenous variables, 
				 which implies the number of leads (nlead=cols(CF)/nx)
		  (d) lists of endogenous (ylist) and exogenous (xlist) variables

  (2) The file 'driver.gss' must contain the matrices RHO and Q
		that describe the driving processes for the system:
		  x(t) = Q delta(t)
		  delta(t) = RHO * delta(t-1) + G e(t)
		The matrix G of this driving process is not essential for
		the current solution.

  (3) The user may optionally decide not to track the solution of the
		model (except for forced error messages). This involves setting 
		track=0 in the call to redkw.prc

  (4) The user may optionally decide not to use the Schur decomposition
		method.  This involves supply a two element vector that involves
		[schur nobalance].  The parameter schur = 0 involves computation via
		the eigenvector-eigenvalue method; the parameter nobalance = 1
		involves the computation of the eigenvalues by MATLAB's no balance
		option, which leads to more accurate computations in some situations.
	Note:  at this point, GAUSS code versions DO NOT support this last
	option */

new;
output file=control.out reset; output on;
library pgraph,nlsys;



/* **********************
		START USER INPUTS       
************************ */
#include parms.gss;
#include ss1.gss;



@systm="ss_rb4a";@   @ kw @
systm="linold1";     @ aw @

@driver="dr_rb4a";@  @ kw @
driver="drv";        @ aw @

@#include ss_rb4a.gss;@    @ kw @
@#include dr_rb4a.gss;@   @ kw @

#include linold1.gss;  @ aw @
#include drv.gss;  @ aw @

track=0;
comp=zeros(2,1);
toli=10^(-10);
@ tolerances? @

@ could put "important switches" or some such thing here @

/* **********************
		END USER INPUTS       
************************ */

/* --------------------------------------------------------------------------
Select details of computational method: 
	standard eigenvalue (schuryn=0, nb=0 or not included in mdrkw call)
	unbalanced eigenvalue (schuryn=0, nb=1)
	schur (schuryn=1)
-------------------------------------------------------------------------- */
schuryn=comp[1,1];
nb=comp[2,1];

if track==1;
	@dr="c:\\steve\\"$+systm$+".out";@   @ kw @
	dr="c:\\mydocs\\download\\gkwre\\"$+systm$+".out";
	print "Tracking option on";
	print "output stored in "$+dr; 
	output file=^dr reset;   
endif;

/* **************************************************************************
Step 0: Settings for computational parameters and methods
************************************************************************** */

/* --------------------------------------------------------------------------
Choose critical value for roots: mu is unstable if bcrit*mu >= 1
-------------------------------------------------------------------------- */

bcrit=1/1.00001;


/* **************************************************************************
Step 1: Specifying dynamic system                                           
************************************************************************** */

/* Run the file 'systm.gss', which contains the setup of the dynamic system */
dr=systm$+".gss";
/* run ^dr; */

/* **************************************************************************
Optional information on setup specified model                        
************************************************************************** */

if track==1;
	print "A matrix";
	print A;
	print "B matrix";
	print B;
	print "transformed C(F) matrix";
	print CF;
	nlead=cols(CF)/nx;
	print "Number of leads: " nlead;
	print "Predetermined endogenous variables: ";
	print $ylist[lpd,.];
	print "Full vector of endogenous variables";
	print $ylist; 
	print "Exogenous variables";
	print $xlist;
	print "  ";
@    saveall ^systm; @
	dr=systm$+".gcg";
	print "Original system information saved in " $dr;
	print "";
	print "running system reduction program (redkw.prc)";  
endif;

@ ensure lpd is a row vector, to match Matlab @
if rows(lpd)>cols(lpd);
	lpd=lpd';
endif;

/* **************************************************************************
Step 2: Call to reduction program                                                
************************************************************************** */

#include redkw.prc;
#include csr.prc;
#include mconv.prc;
{ nd,nf,AN,BN,reord,CFN }=redkw(A,B,CF,nx,lpd,track,&mconv,&csr);

/* **************************************************************************
Optional information on output of reduction program                      *  
************************************************************************** */

if track==1;
	print "transformed A matrix";
	print AN;
	print "transformed B matrix";
	print BN;
	print "Reordering of variables";
	print reord;
	print "transformed C(F) matrix";
	print CFN;
	print "flow variables";
	print $ylist[reord[1,1:nf],.];
	print "dynamic variables";
	print $ylist[reord[1,nf+1:nf+nd],.];
	print "";
	print "reordering vector";
	 print "  ";
	dr="r"$+systm;
@    saveall ^dr; @
	dr=dr$+".gcg";
	print "Reduced system information saved in " $dr;
	print "";
	print "running reduced system transformation program (dynkw.prc)"; 
endif;

/* **************************************************************************
Step 3: Translation matrices for fundamental dynamic equation
************************************************************************** */

#include dynkw.prc;
#include eigl.prc;
#include schurl.prc;
#include tqswitch.prc;
#include givens.prc;
#include rowrot.prc;
#include colrot.prc;

{ LER,MU }=dynkw(BN,nf,ny,lpd,bcrit,track,schuryn,nb);

if track==1;
	print "reduced system translation completed";
	print "";
	print "unstable eigenvalues are: ";
	print diag(MU);
	print "LE matrix that isolates unstable canonical variables is:";
	print LER;
	dr="d"$+systm;
@    saveall ^dr;  @
	dr=dr$+".gcg";
	print "Transformed system information saved in " $dr;
	print "";
	print "running Markov decision rule program (mdrkw.prc)"; 
endif;


/* **************************************************************************
Step 4: Solution for Markov decision rule                                        
************************************************************************** */

 

/* --------------------------------------------------------------------------
  1. Specification of forcing process                 
-------------------------------------------------------------------------- */

dr=driver$+".gss";
/* run ^dr; */

/* --------------------------------------------------------------------------
  2. Call solution program and get output                                
-------------------------------------------------------------------------- */

#include mdrkw.prc;

{ M,PII }=mdrkw(BN,CFN,Q,RHO,LER,MU,reord,nd,lpd,nx,bcrit,schuryn,toli);

/* --------------------------------------------------------------------------
  Optional information on output of Markov decision rules                 
-------------------------------------------------------------------------- */

if track==1;
  print "Markov decision rules computed";
  print "";
  print $ylist[1:ny,.];
  print " ";
  print "PI matrix: y(t) elements";
  print "";
  print PII[1:ny,.];
  print $xlist[1:nx,.];
  print "";
  print "PI matrix: x(t) elements";
  print "";
  print PII[ny+1:ny+nx,.];
  print "elements of k(t)";
  print $ylist[lpd,.];
  print "";
  print "M matrix:  elements ordered k(t), then delta(t)";
  print "";
  print M;
  dr="m"$+systm;
@   saveall ^dr; @
	dr=dr$+".gcg";
  print "Decision rule information saved in " $dr;
  print "";
  print "model solution completed by resolkw.prc";
endif;

output off;

/* list=ylist;
Q=eye(3);
nss=rows(Q);
sm=rows(M);
nda=sm-nss;
Vii=zeros(nda,sm)|(zeros(nss,nda)~Q);
#include fdfkw.prc;
#include facvskw.prc;
end; */

/* retp(M,PI,ylist,xlist,lpd);
endp; */