% Construct AR forecasts for the state employment data
clear all;
small = 1.0e-8;
big = 1.0e+8;
this_date = datestr(now,'yyyymmdd');

% -- File Directories  
datadir='../Data/'; 
outdir = 'out/';
figdir = 'fig/';
matdir = 'mat/';

% Add paths for Matlab Utility Functions
addpath('../Matlab_Utility_Programs/');

% Parameters for the exercise
n_arlags = 12;  % Number of AR lags
i_constant = 1; % Include a constant term
first_fcst = [1999 12]; % First forecast period
n_h = 12; % Number of forecast periods

% Read Employment Data
SData = readtable([datadir 'StateEmployment.xlsx']);
% Get the state names
states = SData.Properties.VariableNames(3:end);
% Get the data
nper = 12; 
[dnobs,calvec,calds] = calendar_make([SData.Year(1) SData.Month(1)],[SData.Year(end) SData.Month(end)],nper);
% Number of states
nstates = length(states);
dnobs = length(calvec);

% Matrices for saving the results
rslt_fcsts = NaN(n_h,nstates,dnobs);      % Forecasted values
rslt_actual = NaN(n_h,nstates,dnobs);     % Actual values
sd_fcsts = NaN(n_h,nstates,dnobs);        % SD of forecast
sd_fcsts_accum = NaN(n_h,nstates,dnobs);  % STD of forecast accumulated values

% State Employment Data
lev_data = SData{:,3:end};
lfd_data = dif(log(lev_data),1);

% Find first forecast period .. find first_fcst in calds
t_first = find(calds(:,1) == first_fcst(1) & calds(:,2) == first_fcst(2));

for i_state = 1:nstates
    for t = t_first:dnobs
        % Construct the forecast for period t
        tmp_data = lfd_data(1:t,i_state);
        [arcoef,constant,ser] = uar_c(tmp_data,n_arlags,i_constant);
        % Convert to state space form
        [s,phi,theta] = ar_withconstant_to_ss(arcoef,constant,ser);
        n_state = size(phi,1);
        % Forecast -- levels
        % State Vector
        x = zeros(n_state,1);
        x(end) = 1;  % Constant term
        x(1:n_arlags) = flipud(lfd_data(t-n_arlags+1:t,i_state));
        v = zeros(n_state,n_state);
        for i_h = 1:n_h
            x = phi*x;
            rslt_fcsts(i_h,i_state,t) = s*x;
            if (t+i_h <= dnobs)
                rslt_actual(i_h,i_state,t) = lfd_data(t+i_h,i_state);
            end
        end
        % Comput Covariance Matrix of forecast errors
        P_h = zeros(n_state,n_state);
        Sigma_Fcst_Errors = SS_Model_Fcst_Error_Covmatrix(s,phi,theta,n_h,P_h);
        sd_fcsts(:,i_state,t) = sqrt(diag(Sigma_Fcst_Errors));
        % Accumulated values
        A = tril(ones(n_h,n_h));
        sd_fcsts_accum(:,i_state,t) = sqrt(diag(A*Sigma_Fcst_Errors*A'));
    end
end
% Save the results
% Variables to save
var_save = {'rslt_fcsts','rslt_actual','sd_fcsts','sd_fcsts_accum','calvec','calds','dnobs','states'};
save([matdir 'ar_fcst_state_emp.mat'],var_save{:});
