function [y_interp] = interpolate_annual_to_monthly(x)
% x = annual observations (no missing values), assumed to be averages of
% monthly values. Monthly values follow a random walk with drift, with and
% mean and variance estimated from the x process. Output is y_interp, which
% is the monthly estimates.

  % Annual observations: X
  dx = x(2:end)-x(1:end-1);
  mdx = mean(dx);
  sdx = std(dx);
  vdx = sdx^2;
  % Estimate mean and variance of Monthly Observations Y
  mdy = 12*mdx;
  % 24 monthly obs RW obs
  A = tril(ones(24,24));
  V = A*A';
  B = ones(24,1);
  B(1:12) = -B(1:12);
  B = B/12;
  VA = B'*V*B;
  vdy = vdx/VA;

  % Carry out interpolation
  nyears = size(x,1);
  nmonths = 12*nyears;
  Mean_y = mdy*ones(nmonths,1);
  A = tril(ones(nmonths,nmonths));
  Sigma_yy = vdy*(A*A' + 10000*ones(nmonths,nmonths));

  % Mapping from y to x
  M = zeros(nyears,nmonths);
  for i = 1:nyears;
     jj = (i-1)*12;
     M(i,jj+1:jj+12) = ones(1,12)/12;
  end;

  Mean_x = M*Mean_y;
  Sigma_xx = M*Sigma_yy*M';
  Sigma_yx = Sigma_yy*M';

  y_interp = Mean_y + Sigma_yx*inv(Sigma_xx)*(x-Mean_x);

end