clear all;

% Monte Carlo simulations for "Consistent Factor Estimation..." (Bates et al., 2012)
% MPM 2012-06-14, 2012-11-12

numrep = 5000; %Number of simulations
Ts = [50 100 200]; %50:50:400; %Array of T values
ms = [0.5 1 2]; %Array of m values: N = m*T
ds = [0 0.4 0.7]; %Array of d values for white noise: h_NT = d
cs = [2 3.5]; %Array of c values for RW: h_NT = c/T^(3/4)
bs = [3.5 7]; %Array of b values for large break: #breaks = b*sqrt(N)
r = 5; %Number of factors

rho = 0.9; %Common AR(1) coefficient for factors
a = 0.5; %AR(1) coefficient for idiosyncratic errors
beta = 0.5; %Cross-sectional correlation coefficient for idiosyncratic errors
R2_dom = [0 0.8]; %Interval on which the R^2 for factors is uniformly distributed
max_rank = 10; %Maximum rank for estimation (=0 if rank estimation isn't desired)
tau_bar = 0.5; %Fraction of sample at which break occurs in large break model

filename = ''; %Filename for .mat file (equal to '' if results shouldn't be saved)

tic; %Start timer

%Define arrays with simulation data
R2s_wn = zeros(numrep,length(Ts),length(ms),2,length(ds),2);
S2s_wn = R2s_wn;
ranks_wn = zeros(numrep,length(Ts),length(ms),2,length(ds));
R2s_rw = zeros(numrep,length(Ts),length(ms),2,length(cs),2);
S2s_rw = R2s_rw;
ranks_rw = zeros(numrep,length(Ts),length(ms),2,length(cs));
R2s_br = zeros(numrep,length(Ts),length(ms),2,length(bs),2);
S2s_br = R2s_br;
ranks_br = zeros(numrep,length(Ts),length(ms),2,length(bs));

for n = 1:numrep; %For each repetition...
    for nT = 1:length(Ts); %For each T...
        for nm = 1:length(ms); %For each N...
            T = Ts(nT); %T
            N = floor(T*ms(nm)); %N
            
            %Calculate the matrix square root of cov. matrix Omega = (beta^{|i-j|})_{i,j}
            circul = reshape(mod(0:N^2-1,N+1),N,N);
            uind = (triu(ones(N))==1);
            circul_tp = circul';
            circul(uind) = circul_tp(uind); %Equal to matrix (|i-j|)_{i,j}
            Omega = beta.^circul; %Define Omega
            [OV,OD]=eig(Omega);
            sqrtOmega = OV*sqrt(OD); %Matrix square root: sqrtOmega*sqrtOmega'=Omega
            
            for nk = 1:min(1+max_rank,2); %nk=1 imposes true rank, nk=2 estimates rank (if desired)
                max_rank_estim = (nk-1)*max_rank; %=0 if true rank is imposed
                
                for nd = 1:length(ds); %For each value of d...
                    [F,y,~,e,Lambda_0] = genvars(T,N,r,rho,a,sqrtOmega,R2_dom); %Generate variables
                    
                    d = ds(nd);
                    h = d; %h_NT
                    %h = d*T^(1/4); %Alternative rate
                    xi = randn(T,N*r);
                    Lambda = repmat(Lambda_0(:)',T,1)+h*xi; %Array of lambda_t values

                    [R2s,S2s,khat] = summ_stats(F,Lambda_0,Lambda,y,e,max_rank_estim); %Calculate summ. stats
                    %Store
                    R2s_wn(n,nT,nm,nk,nd,:) = R2s;
                    S2s_wn(n,nT,nm,nk,nd,:) = S2s;
                    ranks_wn(n,nT,nm,nk,nd) = khat;
                end;

                for nc = 1:length(cs); %For each value of c...
                    [F,y,~,e,Lambda_0] = genvars(T,N,r,rho,a,sqrtOmega,R2_dom); %Generate variables

                    c = cs(nc);
                    h = c/T^(3/4); %h_NT
                    %h = c/T^(1/2); %Alternative rate
                    zeta = randn(T,N*r);
                    xi = filter(1,[1 -1],zeta); %Random walk
                    Lambda = repmat(Lambda_0(:)',T,1)+h*xi; %Array of lambda_t values

                    [R2s,S2s,khat] = summ_stats(F,Lambda_0,Lambda,y,e,max_rank_estim); %Calculate summ. stats
                    %Store
                    R2s_rw(n,nT,nm,nk,nc,:) = R2s;
                    S2s_rw(n,nT,nm,nk,nc,:) = S2s;
                    ranks_rw(n,nT,nm,nk,nc) = khat;
                end;

                for nb = 1:length(bs); %For each value of b...
                    [F,y,~,e,Lambda_0,lambda_star] = genvars(T,N,r,rho,a,sqrtOmega,R2_dom); %Generate variables

                    numbreak = floor(min(bs(nb)*sqrt(N),N)); %Number of breaking series
                    %numbreak = floor(min(bs(nb)*N,N)); %Alternative rate
                    breakinds = randsample(N,numbreak); %Draw break indices uniformly at random
                    breaktime = floor(tau_bar*T); %Time at which break occurs
                    
                    %Break parameter delta
                    Delta = zeros(N,r);
                    Delta(breakinds,:) = lambda_star(mean(R2_dom))*repmat(randn(1,r),numbreak,1);
                    
                    Lambda = [repmat(Lambda_0(:)',breaktime,1);
                              repmat(Lambda_0(:)'+Delta(:)',T-breaktime,1)]; %Array of lambda_t values

                    [R2s,S2s,khat] = summ_stats(F,Lambda_0,Lambda,y,e,max_rank_estim); %Calculate summ. stats
                    %Store
                    R2s_br(n,nT,nm,nk,nb,:) = R2s;
                    S2s_br(n,nT,nm,nk,nb,:) = S2s;
                    ranks_br(n,nT,nm,nk,nb) = khat;
                end;
            end;
        end;
    end;
    
    %Print progress
    if mod(n,ceil(numrep/100))==0;
        fprintf('%3d ',100*n/numrep);
    end;
    if mod(n,ceil(numrep/10))==0;
        fprintf('\n');
    end;
end;
tstop=toc; %Stop timer
disp('Elapsed time');
disp(tstop);

%Calculate trace R-square and S-square statistics for each model
%White noise
R2rat_wn = mean(R2s_wn(:,:,:,:,:,1),1)./mean(R2s_wn(:,:,:,:,:,2),1);
S2rat_wn = 1-mean(S2s_wn(:,:,:,:,:,1),1)./mean(S2s_wn(:,:,:,:,:,2),1);
avgrank_wn = mean(ranks_wn(:,:,:,:,:),1);
pctrank_wn = mean(ranks_wn(:,:,:,:,:)==r,1);

%Random walk
R2rat_rw = mean(R2s_rw(:,:,:,:,:,1),1)./mean(R2s_rw(:,:,:,:,:,2),1);
S2rat_rw = 1-mean(S2s_rw(:,:,:,:,:,1),1)./mean(S2s_rw(:,:,:,:,:,2),1);
avgrank_rw = mean(ranks_rw(:,:,:,:,:),1);
pctrank_rw = mean(ranks_rw(:,:,:,:,:)==r,1);

%Large break
R2rat_br = mean(R2s_br(:,:,:,:,:,1),1)./mean(R2s_br(:,:,:,:,:,2),1);
S2rat_br = 1-mean(S2s_br(:,:,:,:,:,1),1)./mean(S2s_br(:,:,:,:,:,2),1);
avgrank_br = mean(ranks_br(:,:,:,:,:),1);
pctrank_br = mean(ranks_br(:,:,:,:,:)==r,1);

%Reshape arrays
R2rat_wn = reshape(R2rat_wn,length(Ts),length(ms),2,length(ds));
S2rat_wn = reshape(S2rat_wn,length(Ts),length(ms),2,length(ds));
avgrank_wn = reshape(avgrank_wn,length(Ts),length(ms),2,length(ds));
pctrank_wn = reshape(pctrank_wn,length(Ts),length(ms),2,length(ds));
R2rat_rw = reshape(R2rat_rw,length(Ts),length(ms),2,length(cs));
S2rat_rw = reshape(S2rat_rw,length(Ts),length(ms),2,length(cs));
avgrank_rw = reshape(avgrank_rw,length(Ts),length(ms),2,length(cs));
pctrank_rw = reshape(pctrank_rw,length(Ts),length(ms),2,length(cs));
R2rat_br = reshape(R2rat_br,length(Ts),length(ms),2,length(bs));
S2rat_br = reshape(S2rat_br,length(Ts),length(ms),2,length(bs));
avgrank_br = reshape(avgrank_br,length(Ts),length(ms),2,length(bs));
pctrank_br = reshape(pctrank_br,length(Ts),length(ms),2,length(bs));

if ~strcmp(filename,''); %If results are to be saved...
    save([filename '.mat']);
end;